"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyProjectDefinition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_aws_pdk_prelude_1 = require("./components/smithy-aws-pdk-prelude");
const smithy_build_gradle_file_1 = require("./components/smithy-build-gradle-file");
const smithy_settings_gradle_file_1 = require("./components/smithy-settings-gradle-file");
const version_1 = require("./version");
const utils_1 = require("../../codegen/components/utils");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyProjectDefinition extends projen_1.Component {
    constructor(project, options) {
        super(project);
        const { smithyOptions } = options;
        // Ignore gradle wrapper by default
        if (smithyOptions.ignoreGradleWrapper ?? true) {
            project.gitignore.addPatterns("gradle");
            project.gitignore.addPatterns("gradlew");
            project.gitignore.addPatterns("gradlew.bat");
        }
        // Always ignore the .gradle dir which the wrapper downloads gradle into
        project.gitignore.addPatterns(".gradle");
        this.gradleProjectName = project.name.replace(/[\/\\:<>"?\*|]/g, "-");
        // Add settings.gradle
        new smithy_settings_gradle_file_1.SmithySettingsGradleFile(project, {
            gradleProjectName: this.gradleProjectName,
        });
        const modelDir = "src/main/smithy";
        this.modelDir = modelDir;
        // Always add the following required dependencies
        const requiredDependencies = [
            "software.amazon.smithy:smithy-cli",
            "software.amazon.smithy:smithy-model",
            "software.amazon.smithy:smithy-openapi",
            "software.amazon.smithy:smithy-aws-traits",
        ];
        // Ensure dependencies always include the required dependencies, allowing users to customise the version
        const userSpecifiedDependencies = smithyOptions.smithyBuildOptions?.maven?.dependencies ?? [];
        const userSpecifiedDependencySet = new Set(userSpecifiedDependencies.map((dep) => dep.split(":").slice(0, -1).join(":")));
        const dependencies = [
            ...requiredDependencies
                .filter((requiredDep) => !userSpecifiedDependencySet.has(requiredDep))
                .map((dep) => `${dep}:${version_1.DEFAULT_SMITHY_VERSION}`),
            ...userSpecifiedDependencies,
        ];
        // Add build.gradle
        this.smithyBuildGradleFile = new smithy_build_gradle_file_1.SmithyBuildGradleFile(project, {
            modelDir,
            dependencies,
            repositoryUrls: smithyOptions.smithyBuildOptions?.maven?.repositoryUrls,
        });
        const { namespace: serviceNamespace, serviceName } = smithyOptions.serviceName;
        // Create the smithy build json file
        this.smithyBuild = new smithy_build_1.SmithyBuild(project, {
            version: "2.0",
            ...smithyOptions.smithyBuildOptions,
            sources: [
                modelDir,
                ...this.asRelativePathsToProject(smithyOptions.smithyBuildOptions?.additionalSources ?? []),
            ],
            projections: {
                ...smithyOptions.smithyBuildOptions?.projections,
                openapi: {
                    ...smithyOptions.smithyBuildOptions?.projections?.openapi,
                    plugins: {
                        openapi: {
                            service: `${serviceNamespace}#${serviceName}`,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            // By default, use integer types as this is more intuitive when smithy distinguishes between Integers and Doubles.
                            // Users may also override this.
                            useIntegerType: true,
                            ...smithyOptions.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
            maven: {
                // Filter out any file dependencies since these aren't supported in smithy-build.json
                dependencies: dependencies.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)),
                repositories: (smithyOptions.smithyBuildOptions?.maven?.repositoryUrls ?? [
                    "https://repo.maven.apache.org/maven2/",
                    "file://~/.m2/repository",
                ]).map((url) => ({ url })),
            },
        });
        // Add the smithy prelude (managed by aws-pdk)
        this.generatedModelDir = path.join("generated", "main", "smithy");
        new smithy_aws_pdk_prelude_1.SmithyAwsPdkPrelude(project, {
            generatedModelDir: this.generatedModelDir,
            serviceNamespace,
            handlerLanguages: options.handlerLanguages,
        });
        this.addSources(this.generatedModelDir);
        const projectionOutputPath = path.join("build", "smithyprojections", this.gradleProjectName, "openapi");
        this.openApiSpecificationPath = path.join(projectionOutputPath, "openapi", `${serviceName}.openapi.json`);
        this.smithyJsonModelPath = path.join(projectionOutputPath, "model", "model.json");
        // Copy the gradle files during build if they don't exist. We don't overwrite to allow users to BYO gradle wrapper
        // and set `ignoreGradleWrapper: false`.
        project.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.COPY_GRADLE_WRAPPER));
        // Build with gradle to generate smithy projections, and any other tasks
        project.generateTask.exec("./gradlew build");
        if (smithyOptions.ignoreSmithyBuildOutput ?? true) {
            // Ignore the build directory, and smithy-output which was the old build directory for the cli-based generation
            project.gitignore.addPatterns("build", "smithy-output");
        }
    }
    /**
     * Add maven-style or local file dependencies to the smithy model project
     * @param deps dependencies to add, eg "software.amazon.smithy:smithy-validation-model:1.27.2" or "file://../some/path/build/lib/my-shapes.jar
     */
    addDeps(...deps) {
        this.smithyBuildGradleFile.addDeps(...deps);
        this.smithyBuild.addMavenDependencies(...deps.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)));
    }
    /**
     * Add dependencies on other smithy models, such that their shapes can be imported in this project
     * @param deps smithy definitions to depend on
     */
    addSmithyDeps(...deps) {
        this.addDeps(...deps.map((dep) => `${smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix}${path.join(path.relative(this.project.outdir, dep.project.outdir), "build", "libs", `${dep.gradleProjectName}.jar`)}`));
    }
    /**
     * Add additional paths to model source files or directories.
     * Paths should be relative to the project outdir. Any absolute paths will be
     * resolved as relative paths.
     */
    addSources(...sources) {
        const relativeSources = this.asRelativePathsToProject(sources);
        this.smithyBuild.addSources(...relativeSources);
        this.smithyBuildGradleFile.addSources(...relativeSources);
    }
    /**
     * Convert any given absolute paths to relative paths to the project outdir
     * @private
     */
    asRelativePathsToProject(paths) {
        return paths.map((p) => path.isAbsolute(p) ? path.relative(this.project.outdir, p) : p);
    }
}
exports.SmithyProjectDefinition = SmithyProjectDefinition;
_a = JSII_RTTI_SYMBOL_1;
SmithyProjectDefinition[_a] = { fqn: "@aws/pdk.type_safe_api.SmithyProjectDefinition", version: "0.23.72" };
//# sourceMappingURL=data:application/json;base64,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