"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.voiceConnectorValidator = void 0;
const voiceConnector_1 = require("./voiceConnector");
var ISO_3166_ALPHA_2 = /^A[^ABCHJKNPVY]|B[^CKPUX]|C[^BEJPQST]|D[EJKMOZ]|E[CEGHRST]|F[IJKMOR]|G[^CJKOVXZ]|H[KMNRTU]|I[DEL-OQ-T]|J[EMOP]|K[EGHIMNPRWYZ]|L[ABCIKR-VY]|M[^BIJ]|N[ACEFGILOPRUZ]|OM|P[AE-HK-NRSTWY]|QA|R[EOSUW]|S[^FPQUW]|T[^ABEIPQSUXY]|U[AGMSYZ]|V[ACEGINU]|WF|WS|YE|YT|Z[AMW]$/;
var RFC_1918 = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)/;
var IP_ADDRESS = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
var VALID_CIDR = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\/(2[7-9])|(3[0-2])$/;
var RFC_1918_CIDR = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$)$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))/;
var FQDN = /^(?!.*?_.*?)(?!(?:[\w]+?\.)?\-[\w\.\-]*?)(?![\w]+?\-\.(?:[\w\.\-]+?))(?=[\w])(?=[\w\.\-]*?\.+[\w\.\-]*?)(?![\w\.\-]{254})(?!(?:\.?[\w\-\.]*?[\w\-]{64,}\.)+?)[\w\.\-]+?(?<![\w\-\.]*?\.[\d]+?)(?<=[\w\-]{2,})(?<![\w\-]{25})$/;
function voiceConnectorValidator(props) {
    if (props.region) {
        if (!(props.region.includes('Token[AWS.Region.') ||
            props.region === 'us-east-1' ||
            props.region === 'us-west-2')) {
            throw new Error('Region must be us-east-1 or us-west-2');
        }
    }
    if (props.termination) {
        if ('cps' in props.termination) {
            if (props.termination.cps < 1 || props.termination.cps > 256) {
                throw new Error('CPS must be between 1 and 256');
            }
        }
    }
    if (props.termination) {
        for (var country of props.termination.callingRegions) {
            if (!ISO_3166_ALPHA_2.test(country)) {
                throw new Error(`Invalid Country: ${country}`);
            }
        }
    }
    if (props.termination) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (terminationCidr.includes('Token')) {
            }
            else if (!VALID_CIDR.test(terminationCidr)) {
                throw new Error('Termination CIDR must be a valid non-RFC1918 IPv4 CIDR block (/27-/32)');
            }
        }
    }
    if (props.termination) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (RFC_1918_CIDR.test(terminationCidr)) {
                throw new Error('Termination CDIR must not be RFC1918 CIDR block (/27-/32)');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.host.includes('Token')) {
            }
            else if (IP_ADDRESS.test(route.host)) {
                if (RFC_1918.test(route.host)) {
                    throw new Error('Origination IP must not be RFC1918 IP Address');
                }
            }
            else if (!FQDN.test(route.host)) {
                throw new Error('Origination IP must be valid, non-RFC1918 IPv4 Address or FQDN');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.port < 1 || route.port > 65535) {
                throw new Error('Port range must be valid: 1-65535');
            }
        }
    }
    if (props.origination && props.encryption) {
        for (var route of props.origination) {
            if (route.protocol === voiceConnector_1.Protocol.UDP) {
                throw new Error('TCP must be used with Encryption enabled');
            }
        }
    }
    return true;
}
exports.voiceConnectorValidator = voiceConnectorValidator;
//# sourceMappingURL=data:application/json;base64,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