/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package mozilla.telemetry.glean.private

import android.os.SystemClock
import androidx.annotation.VisibleForTesting
import com.sun.jna.StringArray
import mozilla.telemetry.glean.Dispatchers
import mozilla.telemetry.glean.rust.LibGleanFFI
import mozilla.telemetry.glean.rust.toBoolean
import mozilla.telemetry.glean.rust.toByte
import mozilla.telemetry.glean.testing.ErrorType

/**
 * This implements the developer facing API for recording timespans.
 *
 * Instances of this class type are automatically generated by the parsers at build time,
 * allowing developers to record values that were previously registered in the metrics.yaml file.
 *
 * The timespans API exposes the [start], [stop] and [cancel] methods.
 */
class TimespanMetricType internal constructor(
    private var handle: Long,
    private val disabled: Boolean,
    private val sendInPings: List<String>
) {
    /**
     * The public constructor used by automatically generated metrics.
     */
    constructor(
        disabled: Boolean,
        category: String,
        lifetime: Lifetime,
        name: String,
        sendInPings: List<String>,
        timeUnit: TimeUnit = TimeUnit.Minute
    ) : this(handle = 0, disabled = disabled, sendInPings = sendInPings) {
        val ffiPingsList = StringArray(sendInPings.toTypedArray(), "utf-8")
        this.handle = LibGleanFFI.INSTANCE.glean_new_timespan_metric(
            category = category,
            name = name,
            send_in_pings = ffiPingsList,
            send_in_pings_len = sendInPings.size,
            lifetime = lifetime.ordinal,
            disabled = disabled.toByte(),
            time_unit = timeUnit.ordinal
        )
    }

    /**
     * Start tracking time for the provided metric.
     * This records an error if it’s already tracking time (i.e. start was already
     * called with no corresponding [stop]): in that case the original
     * start time will be preserved.
     */
    fun start() {
        if (disabled) {
            return
        }

        val startTime = SystemClock.elapsedRealtimeNanos()

        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.launch {
            LibGleanFFI.INSTANCE.glean_timespan_set_start(this@TimespanMetricType.handle, startTime)
        }
    }

    /**
     * Stop tracking time for the provided metric.
     * Sets the metric to the elapsed time, but does not overwrite an already
     * existing value.
     * This will record an error if no [start] was called or there is an already
     * existing value.
     */
    fun stop() {
        if (disabled) {
            return
        }

        val stopTime = SystemClock.elapsedRealtimeNanos()

        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.launch {
            LibGleanFFI.INSTANCE.glean_timespan_set_stop(this@TimespanMetricType.handle, stopTime)
        }
    }

    /**
     * Convenience method to simplify measuring a function or block of code
     *
     * If the measured function throws, the measurement is canceled and the exception rethrown.
     */
    @Suppress("TooGenericExceptionCaught")
    fun <U> measure(funcToMeasure: () -> U): U {
        start()

        val returnValue = try {
            funcToMeasure()
        } catch (e: Exception) {
            cancel()
            throw e
        }

        stop()
        return returnValue
    }

    /**
     * Abort a previous [start] call. No error is recorded if no [start] was called.
     */
    fun cancel() {
        if (disabled) {
            return
        }

        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.launch {
            LibGleanFFI.INSTANCE.glean_timespan_cancel(this@TimespanMetricType.handle)
        }
    }

    /**
     * Explicitly set the timespan value, in nanoseconds.
     *
     * This API should only be used if your library or application requires recording
     * times in a way that can not make use of [start]/[stop]/[cancel].
     *
     * [setRawNanos] does not overwrite a running timer or an already existing value.
     *
     * @param elapsedNanos The elapsed time to record, in nanoseconds.
     */
    fun setRawNanos(elapsedNanos: Long) {
        if (disabled) {
            return
        }

        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.launch {
            LibGleanFFI.INSTANCE.glean_timespan_set_raw_nanos(
                this@TimespanMetricType.handle,
                elapsedNanos)
        }
    }

    /**
     * Tests whether a value is stored for the metric for testing purposes only
     *
     * @param pingName represents the name of the ping to retrieve the metric for.
     *                 Defaults to the first value in `sendInPings`.
     * @return true if metric value exists, otherwise false
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    @JvmOverloads
    fun testHasValue(pingName: String = sendInPings.first()): Boolean {
        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.assertInTestingMode()

        return LibGleanFFI
            .INSTANCE.glean_timespan_test_has_value(this.handle, pingName)
            .toBoolean()
    }

    /**
     * Returns the stored value for testing purposes only
     *
     * @param pingName represents the name of the ping to retrieve the metric for.
     *                 Defaults to the first value in `sendInPings`.
     * @return value of the stored metric
     * @throws [NullPointerException] if no value is stored
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    @JvmOverloads
    fun testGetValue(pingName: String = sendInPings.first()): Long {
        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.assertInTestingMode()

        if (!testHasValue(pingName)) {
            throw NullPointerException("Metric has no value")
        }
        return LibGleanFFI.INSTANCE.glean_timespan_test_get_value(this.handle, pingName)
    }

    /**
     * Returns the number of errors recorded for the given metric.
     *
     * @param errorType The type of the error recorded.
     * @param pingName represents the name of the ping to retrieve the metric for.
     *                 Defaults to the first value in `sendInPings`.
     * @return the number of errors recorded for the metric.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    @JvmOverloads
    fun testGetNumRecordedErrors(errorType: ErrorType, pingName: String = sendInPings.first()): Int {
        @Suppress("EXPERIMENTAL_API_USAGE")
        Dispatchers.API.assertInTestingMode()

        return LibGleanFFI.INSTANCE.glean_timespan_test_get_num_recorded_errors(
            this.handle, errorType.ordinal, pingName
        )
    }
}
