﻿// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

using Mozilla.Glean.FFI;
using Mozilla.Glean.Utils;
using System;

namespace Mozilla.Glean.Private
{
    /// <summary>
    /// This implements the developer facing API for recording timespan metrics.
    /// 
    /// Instances of this class type are automatically generated by the parsers at build time,
    /// allowing developers to record values that were previously registered in the metrics.yaml file.
    /// </summary>
    public sealed class TimespanMetricType
    {
        private bool disabled;
        private string[] sendInPings;
        private UInt64 handle;

        /// <summary>
        /// The public constructor used by automatically generated metrics.
        /// </summary>
        public TimespanMetricType(
            bool disabled,
            string category,
            Lifetime lifetime,
            string name,
            string[] sendInPings,
            TimeUnit timeUnit = TimeUnit.Minute
            ) : this(0, disabled, sendInPings)
        {
            handle = LibGleanFFI.glean_new_timespan_metric(
                        category: category,
                        name: name,
                        send_in_pings: sendInPings,
                        send_in_pings_len: sendInPings.Length,
                        lifetime: (int)lifetime,
                        disabled: disabled,
                        time_unit: (int)timeUnit
                        );
        }

        /// <summary>
        /// The internal constructor is only used by `LabeledMetricType` directly.
        /// </summary>
        internal TimespanMetricType(
            UInt64 handle,
            bool disabled,
            string[] sendInPings
            )
        {
            this.disabled = disabled;
            this.sendInPings = sendInPings;
            this.handle = handle;
        }

        /// <summary>
        /// Start tracking time for the provided metric.
        /// This records an error if it’s already tracking time (i.e. `Start` was already
        /// called with no corresponding `Stop`): in that case the original
        /// start time will be preserved.
        /// </summary>
        public void Start()
        {
            if (disabled)
            {
                return;
            }

            ulong startTime = HighPrecisionTimestamp.GetTimestamp(TimeUnit.Nanosecond);

            Dispatchers.LaunchAPI(() => {
                LibGleanFFI.glean_timespan_set_start(handle, startTime);
            });
        }

        /// <summary>
        /// Stop tracking time for the provided metric.
        /// Sets the metric to the elapsed time, but does not overwrite an already
        /// existing value.
        /// This will record an error if no `Start` was called or there is an already
        /// existing value.
        /// </summary>
        public void Stop()
        {
            if (disabled)
            {
                return;
            }

            ulong stopTime = HighPrecisionTimestamp.GetTimestamp(TimeUnit.Nanosecond);

            Dispatchers.LaunchAPI(() => {
                LibGleanFFI.glean_timespan_set_stop(handle, stopTime);
            });
        }

        ///<summary>
        /// Convenience method to simplify measuring a function or block of code
        ///
        /// If the measured function throws, the measurement is canceled and the exception rethrown.
        /// </summary>
        /// <exception>If the measured function throws, the measurement is
        /// canceled and the exception rethrown.</exception>
        public T Measure<T>(Func<T> funcToMeasure)
        {
            Start();

            T returnValue;

            try {
                returnValue = funcToMeasure();
            } catch (Exception e) {
                Cancel();
                throw e;
            }

            Stop();

            return returnValue;
        }

        /// <summary>
        /// Abort a previous `Start` call. No error is recorded if no `Start` was called.
        /// </summary>
        public void Cancel()
        {
            if (disabled)
            {
                return;
            }

            Dispatchers.LaunchAPI(() => {
                LibGleanFFI.glean_timespan_cancel(handle);
            });
        }

        /// <summary>
        /// Explicitly set the timespan value, in nanoseconds.
        /// 
        /// This API should only be used if your library or application requires recording
        /// times in a way that can not make use of `Start`/`Stop`/`Cancel`.
        /// 
        /// `SetRawNanos` does not overwrite a running timer or an already existing value.
        /// </summary>
        /// <param name="elapsedNanos">The elapsed time to record, in nanoseconds.</param>
        public void SetRawNanos(ulong elapsedNanos)
        {
            if (disabled)
            {
                return;
            }

            Dispatchers.LaunchAPI(() => {
                LibGleanFFI.glean_timespan_set_raw_nanos(handle, elapsedNanos);
            });
        }


        /// <summary>
        /// Tests whether a value is stored for the metric for testing purposes only.
        /// </summary>
        /// <param name="pingName">represents the name of the ping to retrieve the metric for.
        /// Defaults to the first value in `sendInPings`</param>
        /// <returns>true if metric value exists, otherwise false</returns>
        public bool TestHasValue(string pingName = null)
        {
            Dispatchers.AssertInTestingMode();

            string ping = pingName ?? sendInPings[0];
            return LibGleanFFI.glean_timespan_test_has_value(this.handle, ping) != 0;
        }

        /// <summary>
        /// Returns the stored value for testing purposes only.
        /// </summary>
        /// <param name="pingName">represents the name of the ping to retrieve the metric for.
        /// Defaults to the first value in `sendInPings`</param>
        /// <returns>value of the stored metric</returns>
        /// <exception cref="System.NullReferenceException">Thrown when the metric contains no value</exception>
        public ulong TestGetValue(string pingName = null)
        {
            Dispatchers.AssertInTestingMode();

            if (!TestHasValue(pingName))
            {
                throw new NullReferenceException();
            }

            string ping = pingName ?? sendInPings[0];
            return LibGleanFFI.glean_timespan_test_get_value(this.handle, ping);
        }

        /// <summary>
        /// Returns the number of errors recorded for the given metric.
        /// </summary>
        /// <param name="errorType">The type of the error recorded.</param>
        /// <param name="pingName">represents the name of the ping to retrieve the metric for.
        /// Defaults to the first value in `sendInPings`.</param>
        /// <returns>the number of errors recorded for the metric.</returns>
        public Int32 TestGetNumRecordedErrors(Testing.ErrorType errorType, string pingName = null)
        {
            Dispatchers.AssertInTestingMode();

            string ping = pingName ?? sendInPings[0];
            return LibGleanFFI.glean_timespan_test_get_num_recorded_errors(
                this.handle, (int)errorType, ping
            );
        }
    }
}
