#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "replication.sourcestate"
"""
from urllib.parse import urlencode
from delphixpy.v1_11_11 import response_validator

def get(engine, ref):
    """
    Retrieve the specified ReplicationSourceState object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_11.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_11_11.web.objects.Replic
        ationSourceState.ReplicationSourceState` object
    :type ref: ``str``
    :rtype: :py:class:`v1_11_11.web.vo.ReplicationSourceState`
    """
    url = "/resources/json/delphix/replication/sourcestate/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['ReplicationSourceState'], returns_list=False, raw_result=raw_result)

def get_all(engine):
    """
    List ReplicationSourceState objects on the system.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_11.delphix_engine.DelphixEngine`
    :rtype: ``list`` of :py:class:`v1_11_11.web.vo.ReplicationSourceState`
    """
    url = "/resources/json/delphix/replication/sourcestate"
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['ReplicationSourceState'], returns_list=True, raw_result=raw_result)

def get_by_spec(engine, spec=None):
    """
    Find the replication source state associated with a given replication spec.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_11.delphix_engine.DelphixEngine`
    :param spec: Replication spec object reference.
    :type spec: ``str``
    :rtype: :py:class:`v1_11_11.web.vo.ReplicationSourceState`
    """
    url = "/resources/json/delphix/replication/sourcestate/getBySpec"
    query_params = {"spec": spec}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['ReplicationSourceState'], returns_list=False, raw_result=raw_result)

