"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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