"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope) {
        throw new Error('prepareApp must be called on the root node');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare it's template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (stack_1.Stack.isStack(stack) && stack.nested) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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