"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const asset_schema = require("@aws-cdk/cdk-assets-schema");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const assets_1 = require("../assets");
const cfn_fn_1 = require("../cfn-fn");
const token_1 = require("../token");
const _shared_1 = require("./_shared");
exports.BOOTSTRAP_QUALIFIER_CONTEXT = '@aws-cdk/core:bootstrapQualifier';
/**
 * Uses conventionally named roles and reify asset storage locations
 *
 * This synthesizer is the only StackSynthesizer that generates
 * an asset manifest, and is required to deploy CDK applications using the
 * `@aws-cdk/app-delivery` CI/CD library.
 *
 * Requires the environment to have been bootstrapped with Bootstrap Stack V2.
 */
class DefaultStackSynthesizer {
    constructor(props = {}) {
        this.props = props;
        this.files = {};
        this.dockerImages = {};
    }
    bind(stack) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.stack = stack;
        const qualifier = (_b = (_a = this.props.qualifier) !== null && _a !== void 0 ? _a : stack.node.tryGetContext(exports.BOOTSTRAP_QUALIFIER_CONTEXT)) !== null && _b !== void 0 ? _b : DefaultStackSynthesizer.DEFAULT_QUALIFIER;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${Qualifier}: always
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            s = replaceAll(s, '${Qualifier}', qualifier);
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        // tslint:disable:max-line-length
        this.bucketName = specialize((_c = this.props.fileAssetsBucketName) !== null && _c !== void 0 ? _c : DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME);
        this.repositoryName = specialize((_d = this.props.imageAssetsRepositoryName) !== null && _d !== void 0 ? _d : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME);
        this.deployRoleArn = specialize((_e = this.props.deployRoleArn) !== null && _e !== void 0 ? _e : DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN);
        this.cloudFormationExecutionRoleArn = specialize((_f = this.props.cloudFormationExecutionRole) !== null && _f !== void 0 ? _f : DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN);
        this.assetPublishingRoleArn = specialize((_g = this.props.assetPublishingRoleArn) !== null && _g !== void 0 ? _g : DefaultStackSynthesizer.DEFAULT_ASSET_PUBLISHING_ROLE_ARN);
        // tslint:enable:max-line-length
    }
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.bucketName);
        const objectKey = asset.sourceHash + (asset.packaging === assets_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations: {
                [this.manifestEnvName]: {
                    bucketName: this.bucketName,
                    objectKey,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.assetPublishingRoleArn,
                    assumeRoleExternalId: this.props.assetPublishingExternalId,
                },
            },
        };
        const httpUrl = cfnify(`https://s3.${this.stack.region}.${this.stack.urlSuffix}/${this.bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${this.bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(this.bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
            s3Url: httpUrl,
        };
    }
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.repositoryName);
        const imageTag = asset.sourceHash;
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations: {
                [this.manifestEnvName]: {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.assetPublishingRoleArn,
                    assumeRoleExternalId: this.props.assetPublishingExternalId,
                },
            },
        };
        // Return CFN expression
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    synthesizeStackArtifacts(session) {
        _shared_1.assertBound(this.stack);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        _shared_1.addStackArtifactToAssembly(session, this.stack, {
            assumeRoleArn: this.deployRoleArn,
            cloudFormationExecutionRoleArn: this.cloudFormationExecutionRoleArn,
            stackTemplateAssetObjectUrl: templateManifestUrl,
            requiresBootstrapStackVersion: 1,
        }, [artifactId]);
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const templatePath = path.join(session.assembly.outdir, this.stack.templateFile);
        const template = fs.readFileSync(templatePath, { encoding: 'utf-8' });
        const sourceHash = _shared_1.contentHash(template);
        this.addFileAsset({
            fileName: this.stack.templateFile,
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash,
        });
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        const manifest = {
            version: asset_schema.AssetManifestSchema.currentVersion(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        fs.writeFileSync(outPath, JSON.stringify(manifest, undefined, 2));
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
                requiresBootstrapStackVersion: 1,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        _shared_1.assertBound(this.stack);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.DefaultStackSynthesizer = DefaultStackSynthesizer;
/**
 * Default ARN qualifier
 */
DefaultStackSynthesizer.DEFAULT_QUALIFIER = 'hnb659fds';
/**
 * Default CloudFormation role ARN.
 */
DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-cfn-exec-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default deploy role ARN.
 */
DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-deploy-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN.
 */
DefaultStackSynthesizer.DEFAULT_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default image assets repository name
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME = 'cdk-${Qualifier}-container-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Default file assets bucket name
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME = 'cdk-${Qualifier}-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return token_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deploymen time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? cfn_fn_1.Fn.sub(s) : s;
}
//# sourceMappingURL=data:application/json;base64,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