"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const fs_1 = require("../../lib/fs");
module.exports = {
    files: {
        'does not change with the file name'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const content = 'Hello, world!';
            const input1 = path.join(workdir, 'input1.txt');
            const input2 = path.join(workdir, 'input2.txt');
            const input3 = path.join(workdir, 'input3.txt');
            fs.writeFileSync(input1, content);
            fs.writeFileSync(input2, content);
            fs.writeFileSync(input3, content + '.'); // add one character, hash should be different
            // WHEN
            const hash1 = fs_1.FileSystem.fingerprint(input1);
            const hash2 = fs_1.FileSystem.fingerprint(input2);
            const hash3 = fs_1.FileSystem.fingerprint(input3);
            // THEN
            test.deepEqual(hash1, hash2);
            test.notDeepEqual(hash3, hash1);
            test.done();
        },
        'works on empty files'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const input1 = path.join(workdir, 'empty');
            const input2 = path.join(workdir, 'empty');
            fs.writeFileSync(input1, '');
            fs.writeFileSync(input2, '');
            // WHEN
            const hash1 = fs_1.FileSystem.fingerprint(input1);
            const hash2 = fs_1.FileSystem.fingerprint(input2);
            // THEN
            test.deepEqual(hash1, hash2);
            test.done();
        },
    },
    directories: {
        'works on directories'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir);
            const hashCopy = fs_1.FileSystem.fingerprint(outdir);
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'ignores requested files'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir, { exclude: ['*.ignoreme'] });
            fs.writeFileSync(path.join(outdir, `${hashSrc}.ignoreme`), 'Ignore me!');
            const hashCopy = fs_1.FileSystem.fingerprint(outdir, { exclude: ['*.ignoreme'] });
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'changes with file names'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const cpydir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, cpydir);
            // be careful not to break a symlink
            fs.renameSync(path.join(cpydir, 'normal-dir', 'file-in-subdir.txt'), path.join(cpydir, 'move-me.txt'));
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir);
            const hashCopy = fs_1.FileSystem.fingerprint(cpydir);
            // THEN
            test.notDeepEqual(hashSrc, hashCopy);
            test.done();
        },
    },
    symlinks: {
        'changes with the contents of followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = fs_1.FileSystem.fingerprint(dir2);
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = fs_1.FileSystem.fingerprint(dir2);
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = fs_1.FileSystem.fingerprint(dir2);
            // THEN
            test.notDeepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
        'does not change with the contents of un-followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // THEN
            test.deepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
    },
    exclude: {
        'encodes exclude patterns'(test) {
            // GIVEN
            const dir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const options1 = { path: dir, exclude: ['**', '!file.py'], sourcePath: dir };
            const options2 = { path: dir, exclude: ['**', '!otherfile.py'], sourcePath: dir };
            // WHEN
            const f1 = fs_1.FileSystem.fingerprint(dir, options1);
            const f2 = fs_1.FileSystem.fingerprint(dir, options2);
            // THEN
            test.notDeepEqual(f1, f2);
            test.done();
        },
        'considers negated exclude patterns for fingerprint'(test) {
            // GIVEN
            const dir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const options = { path: dir, exclude: ['**', '!file.txt'], sourcePath: dir };
            // WHEN
            const f1 = fs_1.FileSystem.fingerprint(dir, options);
            fs.writeFileSync(path.join(dir, 'file.txt'), 'data');
            const f2 = fs_1.FileSystem.fingerprint(dir, options);
            // THEN
            test.notDeepEqual(f1, f2);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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