"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        if (props.maxCapacity < 0) {
            throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
        }
        if (props.minCapacity < 0) {
            throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
        }
        if (props.maxCapacity < props.minCapacity) {
            throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
        }
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: this.actions,
            serviceNamespace: props.serviceNamespace
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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