"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Schedule for scheduled scaling actions
 */
class Schedule {
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that Application AutoScaling will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Construct a Schedule from a moment in time
     */
    static at(moment) {
        return new LiteralSchedule(`at(${formatISO(moment)})`);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error(`Cannot supply both 'day' and 'weekDay', use at most one`);
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
    constructor() {
    }
}
exports.Schedule = Schedule;
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
function formatISO(date) {
    if (!date) {
        return undefined;
    }
    return date.getUTCFullYear() +
        '-' + pad(date.getUTCMonth() + 1) +
        '-' + pad(date.getUTCDate()) +
        'T' + pad(date.getUTCHours()) +
        ':' + pad(date.getUTCMinutes()) +
        ':' + pad(date.getUTCSeconds());
    function pad(num) {
        if (num < 10) {
            return '0' + num;
        }
        return num;
    }
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZWR1bGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzY2hlZHVsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUVBOztHQUVHO0FBQ0gsTUFBc0IsUUFBUTtJQUM1Qjs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxVQUFrQjtRQUN6QyxPQUFPLElBQUksZUFBZSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBa0I7UUFDbkMsSUFBSSxRQUFRLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsQ0FBQztTQUN6QztRQUVELElBQUksSUFBSSxHQUFHLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDbEUsSUFBSSxJQUFJLEtBQUssU0FBUyxFQUFFO1lBQUUsSUFBSSxHQUFHLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7U0FBRTtRQUM1RixJQUFJLElBQUksS0FBSyxTQUFTLEVBQUU7WUFBRSxJQUFJLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQztTQUFFO1FBQzlGLE9BQU8sSUFBSSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxNQUFZO1FBQzNCLE9BQU8sSUFBSSxlQUFlLENBQUMsTUFBTSxTQUFTLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBb0I7UUFDckMsSUFBSSxPQUFPLENBQUMsT0FBTyxLQUFLLFNBQVMsSUFBSSxPQUFPLENBQUMsR0FBRyxLQUFLLFNBQVMsRUFBRTtZQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLHlEQUF5RCxDQUFDLENBQUM7U0FDNUU7UUFFRCxNQUFNLE1BQU0sR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsQ0FBQztRQUM3QyxNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN6QyxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztRQUMzQyxNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQztRQUV6QyxrRkFBa0Y7UUFDbEYsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDN0UsTUFBTSxPQUFPLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFFL0MsT0FBTyxJQUFJLGVBQWUsQ0FBQyxRQUFRLE1BQU0sSUFBSSxJQUFJLElBQUksR0FBRyxJQUFJLEtBQUssSUFBSSxPQUFPLElBQUksSUFBSSxHQUFHLENBQUMsQ0FBQztJQUMzRixDQUFDO0lBT0Q7SUFDQSxDQUFDO0NBQ0Y7QUExREQsNEJBMERDO0FBc0RELE1BQU0sZUFBZ0IsU0FBUSxRQUFRO0lBQ3BDLFlBQTRCLGdCQUF3QjtRQUNsRCxLQUFLLEVBQUUsQ0FBQztRQURrQixxQkFBZ0IsR0FBaEIsZ0JBQWdCLENBQVE7SUFFcEQsQ0FBQztDQUNGO0FBRUQsU0FBUyxRQUFRLENBQUksQ0FBZ0IsRUFBRSxHQUFNO0lBQzNDLE9BQU8sQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDbkMsQ0FBQztBQUVELFNBQVMsU0FBUyxDQUFDLElBQVc7SUFDNUIsSUFBSSxDQUFDLElBQUksRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFFaEMsT0FBTyxJQUFJLENBQUMsY0FBYyxFQUFFO1FBQzFCLEdBQUcsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUNqQyxHQUFHLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUM1QixHQUFHLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUM3QixHQUFHLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO0lBRWxDLFNBQVMsR0FBRyxDQUFDLEdBQVc7UUFDdEIsSUFBSSxHQUFHLEdBQUcsRUFBRSxFQUFFO1lBQ1osT0FBTyxHQUFHLEdBQUcsR0FBRyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0FBQ0gsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxTQUFTLENBQUMsUUFBZ0IsRUFBRSxRQUFnQjtJQUNuRCxJQUFJLFFBQVEsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQUUsT0FBTyxTQUFTLENBQUM7S0FBRTtJQUN4RSxPQUFPLFFBQVEsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7QUFDdEMsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxRQUFRLENBQUMsUUFBZ0IsRUFBRSxRQUFnQjtJQUNsRCxPQUFPLFFBQVEsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsUUFBUSxHQUFHLENBQUMsQ0FBQyxDQUFDLFFBQVEsUUFBUSxJQUFJLFFBQVEsSUFBSSxDQUFDO0FBQ25GLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4vKipcbiAqIFNjaGVkdWxlIGZvciBzY2hlZHVsZWQgc2NhbGluZyBhY3Rpb25zXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBTY2hlZHVsZSB7XG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBzY2hlZHVsZSBmcm9tIGEgbGl0ZXJhbCBzY2hlZHVsZSBleHByZXNzaW9uXG4gICAqXG4gICAqIEBwYXJhbSBleHByZXNzaW9uIFRoZSBleHByZXNzaW9uIHRvIHVzZS4gTXVzdCBiZSBpbiBhIGZvcm1hdCB0aGF0IEFwcGxpY2F0aW9uIEF1dG9TY2FsaW5nIHdpbGwgcmVjb2duaXplXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGV4cHJlc3Npb24oZXhwcmVzc2lvbjogc3RyaW5nKTogU2NoZWR1bGUge1xuICAgIHJldHVybiBuZXcgTGl0ZXJhbFNjaGVkdWxlKGV4cHJlc3Npb24pO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIHNjaGVkdWxlIGZyb20gYW4gaW50ZXJ2YWwgYW5kIGEgdGltZSB1bml0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJhdGUoZHVyYXRpb246IER1cmF0aW9uKTogU2NoZWR1bGUge1xuICAgIGlmIChkdXJhdGlvbi50b1NlY29uZHMoKSA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdEdXJhdGlvbiBjYW5ub3QgYmUgMCcpO1xuICAgIH1cblxuICAgIGxldCByYXRlID0gbWF5YmVSYXRlKGR1cmF0aW9uLnRvRGF5cyh7IGludGVncmFsOiBmYWxzZSB9KSwgJ2RheScpO1xuICAgIGlmIChyYXRlID09PSB1bmRlZmluZWQpIHsgcmF0ZSA9IG1heWJlUmF0ZShkdXJhdGlvbi50b0hvdXJzKHsgaW50ZWdyYWw6IGZhbHNlIH0pLCAnaG91cicpOyB9XG4gICAgaWYgKHJhdGUgPT09IHVuZGVmaW5lZCkgeyByYXRlID0gbWFrZVJhdGUoZHVyYXRpb24udG9NaW51dGVzKHsgaW50ZWdyYWw6IHRydWUgfSksICdtaW51dGUnKTsgfVxuICAgIHJldHVybiBuZXcgTGl0ZXJhbFNjaGVkdWxlKHJhdGUpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIFNjaGVkdWxlIGZyb20gYSBtb21lbnQgaW4gdGltZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhdChtb21lbnQ6IERhdGUpOiBTY2hlZHVsZSB7XG4gICAgcmV0dXJuIG5ldyBMaXRlcmFsU2NoZWR1bGUoYGF0KCR7Zm9ybWF0SVNPKG1vbWVudCl9KWApO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHNjaGVkdWxlIGZyb20gYSBzZXQgb2YgY3JvbiBmaWVsZHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY3JvbihvcHRpb25zOiBDcm9uT3B0aW9ucyk6IFNjaGVkdWxlIHtcbiAgICBpZiAob3B0aW9ucy53ZWVrRGF5ICE9PSB1bmRlZmluZWQgJiYgb3B0aW9ucy5kYXkgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3Qgc3VwcGx5IGJvdGggJ2RheScgYW5kICd3ZWVrRGF5JywgdXNlIGF0IG1vc3Qgb25lYCk7XG4gICAgfVxuXG4gICAgY29uc3QgbWludXRlID0gZmFsbGJhY2sob3B0aW9ucy5taW51dGUsICcqJyk7XG4gICAgY29uc3QgaG91ciA9IGZhbGxiYWNrKG9wdGlvbnMuaG91ciwgJyonKTtcbiAgICBjb25zdCBtb250aCA9IGZhbGxiYWNrKG9wdGlvbnMubW9udGgsICcqJyk7XG4gICAgY29uc3QgeWVhciA9IGZhbGxiYWNrKG9wdGlvbnMueWVhciwgJyonKTtcblxuICAgIC8vIFdlZWtkYXkgZGVmYXVsdHMgdG8gJz8nIGlmIG5vdCBzdXBwbGllZC4gSWYgaXQgaXMgc3VwcGxpZWQsIGRheSBtdXN0IGJlY29tZSAnPydcbiAgICBjb25zdCBkYXkgPSBmYWxsYmFjayhvcHRpb25zLmRheSwgb3B0aW9ucy53ZWVrRGF5ICE9PSB1bmRlZmluZWQgPyAnPycgOiAnKicpO1xuICAgIGNvbnN0IHdlZWtEYXkgPSBmYWxsYmFjayhvcHRpb25zLndlZWtEYXksICc/Jyk7XG5cbiAgICByZXR1cm4gbmV3IExpdGVyYWxTY2hlZHVsZShgY3Jvbigke21pbnV0ZX0gJHtob3VyfSAke2RheX0gJHttb250aH0gJHt3ZWVrRGF5fSAke3llYXJ9KWApO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIHRoZSBleHByZXNzaW9uIGZvciB0aGlzIHNjaGVkdWxlXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZXhwcmVzc2lvblN0cmluZzogc3RyaW5nO1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcigpIHtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gY29uZmlndXJlIGEgY3JvbiBleHByZXNzaW9uXG4gKlxuICogQWxsIGZpZWxkcyBhcmUgc3RyaW5ncyBzbyB5b3UgY2FuIHVzZSBjb21wbGV4IGV4cHJlc2lvbnMuIEFic2VuY2Ugb2ZcbiAqIGEgZmllbGQgaW1wbGllcyAnKicgb3IgJz8nLCB3aGljaGV2ZXIgb25lIGlzIGFwcHJvcHJpYXRlLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L2V2ZW50cy9TY2hlZHVsZWRFdmVudHMuaHRtbCNDcm9uRXhwcmVzc2lvbnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDcm9uT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbWludXRlIHRvIHJ1biB0aGlzIHJ1bGUgYXRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBFdmVyeSBtaW51dGVcbiAgICovXG4gIHJlYWRvbmx5IG1pbnV0ZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGhvdXIgdG8gcnVuIHRoaXMgcnVsZSBhdFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEV2ZXJ5IGhvdXJcbiAgICovXG4gIHJlYWRvbmx5IGhvdXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkYXkgb2YgdGhlIG1vbnRoIHRvIHJ1biB0aGlzIHJ1bGUgYXRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBFdmVyeSBkYXkgb2YgdGhlIG1vbnRoXG4gICAqL1xuICByZWFkb25seSBkYXk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBtb250aCB0byBydW4gdGhpcyBydWxlIGF0XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRXZlcnkgbW9udGhcbiAgICovXG4gIHJlYWRvbmx5IG1vbnRoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgeWVhciB0byBydW4gdGhpcyBydWxlIGF0XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRXZlcnkgeWVhclxuICAgKi9cbiAgcmVhZG9ubHkgeWVhcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRheSBvZiB0aGUgd2VlayB0byBydW4gdGhpcyBydWxlIGF0XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQW55IGRheSBvZiB0aGUgd2Vla1xuICAgKi9cbiAgcmVhZG9ubHkgd2Vla0RheT86IHN0cmluZztcbn1cblxuY2xhc3MgTGl0ZXJhbFNjaGVkdWxlIGV4dGVuZHMgU2NoZWR1bGUge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgZXhwcmVzc2lvblN0cmluZzogc3RyaW5nKSB7XG4gICAgc3VwZXIoKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBmYWxsYmFjazxUPih4OiBUIHwgdW5kZWZpbmVkLCBkZWY6IFQpOiBUIHtcbiAgcmV0dXJuIHggPT09IHVuZGVmaW5lZCA/IGRlZiA6IHg7XG59XG5cbmZ1bmN0aW9uIGZvcm1hdElTTyhkYXRlPzogRGF0ZSkge1xuICBpZiAoIWRhdGUpIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuXG4gIHJldHVybiBkYXRlLmdldFVUQ0Z1bGxZZWFyKCkgK1xuICAgICctJyArIHBhZChkYXRlLmdldFVUQ01vbnRoKCkgKyAxKSArXG4gICAgJy0nICsgcGFkKGRhdGUuZ2V0VVRDRGF0ZSgpKSArXG4gICAgJ1QnICsgcGFkKGRhdGUuZ2V0VVRDSG91cnMoKSkgK1xuICAgICc6JyArIHBhZChkYXRlLmdldFVUQ01pbnV0ZXMoKSkgK1xuICAgICc6JyArIHBhZChkYXRlLmdldFVUQ1NlY29uZHMoKSk7XG5cbiAgZnVuY3Rpb24gcGFkKG51bTogbnVtYmVyKSB7XG4gICAgaWYgKG51bSA8IDEwKSB7XG4gICAgICByZXR1cm4gJzAnICsgbnVtO1xuICAgIH1cbiAgICByZXR1cm4gbnVtO1xuICB9XG59XG5cbi8qKlxuICogUmV0dXJuIHRoZSByYXRlIGlmIHRoZSByYXRlIGlzIHdob2xlIG51bWJlclxuICovXG5mdW5jdGlvbiBtYXliZVJhdGUoaW50ZXJ2YWw6IG51bWJlciwgc2luZ3VsYXI6IHN0cmluZykge1xuICBpZiAoaW50ZXJ2YWwgPT09IDAgfHwgIU51bWJlci5pc0ludGVnZXIoaW50ZXJ2YWwpKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgcmV0dXJuIG1ha2VSYXRlKGludGVydmFsLCBzaW5ndWxhcik7XG59XG5cbi8qKlxuICogUmV0dXJuICdyYXRlKCR7aW50ZXJ2YWx9ICR7c2luZ3VsYXJ9KHMpKWAgZm9yIHRoZSBpbnRlcnZhbFxuICovXG5mdW5jdGlvbiBtYWtlUmF0ZShpbnRlcnZhbDogbnVtYmVyLCBzaW5ndWxhcjogc3RyaW5nKSB7XG4gIHJldHVybiBpbnRlcnZhbCA9PT0gMSA/IGByYXRlKDEgJHtzaW5ndWxhcn0pYCA6IGByYXRlKCR7aW50ZXJ2YWx9ICR7c2luZ3VsYXJ9cylgO1xufSJdfQ==