"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            }
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RlcC1zY2FsaW5nLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0ZXAtc2NhbGluZy1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxxQ0FBc0M7QUFDdEMseUZBQXNFO0FBMER0RTs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsaUJBQWtCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFRbEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUN6RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSEYsZ0JBQVcsR0FBRyxJQUFJLEtBQUssRUFBMkMsQ0FBQztRQUtsRix5RkFBeUY7UUFDekYsNkRBQTZEO1FBQzdELHdIQUF3SDtRQUN4SCxNQUFNLFFBQVEsR0FBRyxJQUFJLG1EQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDdEQsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRO1lBQ2xELFVBQVUsRUFBRSxhQUFhO1lBQ3pCLGVBQWUsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLGdCQUFnQjtZQUNyRCw4QkFBOEIsRUFBRTtnQkFDOUIsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO2dCQUNwQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtnQkFDdEQsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtnQkFDcEQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtnQkFDbEQsZUFBZSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQzthQUNiO1NBQzdELENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7T0FFRztJQUNJLGFBQWEsQ0FBQyxVQUEwQjtRQUM3QyxJQUFJLFVBQVUsQ0FBQyxVQUFVLEtBQUssU0FBUyxJQUFJLFVBQVUsQ0FBQyxVQUFVLEtBQUssU0FBUyxFQUFFO1lBQzlFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztTQUN6RTtRQUNELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDO1lBQ3BCLHdCQUF3QixFQUFFLFVBQVUsQ0FBQyxVQUFVO1lBQy9DLHdCQUF3QixFQUFFLFVBQVUsQ0FBQyxVQUFVO1lBQy9DLGlCQUFpQixFQUFFLFVBQVUsQ0FBQyxVQUFVO1NBQ3pDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQTNDRCw4Q0EyQ0M7QUFFRDs7R0FFRztBQUNILElBQVksY0FvQlg7QUFwQkQsV0FBWSxjQUFjO0lBQ3hCOzs7O09BSUc7SUFDSCx5REFBdUMsQ0FBQTtJQUV2Qzs7Ozs7T0FLRztJQUNILHdFQUFzRCxDQUFBO0lBRXREOztPQUVHO0lBQ0gsa0RBQWdDLENBQUE7QUFDbEMsQ0FBQyxFQXBCVyxjQUFjLEdBQWQsc0JBQWMsS0FBZCxzQkFBYyxRQW9CekI7QUFFRDs7R0FFRztBQUNILElBQVkscUJBZVg7QUFmRCxXQUFZLHFCQUFxQjtJQUMvQjs7T0FFRztJQUNILDRDQUFtQixDQUFBO0lBRW5COztPQUVHO0lBQ0gsNENBQW1CLENBQUE7SUFFbkI7O09BRUc7SUFDSCw0Q0FBbUIsQ0FBQTtBQUNyQixDQUFDLEVBZlcscUJBQXFCLEdBQXJCLDZCQUFxQixLQUFyQiw2QkFBcUIsUUFlaEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY29yZScpO1xuaW1wb3J0IHsgQ2ZuU2NhbGluZ1BvbGljeSB9IGZyb20gJy4vYXBwbGljYXRpb25hdXRvc2NhbGluZy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVNjYWxhYmxlVGFyZ2V0IH0gZnJvbSAnLi9zY2FsYWJsZS10YXJnZXQnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgc2NhbGluZyBwb2xpY3lcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdGVwU2NhbGluZ0FjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBzY2FsYWJsZSB0YXJnZXRcbiAgICovXG4gIHJlYWRvbmx5IHNjYWxpbmdUYXJnZXQ6IElTY2FsYWJsZVRhcmdldDtcblxuICAvKipcbiAgICogQSBuYW1lIGZvciB0aGUgc2NhbGluZyBwb2xpY3lcbiAgICpcbiAgICogQGRlZmF1bHQgQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogSG93IHRoZSBhZGp1c3RtZW50IG51bWJlcnMgYXJlIGludGVycHJldGVkXG4gICAqXG4gICAqIEBkZWZhdWx0IENoYW5nZUluQ2FwYWNpdHlcbiAgICovXG4gIHJlYWRvbmx5IGFkanVzdG1lbnRUeXBlPzogQWRqdXN0bWVudFR5cGU7XG5cbiAgLyoqXG4gICAqIEdyYWNlIHBlcmlvZCBhZnRlciBzY2FsaW5nIGFjdGl2aXR5LlxuICAgKlxuICAgKiBGb3Igc2NhbGUgb3V0IHBvbGljaWVzLCBtdWx0aXBsZSBzY2FsZSBvdXRzIGR1cmluZyB0aGUgY29vbGRvd24gcGVyaW9kIGFyZVxuICAgKiBzcXVhc2hlZCBzbyB0aGF0IG9ubHkgdGhlIGJpZ2dlc3Qgc2NhbGUgb3V0IGhhcHBlbnMuXG4gICAqXG4gICAqIEZvciBzY2FsZSBpbiBwb2xpY2llcywgc3Vic2VxdWVudCBzY2FsZSBpbnMgZHVyaW5nIHRoZSBjb29sZG93biBwZXJpb2QgYXJlXG4gICAqIGlnbm9yZWQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfU3RlcFNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uLmh0bWxcbiAgICogQGRlZmF1bHQgTm8gY29vbGRvd24gcGVyaW9kXG4gICAqL1xuICByZWFkb25seSBjb29sZG93bj86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogTWluaW11bSBhYnNvbHV0ZSBudW1iZXIgdG8gYWRqdXN0IGNhcGFjaXR5IHdpdGggYXMgcmVzdWx0IG9mIHBlcmNlbnRhZ2Ugc2NhbGluZy5cbiAgICpcbiAgICogT25seSB3aGVuIHVzaW5nIEFkanVzdG1lbnRUeXBlID0gUGVyY2VudENoYW5nZUluQ2FwYWNpdHksIHRoaXMgbnVtYmVyIGNvbnRyb2xzXG4gICAqIHRoZSBtaW5pbXVtIGFic29sdXRlIGVmZmVjdCBzaXplLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBtaW5pbXVtIHNjYWxpbmcgZWZmZWN0XG4gICAqL1xuICByZWFkb25seSBtaW5BZGp1c3RtZW50TWFnbml0dWRlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgYWdncmVnYXRpb24gdHlwZSBmb3IgdGhlIENsb3VkV2F0Y2ggbWV0cmljcy5cbiAgICpcbiAgICogQGRlZmF1bHQgQXZlcmFnZVxuICAgKi9cbiAgcmVhZG9ubHkgbWV0cmljQWdncmVnYXRpb25UeXBlPzogTWV0cmljQWdncmVnYXRpb25UeXBlO1xufVxuXG4vKipcbiAqIERlZmluZSBhIHN0ZXAgc2NhbGluZyBhY3Rpb25cbiAqXG4gKiBUaGlzIGtpbmQgb2Ygc2NhbGluZyBwb2xpY3kgYWRqdXN0cyB0aGUgdGFyZ2V0IGNhcGFjaXR5IGluIGNvbmZpZ3VyYWJsZVxuICogc3RlcHMuIFRoZSBzaXplIG9mIHRoZSBzdGVwIGlzIGNvbmZpZ3VyYWJsZSBiYXNlZCBvbiB0aGUgbWV0cmljJ3MgZGlzdGFuY2VcbiAqIHRvIGl0cyBhbGFybSB0aHJlc2hvbGQuXG4gKlxuICogVGhpcyBBY3Rpb24gbXVzdCBiZSB1c2VkIGFzIHRoZSB0YXJnZXQgb2YgYSBDbG91ZFdhdGNoIGFsYXJtIHRvIHRha2UgZWZmZWN0LlxuICovXG5leHBvcnQgY2xhc3MgU3RlcFNjYWxpbmdBY3Rpb24gZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgc2NhbGluZyBwb2xpY3lcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzY2FsaW5nUG9saWN5QXJuOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhZGp1c3RtZW50cyA9IG5ldyBBcnJheTxDZm5TY2FsaW5nUG9saWN5LlN0ZXBBZGp1c3RtZW50UHJvcGVydHk+KCk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTdGVwU2NhbGluZ0FjdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vIENsb3VkZm9ybWF0aW9uIHJlcXVpcmVzIGVpdGhlciB0aGUgUmVzb3VyY2VJZCwgU2NhbGFibGVEaW1lbnNpb24sIGFuZCBTZXJ2aWNlTmFtZXNwYWNlXG4gICAgLy8gcHJvcGVydGllcywgb3IgdGhlIFNjYWxpbmdUYXJnZXRJZCBwcm9wZXJ0eSwgYnV0IG5vdCBib3RoLlxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1hcHBsaWNhdGlvbmF1dG9zY2FsaW5nLXNjYWxpbmdwb2xpY3kuaHRtbFxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblNjYWxpbmdQb2xpY3kodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgcG9saWN5TmFtZTogcHJvcHMucG9saWN5TmFtZSB8fCB0aGlzLm5vZGUudW5pcXVlSWQsXG4gICAgICBwb2xpY3lUeXBlOiAnU3RlcFNjYWxpbmcnLFxuICAgICAgc2NhbGluZ1RhcmdldElkOiBwcm9wcy5zY2FsaW5nVGFyZ2V0LnNjYWxhYmxlVGFyZ2V0SWQsXG4gICAgICBzdGVwU2NhbGluZ1BvbGljeUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgYWRqdXN0bWVudFR5cGU6IHByb3BzLmFkanVzdG1lbnRUeXBlLFxuICAgICAgICBjb29sZG93bjogcHJvcHMuY29vbGRvd24gJiYgcHJvcHMuY29vbGRvd24udG9TZWNvbmRzKCksXG4gICAgICAgIG1pbkFkanVzdG1lbnRNYWduaXR1ZGU6IHByb3BzLm1pbkFkanVzdG1lbnRNYWduaXR1ZGUsXG4gICAgICAgIG1ldHJpY0FnZ3JlZ2F0aW9uVHlwZTogcHJvcHMubWV0cmljQWdncmVnYXRpb25UeXBlLFxuICAgICAgICBzdGVwQWRqdXN0bWVudHM6IGNkay5MYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5hZGp1c3RtZW50cyB9KSxcbiAgICAgIH0gYXMgQ2ZuU2NhbGluZ1BvbGljeS5TdGVwU2NhbGluZ1BvbGljeUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eVxuICAgIH0pO1xuXG4gICAgdGhpcy5zY2FsaW5nUG9saWN5QXJuID0gcmVzb3VyY2UucmVmO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBhZGp1c21lbnQgaW50ZXJ2YWwgdG8gdGhlIFNjYWxpbmdBY3Rpb25cbiAgICovXG4gIHB1YmxpYyBhZGRBZGp1c3RtZW50KGFkanVzdG1lbnQ6IEFkanVzdG1lbnRUaWVyKSB7XG4gICAgaWYgKGFkanVzdG1lbnQubG93ZXJCb3VuZCA9PT0gdW5kZWZpbmVkICYmIGFkanVzdG1lbnQudXBwZXJCb3VuZCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBvZiBsb3dlckJvdW5kIG9yIHVwcGVyQm91bmQgaXMgcmVxdWlyZWQnKTtcbiAgICB9XG4gICAgdGhpcy5hZGp1c3RtZW50cy5wdXNoKHtcbiAgICAgIG1ldHJpY0ludGVydmFsTG93ZXJCb3VuZDogYWRqdXN0bWVudC5sb3dlckJvdW5kLFxuICAgICAgbWV0cmljSW50ZXJ2YWxVcHBlckJvdW5kOiBhZGp1c3RtZW50LnVwcGVyQm91bmQsXG4gICAgICBzY2FsaW5nQWRqdXN0bWVudDogYWRqdXN0bWVudC5hZGp1c3RtZW50LFxuICAgIH0pO1xuICB9XG59XG5cbi8qKlxuICogSG93IGFkanVzdG1lbnQgbnVtYmVycyBhcmUgaW50ZXJwcmV0ZWRcbiAqL1xuZXhwb3J0IGVudW0gQWRqdXN0bWVudFR5cGUge1xuICAvKipcbiAgICogQWRkIHRoZSBhZGp1c3RtZW50IG51bWJlciB0byB0aGUgY3VycmVudCBjYXBhY2l0eS5cbiAgICpcbiAgICogQSBwb3NpdGl2ZSBudW1iZXIgaW5jcmVhc2VzIGNhcGFjaXR5LCBhIG5lZ2F0aXZlIG51bWJlciBkZWNyZWFzZXMgY2FwYWNpdHkuXG4gICAqL1xuICBDSEFOR0VfSU5fQ0FQQUNJVFkgPSAnQ2hhbmdlSW5DYXBhY2l0eScsXG5cbiAgLyoqXG4gICAqIEFkZCB0aGlzIHBlcmNlbnRhZ2Ugb2YgdGhlIGN1cnJlbnQgY2FwYWNpdHkgdG8gaXRzZWxmLlxuICAgKlxuICAgKiBUaGUgbnVtYmVyIG11c3QgYmUgYmV0d2VlbiAtMTAwIGFuZCAxMDA7IGEgcG9zaXRpdmUgbnVtYmVyIGluY3JlYXNlc1xuICAgKiBjYXBhY2l0eSBhbmQgYSBuZWdhdGl2ZSBudW1iZXIgZGVjcmVhc2VzIGl0LlxuICAgKi9cbiAgUEVSQ0VOVF9DSEFOR0VfSU5fQ0FQQUNJVFkgPSAnUGVyY2VudENoYW5nZUluQ2FwYWNpdHknLFxuXG4gIC8qKlxuICAgKiBNYWtlIHRoZSBjYXBhY2l0eSBlcXVhbCB0byB0aGUgZXhhY3QgbnVtYmVyIGdpdmVuLlxuICAgKi9cbiAgRVhBQ1RfQ0FQQUNJVFkgPSAnRXhhY3RDYXBhY2l0eScsXG59XG5cbi8qKlxuICogSG93IHRoZSBzY2FsaW5nIG1ldHJpYyBpcyBnb2luZyB0byBiZSBhZ2dyZWdhdGVkXG4gKi9cbmV4cG9ydCBlbnVtIE1ldHJpY0FnZ3JlZ2F0aW9uVHlwZSB7XG4gIC8qKlxuICAgKiBBdmVyYWdlXG4gICAqL1xuICBBVkVSQUdFID0gJ0F2ZXJhZ2UnLFxuXG4gIC8qKlxuICAgKiBNaW5pbXVtXG4gICAqL1xuICBNSU5JTVVNID0gJ01pbmltdW0nLFxuXG4gIC8qKlxuICAgKiBNYXhpbXVtXG4gICAqL1xuICBNQVhJTVVNID0gJ01heGltdW0nXG59XG5cbi8qKlxuICogQW4gYWRqdXN0bWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkanVzdG1lbnRUaWVyIHtcbiAgLyoqXG4gICAqIFdoYXQgbnVtYmVyIHRvIGFkanVzdCB0aGUgY2FwYWNpdHkgd2l0aFxuICAgKlxuICAgKiBUaGUgbnVtYmVyIGlzIGludGVycGV0ZWQgYXMgYW4gYWRkZWQgY2FwYWNpdHksIGEgbmV3IGZpeGVkIGNhcGFjaXR5IG9yIGFuXG4gICAqIGFkZGVkIHBlcmNlbnRhZ2UgZGVwZW5kaW5nIG9uIHRoZSBBZGp1c3RtZW50VHlwZSB2YWx1ZSBvZiB0aGVcbiAgICogU3RlcFNjYWxpbmdQb2xpY3kuXG4gICAqXG4gICAqIENhbiBiZSBwb3NpdGl2ZSBvciBuZWdhdGl2ZS5cbiAgICovXG4gIHJlYWRvbmx5IGFkanVzdG1lbnQ6IG51bWJlcjtcblxuICAvKipcbiAgICogTG93ZXIgYm91bmQgd2hlcmUgdGhpcyBzY2FsaW5nIHRpZXIgYXBwbGllcy5cbiAgICpcbiAgICogVGhlIHNjYWxpbmcgdGllciBhcHBsaWVzIGlmIHRoZSBkaWZmZXJlbmNlIGJldHdlZW4gdGhlIG1ldHJpY1xuICAgKiB2YWx1ZSBhbmQgaXRzIGFsYXJtIHRocmVzaG9sZCBpcyBoaWdoZXIgdGhhbiB0aGlzIHZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtSW5maW5pdHkgaWYgdGhpcyBpcyB0aGUgZmlyc3QgdGllciwgb3RoZXJ3aXNlIHRoZSB1cHBlckJvdW5kIG9mIHRoZSBwcmV2aW91cyB0aWVyXG4gICAqL1xuICByZWFkb25seSBsb3dlckJvdW5kPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBVcHBlciBib3VuZCB3aGVyZSB0aGlzIHNjYWxpbmcgdGllciBhcHBsaWVzXG4gICAqXG4gICAqIFRoZSBzY2FsaW5nIHRpZXIgYXBwbGllcyBpZiB0aGUgZGlmZmVyZW5jZSBiZXR3ZWVuIHRoZSBtZXRyaWNcbiAgICogdmFsdWUgYW5kIGl0cyBhbGFybSB0aHJlc2hvbGQgaXMgbG93ZXIgdGhhbiB0aGlzIHZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCArSW5maW5pdHlcbiAgICovXG4gIHJlYWRvbmx5IHVwcGVyQm91bmQ/OiBudW1iZXI7XG59XG4iXX0=