"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_autoscaling_common_1 = require("@aws-cdk/aws-autoscaling-common");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                period: cdk.Duration.minutes(1),
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                period: cdk.Duration.minutes(1),
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    const statistic = metric.toAlarmConfig().statistic;
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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