"use strict";
const assert_1 = require("@aws-cdk/assert");
const assert_2 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const fc = require("fast-check");
const util_1 = require("./util");
const util_2 = require("./util");
/**
 * Synthesize the given step scaling setup to a template
 */
function setupStepScaling(intervals) {
    const stack = new cdk.Stack();
    const target = util_2.createScalableTarget(stack);
    target.scaleOnMetric('ScaleInterval', {
        metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Success' }),
        scalingSteps: intervals
    });
    return new ScalingStackTemplate(assert_2.SynthUtils.synthesize(stack).template);
}
class ScalingStackTemplate {
    constructor(template) {
        this.template = template;
        this.lowerPolicy = 'TargetScaleIntervalLowerPolicy6F26D597';
        this.lowerAlarm = 'TargetScaleIntervalLowerAlarm4B5CE869';
        this.upperPolicy = 'TargetScaleIntervalUpperPolicy7C751132';
        this.upperAlarm = 'TargetScaleIntervalUpperAlarm69FD1BBB';
    }
    get lowerThreshold() {
        return this.threshold(this.lowerAlarm);
    }
    get upperThreshold() {
        return this.threshold(this.upperAlarm);
    }
    get lowerSteps() {
        return this.steps(this.lowerPolicy);
    }
    get upperSteps() {
        return this.steps(this.upperPolicy);
    }
    allStepsAbsolute() {
        const ret = new Array();
        const lowerThreshold = this.lowerThreshold;
        if (lowerThreshold !== undefined) {
            ret.push(...this.lowerSteps.map(x => makeAbsolute(lowerThreshold, x)));
        }
        const upperThreshold = this.upperThreshold;
        if (upperThreshold !== undefined) {
            ret.push(...this.upperSteps.map(x => makeAbsolute(upperThreshold, x)));
        }
        return ret;
    }
    resource(id) {
        return this.template.Resources[id];
    }
    threshold(id) {
        return apply(this.resource(id), x => x.Properties.Threshold);
    }
    steps(id) {
        return apply(this.resource(id), x => x.Properties.StepScalingPolicyConfiguration.StepAdjustments);
    }
}
function makeAbsolute(threshold, step) {
    return concrete({
        MetricIntervalLowerBound: apply(step.MetricIntervalLowerBound, x => x + threshold),
        MetricIntervalUpperBound: apply(step.MetricIntervalUpperBound, x => x + threshold),
        ScalingAdjustment: step.ScalingAdjustment
    });
}
function overlaps(a, b) {
    return (a.MetricIntervalLowerBound < b.MetricIntervalUpperBound
        && a.MetricIntervalUpperBound > b.MetricIntervalLowerBound);
}
function concrete(step) {
    return {
        MetricIntervalLowerBound: ifUndefined(step.MetricIntervalLowerBound, -Infinity),
        MetricIntervalUpperBound: ifUndefined(step.MetricIntervalUpperBound, Infinity),
        ScalingAdjustment: step.ScalingAdjustment
    };
}
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
function apply(x, f) {
    if (x === undefined) {
        return undefined;
    }
    return f(x);
}
/**
 * Helper function to print variables in case of a failing property check
 */
function reportFalse(cond, ...repr) {
    if (!cond) {
        // tslint:disable-next-line:no-console
        console.error('PROPERTY FAILS ON:', ...repr);
    }
    return cond;
}
module.exports = {
    'alarm thresholds are valid numbers'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const lowerThreshold = template.lowerThreshold;
            const upperThreshold = template.upperThreshold;
            return reportFalse((lowerThreshold === undefined || (lowerThreshold > 0 && lowerThreshold !== Infinity))
                && (upperThreshold === undefined || (upperThreshold > 0 && upperThreshold !== Infinity)), lowerThreshold, upperThreshold);
        }));
        test.done();
    },
    'generated step intervals are valid intervals'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return reportFalse(steps.every(step => {
                return step.MetricIntervalLowerBound < step.MetricIntervalUpperBound;
            }), steps, 'template', JSON.stringify(template, undefined, 2));
        }));
        test.done();
    },
    'generated step intervals are nonoverlapping'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            for (let i = 0; i < steps.length; i++) {
                const compareTo = steps.slice(i + 1);
                if (compareTo.some(x => overlaps(steps[i], x))) {
                    return reportFalse(false, steps);
                }
            }
            return true;
        }), { verbose: true });
        test.done();
    },
    'all template intervals occur in input array'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return steps.every(step => {
                return reportFalse(intervals.find(interval => {
                    const acceptableLowerBounds = step.MetricIntervalLowerBound === -Infinity ? [undefined, 0] : [undefined, step.MetricIntervalLowerBound];
                    // tslint:disable-next-line:max-line-length
                    const acceptableUpperBounds = step.MetricIntervalUpperBound === Infinity ? [undefined, Infinity] : [undefined, step.MetricIntervalUpperBound];
                    return (acceptableLowerBounds.includes(interval.lower) && acceptableUpperBounds.includes(interval.upper));
                }) !== undefined, step, intervals);
            });
        }));
        test.done();
    },
    'lower alarm uses lower policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.lowerAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.lowerPolicy, alarm);
        }));
        test.done();
    },
    'upper alarm uses upper policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.upperAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.upperPolicy, alarm);
        }));
        test.done();
    },
    'test step scaling on metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const target = util_2.createScalableTarget(stack);
        // WHEN
        target.scaleOnMetric('Tracking', {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            scalingSteps: [
                { upper: 0, change: -1 },
                { lower: 100, change: +1 },
                { lower: 500, change: +5 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "StepScaling",
            ScalingTargetId: {
                Ref: "Target3191CF44"
            },
            StepScalingPolicyConfiguration: {
                AdjustmentType: "ChangeInCapacity",
                MetricAggregationType: "Average",
                StepAdjustments: [
                    {
                        MetricIntervalUpperBound: 0,
                        ScalingAdjustment: -1
                    }
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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