"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsPublish = exports.MessageAttributeDataType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The data type set for the SNS message attributes
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
 */
var MessageAttributeDataType;
(function (MessageAttributeDataType) {
    /**
     * Strings are Unicode with UTF-8 binary encoding
     */
    MessageAttributeDataType["STRING"] = "String";
    /**
     * An array, formatted as a string
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["STRING_ARRAY"] = "String.Array";
    /**
     * Numbers are positive or negative integers or floating-point numbers
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["NUMBER"] = "Number";
    /**
     * Binary type attributes can store any binary data
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["BINARY"] = "Binary";
})(MessageAttributeDataType = exports.MessageAttributeDataType || (exports.MessageAttributeDataType = {}));
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 */
class SnsPublish extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SnsPublishProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SnsPublish);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SnsPublish.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is required in `message` Use JsonPath.taskToken to set the token.');
            }
        }
        this.taskPolicies = [
            new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [this.props.topic.topicArn],
            }),
        ];
    }
    /**
     * Provides the SNS Publish service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sns', 'publish', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                TopicArn: this.props.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                MessageAttributes: renderMessageAttributes(this.props.messageAttributes),
                Subject: this.props.subject,
            }),
        };
    }
}
exports.SnsPublish = SnsPublish;
_a = JSII_RTTI_SYMBOL_1;
SnsPublish[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SnsPublish", version: "1.198.1" };
SnsPublish.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderMessageAttributes(attributes) {
    if (attributes === undefined) {
        return undefined;
    }
    const renderedAttributes = {};
    Object.entries(attributes).map(([key, val]) => {
        renderedAttributes[key] = renderMessageAttributeValue(val);
    });
    return sfn.TaskInput.fromObject(renderedAttributes).value;
}
function renderMessageAttributeValue(attribute) {
    const dataType = attribute.dataType;
    if (attribute.value instanceof sfn.TaskInput) {
        return {
            DataType: dataType ?? MessageAttributeDataType.STRING,
            StringValue: dataType !== MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
            BinaryValue: dataType === MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
        };
    }
    if (dataType === MessageAttributeDataType.BINARY) {
        return { DataType: dataType, BinaryValue: `${attribute.value}` };
    }
    if (core_1.Token.isUnresolved(attribute.value)) {
        return { DataType: dataType ?? MessageAttributeDataType.STRING, StringValue: attribute.value };
    }
    validateMessageAttribute(attribute);
    if (Array.isArray(attribute.value)) {
        return { DataType: MessageAttributeDataType.STRING_ARRAY, StringValue: JSON.stringify(attribute.value) };
    }
    const value = attribute.value;
    if (typeof value === 'number') {
        return { DataType: MessageAttributeDataType.NUMBER, StringValue: `${value}` };
    }
    else {
        return { DataType: MessageAttributeDataType.STRING, StringValue: `${value}` };
    }
}
function validateMessageAttribute(attribute) {
    const dataType = attribute.dataType;
    const value = attribute.value;
    if (dataType === undefined) {
        return;
    }
    if (Array.isArray(value)) {
        if (dataType !== MessageAttributeDataType.STRING_ARRAY) {
            throw new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type Array`);
        }
        const validArrayTypes = ['string', 'boolean', 'number'];
        value.forEach((v) => {
            if (v !== null || !validArrayTypes.includes(typeof v)) {
                throw new Error(`Requested SNS message attribute type was ${typeof value} but Array values must be one of ${validArrayTypes}`);
            }
        });
        return;
    }
    const error = new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type ${typeof value}`);
    switch (typeof value) {
        case 'string':
            // trust the user or will default to string
            if (sfn.JsonPath.isEncodedJsonPath(attribute.value)) {
                return;
            }
            if (dataType === MessageAttributeDataType.STRING ||
                dataType === MessageAttributeDataType.BINARY) {
                return;
            }
            throw error;
        case 'number':
            if (dataType === MessageAttributeDataType.NUMBER) {
                return;
            }
            throw error;
        case 'boolean':
            if (dataType === MessageAttributeDataType.STRING) {
                return;
            }
            throw error;
        default:
            throw error;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHVibGlzaC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInB1Ymxpc2gudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsd0NBQXdDO0FBRXhDLGtEQUFrRDtBQUNsRCx3Q0FBc0M7QUFFdEMsc0RBQXlGO0FBRXpGOzs7O0dBSUc7QUFDSCxJQUFZLHdCQTBCWDtBQTFCRCxXQUFZLHdCQUF3QjtJQUNsQzs7T0FFRztJQUNILDZDQUFpQixDQUFBO0lBRWpCOzs7O09BSUc7SUFDSCx5REFBNkIsQ0FBQTtJQUU3Qjs7OztPQUlHO0lBQ0gsNkNBQWlCLENBQUE7SUFFakI7Ozs7T0FJRztJQUNILDZDQUFpQixDQUFBO0FBQ25CLENBQUMsRUExQlcsd0JBQXdCLEdBQXhCLGdDQUF3QixLQUF4QixnQ0FBd0IsUUEwQm5DO0FBOEVEOzs7R0FHRztBQUNILE1BQWEsVUFBVyxTQUFRLEdBQUcsQ0FBQyxhQUFhO0lBWS9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQW1CLEtBQXNCO1FBQy9FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRGlDLFVBQUssR0FBTCxLQUFLLENBQWlCOzs7Ozs7K0NBWnRFLFVBQVU7Ozs7UUFjbkIsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7UUFFOUYscUNBQXdCLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBRTdGLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRTtZQUMxRSxJQUFJLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQ3BELE1BQU0sSUFBSSxLQUFLLENBQUMsOEVBQThFLENBQUMsQ0FBQzthQUNqRztTQUNGO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRztZQUNsQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRSxDQUFDLGFBQWEsQ0FBQztnQkFDeEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO2FBQ3ZDLENBQUM7U0FDSCxDQUFDO0tBQ0g7SUFFRDs7T0FFRztJQUNIOztPQUVHO0lBQ08sV0FBVztRQUNuQixPQUFPO1lBQ0wsUUFBUSxFQUFFLG1DQUFzQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQzNFLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQztnQkFDdEMsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFFBQVE7Z0JBQ25DLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLO2dCQUNqQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQzVFLGlCQUFpQixFQUFFLHVCQUF1QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7Z0JBQ3hFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU87YUFDNUIsQ0FBQztTQUNILENBQUM7S0FDSDs7QUFqREgsZ0NBa0RDOzs7QUFoRHlCLHlDQUE4QixHQUE2QjtJQUNqRixHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCO0lBQ3ZDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7Q0FDM0MsQ0FBQztBQXFESixTQUFTLHVCQUF1QixDQUFDLFVBQWdEO0lBQy9FLElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFDbkQsTUFBTSxrQkFBa0IsR0FBNkMsRUFBRSxDQUFDO0lBQ3hFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsRUFBRTtRQUM1QyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsR0FBRywyQkFBMkIsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM3RCxDQUFDLENBQUMsQ0FBQztJQUNILE9BQU8sR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxLQUFLLENBQUM7QUFDNUQsQ0FBQztBQUVELFNBQVMsMkJBQTJCLENBQUMsU0FBMkI7SUFDOUQsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLFFBQVEsQ0FBQztJQUNwQyxJQUFJLFNBQVMsQ0FBQyxLQUFLLFlBQVksR0FBRyxDQUFDLFNBQVMsRUFBRTtRQUM1QyxPQUFPO1lBQ0wsUUFBUSxFQUFFLFFBQVEsSUFBSSx3QkFBd0IsQ0FBQyxNQUFNO1lBQ3JELFdBQVcsRUFBRSxRQUFRLEtBQUssd0JBQXdCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUztZQUM3RixXQUFXLEVBQUUsUUFBUSxLQUFLLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDOUYsQ0FBQztLQUNIO0lBRUQsSUFBSSxRQUFRLEtBQUssd0JBQXdCLENBQUMsTUFBTSxFQUFFO1FBQ2hELE9BQU8sRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxHQUFHLFNBQVMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDO0tBQ2xFO0lBRUQsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUN2QyxPQUFPLEVBQUUsUUFBUSxFQUFFLFFBQVEsSUFBSSx3QkFBd0IsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLFNBQVMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztLQUNoRztJQUVELHdCQUF3QixDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3BDLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7UUFDbEMsT0FBTyxFQUFFLFFBQVEsRUFBRSx3QkFBd0IsQ0FBQyxZQUFZLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7S0FDMUc7SUFDRCxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDO0lBQzlCLElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFO1FBQzdCLE9BQU8sRUFBRSxRQUFRLEVBQUUsd0JBQXdCLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxHQUFHLEtBQUssRUFBRSxFQUFFLENBQUM7S0FDL0U7U0FBTTtRQUNMLE9BQU8sRUFBRSxRQUFRLEVBQUUsd0JBQXdCLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxHQUFHLEtBQUssRUFBRSxFQUFFLENBQUM7S0FDL0U7QUFDSCxDQUFDO0FBRUQsU0FBUyx3QkFBd0IsQ0FBQyxTQUEyQjtJQUMzRCxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsUUFBUSxDQUFDO0lBQ3BDLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUM7SUFDOUIsSUFBSSxRQUFRLEtBQUssU0FBUyxFQUFFO1FBQzFCLE9BQU87S0FDUjtJQUNELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUN4QixJQUFJLFFBQVEsS0FBSyx3QkFBd0IsQ0FBQyxZQUFZLEVBQUU7WUFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsUUFBUSxRQUFRLEtBQUssb0JBQW9CLENBQUMsQ0FBQztTQUN4RztRQUNELE1BQU0sZUFBZSxHQUFHLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4RCxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7WUFDbEIsSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxFQUFFO2dCQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxPQUFPLEtBQUssb0NBQW9DLGVBQWUsRUFBRSxDQUFDLENBQUM7YUFDaEk7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILE9BQU87S0FDUjtJQUNELE1BQU0sS0FBSyxHQUFHLElBQUksS0FBSyxDQUFDLDRDQUE0QyxRQUFRLFFBQVEsS0FBSyxnQkFBZ0IsT0FBTyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3pILFFBQVEsT0FBTyxLQUFLLEVBQUU7UUFDcEIsS0FBSyxRQUFRO1lBQ1gsMkNBQTJDO1lBQzNDLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ25ELE9BQU87YUFDUjtZQUNELElBQUksUUFBUSxLQUFLLHdCQUF3QixDQUFDLE1BQU07Z0JBQzlDLFFBQVEsS0FBSyx3QkFBd0IsQ0FBQyxNQUFNLEVBQUU7Z0JBQzlDLE9BQU87YUFDUjtZQUNELE1BQU0sS0FBSyxDQUFDO1FBQ2QsS0FBSyxRQUFRO1lBQ1gsSUFBSSxRQUFRLEtBQUssd0JBQXdCLENBQUMsTUFBTSxFQUFFO2dCQUFFLE9BQU87YUFBRTtZQUM3RCxNQUFNLEtBQUssQ0FBQztRQUNkLEtBQUssU0FBUztZQUNaLElBQUksUUFBUSxLQUFLLHdCQUF3QixDQUFDLE1BQU0sRUFBRTtnQkFBRSxPQUFPO2FBQUU7WUFDN0QsTUFBTSxLQUFLLENBQUM7UUFDZDtZQUNFLE1BQU0sS0FBSyxDQUFDO0tBQ2Y7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IGludGVncmF0aW9uUmVzb3VyY2VBcm4sIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCB9IGZyb20gJy4uL3ByaXZhdGUvdGFzay11dGlscyc7XG5cbi8qKlxuICogVGhlIGRhdGEgdHlwZSBzZXQgZm9yIHRoZSBTTlMgbWVzc2FnZSBhdHRyaWJ1dGVzXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc25zL2xhdGVzdC9kZy9zbnMtbWVzc2FnZS1hdHRyaWJ1dGVzLmh0bWwjU05TTWVzc2FnZUF0dHJpYnV0ZXMuRGF0YVR5cGVzXG4gKi9cbmV4cG9ydCBlbnVtIE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZSB7XG4gIC8qKlxuICAgKiBTdHJpbmdzIGFyZSBVbmljb2RlIHdpdGggVVRGLTggYmluYXJ5IGVuY29kaW5nXG4gICAqL1xuICBTVFJJTkcgPSAnU3RyaW5nJyxcblxuICAvKipcbiAgICogQW4gYXJyYXksIGZvcm1hdHRlZCBhcyBhIHN0cmluZ1xuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zbnMvbGF0ZXN0L2RnL3Nucy1tZXNzYWdlLWF0dHJpYnV0ZXMuaHRtbCNTTlNNZXNzYWdlQXR0cmlidXRlcy5EYXRhVHlwZXNcbiAgICovXG4gIFNUUklOR19BUlJBWSA9ICdTdHJpbmcuQXJyYXknLFxuXG4gIC8qKlxuICAgKiBOdW1iZXJzIGFyZSBwb3NpdGl2ZSBvciBuZWdhdGl2ZSBpbnRlZ2VycyBvciBmbG9hdGluZy1wb2ludCBudW1iZXJzXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nucy9sYXRlc3QvZGcvc25zLW1lc3NhZ2UtYXR0cmlidXRlcy5odG1sI1NOU01lc3NhZ2VBdHRyaWJ1dGVzLkRhdGFUeXBlc1xuICAgKi9cbiAgTlVNQkVSID0gJ051bWJlcicsXG5cbiAgLyoqXG4gICAqIEJpbmFyeSB0eXBlIGF0dHJpYnV0ZXMgY2FuIHN0b3JlIGFueSBiaW5hcnkgZGF0YVxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zbnMvbGF0ZXN0L2RnL3Nucy1tZXNzYWdlLWF0dHJpYnV0ZXMuaHRtbCNTTlNNZXNzYWdlQXR0cmlidXRlcy5EYXRhVHlwZXNcbiAgICovXG4gIEJJTkFSWSA9ICdCaW5hcnknXG59XG5cbi8qKlxuICogQSBtZXNzYWdlIGF0dHJpYnV0ZSB0byBhZGQgdG8gdGhlIFNOUyBtZXNzYWdlXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc25zL2xhdGVzdC9kZy9zbnMtbWVzc2FnZS1hdHRyaWJ1dGVzLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNZXNzYWdlQXR0cmlidXRlIHtcbiAgLyoqXG4gICAqIFRoZSB2YWx1ZSBvZiB0aGUgYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSB2YWx1ZTogYW55O1xuXG4gIC8qKlxuICAgKiBUaGUgZGF0YSB0eXBlIGZvciB0aGUgYXR0cmlidXRlXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nucy9sYXRlc3QvZGcvc25zLW1lc3NhZ2UtYXR0cmlidXRlcy5odG1sI1NOU01lc3NhZ2VBdHRyaWJ1dGVzLkRhdGFUeXBlc1xuICAgKiBAZGVmYXVsdCBkZXRlcm1pbmVkIGJ5IHR5cGUgaW5zcGVjdGlvbiBpZiBwb3NzaWJsZSwgZmFsbGJhY2sgaXMgU3RyaW5nXG4gICAqL1xuICByZWFkb25seSBkYXRhVHlwZT86IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHB1Ymxpc2hpbmcgYSBtZXNzYWdlIHRvIGFuIFNOUyB0b3BpY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFNuc1B1Ymxpc2hQcm9wcyBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlUHJvcHMge1xuXG4gIC8qKlxuICAgKiBUaGUgU05TIHRvcGljIHRoYXQgdGhlIHRhc2sgd2lsbCBwdWJsaXNoIHRvLlxuICAgKi9cbiAgcmVhZG9ubHkgdG9waWM6IHNucy5JVG9waWM7XG5cbiAgLyoqXG4gICAqIFRoZSBtZXNzYWdlIHlvdSB3YW50IHRvIHNlbmQuXG4gICAqXG4gICAqIFdpdGggdGhlIGV4Y2VwdGlvbiBvZiBTTVMsIG1lc3NhZ2VzIG11c3QgYmUgVVRGLTggZW5jb2RlZCBzdHJpbmdzIGFuZFxuICAgKiBhdCBtb3N0IDI1NiBLQiBpbiBzaXplLlxuICAgKiBGb3IgU01TLCBlYWNoIG1lc3NhZ2UgY2FuIGNvbnRhaW4gdXAgdG8gMTQwIGNoYXJhY3RlcnMuXG4gICAqL1xuICByZWFkb25seSBtZXNzYWdlOiBzZm4uVGFza0lucHV0O1xuXG4gIC8qKlxuICAgKiBBZGQgbWVzc2FnZSBhdHRyaWJ1dGVzIHdoZW4gcHVibGlzaGluZy5cbiAgICpcbiAgICogVGhlc2UgYXR0cmlidXRlcyBjYXJyeSBhZGRpdGlvbmFsIG1ldGFkYXRhIGFib3V0IHRoZSBtZXNzYWdlIGFuZCBtYXkgYmUgdXNlZFxuICAgKiBmb3Igc3Vic2NyaXB0aW9uIGZpbHRlcnMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nucy9sYXRlc3QvZGcvc25zLW1lc3NhZ2UtYXR0cmlidXRlcy5odG1sXG4gICAqIEBkZWZhdWx0IHt9XG4gICAqL1xuICByZWFkb25seSBtZXNzYWdlQXR0cmlidXRlcz86IHsgW2tleTogc3RyaW5nXTogTWVzc2FnZUF0dHJpYnV0ZSB9O1xuXG4gIC8qKlxuICAgKiBTZW5kIGRpZmZlcmVudCBtZXNzYWdlcyBmb3IgZWFjaCB0cmFuc3BvcnQgcHJvdG9jb2wuXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCB5b3UgbWlnaHQgd2FudCB0byBzZW5kIGEgc2hvcnRlciBtZXNzYWdlIHRvIFNNUyBzdWJzY3JpYmVyc1xuICAgKiBhbmQgYSBtb3JlIHZlcmJvc2UgbWVzc2FnZSB0byBlbWFpbCBhbmQgU1FTIHN1YnNjcmliZXJzLlxuICAgKlxuICAgKiBZb3VyIG1lc3NhZ2UgbXVzdCBiZSBhIEpTT04gb2JqZWN0IHdpdGggYSB0b3AtbGV2ZWwgSlNPTiBrZXkgb2ZcbiAgICogXCJkZWZhdWx0XCIgd2l0aCBhIHZhbHVlIHRoYXQgaXMgYSBzdHJpbmdcbiAgICogWW91IGNhbiBkZWZpbmUgb3RoZXIgdG9wLWxldmVsIGtleXMgdGhhdCBkZWZpbmUgdGhlIG1lc3NhZ2UgeW91IHdhbnQgdG9cbiAgICogc2VuZCB0byBhIHNwZWNpZmljIHRyYW5zcG9ydCBwcm90b2NvbCAoaS5lLiBcInNxc1wiLCBcImVtYWlsXCIsIFwiaHR0cFwiLCBldGMpXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nucy9sYXRlc3QvYXBpL0FQSV9QdWJsaXNoLmh0bWwjQVBJX1B1Ymxpc2hfUmVxdWVzdFBhcmFtZXRlcnNcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG1lc3NhZ2VQZXJTdWJzY3JpcHRpb25UeXBlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVXNlZCBhcyB0aGUgXCJTdWJqZWN0XCIgbGluZSB3aGVuIHRoZSBtZXNzYWdlIGlzIGRlbGl2ZXJlZCB0byBlbWFpbCBlbmRwb2ludHMuXG4gICAqIFRoaXMgZmllbGQgd2lsbCBhbHNvIGJlIGluY2x1ZGVkLCBpZiBwcmVzZW50LCBpbiB0aGUgc3RhbmRhcmQgSlNPTiBtZXNzYWdlc1xuICAgKiBkZWxpdmVyZWQgdG8gb3RoZXIgZW5kcG9pbnRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHN1YmplY3RcbiAgICovXG4gIHJlYWRvbmx5IHN1YmplY3Q/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSBTdGVwIEZ1bmN0aW9ucyBUYXNrIHRvIHB1Ymxpc2ggbWVzc2FnZXMgdG8gU05TIHRvcGljLlxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIFNuc1B1Ymxpc2ggZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZSB7XG5cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU1VQUE9SVEVEX0lOVEVHUkFUSU9OX1BBVFRFUk5TOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuW10gPSBbXG4gICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTixcbiAgXTtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza01ldHJpY3M6IHNmbi5UYXNrTWV0cmljc0NvbmZpZyB8IHVuZGVmaW5lZDtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llczogaWFtLlBvbGljeVN0YXRlbWVudFtdIHwgdW5kZWZpbmVkO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFNuc1B1Ymxpc2hQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuID8/IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRTtcblxuICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgU25zUHVibGlzaC5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuXG4gICAgaWYgKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID09PSBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4pIHtcbiAgICAgIGlmICghc2ZuLkZpZWxkVXRpbHMuY29udGFpbnNUYXNrVG9rZW4ocHJvcHMubWVzc2FnZSkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUYXNrIFRva2VuIGlzIHJlcXVpcmVkIGluIGBtZXNzYWdlYCBVc2UgSnNvblBhdGgudGFza1Rva2VuIHRvIHNldCB0aGUgdG9rZW4uJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy50YXNrUG9saWNpZXMgPSBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnc25zOlB1Ymxpc2gnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5wcm9wcy50b3BpYy50b3BpY0Fybl0sXG4gICAgICB9KSxcbiAgICBdO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIHRoZSBTTlMgUHVibGlzaCBzZXJ2aWNlIGludGVncmF0aW9uIHRhc2sgY29uZmlndXJhdGlvblxuICAgKi9cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHJvdGVjdGVkIF9yZW5kZXJUYXNrKCk6IGFueSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFJlc291cmNlOiBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdzbnMnLCAncHVibGlzaCcsIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSxcbiAgICAgIFBhcmFtZXRlcnM6IHNmbi5GaWVsZFV0aWxzLnJlbmRlck9iamVjdCh7XG4gICAgICAgIFRvcGljQXJuOiB0aGlzLnByb3BzLnRvcGljLnRvcGljQXJuLFxuICAgICAgICBNZXNzYWdlOiB0aGlzLnByb3BzLm1lc3NhZ2UudmFsdWUsXG4gICAgICAgIE1lc3NhZ2VTdHJ1Y3R1cmU6IHRoaXMucHJvcHMubWVzc2FnZVBlclN1YnNjcmlwdGlvblR5cGUgPyAnanNvbicgOiB1bmRlZmluZWQsXG4gICAgICAgIE1lc3NhZ2VBdHRyaWJ1dGVzOiByZW5kZXJNZXNzYWdlQXR0cmlidXRlcyh0aGlzLnByb3BzLm1lc3NhZ2VBdHRyaWJ1dGVzKSxcbiAgICAgICAgU3ViamVjdDogdGhpcy5wcm9wcy5zdWJqZWN0LFxuICAgICAgfSksXG4gICAgfTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgTWVzc2FnZUF0dHJpYnV0ZVZhbHVlIHtcbiAgRGF0YVR5cGU6IHN0cmluZztcbiAgU3RyaW5nVmFsdWU/OiBzdHJpbmc7XG4gIEJpbmFyeVZhbHVlPzogc3RyaW5nO1xufVxuXG5mdW5jdGlvbiByZW5kZXJNZXNzYWdlQXR0cmlidXRlcyhhdHRyaWJ1dGVzPzogeyBba2V5OiBzdHJpbmddOiBNZXNzYWdlQXR0cmlidXRlIH0pOiBhbnkge1xuICBpZiAoYXR0cmlidXRlcyA9PT0gdW5kZWZpbmVkKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgY29uc3QgcmVuZGVyZWRBdHRyaWJ1dGVzOiB7IFtrZXk6IHN0cmluZ106IE1lc3NhZ2VBdHRyaWJ1dGVWYWx1ZSB9ID0ge307XG4gIE9iamVjdC5lbnRyaWVzKGF0dHJpYnV0ZXMpLm1hcCgoW2tleSwgdmFsXSkgPT4ge1xuICAgIHJlbmRlcmVkQXR0cmlidXRlc1trZXldID0gcmVuZGVyTWVzc2FnZUF0dHJpYnV0ZVZhbHVlKHZhbCk7XG4gIH0pO1xuICByZXR1cm4gc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHJlbmRlcmVkQXR0cmlidXRlcykudmFsdWU7XG59XG5cbmZ1bmN0aW9uIHJlbmRlck1lc3NhZ2VBdHRyaWJ1dGVWYWx1ZShhdHRyaWJ1dGU6IE1lc3NhZ2VBdHRyaWJ1dGUpOiBNZXNzYWdlQXR0cmlidXRlVmFsdWUge1xuICBjb25zdCBkYXRhVHlwZSA9IGF0dHJpYnV0ZS5kYXRhVHlwZTtcbiAgaWYgKGF0dHJpYnV0ZS52YWx1ZSBpbnN0YW5jZW9mIHNmbi5UYXNrSW5wdXQpIHtcbiAgICByZXR1cm4ge1xuICAgICAgRGF0YVR5cGU6IGRhdGFUeXBlID8/IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZS5TVFJJTkcsXG4gICAgICBTdHJpbmdWYWx1ZTogZGF0YVR5cGUgIT09IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZS5CSU5BUlkgPyBhdHRyaWJ1dGUudmFsdWUudmFsdWUgOiB1bmRlZmluZWQsXG4gICAgICBCaW5hcnlWYWx1ZTogZGF0YVR5cGUgPT09IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZS5CSU5BUlkgPyBhdHRyaWJ1dGUudmFsdWUudmFsdWUgOiB1bmRlZmluZWQsXG4gICAgfTtcbiAgfVxuXG4gIGlmIChkYXRhVHlwZSA9PT0gTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLkJJTkFSWSkge1xuICAgIHJldHVybiB7IERhdGFUeXBlOiBkYXRhVHlwZSwgQmluYXJ5VmFsdWU6IGAke2F0dHJpYnV0ZS52YWx1ZX1gIH07XG4gIH1cblxuICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGF0dHJpYnV0ZS52YWx1ZSkpIHtcbiAgICByZXR1cm4geyBEYXRhVHlwZTogZGF0YVR5cGUgPz8gTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLlNUUklORywgU3RyaW5nVmFsdWU6IGF0dHJpYnV0ZS52YWx1ZSB9O1xuICB9XG5cbiAgdmFsaWRhdGVNZXNzYWdlQXR0cmlidXRlKGF0dHJpYnV0ZSk7XG4gIGlmIChBcnJheS5pc0FycmF5KGF0dHJpYnV0ZS52YWx1ZSkpIHtcbiAgICByZXR1cm4geyBEYXRhVHlwZTogTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLlNUUklOR19BUlJBWSwgU3RyaW5nVmFsdWU6IEpTT04uc3RyaW5naWZ5KGF0dHJpYnV0ZS52YWx1ZSkgfTtcbiAgfVxuICBjb25zdCB2YWx1ZSA9IGF0dHJpYnV0ZS52YWx1ZTtcbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gJ251bWJlcicpIHtcbiAgICByZXR1cm4geyBEYXRhVHlwZTogTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLk5VTUJFUiwgU3RyaW5nVmFsdWU6IGAke3ZhbHVlfWAgfTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4geyBEYXRhVHlwZTogTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLlNUUklORywgU3RyaW5nVmFsdWU6IGAke3ZhbHVlfWAgfTtcbiAgfVxufVxuXG5mdW5jdGlvbiB2YWxpZGF0ZU1lc3NhZ2VBdHRyaWJ1dGUoYXR0cmlidXRlOiBNZXNzYWdlQXR0cmlidXRlKTogdm9pZCB7XG4gIGNvbnN0IGRhdGFUeXBlID0gYXR0cmlidXRlLmRhdGFUeXBlO1xuICBjb25zdCB2YWx1ZSA9IGF0dHJpYnV0ZS52YWx1ZTtcbiAgaWYgKGRhdGFUeXBlID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm47XG4gIH1cbiAgaWYgKEFycmF5LmlzQXJyYXkodmFsdWUpKSB7XG4gICAgaWYgKGRhdGFUeXBlICE9PSBNZXNzYWdlQXR0cmlidXRlRGF0YVR5cGUuU1RSSU5HX0FSUkFZKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFJlcXVlc3RlZCBTTlMgbWVzc2FnZSBhdHRyaWJ1dGUgdHlwZSB3YXMgJHtkYXRhVHlwZX0gYnV0ICR7dmFsdWV9IHdhcyBvZiB0eXBlIEFycmF5YCk7XG4gICAgfVxuICAgIGNvbnN0IHZhbGlkQXJyYXlUeXBlcyA9IFsnc3RyaW5nJywgJ2Jvb2xlYW4nLCAnbnVtYmVyJ107XG4gICAgdmFsdWUuZm9yRWFjaCgodikgPT4ge1xuICAgICAgaWYgKHYgIT09IG51bGwgfHwgIXZhbGlkQXJyYXlUeXBlcy5pbmNsdWRlcyh0eXBlb2YgdikpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBSZXF1ZXN0ZWQgU05TIG1lc3NhZ2UgYXR0cmlidXRlIHR5cGUgd2FzICR7dHlwZW9mIHZhbHVlfSBidXQgQXJyYXkgdmFsdWVzIG11c3QgYmUgb25lIG9mICR7dmFsaWRBcnJheVR5cGVzfWApO1xuICAgICAgfVxuICAgIH0pO1xuICAgIHJldHVybjtcbiAgfVxuICBjb25zdCBlcnJvciA9IG5ldyBFcnJvcihgUmVxdWVzdGVkIFNOUyBtZXNzYWdlIGF0dHJpYnV0ZSB0eXBlIHdhcyAke2RhdGFUeXBlfSBidXQgJHt2YWx1ZX0gd2FzIG9mIHR5cGUgJHt0eXBlb2YgdmFsdWV9YCk7XG4gIHN3aXRjaCAodHlwZW9mIHZhbHVlKSB7XG4gICAgY2FzZSAnc3RyaW5nJzpcbiAgICAgIC8vIHRydXN0IHRoZSB1c2VyIG9yIHdpbGwgZGVmYXVsdCB0byBzdHJpbmdcbiAgICAgIGlmIChzZm4uSnNvblBhdGguaXNFbmNvZGVkSnNvblBhdGgoYXR0cmlidXRlLnZhbHVlKSkge1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgICBpZiAoZGF0YVR5cGUgPT09IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZS5TVFJJTkcgfHxcbiAgICAgICAgZGF0YVR5cGUgPT09IE1lc3NhZ2VBdHRyaWJ1dGVEYXRhVHlwZS5CSU5BUlkpIHtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuICAgICAgdGhyb3cgZXJyb3I7XG4gICAgY2FzZSAnbnVtYmVyJzpcbiAgICAgIGlmIChkYXRhVHlwZSA9PT0gTWVzc2FnZUF0dHJpYnV0ZURhdGFUeXBlLk5VTUJFUikgeyByZXR1cm47IH1cbiAgICAgIHRocm93IGVycm9yO1xuICAgIGNhc2UgJ2Jvb2xlYW4nOlxuICAgICAgaWYgKGRhdGFUeXBlID09PSBNZXNzYWdlQXR0cmlidXRlRGF0YVR5cGUuU1RSSU5HKSB7IHJldHVybjsgfVxuICAgICAgdGhyb3cgZXJyb3I7XG4gICAgZGVmYXVsdDpcbiAgICAgIHRocm93IGVycm9yO1xuICB9XG59XG4iXX0=