"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const destinations = require("../lib");
// Test success case with:
// 1. Invoke first function in the chain
//   aws lambda invoke --function-name <first function name> --invocation-type Event --payload '"OK"' response.json
// 2. Check logs of third function (should show 'Event: "OK"')
//   aws logs filter-log-events --log-group-name /aws/lambda/<third function name>
//
// Test failure case with:
// 1. Invoke first function in the chain
//   aws lambda invoke --function-name <first function name> --invocation-type Event --payload '"error"' response.json
// 2. Check logs of error function (should show 'Event: {"errorType": "Error", "errorMessage": "UnkownError", "trace":"..."}')
//   aws logs filter-log-events --log-group-name /aws/lambda/<error function name>
class TestStack extends core_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const lambdaProps = {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        console.log('Event: %j', event);
        if (event === 'error') throw new Error('UnkownError');
        return event;
      };`),
        };
        const first = new lambda.Function(this, 'First', lambdaProps);
        const second = new lambda.Function(this, 'Second', lambdaProps);
        const third = new lambda.Function(this, 'Third', lambdaProps);
        const error = new lambda.Function(this, 'Error', lambdaProps);
        first.configureAsyncInvoke({
            onSuccess: new destinations.LambdaDestination(second, { responseOnly: true }),
            onFailure: new destinations.LambdaDestination(error, { responseOnly: true }),
            retryAttempts: 0,
        });
        second.configureAsyncInvoke({
            onSuccess: new destinations.LambdaDestination(third, { responseOnly: true }),
        });
        new core_1.CfnOutput(this, 'FirstFunctionName', { value: first.functionName });
        new core_1.CfnOutput(this, 'ThirdFunctionName', { value: third.functionName });
        new core_1.CfnOutput(this, 'ErrorFunctionName', { value: error.functionName });
    }
}
const app = new core_1.App();
new TestStack(app, 'aws-cdk-lambda-chain');
app.synth();
//# sourceMappingURL=data:application/json;base64,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