/////////////////////////////////////////////////////////////////////////////
// This C file has been created automatically. Do not edit!!!
/////////////////////////////////////////////////////////////////////////////

/** @file gen_leech.c
The functions in file ``gen_leech.c`` implement operations on the
vectors of the Leech lattice modulo 2 and on the
subgroup \f$Q_{x0}\f$. We use the terminology defined in
the document *The C interface of the mmgroup project*, 
section *Description of the mmgroup.generators extension*.
*/


/*************************************************************************
** External references 
*************************************************************************/

/// @cond DO_NOT_DOCUMENT 
#include <string.h>
#include "mat24_functions.h"
#define MMGROUP_GENERATORS_INTERN
#include "mmgroup_generators.h"
/// @endcond 




// %%EXPORT_KWD MAT24_API


// %%GEN ch
#ifdef __cplusplus
extern "C" {
#endif
// %%GEN c


//  %%GEN h
//  %%GEN c



/*************************************************************************
*** Multiplication and exponentiation in the group Q_{x0}
*************************************************************************/


/**
  @brief Return product of two elements the group \f$Q_{x0}\f$.

  Here all elements of the group \f$Q_{x0}\f$ are encoded in
  ``Leech lattice encoding``. The function returns the
  product of the elements ``x1`` and ``x2`` of \f$Q_{x0}\f$.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_mul(uint32_t x1, uint32_t x2)
{
    uint_fast32_t scalar;      
    scalar = (x2 >> 12) &  x1 & 0xfff;
    scalar ^= scalar >> 6;
    scalar ^= scalar >> 3;
    scalar = (0x96 >> (scalar & 7)) & 1;
    x1 ^= x2 ^ (scalar << 24);
    return x1;
}


/**
  @brief Return power of element the group \f$Q_{x0}\f$.

  Here all elements of the group \f$Q_{x0}\f$ are encoded in
  ``Leech lattice encoding``. The function returns the
  power ``x1**e`` of the element ``x1`` of \f$Q_{x0}\f$.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_pow(uint32_t x1, uint32_t e)
{
    uint_fast32_t scalar = 0; 
    x1 &= 0x1ffffff;
    if (e & 2) {
        scalar = (x1 >> 12) &  x1 & 0xfff;
        scalar ^= scalar >> 6;
        scalar ^= scalar >> 3;
        scalar = ((0x96 >> (scalar & 7)) & 1) << 24;
    }
    return (e & 1) ? x1 ^ scalar : scalar;
}



/**
  @brief Return scalar product in the Leech lattice modulo 2.

  Here all elements of Leech lattice modulo 2 are encoded in
  ``Leech lattice encoding``. The function returns the
  scalar product of the vectors ``x1`` and ``x2`` in the
  Leech lattice modulo 2, which may be 0 or 1.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_scalprod(uint32_t x1, uint32_t x2)
{
    uint_fast32_t scalar; 
    scalar = (((x1 >> 12) & x2) ^ ((x2 >> 12) & x1)) & 0xfff;    
    scalar ^= scalar >> 6;
    scalar ^= scalar >> 3;
    return (0x96 >> (scalar & 7)) & 1;
}



/*************************************************************************
*** Auxiliary functions for computing subtype of vector
*************************************************************************/

/// @cond DO_NOT_DOCUMENT 

/**
  @brief Return suboctad type.

  Let ``octad`` be an octad, i.e. a bit vector of length 8. Let
  w = 1 (mod 2) if ``octad`` denotes an octad and ``w = 0`` 
  if ``octad`` denotes a complemented octad. Let ``coc`` be
  an even cocode vector in cocode representation.

  The function returns 0 in bit 1 of the return value if the 
  cocode word ``coc`` can be written as a subset of the octad,
  and 1 in bit 1 otherwise.

  the function returns ``1 + w + bit_weight(coc)/2`` in bit 0
  of the return value.

  Then Leech lattice vector ``x_octad + x_coc`` is of subtype 
  0x22 if the return value is zero. Otherwise it is of
  subtype 0x44 (or 0x46) if bit 1 of the return value is 0
  (or 1).
*/
static inline int suboctad_type(
    uint_fast32_t octad, // An octad as a bit vector of length 8
    uint_fast32_t w,     // 1 for an octad, 0 for acomplemented octad
    uint_fast32_t coc    // even cocode vector in cocode representation       
)
{
    uint_fast32_t cw, lsb, syn, cocodev, sub;

    // Let cw be the halved bit weight of coc
    cw = MAT24_SYNDROME_TABLE[coc & 0x7ff] >> 15;
    // Put cocodev = cocode word of v (in vector rep), such 
    // that the cocode word is a suboctad of octad if possible.
    lsb = mat24_def_lsbit24(octad);
    coc ^= MAT24_RECIP_BASIS[lsb];
    syn = MAT24_SYNDROME_TABLE[coc & 0x7ff];
    cocodev = mat24_def_syndrome_from_table(syn) ^ (1UL << lsb);
    // Set sub = 0 iff cocodev is a subset of octad.
    sub = (octad & cocodev) != cocodev;
    return ((~w ^ cw) & 1) + 2 * sub;
}

/// @endcond  


/*************************************************************************
*** Subtype of a vector in the Leech lattice mod 2
*************************************************************************/



/**
  @brief Return subtype of vector in Leech lattice mod 2

  The function returns the subtype of the vector \f$v_2\f$
  in the Leech lattice modulo 2 as a BCD-coded two-digit
  integer. \f$v_2\f$ must be given in Leech lattice encoding.
  
  The subtype of a vector in the Leech lattice mod 2 is 
  defined in **The mmgroup guide for developers**, 
  section **Computations in the Leech lattice modulo 2**.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_type(uint64_t v2)
// Return ``0x10 * type(v2) + subtype(v2)`` for a vector ``x``
// in the Leech lattice modulo ``2``.
// The type of a vector is half the length of the shortest 
// representative of a vector in the Leech lattice.
// The subtype decsribes the double coset of the vector in
// in ``N_x \ Co_1 / N_x``, where ``Co``_1 is the automorphism 
// group of the Leech lattice modulo ``2``, and ``N_x`` is the 
// automorphism group of a vector ``(2,...,2)`` in the standard 
// basis of the Leech lattice modulo ``N_x`` has structure 
// ``2^11. Mat_24``, with ``Mat_24`` the Mathieu group operating
// on 24 elements.
//
// In :cite`:`Con85` a vector in the Leech lattice is
// denotes by ``x_d * x_delta``, where ``d`` is an element of
// the Golay code and ``delta`` is an element of it cocode. 
// Subtypes can be described as follows:
//
// 0: The set contains an element with ``delta = 0``
// 1: ``delta`` has weight ``1`` for all elements in the set.
// 2: ``d`` or its complement is an octad and ``delta`` is
//     even and contained in that octad.
// 3: ``delta`` has weight ``1`` for all elements in the set.  
// 4: ``d`` or its complement is an octad and ``delta`` is
//     even and not contained in that octad.
// 6: ``d`` has weight 12
// 8: The singleton containing ``(2,...,2)`` 
//
// Valid combinations of type and subtype (hex) are
//
// 00,   20, 21, 22,   31, 33, 34, 36,  
// 40, 42, 43, 44, 46, 48.
{
    static uint8_t tab_odd[4] = {
        0x21, 0x31, 0x43, 0x33
    };
    static uint8_t tab_even_scalar1[7] = {
        0xff, 0xff, 0x34, 0x36, 0x34, 0xff, 0xff
    };
    uint_fast32_t theta, coc, scalar, syn, cw, w, octad;

    theta = MAT24_THETA_TABLE[(v2 >> 12) & 0x7ff]; 
    // put syn = syndrome table[cocode word]
    coc = (v2 ^ theta) & 0xfff;
    // Put scalar = scalar product (code, cocode)
    scalar = (v2 >> 12) &  v2 & 0xfff;
    scalar ^= scalar >> 6;
    scalar ^= scalar >> 3;
    scalar = (0x96 >> (scalar & 7)) & 1;

    syn =  MAT24_SYNDROME_TABLE[coc & 0x7ff];

    
    // Deal with odd cocode first
    if (v2 & 0x800) {
        // Let cw be the weight of the cocode word
        cw =  3 - ((((syn & 0x7fff) + 0x2000) >> 15) << 1);
        return tab_odd[cw-1+scalar]; 
    }

    // Put w = weight(Golay code word of v2) / 4
    w = 0 - ((v2 >> 23) & 1);
    w = (((theta >> 12) & 7) ^ w) + (w & 7);  

    // Let cw be the weight of the cocode word (modulo 4)
    cw = (syn >> 15) << 1; 


    // Deal with even cocode and scalar == 1
    if (scalar) return tab_even_scalar1[w];
   
    switch (w) {
        case 6:
            if (coc == 0) return 0x48;
        case 0: 
            // Compute the exact cocode weight in cw
            cw = (4 - cw) & ((0UL - coc) >> 16);
            return cw << 4;
        case 3:
            return 0x46;
        case 4:
            // Add 1....1 to bit vector x (in vector rep)
            v2 ^= 0x800000; 
        default:   // This must be case 2:
            // Compute subtype with function suboctad_type()
            octad = mat24_def_gcode_to_vect(v2 >> 12); 
            w = suboctad_type(octad, w >> 1, coc);
            return (0x44444222 >> (8*w)) & 0xff;
    }
}



/*************************************************************************
*** Subtype of a type-2 vector in the Leech lattice mod 2
*************************************************************************/


/**
  @brief Compute subtype if vector in Leech lattice mod 2 is of type 2

  This function returns the subtype ofthe vector \f$v_2\f$ in the 
  Leech lattice modulo 2 if \f$v_2\f$ is of type 2 and 0 otherwise.

  It is faster than function ``gen_leech2_type()``.
  
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_type2(uint64_t v2)
{
    uint_fast32_t theta, syn, scalar, w, coc, octad;
    // Deal with odd cocode words
    if (v2 & 0x800) {   // Deal with odd cocode words
         // Let syn be the syndrome table entry for the cocode part
         theta = MAT24_THETA_TABLE[(v2 >> 12) & 0x7ff];
         syn = MAT24_SYNDROME_TABLE[(theta ^ v2) & 0x7ff];
         // Return 0 if syn does not encode a cocode word of length 1
         if ((syn & 0x3ff) < (24 << 5)) return 0;
         // Put scalar = scalar product <code, cocode> + 1   (mod 2)
         scalar = (v2 >> 12) &  v2 & 0xfff;
         scalar ^= scalar >> 6;
         scalar ^= scalar >> 3;
         scalar = (0x69 >> (scalar & 7)) & 1;
         // Return type 0x21 if scalar product is 0, and 0 otherwise
         return 0x21 & (0 - scalar);
    }
    // Deal with Golay code word 0
    if ((v2 & 0x7ff000L) == 0) {
         // Let syn be the syndrome table entry for the cocode part 
         syn = MAT24_SYNDROME_TABLE[v2 & 0x7ff];
         // Return 0x20 if syn encodes a cocode word of length 2
         // and 0 otherwise
         return 0x20 & (0 - ((syn >> 15) & 1));
    }

    theta = MAT24_THETA_TABLE[(v2 >> 12) & 0x7ff];

    // Here (theta >> 12) & 7 is the bit weight of 
    // v2 + Omega * b23, where b23 is bit 23 of v2.
    // If bit 12 of theta is odd then v2 is a dodecad
    // and hence never short.
    if (theta & 0x1000) return 0;
    // Now v2 is a (possibly complemented) octad.
    // Put w = 1 if v2 is an octad and w = 0 otherwise.
    w = ((theta >> 13) ^ (v2 >> 23)) & 1;
    // XOR v2 with Omega if it has weight 16
    v2 ^= (1 - w) << 23;
    // Let coc be the cocode part of v2 (this is even)
    coc = (v2 ^ theta) & 0x7ff;
    // Compute subtype with function suboctad_type() 
    octad = mat24_def_gcode_to_vect(v2 >> 12); 
    return suboctad_type(octad, w, coc) ? 0 : 0x22;
}



/// @cond DO_NOT_DOCUMENT 
// In the following procedure we'll store 1 << LSTEPS vectors for speed up
#define LSTEPS 6
/// @endcond  


/**
  @brief Count type-2 vectors in an affine subspace of the Leech lattice mod 2

  This function returns the number of type-2 vectors in an affine 
  subspace \f$V\f$  of the Leech lattice mod 2. Subspace \f$V\f$ is
  defined by an array \f$a\f$ of length \f$n\f$ of bit vectors. 
  If \f$a_1,\ldots,a_{n-1}\f$ are linear independent then \f$V\f$ is
  given by:

  \f$V = \{a_0 + \sum_{i=1}^{n-1} \lambda_i a_i \mid \lambda_i=0,1\}\f$.
  
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_count_type2(uint32_t *a, uint32_t n)
{
    uint32_t b[1 << LSTEPS];
    uint_fast32_t count, i, j, l, n0, bigsteps, v;
    
    if (n == 0) return 0;
    b[0] = a[0];
    ++a; --n;
    n0 = n  < LSTEPS ? n : LSTEPS;
    bigsteps = 1 << (n - n0);
    l = 1;  
    for (i = 0; i < n0; ++i) {
        for (j = 0; j < l; ++j) b[l + j] = b[j] ^ a[i];
        l += l;
    }
    a += n0;
    count = v = 0;
    for (i = 1; ; ++i) {
         for (j = 0; j < l; ++j) {
             count += gen_leech2_type2(v ^ b[j]) > 0;
         }
         if (i == bigsteps) break;
         v ^= a[mat24_def_lsbit24(i)];
    }
    return count;
}


#undef LSTEPS

/*************************************************************************
*** Auxiliary functions for reducing a certain type-2 vector  mod 2
*************************************************************************/

/**
  @brief Auxiliary function for function gen_leech2_reduce_type4
  
  The function returns the subtype of a vector \f$v\f$ of type 2 in
  the Leech lattice modulo 2, provided that \f$v + \beta\f$ is of
  type 4. It returns 0 in the special case \f$v = \beta + \Omega\f$
  and a negative value if  \f$v\f$ is not of type 2 or \f$v + \beta\f$
  is not of type 4.

  It is used for rotating a type-4 vector \f$v\f$ which is orthogonal
  to \f$\beta\f$ (in the real Leech lattice) into the \f$v + \Omega\f$.
  That rotation will fix the special short vector \f$\beta\f$.
*/
// %%EXPORT px
MAT24_API
int32_t gen_leech2_start_type24(uint32_t v)
{
    uint_fast32_t vtype, theta, syn, w, coc, octad;

    // Return -1 if the scalar product of v and \beta is -1
    if (v & 0x200000) return -1;
    switch (vtype = gen_leech2_type2(v)) {
        case 0x21:
            // Put cocode vector index into variable 'syn'
            theta = MAT24_THETA_TABLE[(v >> 12) & 0x7ff];
            syn = MAT24_SYNDROME_TABLE[(theta ^ v) & 0x7ff];
            // Return 0 if that index is 2 or 3, and 0x21 otherwise
            return (syn & 0x1e) == 2 ? -1 : 0x21;
        case 0x20:
            // Return -1 (or 0) if v is \beta (or \beta + \Omega)
            if ((v & 0x7fffff) == 0x200)
                return v & 0x800000 ? 0 : -1;
            // Put weight of cocode vector of v + \beta into 
            // variable 'syn'
            theta = MAT24_THETA_TABLE[(v >> 12) & 0x7ff];
            syn = MAT24_SYNDROME_TABLE[(theta ^ v ^ 0x200) & 0x7ff];
            // Return -1 if weight is 2 and vtype otherwise
            return syn & 0x8000 ? -1 : vtype;
        case 0x22:
            // Now v is a (possibly complemented) octad.
            // Put w = 1 if v is an octad and w = 0 otherwise.
            theta = MAT24_THETA_TABLE[(v >> 12) & 0x7ff];
            w = ((theta >> 13) ^ (v >> 23)) & 1;
            // XOR v with Omega if it has weight 16
            v ^= (1 - w) << 23;
            // Let coc be the cocode part of v ^ \beta (this is even)
            coc = (v ^ theta ^ 0x200) & 0x7ff;
            // Compute subtype with function suboctad_type() 
            octad = mat24_def_gcode_to_vect(v >> 12); 
            return suboctad_type(octad, w, coc) ? 0x22 : -1;
        default:
            // Return -1 if v s not of type 2.
            return -1;
    }
}


/*************************************************************************
*** Auxiliary functions for reducing a type-4 vector  mod 2
*************************************************************************/

/**
  @brief Auxiliary function for function gen_leech2_reduce_type4
  

  The function returns the subtype of a vector \f$v\f$ of type 4 in
  the Leech lattice modulo 2. Parameter \f$v\f$ must a vector of
  type 4 in Leech lattice encoding. The function returns the subtype
  of \f$v\f$ that will be used for reduction in
  function ``gen_leech2_reduce_type4``.

  This function takes care of the special vectors \f$\Omega\f$
  and \f$\beta\f$ the Leech lattice modulo 2.

  It is used for rotating a type-4 vector \f$v\f$ into 
  \f$\Omega\f$. If this is possible, that rotation will fix the
  special short vector \f$\beta\f$.

  Therefore the function returns 0 in case \f$v = \Omega\f$. It returns 
  the subtype of  \f$v + \beta\f$ if \f$\beta\f$ and \f$v + \beta\f$
  are of type 2 and orthogonal in the real Leech lattice.

  The function returns a negative value if \f$v\f$ is not of type 4.
*/
// %%EXPORT px
MAT24_API
int32_t gen_leech2_start_type4(uint32_t v)
{
    uint_fast32_t syn, scalar, theta, w, octad, coc, sub;

    v &= 0xffffff;
    if ((v & 0x7ff800) == 0) {
        // Then v or v + Omega is an even cocode element.
        // Return 0 if v == Omega and -1 if v ==  0.
        if ((v & 0x7fffff) == 0) {
            return v & 0x800000 ? 0 : -1;
        }
        // Let w be the cocode weight. Return -2 if w == 2.
        syn = MAT24_SYNDROME_TABLE[coc = v & 0x7ff];
        if (syn & 0x8000)  return -2;
        // Here v has type 4. Let beta be the standard type-2 vector.
        // Return 0x20 if v ^ beta has type 2 and 0x40 otherwise.
        syn = MAT24_SYNDROME_TABLE[coc ^ 0x200];
        return (syn & 0x8000)  ? 0x20 : 0x40;
    }
    // Compute type of v (mod 2) in scalar; return -3 if this is odd
    scalar = (v >> 12) &  v & 0xfff;
    scalar ^= scalar >> 6;
    scalar ^= scalar >> 3;
    scalar = (0x96 >> (scalar & 7)) & 1;
    if (scalar) return -3;

    // Let syn be the syndrome table entry for the cocode part
    theta = MAT24_THETA_TABLE[(v >> 12) & 0x7ff];
    syn = MAT24_SYNDROME_TABLE[coc = (theta ^ v) & 0x7ff];
 
    if (v & 0x800) {
        // The v has an odd cocode element, encoded in 'syn'.
        // Return -2 if syn encodes a cocode word of length 1,
        // since then v is of type 2.
        if ((syn & 0x3ff) >= (24 << 5)) return -2;
        // Here v is of type 4. 
        // Get syndrome table entry of v ^ beta
        syn = MAT24_SYNDROME_TABLE[coc ^ 0x200];
        // Return 0x21 if v ^ beta is of type 2 and 0x43 otherwise.
        if (((syn & 0x3ff) >= (24 << 5)) && (v & 0x200000) == 0)
            return 0x21;
        return 0x43;
    }

    // Here (theta >> 12) & 7 is the bit weight of  v + Omega * b23, 
    // where b23 is bit 23 of v. If bit 12 of theta is odd then the
    // Golay code part of v is a dodecad, and v has subtype 0x46.
    if (theta & 0x1000) return 0x46;

    // Put w = 1 if v is an octad and w = 0 otherwise.
    w = ((theta >> 13) ^ (v >> 23)) & 1;
    // XOR v with Omega if it has weight 16
    v ^= (1 - w) << 23;
    // Let octad be the Golay code part of v; this is an octad.
    octad = mat24_def_gcode_to_vect(v >> 12); 
    // Let coc be the cocode part of v; this is even.
    coc = (v ^ theta) & 0x7ff;

    sub = suboctad_type(octad, w, coc);
    // Return -2 if v is of type 2
    if (sub == 0) return -2;
    // Otherwise return 0x22 if v + beta is of type 2
    if (suboctad_type(octad, w, coc ^ 0x200) == 0) return 0x22;
    // Otherwise return the subtype of v
    return (0x44444222 >> (8 * sub)) & 0xff;    
}


/************************************************************************
*************************************************************************
*** Leech lattice mod 2
*************************************************************************
*************************************************************************/


/*************************************************************************
*** Operation of monomial generators on the extraspecial group Q{x0}
*************************************************************************/


/// @cond DO_NOT_DOCUMENT 

/**
  @brief Perform operation \f$x_d x_\delta\f$ on \f$Q_{x0}\f$

  The function returns the element \f$q_0 x_d x_\delta\f$ for
  \f${q_0} \in Q_{x0}\f$. Here  parameters ``d`` and 
  ``delta`` are the numbers of the Golay code element \f$d\f$
  and the cocode element \f$\delta\f$ defined in the API reference  
  in section **The Golay code and its cocode**.
  
  Parameter \f${q_0}\f$ and the result are given Leech lattice 
  encoding.
*/
static inline
uint32_t op_x_d_delta(uint32_t q0, uint32_t d, uint32_t delta)
{
    uint32_t s;
    delta ^= MAT24_THETA_TABLE[d & 0x7ff];
    s = ((q0 >> 12) & delta) ^ (q0 & d);
    s ^= s >> 6; s ^= s >> 3;
    s = (0x96 >> (s & 7)) & 1;
    return q0 ^ (s << 24);
}


/**
  @brief Perform operation \f$x_\delta x_\pi\f$ on \f$Q_{x0}\f$

  The function returns the element \f$q_0 x_\delta x_\pi\f$ for
  \f${q_0} \in Q_{x0}\f$. Here  parameter ``pi`` is the number 
  of the permutation \f$\pi\f$  in \f$_{24}\f$, and ``d`` is the
  number of the cocode element \f$\delta x\f$. These numbers are 
  defined in the API reference  in sections **Automorphisms of the 
  Parker loop** and  **The Golay code and its cocode**, respectively.
  
  Parameter \f${q_0}\f$ and the result are given Leech lattice 
  encoding.
*/
static inline
uint32_t op_delta_pi(uint32_t q0, uint32_t delta, uint32_t pi)
{
    uint32_t xd, xdelta;
    uint8_t perm[24];
    uint32_t autpl[12];
    
    xd = (q0 >> 12) & 0x1fff;
    xdelta =  (q0 ^ MAT24_THETA_TABLE[(q0 >> 12) & 0x7ff]) & 0xfff;
    if (pi == 0 || mat24_m24num_to_perm(pi, perm)) {
        xd ^=  mat24_scalar_prod(xd, delta) << 12;
    } else {
        mat24_perm_to_autpl(delta, perm, autpl);
        xd = mat24_op_ploop_autpl(xd, autpl);
        xdelta =  mat24_op_cocode_perm(xdelta, perm);
    }
    return (xd << 12) ^ xdelta ^ (MAT24_THETA_TABLE[xd & 0x7ff] & 0xfff);
}


/**
  @brief Inverse of function ``op_delta_pi``

  We have 
  ``op_delta_pi(op_delta_pi_inv(q0, delta , pi), delta, pi) = q0``.
*/
static inline
uint32_t op_delta_pi_inv(uint32_t q0, uint32_t delta, uint32_t pi)
{
    uint32_t xd, xdelta;
    uint8_t perm[24], inv_perm[24];
    uint32_t inv_autpl[12];
    
    xd = (q0 >> 12) & 0x1fff;
    xdelta =  (q0 ^ MAT24_THETA_TABLE[(q0 >> 12) & 0x7ff]) & 0xfff;
    if (pi == 0 || mat24_m24num_to_perm(pi, perm)) {
        xd ^=  mat24_scalar_prod(xd, delta) << 12;
    } else {
        mat24_perm_to_iautpl(delta, perm, inv_perm, inv_autpl);
        xd = mat24_op_ploop_autpl(xd, inv_autpl);
        xdelta =  mat24_op_cocode_perm(xdelta, inv_perm);
    }
    return (xd << 12) ^ xdelta ^ (MAT24_THETA_TABLE[xd & 0x7ff] & 0xfff);
}

/**
  @brief Perform operation \f$x_d\f$ on \f$Q_{x0}\f$

  The function returns the element \f$q_0 y_d\f$ for
  \f${q_0} \in Q_{x0}\f$. Here  parameter ``d`` is the number
  of the Golay code element \f$d\f$ defined in the API reference  
  in section **The Golay code and its cocode**.
  
  Parameter \f${q_0}\f$ and the result are given Leech lattice 
  encoding.
*/
static inline
uint32_t op_y(uint32_t q0, uint32_t d)
{
    // We use the formula for conjugation of 
    // \f$`\tilde{x}_d x_\delta\f$ with \f$y_e\f$ 
    // in the **guide**, section 
    // **Implementing generators of the Monster group**.
    uint32_t s, o, theta_q0, theta_y, odd, eps;
    odd = 0 - ((q0 >> 11) & 1);
    theta_q0 = MAT24_THETA_TABLE[(q0 >> 12) & 0x7ff];
    theta_y = MAT24_THETA_TABLE[d & 0x7ff];
    s =  (theta_q0 & d) ^ (~odd &  q0 & d); 
    s ^= s >> 6; s ^= s >> 3;
    s = (0x96 >> (s & 7)) & 1;
    o = (theta_y & (q0 >> 12)) ^ (q0 & d);
    o ^= (theta_y >> 12) & 1 & odd;
    o ^= o >> 6; o ^= o >> 3;
    o = (0x96 >> (o & 7)) & 1;
    eps = theta_q0 ^ (theta_y & ~odd) 
           ^  MAT24_THETA_TABLE[((q0 >> 12) ^ d) & 0x7ff]; 
    q0 ^= (eps & 0xfff) ^ ((d << 12) & 0x1fff000 & odd);
    q0 ^= (s << 24) ^ (o << 23);
    return q0;
}

/// @endcond 


/*************************************************************************
*** Conjugating a vector in the extraspecial group 2^{1+24}
*************************************************************************/


/**
  @brief Perform operation of \f$G_{x0}\f$ on \f$Q_{x0}\f$

  The function returns the element \f$g^{-1} q_0 g\f$ for
  \f$q_0 \in Q_{x0}\f$ and \f$g \in G_{x0}\f$. Here \f$g\f$
  is given as a word of genenators of length \f$n\f$ in the 
  array ``pg``. Each atom of the word \f$g\f$ is encoded as 
  defined in the header file ``mmgroup_generators.h``.
  
  Parameter \f${q_0}\f$ and the result are given Leech lattice 
  encoding.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_op_word(uint32_t q0, uint32_t *pg, uint32_t n)
// Conjugate the element ``q0`` of the Pauli group with element 
// ``e`` of the group ``G_{x1}`` with the atom
// given by ``v``. Atom ``v`` is interpreted as follows:
// Bit 31:      sign of exponent
// Bit 30..28   tag
// Bit 27..0    operarand
// Tag are as follows:
//
//                bit
// Tag  word     length   operand
//  0:  1         -       unit of the group, no operand
//  1:  x_delta   12      delta in C* in 'cocode' rep
//  2:  x_pi      28      pi a permutation number  
//  3:  x_d       13      d an element of the parker loop
//  4:  y_d       13      d an element of the parker loop
//  5:  t**e      28      exponent e, legal in special cases only
//  6:  xi**e     28      exponent e
//  7   illegal                  
// 
// 
{
    uint_fast32_t tag, i, v, y;
    static uint8_t  o[2][4] = {{0,2,3,1},{0,3,1,2}};
 
    q0 &= 0x1ffffff;
    for (i = 0; i < n; ++i) {
        v = pg[i];
        tag = v & MMGROUP_ATOM_TAG_ALL;
        v  &= MMGROUP_ATOM_DATA;
        y = 0;
        switch(tag) {
            case MMGROUP_ATOM_TAG_1:
            case MMGROUP_ATOM_TAG_I1:
               break;
            case MMGROUP_ATOM_TAG_ID:
            case MMGROUP_ATOM_TAG_D:
               q0 = op_x_d_delta(q0, 0, v & 0xfff);
               break;
            case MMGROUP_ATOM_TAG_IP:
               q0 = op_delta_pi_inv(q0, 0, v);
               break;
            case MMGROUP_ATOM_TAG_P:
               q0 = op_delta_pi(q0, 0, v);
               break;
            case MMGROUP_ATOM_TAG_IX:
            case MMGROUP_ATOM_TAG_X:
               q0 = op_x_d_delta(q0, v & 0xfff, 0);
               break;
            case MMGROUP_ATOM_TAG_IY:
               y ^= (MAT24_THETA_TABLE[v & 0x7ff] & 0x1000);
            case MMGROUP_ATOM_TAG_Y:
               y ^= v & 0x1fffUL;
               q0 = op_y(q0, y & 0x1fff);
               break;
            case MMGROUP_ATOM_TAG_IT:
               v ^= 0xfffffff;
            case MMGROUP_ATOM_TAG_T:
               v = v % 3;
               if (v) {
                   if (q0 & 0x7fffffUL) return (uint32_t)(0-1UL);
                   q0 = (uint32_t)(o[v-1][(q0 >> 23) & 3]) << 23UL;
               }
               break;
            case MMGROUP_ATOM_TAG_IL:
               v ^= 0xfffffff;
            case MMGROUP_ATOM_TAG_L:
               q0 = gen_xi_op_xi(q0, v);
               break;
            default:
               return (uint32_t)(0-1UL);
        }
    }
    return q0;
}


/**
  @brief Atomic operation of \f$G_{x0}\f$ on \f$Q_{x0}\f$

  Equivalent to ``gen_leech2_op_word(q0, &g, 1)``.
*/
// %%EXPORT px
MAT24_API
uint32_t gen_leech2_op_atom(uint32_t q0, uint32_t g)
{
    return  gen_leech2_op_word(q0, &g, 1);
}



 
/*************************************************************************
*** Self test
*************************************************************************/






//  %%GEN h
//  %%GEN c



// %%GEN ch
#ifdef __cplusplus
}
#endif




