"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamicPackage = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
class DynamicPackage {
    constructor(
    /**
     * Name of the npm package
     * Version to install, or version constraint
     *
     * @default - no version constraint, install the latest version
     */
    packageSpec, props = {}) {
        const { name, version } = this.parsePackageSpec(packageSpec);
        this.name = name;
        this.version = version;
        this.installPath =
            props.installPath ||
                fs_1.mkdtempSync(path_1.join(os_1.tmpdir(), `cdk-dynamic-${this.spec}-`));
        this.searchPaths = props.searchPaths || [];
    }
    get spec() {
        if (!this.version) {
            return this.name;
        }
        return `${this.name}@${this.version}`;
    }
    tryResolve(paths) {
        try {
            return require.resolve(this.name, paths ? { paths } : undefined);
        }
        catch (_) {
            return;
        }
    }
    auto() {
        return this.tryResolve() || this.findInPaths() || this.install();
    }
    nodeJs() {
        return this.name;
    }
    findIn(paths) {
        paths.forEach((p) => process.stderr.write('trying... ' + p + '\n'));
        process.stderr.write('\n');
        return this.tryResolve([...paths].filter(Boolean));
    }
    findInPaths() {
        return (this.findInSearchPaths() ||
            this.findInLocalPaths() ||
            this.findInGlobalPaths());
    }
    findInSearchPaths() {
        return this.findIn(this.searchPaths);
    }
    findInLocalPaths() {
        return this.findIn([
            process.cwd(),
            process.env.PWD,
            path_1.resolve(process.env.PWD || process.cwd(), 'node_modules'),
            path_1.resolve(process.cwd(), 'node_modules'),
        ].filter(Boolean));
    }
    findInGlobalPaths() {
        return this.findIn([
            process.execPath,
            path_1.resolve(process.execPath, '../..'),
            path_1.resolve(process.execPath, '../../lib'),
            path_1.resolve(process.execPath, '../../node_modules'),
            path_1.resolve(process.execPath, '../../lib/node_modules'),
        ]);
    }
    install() {
        return aws_cdk_lib_1.Lazy.string({
            produce: () => {
                if (!DynamicPackage.installedPackagePath.has(this.spec)) {
                    const args = [
                        'install',
                        this.spec,
                        '--no-save',
                        '--prefix',
                        this.installPath,
                    ];
                    DynamicPackage.log(`Dynamically installing ${this.spec} into "${this.installPath}"...`, 'info');
                    child_process_1.execFileSync('npm', args);
                    DynamicPackage.installedPackagePath.set(this.spec, require.resolve(this.name, {
                        paths: [this.installPath],
                    }));
                }
                return DynamicPackage.installedPackagePath.get(this.spec);
            },
        });
    }
    static log(message, _level = 'info') {
        process.stderr.write(`⬥ ${message}\n`);
    }
    parsePackageSpec(spec) {
        const hasScope = spec.startsWith('@');
        if (hasScope) {
            spec = spec.substring(1);
        }
        const [module, ...version] = spec.split('@');
        const name = hasScope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        return { name, version: version?.join('@') };
    }
}
exports.DynamicPackage = DynamicPackage;
DynamicPackage.installedPackagePath = new Map();
//# sourceMappingURL=data:application/json;base64,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