"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${core_1.Names.uniqueId(this)}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: this.handler.resourceArnsForGrantInvoke,
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.string({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_TokenAuthorizerProps(props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_c = props.authorizerName) !== null && _c !== void 0 ? _c : core_1.Names.uniqueId(this),
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_d = props.assumeRole) === null || _d === void 0 ? void 0 : _d.roleArn,
            authorizerResultTtlInSeconds: (_e = props.resultsCacheTtl) === null || _e === void 0 ? void 0 : _e.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
_a = JSII_RTTI_SYMBOL_1;
TokenAuthorizer[_a] = { fqn: "@aws-cdk/aws-apigateway.TokenAuthorizer", version: "1.150.0" };
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_RequestAuthorizerProps(props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_c = props.authorizerName) !== null && _c !== void 0 ? _c : core_1.Names.uniqueId(this),
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_d = props.assumeRole) === null || _d === void 0 ? void 0 : _d.roleArn,
            authorizerResultTtlInSeconds: (_e = props.resultsCacheTtl) === null || _e === void 0 ? void 0 : _e.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
_b = JSII_RTTI_SYMBOL_1;
RequestAuthorizer[_b] = { fqn: "@aws-cdk/aws-apigateway.RequestAuthorizer", version: "1.150.0" };
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    const { region, partition } = core_1.Arn.split(handler.functionArn, core_1.ArnFormat.COLON_RESOURCE_NAME);
    return `arn:${partition}:apigateway:${region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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