"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const vpc_link_1 = require("./vpc-link");
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IntegrationProps(props);
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.timeout && !options.timeout.isUnresolved() && (options.timeout.toMilliseconds() < 50 || options.timeout.toMilliseconds() > 29000)) {
            throw new Error('Integration timeout must be between 50 milliseconds and 29 seconds.');
        }
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_Method(_method);
        let uri = this.props.uri;
        const options = this.props.options;
        if ((options === null || options === void 0 ? void 0 : options.connectionType) === ConnectionType.VPC_LINK && uri === undefined) {
            uri = core_1.Lazy.string({
                // needs to be a lazy since the targets can be added to the VpcLink construct after initialization.
                produce: () => {
                    const vpcLink = options.vpcLink;
                    if (vpcLink instanceof vpc_link_1.VpcLink) {
                        const targets = vpcLink._targetDnsNames;
                        if (targets.length > 1) {
                            throw new Error("'uri' is required when there are more than one NLBs in the VPC Link");
                        }
                        else {
                            return `http://${targets[0]}`;
                        }
                    }
                    else {
                        throw new Error("'uri' is required when the 'connectionType' is VPC_LINK");
                    }
                },
            });
        }
        return {
            options: {
                ...options,
                connectionType: (options === null || options === void 0 ? void 0 : options.vpcLink) ? ConnectionType.VPC_LINK : options === null || options === void 0 ? void 0 : options.connectionType,
            },
            type: this.props.type,
            uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
_a = JSII_RTTI_SYMBOL_1;
Integration[_a] = { fqn: "@aws-cdk/aws-apigateway.Integration", version: "1.150.0" };
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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