"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaMapper = exports.validateDouble = exports.validateInteger = exports.parseAwsApiCall = exports.parseMethodOptionsPath = exports.validateHttpMethod = exports.ALL_METHODS = void 0;
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
function validateDouble(property, messagePrefix) {
    if (property && isNaN(property) && isNaN(parseFloat(property.toString()))) {
        throw new Error(`${messagePrefix} should be an double`);
    }
}
exports.validateDouble = validateDouble;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTBDO0FBQzFDLDRDQUE0QztBQUUvQixRQUFBLFdBQVcsR0FBRyxDQUFDLFNBQVMsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0FBRXhGLE1BQU0sZUFBZSxHQUFHLENBQUMsS0FBSyxFQUFFLEdBQUcsbUJBQVcsQ0FBQyxDQUFDO0FBRWhELFNBQWdCLGtCQUFrQixDQUFDLE1BQWMsRUFBRSxnQkFBd0IsRUFBRTtJQUMzRSxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUNyQyxNQUFNLElBQUksS0FBSyxDQUFDLEdBQUcsYUFBYSx3QkFBd0IsTUFBTSx1QkFBdUIsZUFBZSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7S0FDbkg7QUFDSCxDQUFDO0FBSkQsZ0RBSUM7QUFFRCxTQUFnQixzQkFBc0IsQ0FBQyxZQUFvQjtJQUN6RCxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxZQUFZLEVBQUUsQ0FBQyxDQUFDO0tBQzdFO0lBRUQsTUFBTSxJQUFJLEdBQUcsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjtJQUV6RCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRW5DLElBQUksVUFBVSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDekIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1R0FBdUcsSUFBSSxFQUFFLENBQUMsQ0FBQztLQUNoSTtJQUVELE1BQU0sVUFBVSxHQUFHLFVBQVUsQ0FBQyxHQUFHLEVBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLG1DQUFtQztJQUN2RixJQUFJLFVBQVUsS0FBSyxHQUFHLEVBQUU7UUFDdEIsa0JBQWtCLENBQUMsVUFBVSxFQUFFLEdBQUcsWUFBWSxJQUFJLENBQUMsQ0FBQztLQUNyRDtJQUVELElBQUksWUFBWSxHQUFHLEtBQUssR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pELElBQUksVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsRUFBRTtRQUNwRCxZQUFZLEdBQUcsSUFBSSxDQUFDO0tBQ3JCO1NBQU0sSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxVQUFVLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFO1FBQzFELFlBQVksR0FBRyxHQUFHLENBQUM7S0FDcEI7SUFFRCxPQUFPO1FBQ0wsVUFBVTtRQUNWLFlBQVk7S0FDYixDQUFDO0FBQ0osQ0FBQztBQTdCRCx3REE2QkM7QUFFRCxTQUFnQixlQUFlLENBQUMsSUFBYSxFQUFFLE1BQWUsRUFBRSxZQUF3QztJQUN0RyxJQUFJLFlBQVksSUFBSSxDQUFDLE1BQU0sRUFBRTtRQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7S0FDdEU7SUFFRCxJQUFJLElBQUksSUFBSSxNQUFNLEVBQUU7UUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsSUFBSSxjQUFjLE1BQU0sSUFBSSxDQUFDLENBQUM7S0FDcEc7SUFFRCxJQUFJLElBQUksRUFBRTtRQUNSLE9BQU87WUFDTCxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxJQUFJO1NBQ2YsQ0FBQztLQUNIO0lBRUQsSUFBSSxNQUFNLEVBQUU7UUFDVixJQUFJLFlBQVksRUFBRTtZQUNoQixNQUFNLElBQUksR0FBRyxHQUFHLFlBQVMsQ0FBQyxFQUFFLEtBQUssRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUM5RDtRQUVELE9BQU87WUFDTCxPQUFPLEVBQUUsUUFBUTtZQUNqQixRQUFRLEVBQUUsTUFBTTtTQUNqQixDQUFDO0tBQ0g7SUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLHdDQUF3QyxDQUFDLENBQUM7QUFDNUQsQ0FBQztBQTVCRCwwQ0E0QkM7QUFFRCxTQUFnQixlQUFlLENBQUMsUUFBNEIsRUFBRSxhQUFxQjtJQUNqRixJQUFJLFFBQVEsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLEVBQUU7UUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLGFBQWEsdUJBQXVCLENBQUMsQ0FBQztLQUMxRDtBQUNILENBQUM7QUFKRCwwQ0FJQztBQUVELFNBQWdCLGNBQWMsQ0FBQyxRQUE0QixFQUFFLGFBQXFCO0lBQ2hGLElBQUksUUFBUSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLEVBQUU7UUFDekUsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLGFBQWEsc0JBQXNCLENBQUMsQ0FBQztLQUN6RDtBQUNILENBQUM7QUFKRCx3Q0FJQztBQUVELE1BQWEsZ0JBQWdCO0lBQzNCOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsZUFBZSxDQUFDLE1BQTZCO1FBQ3pELE1BQU0sTUFBTSxHQUFHLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBRSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsRUFBRTtZQUMzQixNQUFNLENBQUMsT0FBTyxHQUFHLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUM7U0FDdEQ7UUFDRCxPQUFPLE1BQU0sQ0FBQztLQUNmO0lBZU8sTUFBTSxDQUFDLGdCQUFnQixDQUFDLE1BQVcsRUFBRSxZQUFZLEdBQUcsS0FBSztRQUMvRCxJQUFJLE1BQU0sSUFBSSxJQUFJLElBQUksT0FBTyxNQUFNLEtBQUssUUFBUSxFQUFFO1lBQ2hELE9BQU8sTUFBTSxDQUFDO1NBQ2Y7UUFDRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDekIsT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztTQUN0RTtRQUNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUU7WUFDdEUsTUFBTSxNQUFNLEdBQUcsQ0FBQyxZQUFZLElBQUksQ0FBQyxHQUFHLElBQUksZ0JBQWdCLENBQUMscUJBQXFCLENBQUMsQ0FBQztZQUNoRixNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLHFCQUFxQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUM7WUFDMUUsb0hBQW9IO1lBQ3BILE1BQU0sUUFBUSxHQUFHLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxDQUFDLFlBQVksSUFBSSxnQkFBZ0IsQ0FBQyxrQ0FBa0MsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3JJLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQyxFQUFFLFFBQVEsRUFBRSxDQUFDO1FBQ2hDLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDTDs7QUF6Q0gsNENBMENDO0FBNUJ5QixzQ0FBcUIsR0FBOEI7SUFDekUsTUFBTSxFQUFFLFNBQVM7SUFDakIsR0FBRyxFQUFFLE1BQU07SUFDWCxFQUFFLEVBQUUsS0FBSztDQUNWLENBQUM7QUFDRixvRUFBb0U7QUFDNUMsbURBQWtDLEdBQStCO0lBQ3ZGLFdBQVcsRUFBRSxJQUFJO0lBQ2pCLFVBQVUsRUFBRSxJQUFJO0lBQ2hCLGlCQUFpQixFQUFFLElBQUk7SUFDdkIsWUFBWSxFQUFFLElBQUk7Q0FDbkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGZvcm1hdCBhcyBmb3JtYXRVcmwgfSBmcm9tICd1cmwnO1xuaW1wb3J0ICogYXMganNvblNjaGVtYSBmcm9tICcuL2pzb24tc2NoZW1hJztcblxuZXhwb3J0IGNvbnN0IEFMTF9NRVRIT0RTID0gWydPUFRJT05TJywgJ0dFVCcsICdQVVQnLCAnUE9TVCcsICdERUxFVEUnLCAnUEFUQ0gnLCAnSEVBRCddO1xuXG5jb25zdCBBTExPV0VEX01FVEhPRFMgPSBbJ0FOWScsIC4uLkFMTF9NRVRIT0RTXTtcblxuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlSHR0cE1ldGhvZChtZXRob2Q6IHN0cmluZywgbWVzc2FnZVByZWZpeDogc3RyaW5nID0gJycpIHtcbiAgaWYgKCFBTExPV0VEX01FVEhPRFMuaW5jbHVkZXMobWV0aG9kKSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgJHttZXNzYWdlUHJlZml4fUludmFsaWQgSFRUUCBtZXRob2QgXCIke21ldGhvZH1cIi4gQWxsb3dlZCBtZXRob2RzOiAke0FMTE9XRURfTUVUSE9EUy5qb2luKCcsJyl9YCk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHBhcnNlTWV0aG9kT3B0aW9uc1BhdGgob3JpZ2luYWxQYXRoOiBzdHJpbmcpOiB7IHJlc291cmNlUGF0aDogc3RyaW5nLCBodHRwTWV0aG9kOiBzdHJpbmcgfSB7XG4gIGlmICghb3JpZ2luYWxQYXRoLnN0YXJ0c1dpdGgoJy8nKSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgTWV0aG9kIG9wdGlvbnMgcGF0aCBtdXN0IHN0YXJ0IHdpdGggJy8nOiAke29yaWdpbmFsUGF0aH1gKTtcbiAgfVxuXG4gIGNvbnN0IHBhdGggPSBvcmlnaW5hbFBhdGguc3Vic3RyKDEpOyAvLyB0cmltIHRyYWlsaW5nICcvJ1xuXG4gIGNvbnN0IGNvbXBvbmVudHMgPSBwYXRoLnNwbGl0KCcvJyk7XG5cbiAgaWYgKGNvbXBvbmVudHMubGVuZ3RoIDwgMikge1xuICAgIHRocm93IG5ldyBFcnJvcihgTWV0aG9kIG9wdGlvbnMgcGF0aCBtdXN0IGluY2x1ZGUgYXQgbGVhc3QgdHdvIGNvbXBvbmVudHM6IC97cmVzb3VyY2V9L3ttZXRob2R9IChpLmUuIC9mb28vYmFyL0dFVCk6ICR7cGF0aH1gKTtcbiAgfVxuXG4gIGNvbnN0IGh0dHBNZXRob2QgPSBjb21wb25lbnRzLnBvcCgpIS50b1VwcGVyQ2FzZSgpOyAvLyBsYXN0IGNvbXBvbmVudCBpcyBhbiBIVFRQIG1ldGhvZFxuICBpZiAoaHR0cE1ldGhvZCAhPT0gJyonKSB7XG4gICAgdmFsaWRhdGVIdHRwTWV0aG9kKGh0dHBNZXRob2QsIGAke29yaWdpbmFsUGF0aH06IGApO1xuICB9XG5cbiAgbGV0IHJlc291cmNlUGF0aCA9ICcvfjEnICsgY29tcG9uZW50cy5qb2luKCd+MScpO1xuICBpZiAoY29tcG9uZW50cy5sZW5ndGggPT09IDEgJiYgY29tcG9uZW50c1swXSA9PT0gJyonKSB7XG4gICAgcmVzb3VyY2VQYXRoID0gJy8qJztcbiAgfSBlbHNlIGlmIChjb21wb25lbnRzLmxlbmd0aCA9PT0gMSAmJiBjb21wb25lbnRzWzBdID09PSAnJykge1xuICAgIHJlc291cmNlUGF0aCA9ICcvJztcbiAgfVxuXG4gIHJldHVybiB7XG4gICAgaHR0cE1ldGhvZCxcbiAgICByZXNvdXJjZVBhdGgsXG4gIH07XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBwYXJzZUF3c0FwaUNhbGwocGF0aD86IHN0cmluZywgYWN0aW9uPzogc3RyaW5nLCBhY3Rpb25QYXJhbXM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9KTogeyBhcGlUeXBlOiBzdHJpbmcsIGFwaVZhbHVlOiBzdHJpbmcgfSB7XG4gIGlmIChhY3Rpb25QYXJhbXMgJiYgIWFjdGlvbikge1xuICAgIHRocm93IG5ldyBFcnJvcignXCJhY3Rpb25QYXJhbXNcIiByZXF1aXJlcyB0aGF0IFwiYWN0aW9uXCIgd2lsbCBiZSBzZXQnKTtcbiAgfVxuXG4gIGlmIChwYXRoICYmIGFjdGlvbikge1xuICAgIHRocm93IG5ldyBFcnJvcihgXCJwYXRoXCIgYW5kIFwiYWN0aW9uXCIgYXJlIG11dHVhbGx5IGV4Y2x1c2l2ZSAocGF0aD1cIiR7cGF0aH1cIiwgYWN0aW9uPVwiJHthY3Rpb259XCIpYCk7XG4gIH1cblxuICBpZiAocGF0aCkge1xuICAgIHJldHVybiB7XG4gICAgICBhcGlUeXBlOiAncGF0aCcsXG4gICAgICBhcGlWYWx1ZTogcGF0aCxcbiAgICB9O1xuICB9XG5cbiAgaWYgKGFjdGlvbikge1xuICAgIGlmIChhY3Rpb25QYXJhbXMpIHtcbiAgICAgIGFjdGlvbiArPSAnJicgKyBmb3JtYXRVcmwoeyBxdWVyeTogYWN0aW9uUGFyYW1zIH0pLnN1YnN0cigxKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgYXBpVHlwZTogJ2FjdGlvbicsXG4gICAgICBhcGlWYWx1ZTogYWN0aW9uLFxuICAgIH07XG4gIH1cblxuICB0aHJvdyBuZXcgRXJyb3IoJ0VpdGhlciBcInBhdGhcIiBvciBcImFjdGlvblwiIGFyZSByZXF1aXJlZCcpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gdmFsaWRhdGVJbnRlZ2VyKHByb3BlcnR5OiBudW1iZXIgfCB1bmRlZmluZWQsIG1lc3NhZ2VQcmVmaXg6IHN0cmluZykge1xuICBpZiAocHJvcGVydHkgJiYgIU51bWJlci5pc0ludGVnZXIocHJvcGVydHkpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGAke21lc3NhZ2VQcmVmaXh9IHNob3VsZCBiZSBhbiBpbnRlZ2VyYCk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlRG91YmxlKHByb3BlcnR5OiBudW1iZXIgfCB1bmRlZmluZWQsIG1lc3NhZ2VQcmVmaXg6IHN0cmluZykge1xuICBpZiAocHJvcGVydHkgJiYgaXNOYU4ocHJvcGVydHkpICYmIGlzTmFOKHBhcnNlRmxvYXQocHJvcGVydHkudG9TdHJpbmcoKSkpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGAke21lc3NhZ2VQcmVmaXh9IHNob3VsZCBiZSBhbiBkb3VibGVgKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgSnNvblNjaGVtYU1hcHBlciB7XG4gIC8qKlxuICAgKiBUcmFuc2Zvcm1zIG5hbWluZyBvZiBzb21lIHByb3BlcnRpZXMgdG8gcHJlZml4IHdpdGggYSAkLCB3aGVyZSBuZWVkZWRcbiAgICogYWNjb3JkaW5nIHRvIHRoZSBKU09OIHNjaGVtYSBzcGVjXG4gICAqIEBwYXJhbSBzY2hlbWEgVGhlIEpzb25TY2hlbWEgb2JqZWN0IHRvIHRyYW5zZm9ybSBmb3IgQ2xvdWRGb3JtYXRpb24gb3V0cHV0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHRvQ2ZuSnNvblNjaGVtYShzY2hlbWE6IGpzb25TY2hlbWEuSnNvblNjaGVtYSk6IGFueSB7XG4gICAgY29uc3QgcmVzdWx0ID0gSnNvblNjaGVtYU1hcHBlci5fdG9DZm5Kc29uU2NoZW1hKHNjaGVtYSk7XG4gICAgaWYgKCEgKCckc2NoZW1hJyBpbiByZXN1bHQpKSB7XG4gICAgICByZXN1bHQuJHNjaGVtYSA9IGpzb25TY2hlbWEuSnNvblNjaGVtYVZlcnNpb24uRFJBRlQ0O1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU2NoZW1hUHJvcHNXaXRoUHJlZml4OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9ID0ge1xuICAgIHNjaGVtYTogJyRzY2hlbWEnLFxuICAgIHJlZjogJyRyZWYnLFxuICAgIGlkOiAnJGlkJyxcbiAgfTtcbiAgLy8gVGhlIHZhbHVlIGluZGljYXRlcyB3aGV0aGVyIGRpcmVjdCBjaGlsZHJlbiBzaG91bGQgYmUga2V5LW1hcHBlZC5cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU2NoZW1hUHJvcHNXaXRoVXNlckRlZmluZWRDaGlsZHJlbjogeyBba2V5OiBzdHJpbmddOiBib29sZWFuIH0gPSB7XG4gICAgZGVmaW5pdGlvbnM6IHRydWUsXG4gICAgcHJvcGVydGllczogdHJ1ZSxcbiAgICBwYXR0ZXJuUHJvcGVydGllczogdHJ1ZSxcbiAgICBkZXBlbmRlbmNpZXM6IHRydWUsXG4gIH07XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX3RvQ2ZuSnNvblNjaGVtYShzY2hlbWE6IGFueSwgcHJlc2VydmVLZXlzID0gZmFsc2UpOiBhbnkge1xuICAgIGlmIChzY2hlbWEgPT0gbnVsbCB8fCB0eXBlb2Ygc2NoZW1hICE9PSAnb2JqZWN0Jykge1xuICAgICAgcmV0dXJuIHNjaGVtYTtcbiAgICB9XG4gICAgaWYgKEFycmF5LmlzQXJyYXkoc2NoZW1hKSkge1xuICAgICAgcmV0dXJuIHNjaGVtYS5tYXAoZW50cnkgPT4gSnNvblNjaGVtYU1hcHBlci5fdG9DZm5Kc29uU2NoZW1hKGVudHJ5KSk7XG4gICAgfVxuICAgIHJldHVybiBPYmplY3QuYXNzaWduKHt9LCAuLi5PYmplY3QuZW50cmllcyhzY2hlbWEpLm1hcCgoW2tleSwgdmFsdWVdKSA9PiB7XG4gICAgICBjb25zdCBtYXBLZXkgPSAhcHJlc2VydmVLZXlzICYmIChrZXkgaW4gSnNvblNjaGVtYU1hcHBlci5TY2hlbWFQcm9wc1dpdGhQcmVmaXgpO1xuICAgICAgY29uc3QgbmV3S2V5ID0gbWFwS2V5ID8gSnNvblNjaGVtYU1hcHBlci5TY2hlbWFQcm9wc1dpdGhQcmVmaXhba2V5XSA6IGtleTtcbiAgICAgIC8vIElmIGtleXMgd2VyZSBwcmVzZXJ2ZWQsIGRvbid0IGNvbnNpZGVyIFNjaGVtYVByb3BzV2l0aFVzZXJEZWZpbmVkQ2hpbGRyZW4gZm9yIHRob3NlIGtleXMgKHRoZXkgYXJlIHVzZXItZGVmaW5lZCEpXG4gICAgICBjb25zdCBuZXdWYWx1ZSA9IEpzb25TY2hlbWFNYXBwZXIuX3RvQ2ZuSnNvblNjaGVtYSh2YWx1ZSwgIXByZXNlcnZlS2V5cyAmJiBKc29uU2NoZW1hTWFwcGVyLlNjaGVtYVByb3BzV2l0aFVzZXJEZWZpbmVkQ2hpbGRyZW5ba2V5XSk7XG4gICAgICByZXR1cm4geyBbbmV3S2V5XTogbmV3VmFsdWUgfTtcbiAgICB9KSk7XG4gIH1cbn1cbiJdfQ==