# -*- coding: utf-8 -*-

# Copyright 2010-2011 OpenStack Foundation
# Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

"""Base test case for all tests."""

import os

import fixtures
from oslo_concurrency import lockutils
from oslo_config import cfg
from oslo_config import fixture as config_fixture
import six
import testtools

CONF = cfg.CONF

_TRUE_VALUES = ("True", "true", "1", "yes")


class TestCase(testtools.TestCase):
    """Test case base class for all unit tests."""

    REQUIRES_LOCKING = False

    def setUp(self):
        """Run before each test method to initialize test environment."""
        super(TestCase, self).setUp()
        test_timeout = os.environ.get("OS_TEST_TIMEOUT", 0)
        try:
            test_timeout = int(test_timeout)
        except ValueError:
            # If timeout value is invalid do not set a timeout.
            test_timeout = 0
        if test_timeout > 0:
            self.useFixture(fixtures.Timeout(test_timeout, gentle=True))

        self.useFixture(fixtures.NestedTempfile())
        self.useFixture(fixtures.TempHomeDir())

        if os.environ.get("OS_STDOUT_CAPTURE") in _TRUE_VALUES:
            stdout = self.useFixture(fixtures.StringStream("stdout")).stream
            self.useFixture(fixtures.MonkeyPatch("sys.stdout", stdout))
        if os.environ.get("OS_STDERR_CAPTURE") in _TRUE_VALUES:
            stderr = self.useFixture(fixtures.StringStream("stderr")).stream
            self.useFixture(fixtures.MonkeyPatch("sys.stderr", stderr))

        self.log_fixture = self.useFixture(fixtures.FakeLogger())

        if self.REQUIRES_LOCKING:
            lock_path = self.useFixture(fixtures.TempDir()).path
            self.fixture = self.useFixture(config_fixture.Config(lockutils.CONF))
            self.fixture.config(lock_path=lock_path)

    def flags(self, **kw):
        """Override flag variables for a test."""
        group = kw.pop("group", None)
        for k, v in six.iteritems(kw):
            CONF.set_override(k, v, group)

    def reset_flags(self):
        """Reset flags."""
        CONF.reset()
