"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("assert");
const assert_2 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('FSx for Lustre File System', () => {
    let lustreConfiguration;
    let stack;
    let storageCapacity;
    let vpcSubnet;
    let vpc;
    beforeEach(() => {
        stack = new core_1.Stack();
        storageCapacity = 1200;
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        vpcSubnet = new aws_ec2_1.Subnet(stack, 'Subnet', {
            availabilityZone: 'us-west-2',
            cidrBlock: vpc.vpcCidrBlock,
            vpcId: vpc.vpcId,
        });
    });
    test('default file system is created correctly', () => {
        lustreConfiguration = {
            deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
        };
        const fileSystem = new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
            lustreConfiguration,
            storageCapacityGiB: storageCapacity,
            vpc,
            vpcSubnet,
        });
        assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem'));
        assert_2.expect(stack).to(assert_2.haveResource('AWS::EC2::SecurityGroup'));
        assert_1.strictEqual(fileSystem.dnsName, `${fileSystem.fileSystemId}.fsx.${stack.region}.${core_1.Aws.URL_SUFFIX}`);
    });
    test('file system is created correctly when security group is provided', () => {
        lustreConfiguration = {
            deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
        };
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'FsxLustreSecurityGroup', {
            vpc,
        });
        new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
            lustreConfiguration,
            securityGroup,
            storageCapacityGiB: storageCapacity,
            vpc,
            vpcSubnet,
        });
        assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem'));
        assert_2.expect(stack).to(assert_2.haveResource('AWS::EC2::SecurityGroup'));
    });
    test('encrypted file system is created correctly with custom KMS', () => {
        lustreConfiguration = {
            deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
        };
        const key = new aws_kms_1.Key(stack, 'customKeyFS');
        new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
            kmsKey: key,
            lustreConfiguration,
            storageCapacityGiB: storageCapacity,
            vpc,
            vpcSubnet,
        });
        /**
         * CDK appends 8-digit MD5 hash of the resource path to the logical Id of the resource in order to make sure
         * that the id is unique across multiple stacks. There isnt a direct way to identify the exact name of the resource
         * in generated CDK, hence hardcoding the MD5 hash here for assertion. Assumption is that the path of the Key wont
         * change in this UT. Checked the unique id by generating the cloud formation stack.
         */
        assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
            KmsKeyId: {
                Ref: 'customKeyFSDDB87C6D',
            },
        }));
    });
    test('file system is created correctly when weekly maintenance time is provided', () => {
        const maintenanceTime = new lib_1.LustreMaintenanceTime({
            day: lib_1.Weekday.SUNDAY,
            hour: 12,
            minute: 34,
        });
        lustreConfiguration = {
            deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
            weeklyMaintenanceStartTime: maintenanceTime,
        };
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'FsxLustreSecurityGroup', {
            vpc,
        });
        new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
            lustreConfiguration,
            securityGroup,
            storageCapacityGiB: storageCapacity,
            vpc,
            vpcSubnet,
        });
        assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
            LustreConfiguration: {
                DeploymentType: 'SCRATCH_2',
                WeeklyMaintenanceStartTime: '0:12:34',
            },
        }));
        assert_2.expect(stack).to(assert_2.haveResource('AWS::EC2::SecurityGroup'));
    });
    describe('when validating props', () => {
        describe('exportPath', () => {
            test('export path valid', () => {
                const importPath = 's3://import-bucket';
                const exportPath = `${importPath}/export-prefix`;
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                        ExportPath: exportPath,
                        ImportPath: importPath,
                    },
                }));
            });
            test('export and import paths are Tokens', () => {
                const importPathResolved = 'importPath';
                const exportPathResolved = 'exportPath';
                const importPath = core_1.Token.asString(core_1.Token.asAny(importPathResolved));
                const exportPath = core_1.Token.asString(core_1.Token.asAny(exportPathResolved));
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                        ExportPath: exportPathResolved,
                        ImportPath: importPathResolved,
                    },
                }));
            });
            test('only export path is Token', () => {
                const exportPathResolved = 'exportPath';
                const importPath = 's3://import-bucket';
                const exportPath = core_1.Token.asString(core_1.Token.asAny(exportPathResolved));
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
            });
            test('only import path is Token', () => {
                const importPathResolved = 'importPath';
                const importPath = core_1.Token.asString(core_1.Token.asAny(importPathResolved));
                const exportPath = 's3://import-bucket/export';
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
            });
            test('invalid export path length', () => {
                const importPath = 's3://import-bucket';
                const path = Array(902).join('x');
                const exportPath = `${importPath}/${path}`;
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
            });
            test('export path does not start with import path', () => {
                const importPath = 's3://import-bucket';
                const exportPath = 's3://wrong-bucket';
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
            });
            test('export path with no import path', () => {
                const exportPath = 's://import-bucket/export-prefix';
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    exportPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError('Cannot define an export path without also defining an import path');
            });
        });
        describe('importedFileChunkSize', () => {
            test.each([
                1,
                256000,
                512000,
            ])('valid file chunk size of %d', (size) => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importedFileChunkSizeMiB: size,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                        ImportedFileChunkSize: size,
                    },
                }));
            });
            test.each([
                0,
                512001,
            ])('invalid file chunk size of %d', (size) => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importedFileChunkSizeMiB: size,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(`importedFileChunkSize cannot be ${size} MiB. It must be a value from 1 to 512,000 MiB`);
            });
        });
        describe('importPath', () => {
            test('import path valid', () => {
                const importPath = 's3://import-bucket';
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importPath,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                        ImportPath: importPath,
                    },
                }));
            });
            test('import path is Token', () => {
                const importPathResolved = 'importPath';
                const importPath = core_1.Token.asString(core_1.Token.asAny(importPathResolved));
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importPath,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                        ImportPath: importPathResolved,
                    },
                }));
            });
            test('invalid import path format', () => {
                const importPath = 'import-bucket';
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
            });
            test('invalid import path length', () => {
                const path = Array(902).join('x');
                const importPath = `s3://${path}`;
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    importPath,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
            });
        });
        describe('perUnitStorageThroughput', () => {
            test.each([
                50,
                100,
                200,
            ])('valid perUnitStorageThroughput of %d', (throughput) => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.PERSISTENT_1,
                    perUnitStorageThroughput: throughput,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: storageCapacity,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.PERSISTENT_1,
                        PerUnitStorageThroughput: throughput,
                    },
                }));
            });
            test('invalid perUnitStorageThroughput', () => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.PERSISTENT_1,
                    perUnitStorageThroughput: 1,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
            });
            test('setting perUnitStorageThroughput on wrong deploymentType', () => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_2,
                    perUnitStorageThroughput: 50,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: storageCapacity,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
            });
        });
        describe('storageCapacity', () => {
            test.each([
                [1200, lib_1.LustreDeploymentType.SCRATCH_2],
                [2400, lib_1.LustreDeploymentType.SCRATCH_2],
                [4800, lib_1.LustreDeploymentType.SCRATCH_2],
                [1200, lib_1.LustreDeploymentType.PERSISTENT_1],
                [2400, lib_1.LustreDeploymentType.PERSISTENT_1],
                [4800, lib_1.LustreDeploymentType.PERSISTENT_1],
            ])('proper multiple for storage capacity of %d on %s', (value, deploymentType) => {
                lustreConfiguration = {
                    deploymentType,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: value,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: deploymentType,
                    },
                    StorageCapacity: value,
                }));
            });
            test.each([
                [1, lib_1.LustreDeploymentType.SCRATCH_2],
                [2401, lib_1.LustreDeploymentType.SCRATCH_2],
                [1, lib_1.LustreDeploymentType.PERSISTENT_1],
                [2401, lib_1.LustreDeploymentType.PERSISTENT_1],
            ])('invalid value of %d for storage capacity on %s', (invalidValue, deploymentType) => {
                lustreConfiguration = {
                    deploymentType,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: invalidValue,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(/storageCapacity must be 1,200, 2,400, or a multiple of 2,400/);
            });
            test.each([1200, 2400, 3600, 7200])('proper multiple for storage capacity using %d with SCRATCH_1', (validValue) => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_1,
                };
                new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                    lustreConfiguration,
                    storageCapacityGiB: validValue,
                    vpc,
                    vpcSubnet,
                });
                assert_2.expect(stack).to(assert_2.haveResource('AWS::FSx::FileSystem', {
                    LustreConfiguration: {
                        DeploymentType: lib_1.LustreDeploymentType.SCRATCH_1,
                    },
                    StorageCapacity: validValue,
                }));
            });
            test.each([1, 3601])('invalid value of %d for storage capacity with SCRATCH_1', (invalidValue) => {
                lustreConfiguration = {
                    deploymentType: lib_1.LustreDeploymentType.SCRATCH_1,
                };
                expect(() => {
                    new lib_1.LustreFileSystem(stack, 'FsxFileSystem', {
                        lustreConfiguration,
                        storageCapacityGiB: invalidValue,
                        vpc,
                        vpcSubnet,
                    });
                }).toThrowError(/storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600/);
            });
        });
    });
    test('existing file system is imported correctly', () => {
        const fileSystemId = 'fs123';
        const fs = lib_1.LustreFileSystem.fromLustreFileSystemAttributes(stack, 'existingFS', {
            dnsName: `${fileSystemId}.fsx.${stack.region}.${core_1.Aws.URL_SUFFIX}`,
            fileSystemId,
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        fs.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443));
        assert_2.expect(stack).to(assert_2.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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