"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2022.10.1"
__checksum__ = "6bf3e66b289fe057ca9ec5e4ba2da4b116074decac3e1dd0d58a49781533b33c"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fly', b'foo', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'ing', b'insurance', b'meet', b'meme', b'new', b'nexus', b'page', b'phd', b'play', b'prof', b'rsvp', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 10), (21, 16), (37, 61), (98, 26), (124, 12), None, (136, 19), (155, 22), (177, 7), (184, 20), (204, 18), None, (222, 29), (251, 45), (296, 7), (303, 9), (312, 36), (348, 16), (364, 10), (374, 28), None, (402, 62), (464, 8), (472, 18), (490, 19), (509, 13), (522, 14), (536, 14), None, None, (550, 40), (590, 20), (610, 35), (645, 14), (659, 32), (691, 17), (708, 9), (717, 25), (742, 20), (762, 8), (770, 13), (783, 10), None, (793, 17), (810, 6), (816, 26), (842, 10), (852, 5), (857, 19), (876, 14), (890, 11), (901, 12), (913, 27), None, (940, 24), (964, 11), (975, 7), (982, 29), (1011, 18), (1029, 27), (1056, 46), (1102, 25), (1127, 16), (1143, 18), (1161, 5), (1166, 22), (1188, 18), None, (1206, 32), (1238, 15), (1253, 8), (1261, 11), None, (1272, 5), (1277, 16), (1293, 14), (1307, 18), None, (1325, 20), (1345, 26), (1371, 48), (1419, 19), (1438, 12), (1450, 59), (1509, 14), (1523, 14), (1537, 20), None, (1557, 10), (1567, 13), (1580, 20), (1600, 29), None, (1629, 13), (1642, 19), (1661, 11), (1672, 4), (1676, 22), (1698, 10), (1708, 13), (1721, 14), (1735, 28), (1763, 11), (1774, 21), (1795, 12), (1807, 32), None, (1839, 10), (1849, 14), (1863, 26), (1889, 45), (1934, 15), None, (1949, 11), (1960, 23), (1983, 21), (2004, 26), (2030, 6), (2036, 6), (2042, 7), (2049, 5), (2054, 30), (2084, 23), (2107, 35), (2142, 17), (2159, 15), (2174, 29), (2203, 12), (2215, 70), (2285, 55), (2340, 12), (2352, 23), (2375, 16), (2391, 38), (2429, 6), (2435, 20), (2455, 44), (2499, 6), (2505, 41), (2546, 22), (2568, 23), (2591, 36), (2627, 20), (2647, 8), (2655, 15), (2670, 12), (2682, 19), (2701, 25), (2726, 15), None, (2741, 35), (2776, 21), (2797, 17), (2814, 19), (2833, 26), (2859, 5), (2864, 37), (2901, 39), (2940, 16), (2956, 10), (2966, 17), (2983, 23), (3006, 14), (3020, 17), (3037, 8), (3045, 14), (3059, 7), (3066, 29), (3095, 6), (3101, 18), (3119, 32), (3151, 20), (3171, 17), (3188, 24), (3212, 12), (3224, 40), (3264, 40), (3304, 12), (3316, 48), (3364, 32), (3396, 17), None, (3413, 8), (3421, 25), (3446, 25), (3471, 6), (3477, 23), None, (3500, 36), (3536, 33), (3569, 14), (3583, 16), (3599, 27), None, (3626, 30), (3656, 41), (3697, 50), (3747, 15), (3762, 20), (3782, 25), (3807, 21), (3828, 32), (3860, 24), (3884, 20), (3904, 17), (3921, 60), (3981, 6), (3987, 9), (3996, 12), (4008, 12), (4020, 11), (4031, 10), (4041, 48), (4089, 42), None, (4131, 36), (4167, 28), None, (4195, 8), (4203, 8), (4211, 7), None, (4218, 25), (4243, 17), None, (4260, 21), (4281, 35), (4316, 21), (4337, 10), (4347, 41), (4388, 20), (4408, 31), (4439, 23), (4462, 19), (4481, 12), (4493, 5), (4498, 30), (4528, 29), (4557, 14), (4571, 14), (4585, 47), (4632, 52), None, None, (4684, 51), (4735, 42), None, (4777, 18), None, (4795, 15), (4810, 8), (4818, 21), (4839, 6), (4845, 16), (4861, 17)], [(4878, 9897), (14775, 10176), (24951, 10529), (35480, 9513), (44993, 9674), (54667, 9413), (64080, 10512), (74592, 9417), (84009, 10436), (94445, 9744), (104189, 10624), (114813, 9776), (124589, 10347), (134936, 11317), (146253, 10160), (156413, 10629), (167042, 10983), (178025, 10152), (188177, 10091), (198268, 9403), (207671, 10444), (218115, 10101), (228216, 10255), (238471, 10260), (248731, 10608), (259339, 9993), (269332, 10351), (279683, 10358), (290041, 9615), (299656, 10052), (309708, 10703), (320411, 10195), (330606, 10098), (340704, 9978), (350682, 9370), (360052, 10231), (370283, 10066), (380349, 10894), (391243, 10755), (401998, 10678), (412676, 11043), (423719, 9712), (433431, 9908), (443339, 10152), (453491, 10193), (463684, 10252), (473936, 10069), (484005, 11212), (495217, 10140), (505357, 9229), (514586, 9964), (524550, 10504), (535054, 10520), (545574, 10152), (555726, 10586), (566312, 10091), (576403, 10469), (586872, 10264), (597136, 10337), (607473, 8760), (616233, 10082), (626315, 10117), (636432, 9990), (646422, 10398), (656820, 10460), (667280, 10617), (677897, 10048), (687945, 10670), (698615, 10641), (709256, 10414), (719670, 10143), (729813, 9842), (739655, 9214), (748869, 10597), (759466, 10190), (769656, 10699), (780355, 9471), (789826, 11064), (800890, 10071), (810961, 9738), (820699, 10442), (831141, 9095), (840236, 9925), (850161, 10164), (860325, 9769), (870094, 10465), (880559, 11020), (891579, 9981), (901560, 10469), (912029, 10221), (922250, 11128), (933378, 9505), (942883, 10021), (952904, 9872), (962776, 10160), (972936, 10663), (983599, 10351), (993950, 9905), (1003855, 10349), (1014204, 9718), (1023922, 9866), (1033788, 10150), (1043938, 9994), (1053932, 9993), (1063925, 9981), (1073906, 10310), (1084216, 10831), (1095047, 10471), (1105518, 11189), (1116707, 10380), (1127087, 10688), (1137775, 10432), (1148207, 9849), (1158056, 10148), (1168204, 10470), (1178674, 10204), (1188878, 10241), (1199119, 10197), (1209316, 9977), (1219293, 11136), (1230429, 10939), (1241368, 10167), (1251535, 10239), (1261774, 10596), (1272370, 11255), (1283625, 10198), (1293823, 9699), (1303522, 10973), (1314495, 10210), (1324705, 11768), (1336473, 11042), (1347515, 9983), (1357498, 10187), (1367685, 9780), (1377465, 9906), (1387371, 10325), (1397696, 9690), (1407386, 10701), (1418087, 9710), (1427797, 9889), (1437686, 10742), (1448428, 10711), (1459139, 9660), (1468799, 9896), (1478695, 10722), (1489417, 9563), (1498980, 10128), (1509108, 9981), (1519089, 9780), (1528869, 10603), (1539472, 10394), (1549866, 10414), (1560280, 10416), (1570696, 9676), (1580372, 10264), (1590636, 10412), (1601048, 10068), (1611116, 10417), (1621533, 9744), (1631277, 9442), (1640719, 9297), (1650016, 10309), (1660325, 10780), (1671105, 9576), (1680681, 10055), (1690736, 11182), (1701918, 10368), (1712286, 9865), (1722151, 10903), (1733054, 10064), (1743118, 9573), (1752691, 10038), (1762729, 11699), (1774428, 9588), (1784016, 9506), (1793522, 10711), (1804233, 10079), (1814312, 10672), (1824984, 10000), (1834984, 9700), (1844684, 12479), (1857163, 10277), (1867440, 10240), (1877680, 10194), (1887874, 10956), (1898830, 11167), (1909997, 9504), (1919501, 10694), (1930195, 10068), (1940263, 9967), (1950230, 10925), (1961155, 9591), (1970746, 10280), (1981026, 9999), (1991025, 10202), (2001227, 10179), (2011406, 10076), (2021482, 9555), (2031037, 9966), (2041003, 9759), (2050762, 10492), (2061254, 10165), (2071419, 11049), (2082468, 10055), (2092523, 10791), (2103314, 10855), (2114169, 9412), (2123581, 10478), (2134059, 10456), (2144515, 10316), (2154831, 10512), (2165343, 10452), (2175795, 10288), (2186083, 10104), (2196187, 10652), (2206839, 10120), (2216959, 9617), (2226576, 10194), (2236770, 9907), (2246677, 10911), (2257588, 9925), (2267513, 9474), (2276987, 10950), (2287937, 10326), (2298263, 10418), (2308681, 10007), (2318688, 10170), (2328858, 9532), (2338390, 10488), (2348878, 10310), (2359188, 11067), (2370255, 10101), (2380356, 9712), (2390068, 10635), (2400703, 10137), (2410840, 11009), (2421849, 9897), (2431746, 9552), (2441298, 9167), (2450465, 10795), (2461260, 10450), (2471710, 10623), (2482333, 10241), (2492574, 10368), (2502942, 9829), (2512771, 10843), (2523614, 10449), (2534063, 9385), (2543448, 10016), (2553464, 9757), (2563221, 10268), (2573489, 10666), (2584155, 10680), (2594835, 9882), (2604717, 10119), (2614836, 10369)], [(2625205, 1123), (2626328, 888), (2627216, 902), (2628118, 1164), (2629282, 857), (2630139, 1057), (2631196, 816), (2632012, 1216), (2633228, 861), (2634089, 976), (2635065, 698), (2635763, 780), (2636543, 1042), (2637585, 1095), (2638680, 1158), (2639838, 1259), (2641097, 1492), (2642589, 838), (2643427, 1086), (2644513, 1024), (2645537, 1158), (2646695, 983), (2647678, 1201), (2648879, 1007), (2649886, 1062), (2650948, 903), (2651851, 1322), (2653173, 1508), (2654681, 1056), (2655737, 1056), (2656793, 1189), (2657982, 1114), (2659096, 857), (2659953, 983), (2660936, 1178), (2662114, 1040), (2663154, 940), (2664094, 1045), (2665139, 1049), (2666188, 1297), (2667485, 965), (2668450, 1198), (2669648, 971), (2670619, 994), (2671613, 1011), (2672624, 761), (2673385, 1161), (2674546, 1239), (2675785, 1015), (2676800, 668), (2677468, 1007), (2678475, 980), (2679455, 1079), (2680534, 1250), (2681784, 1570), (2683354, 878), (2684232, 940), (2685172, 927), (2686099, 857), (2686956, 1000), (2687956, 1007), (2688963, 1123), (2690086, 1298), (2691384, 1174), (2692558, 898), (2693456, 1074), (2694530, 999), (2695529, 721), (2696250, 905), (2697155, 1118), (2698273, 980), (2699253, 1061), (2700314, 778), (2701092, 993), (2702085, 882), (2702967, 945), (2703912, 879), (2704791, 935), (2705726, 1033), (2706759, 741), (2707500, 1107), (2708607, 846), (2709453, 1161), (2710614, 831), (2711445, 1021), (2712466, 833), (2713299, 966), (2714265, 1104), (2715369, 1075), (2716444, 1073), (2717517, 1216), (2718733, 1360), (2720093, 1212), (2721305, 934), (2722239, 1057), (2723296, 724), (2724020, 1142), (2725162, 1044), (2726206, 833), (2727039, 891), (2727930, 1045), (2728975, 1124), (2730099, 1104), (2731203, 753), (2731956, 896), (2732852, 1051), (2733903, 721), (2734624, 724), (2735348, 1209), (2736557, 1306), (2737863, 1016), (2738879, 967), (2739846, 957), (2740803, 1050), (2741853, 1057), (2742910, 999), (2743909, 890), (2744799, 864), (2745663, 886), (2746549, 848), (2747397, 1315), (2748712, 991), (2749703, 1010), (2750713, 731), (2751444, 1033), (2752477, 1046), (2753523, 1002), (2754525, 1210), (2755735, 909), (2756644, 1247), (2757891, 1041), (2758932, 885), (2759817, 1097), (2760914, 878), (2761792, 1133), (2762925, 1086), (2764011, 994), (2765005, 935), (2765940, 1019), (2766959, 832), (2767791, 937), (2768728, 926), (2769654, 980), (2770634, 784), (2771418, 793), (2772211, 708), (2772919, 939), (2773858, 860), (2774718, 942), (2775660, 850), (2776510, 991), (2777501, 833), (2778334, 700), (2779034, 1174), (2780208, 1020), (2781228, 893), (2782121, 936), (2783057, 1205), (2784262, 980), (2785242, 859), (2786101, 1267), (2787368, 928), (2788296, 904), (2789200, 1083), (2790283, 1340), (2791623, 875), (2792498, 854), (2793352, 1047), (2794399, 967), (2795366, 895), (2796261, 1002), (2797263, 814), (2798077, 1212), (2799289, 1023), (2800312, 1078), (2801390, 1167), (2802557, 961), (2803518, 831), (2804349, 1007), (2805356, 932), (2806288, 2130), (2808418, 805), (2809223, 1008), (2810231, 946), (2811177, 1289), (2812466, 890), (2813356, 975), (2814331, 871), (2815202, 780), (2815982, 1203), (2817185, 870), (2818055, 795), (2818850, 1022), (2819872, 1178), (2821050, 1181), (2822231, 1076), (2823307, 949), (2824256, 848), (2825104, 998), (2826102, 890), (2826992, 1036), (2828028, 882), (2828910, 1045), (2829955, 860), (2830815, 1001), (2831816, 833), (2832649, 1028), (2833677, 1242), (2834919, 900), (2835819, 1208), (2837027, 982), (2838009, 998), (2839007, 1187), (2840194, 1332), (2841526, 1035), (2842561, 961), (2843522, 1165), (2844687, 905), (2845592, 829), (2846421, 631), (2847052, 1011), (2848063, 1192), (2849255, 797), (2850052, 1316), (2851368, 806), (2852174, 1025), (2853199, 1001), (2854200, 1147), (2855347, 1162), (2856509, 956), (2857465, 1103), (2858568, 806), (2859374, 1143), (2860517, 881), (2861398, 779), (2862177, 868), (2863045, 860), (2863905, 530), (2864435, 1078), (2865513, 1220), (2866733, 987), (2867720, 886), (2868606, 863), (2869469, 810), (2870279, 1115), (2871394, 792), (2872186, 802), (2872988, 1124), (2874112, 689), (2874801, 1137), (2875938, 2642), (2878580, 849), (2879429, 896), (2880325, 1184), (2881509, 1163), (2882672, 653)], [(2883325, 48), None, (2883373, 35), (2883408, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (2883450, 42), None, (2883492, 25), (2883517, 44), (2883561, 22), (2883583, 33), None, None, None, None, (2883616, 26), None, None, None, None, (2883642, 21), (2883663, 25), None, (2883688, 21), (2883709, 26), None, None, None, None, (2883735, 71), (2883806, 21), (2883827, 23), None, None, None, None, (2883850, 48), None, None, None, None, None, (2883898, 31), None, None, None, None, (2883929, 42), None, (2883971, 22), None, (2883993, 21), None, (2884014, 26), (2884040, 56), None, None, (2884096, 77), (2884173, 27), None, None, None, None, (2884200, 21), (2884221, 21), None, None, (2884242, 54), (2884296, 42), None, None, None, (2884338, 25), None, None, (2884363, 21), None, None, None, None, None, (2884384, 24), (2884408, 21), None, None, (2884429, 26), None, (2884455, 18), None, (2884473, 54), None, None, None, None, None, None, (2884527, 26), None, None, None, (2884553, 20), None, None, (2884573, 64), (2884637, 42), (2884679, 17), (2884696, 17), (2884713, 26), None, (2884739, 45), None, None, None, (2884784, 26), (2884810, 20), (2884830, 26), None, (2884856, 42), (2884898, 63), None, None, None, (2884961, 40), (2885001, 48), None, None, None, (2885049, 47), None, None, None, None, None, None, None, (2885096, 42), None, (2885138, 80), None, (2885218, 9), None, (2885227, 21), (2885248, 42), None, None, (2885290, 65), (2885355, 82), (2885437, 21), None, (2885458, 72), None, None, (2885530, 24), (2885554, 21), None, None, None, None, None, (2885575, 42), (2885617, 21), (2885638, 21), None, (2885659, 42), (2885701, 25), None, (2885726, 38), (2885764, 21), (2885785, 56), None, None, (2885841, 21), (2885862, 19), (2885881, 26), (2885907, 24), (2885931, 16), None, (2885947, 21), None, None, (2885968, 38), None, (2886006, 22), (2886028, 21), (2886049, 21), (2886070, 21), None, (2886091, 63), None, (2886154, 21), (2886175, 42), None, (2886217, 17), None, None, None, None, (2886234, 21), (2886255, 21), None, None, (2886276, 21), None, None, (2886297, 21), None, (2886318, 26), None, (2886344, 50), (2886394, 22), None, None, (2886416, 50), (2886466, 26), (2886492, 21), (2886513, 21), (2886534, 19), None, (2886553, 35), (2886588, 26), (2886614, 23), (2886637, 39), (2886676, 42), None, None, None, None, None, None, (2886718, 21), None, None, None, (2886739, 21), None, None, (2886760, 90), None, (2886850, 239), (2887089, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
