#!/usr/bin/env python2
# -*- coding: utf-8 -*-
"""
Created on Thu May 17 07:01:17 2018

@author: max
"""
from __future__ import division
import numpy as np
from scipy.interpolate import Rbf
import matplotlib.pyplot as plt


class calc_xsec:
    def __init__(self):

        # create interpolation functions
        self.ione_st_rbf = calc_svione_st()
        self.rec_st_rbf = calc_svrec_st()
        self.cx_st_rbf = calc_svcx_st()
        self.el_st_rbf = calc_svel_st()
        self.eln_st_rbf = calc_sveln_st()

        # self.ione_janev_rbf = calc_svione_janev()
        # self.ioni_janev_rbf = calc_svioni_janev(aion=2, aneut=2)
        # self.cx_janev_rbf = calc_svcx_janev(aion=2, aneut=2)

        self.ione_degas_rbf = calc_svione_degas()
        self.ioni_degas_rbf = calc_svioni_degas()
        self.rec_degas_rbf = calc_svrec_degas()
        self.cx_degas_rbf = calc_svcx_degas()

    # create wrapper functions around the interpolation functions (to add documentation, etc.)
    # you could also access the interpolation functions directly. This is mostly to prompt the user
    # about which inputs go to which cross section calculations
    def ione_st(self, ne, Te):
        """"""
        return 10**self.ione_st_rbf(np.log10(Te), np.log10(ne))

    def rec_st(self, ne, Te):
        """"""
        return 10**self.rec_st_rbf(np.log10(Te), np.log10(ne))

    def cx_st(self, Ti, Tn):
        """"""
        # TODO: make sure rows and columns are right here
        return 10**self.cx_st_rbf(np.log10(Ti), np.log10(Tn))

    def el_st(self, Ti, Tn):
        """"""
        return 10**self.el_st_rbf(np.log10(Ti), np.log10(Tn))

    def eln_st(self, Tn):
        """"""
        return 10**self.eln_st_rbf(np.log10(Tn))

    # def ione_janev(self, logTe):
    #     """"""
    #     return self.ione_janev_rbf(logTe)
    #
    # def ioni_janev(self, logTi, logTn):
    #     """"""
    #     return self.ioni_janev_rbf(logTi, logTn)
    #
    # def cx_janev(self, logTi, logTn):
    #     """"""
    #     return self.cx_janev_rbf(logTi, logTn)

    def ione_degas(self, ne, Te):
        """"""
        return self.ione_degas_rbf(np.log10(Te), np.log10(ne))

    def ioni_degas(self, Ti, Tn):
        """"""
        return self.ioni_degas_rbf(np.log10(Tn), np.log10(Ti))

    def rec_degas(self, ne, Te):
        """"""
        return self.rec_degas_rbf(np.log10(Te), np.log10(ne))

    def cx_degas(self, Ti, Tn):
        """"""
        return self.cx_degas_rbf(np.log10(Tn), np.log10(Ti))

# module level functions
def calc_svione_st():
    """Calculates the ionization cross section using stacey thomas data

    Args:
        ne (float): takes ne in m^-3
        Te (float): takes Te in keV

    Returns:
        svione (float):

    """
    # populate interpolation data
    eion = np.array([[-2.8523E+01, -2.8523E+01, -2.8523E+01, -2.8523E+01, -2.8523E+01],
                     [-1.7745E+01, -1.7745E+01, -1.7745E+01, -1.7745E+01, -1.7745E+01],
                     [-1.3620E+01, -1.3620E+01, -1.3620E+01, -1.3620E+01, -1.3620E+01],
                     [-1.3097E+01, -1.3097E+01, -1.3097E+01, -1.3097E+01, -1.3097E+01],
                     [-1.3301E+01, -1.3301E+01, -1.3301E+01, -1.3301E+01, -1.3301E+01]])

    nint, tint,  = np.meshgrid(np.array([16, 18, 20, 21, 22]),
                               np.array([-4, -3, -2, -1, 0]))

    return Rbf(nint, tint, eion)


def calc_svrec_st():
    """Calculates the

    Args:
        ne (float): takes ne in m^-3
        Te (float): takes Te in keV

    Returns:
        svrec (float): m^3 / s

    """
    # populate interpolation data
    rec = np.array([[-1.7523E+01, -1.6745E+01, -1.5155E+01, -1.4222E+01, -1.3301E+01],
                    [-1.8409E+01, -1.8398E+01, -1.8398E+01, -1.7886E+01, -1.7000E+01],
                    [-1.9398E+01, -1.9398E+01, -1.9398E+01, -1.9398E+01, -1.9398E+01],
                    [-2.0155E+01, -2.0155E+01, -2.0155E+01, -2.0155E+01, -2.0155E+01],
                    [-2.1000E+01, -2.1000E+01, -2.1000E+01, -2.1000E+01, -2.1000E+01]])

    nint, tint = np.meshgrid(np.array([16, 18, 20, 21, 22]),
                             np.array([-4, -3, -2, -1, 0]))

    return Rbf(nint, tint, rec)


def calc_svcx_st():
    """
    Calculates the

    Args:
        Ti (float): takes Ti in keV
        Tn (float): takes Tn in keV

    Returns:
        svcx (float): m^3/s
    """
    # populate interpolation data
    cx = np.array([[-1.4097E+01, -1.3921E+01, -1.3553E+01, -1.4097E+01, -1.3921E+01],
                   [-1.3553E+01, -1.3921E+01, -1.3824E+01, -1.3538E+01, -1.3553E+01],
                   [-1.3538E+01, -1.3432E+01, -1.3553E+01, -1.3538E+01, -1.3432E+01]])

    tint, tnnt = np.meshgrid(np.array([-4, -3, -2, -1, 0]),
                             np.array([-3, -2, -1]))

    return Rbf(tint, tnnt, cx)


def calc_svel_st():
    """Calculates the

    Args:
        Ti (float): takes Ti in keV
        Tn (float): takes Tn in keV

    Returns:
        svel (float):

    """
    # populate interpolation data
    el = np.array([[-1.3569E+01, -1.3337E+01, -1.3036E+01, -1.3569E+01, -1.3337E+01],
                   [-1.3036E+01, -1.3337E+01, -1.3167E+01, -1.3046E+01, -1.3036E+01],
                   [-1.3046E+01, -1.2796E+01, -1.3036E+01, -1.3046E+01, -1.2796E+01]])

    tint, tnnt = np.meshgrid(np.array([-4, -3, -2, -1, 0]),
                             np.array([-3, -2, -1]))

    return Rbf(tint, tnnt, el)


def calc_sveln_st():
    """Calculates the

    Args:
        Tn (float): takes Tn in keV

    Returns:
        sveln (float):
    """
    # populate interpolation data
    eln = np.array([-1.4569E+01, -1.4167E+01, -1.3796E+01])

    tnnt = np.array([-3, -2, -1])

    return Rbf(tnnt, eln)


#############################################################################
#############################################################################


# def calc_svione_janev():
#     """calculates the electron impact ionization reactivity (<sigma*v>)
#
#     Args:
#         Te (float): Electron temperature in eV
#
#     Returns:
#         svione (float): electron impact ionization rate (m^3/s)
#
#     This function evaluates the electron impact ionization
#     reactivity (<sigma*v>) as a function of the electron temperature
#     using the logarithmic polynomial approximation of Janev, et al.
#
#     Reference:
#     ---------
#     R.K. Janev et al. 'Elementary Processes in Hydrogen-Helium Plasmas,
#     Springer-Verlag, 1987
#
#     e + H(1s) --> e + H+ + e (Reaction 2.1.5)
#     Notice that only six significant digits have been kept, per
#     Janev's suggestion (page 256)
#
#     Originally written by John Mandrekas for the GTNEUT code.
#     Adapted for python by Max Hill
#
#     Comments:
#         --------
#     The electron temperature range is: 0.1 eV <= te < 20 keV
#     """
#
#     # convert from kev to ev
#     Te = np.asarray(Te)*1E3
#
#     bn = np.array([-3.271396786375e+01,  1.353655609057e+01, -5.739328757388e+00,
#                     1.563154982022e+00, -2.877056004391e-01,  3.482559773737e-02,
#                    -2.631976175590e-03,  1.119543953861e-04, -2.039149852002e-06])
#
#     # fix values that are outside the interpolation range
#     Te[Te < 1.0E-1] = 1.0E-1
#     Te[Te > 2.0E4] = 2.0E4
#
#     svione = np.zeros(Te.shape)
#     for i,lT in enumerate(np.log(Te)):
#         for j,coef in enumerate(bn):
#             svione[i] = svione[i] + coef * lT**j
#
#     return np.exp(svione)*1E-6


# def calc_svioni_janev(aion=2, aneut=2):
#     """
#     This function evaluates the ion impact ionization reactivity
#     (<sigma*v>) as a function of the ion temperature and the neutral
#     energy using the double logarithmic polynomial approximation of
#     Janev, et al.
#
#     Reference:
#         ---------
#         R.K. Janev et al. 'Elementary Processes in Hydrogen-Helium Plasmas,
#         Springer-Verlag, 1987
#
#     p + H(1s) --> p + H+ + e (Reaction 3.1.6)
#
#     Written by John Mandrekas, GIT, 10/05/2001 for the GTNEUT code
#
#     Parameters:
#     ----------
#     aion  : ion mass (1 for H, 2 for D, etc.)
#     ti    : ion temperature (eV)
#     aneut : neutral mass (1 for H, 2 for D, etc.)
#     e0    : ion temperature (eV)
#
#     Output:
#     ------
#     svioni : ion impact ionization reaction rate (m^3/s)
#
#     Comments:
#     --------
#     The ion temperature  and neutral energy ranges are:
#     0.1 eV <= Ti, E0 <= 20 keV
#     Energies and temperatures are scaled by mass ratios
#     """
#
#     if isinstance(Ti, float):
#         Ti = np.array([Ti])
#     if isinstance(Tn, float):
#         Tn = np.array([Tn])
#
#     #convert from kev to ev
#     Ti = Ti*1E3
#     Tn = Tn*1E3
#
#     an = np.array([-1.617454916209e+02,  1.021458246570e+02, -5.712267930902e+01,
#                     2.140540272484e+01, -4.767517412803e+00,  6.293295208376e-01,
#                    -4.858173640838e-02,  2.031177914273e-03, -3.557982934756e-05,
#                     1.767238902030e+01, -7.102574692619e+01,  4.246688953154e+01,
#                    -1.128638171243e+01,  1.661679851896e+00, -1.476754423056e-01,
#                     8.175790218529e-03, -2.732531913524e-04,  4.398387454014e-06,
#                    -4.334843983767e+01,  3.855259623260e+01, -1.316883030631e+01,
#                     2.145592145856e+00, -1.467281287038e-01, -2.915256218527e-03,
#                     1.092542891192e-03, -6.205102802216e-05,  1.158798945435e-06,
#                     2.464254915383e+01, -1.283426276878e+01,  2.369698902002e+00,
#                    -1.506665823159e-01, -8.144926683660e-03,  2.231505500086e-03,
#                    -2.210941355372e-04,  1.310924337643e-05, -3.431837053957e-07,
#                    -5.439093405254e+00,  2.357085001656e+00, -2.961732508220e-01,
#                    -9.917174972226e-04,  1.935894665907e-03, -1.679264493005e-05,
#                     5.532386419162e-08, -1.121430499351e-06,  5.960280736984e-08,
#                     5.959975304236e-01, -2.391382925527e-01,  2.789277301925e-02,
#                     8.562387824450e-05, -1.340759667335e-04, -5.927455645560e-06,
#                     5.820264508685e-07,  7.694068657107e-08, -4.972708712807e-09,
#                    -3.361958123977e-02,  1.289667246580e-02, -1.858739201548e-03,
#                     9.235982885753e-05,  9.875232214392e-06, -1.680823118052e-06,
#                     3.019916624608e-08,  6.889325889968e-09, -3.171970185702e-10,
#                     8.706597041685e-04, -3.140899683782e-04,  7.343984485463e-05,
#                    -8.601564864429e-06, -6.467790579320e-07,  1.734797315767e-07,
#                     2.523651535182e-09, -1.719633613108e-09,  7.332933714195e-11,
#                    -6.359765062372e-06,  1.742836004704e-06, -1.235536456998e-06,
#                     2.257852760280e-07,  1.608335682237e-08, -3.855914336143e-09,
#                    -3.556222618473e-10,  7.627265694554e-11, -2.960493966948e-12]).reshape((9,9))
#
#     #scale with isotope mass
#     Ti = Ti / aion
#     Tn = Tn / aneut
#
#     #Make sure we are within limits of validity
#     Ti[Ti<1.0E-1] = 1.0E-1
#     Ti[Ti>2.0E4]  = 2.0E4
#     Tn[Tn<1.0E-1] = 1.0E-1
#     Tn[Tn>2.0E4]  = 2.0E4
#
#     logTi = np.log(Ti)
#     logTn = np.log(Tn)
#
#     svioni = np.zeros(Ti.shape)
#     for i,(lTi,lTn) in enumerate(zip(logTi,logTn)):
#         for (ai,aj),coef in np.ndenumerate(an):
#             svioni[i] = svioni[i] + coef * lTi**aj * lTn**ai
#
#     return np.exp(svioni)*1E-6
#
#
# def calc_svcx_janev(aion=2, aneut=2):
#     """
#     This function evaluates the ion impact ionization reactivity
#     (<sigma*v>) as a function of the ion temperature and the neutral
#     energy using the double logarithmic polynomial approximation of
#     Janev, et al.
#
#     Reference:
#         ---------
#         R.K. Janev et al. 'Elementary Processes in Hydrogen-Helium Plasmas,
#         Springer-Verlag, 1987
#
#     p + H(1s) --> p + H+ + e (Reaction 3.1.6)
#
#     Written by John Mandrekas, GIT, 10/05/2001 for the GTNEUT code
#
#     Parameters:
#     ----------
#     aion  : ion mass (1 for H, 2 for D, etc.)
#     ti    : ion temperature (eV)
#     aneut : neutral mass (1 for H, 2 for D, etc.)
#     e0    : ion temperature (eV)
#
#     Output:
#     ------
#     svioni : ion impact ionization reaction rate (m^3/s)
#
#     Comments:
#     --------
#     The ion temperature  and neutral energy ranges are:
#     0.1 eV <= Ti, E0 <= 20 keV
#     Energies and temperatures are scaled by mass ratios
#     """
#
#     if isinstance(Ti, float):
#         Ti = np.array([Ti])
#     if isinstance(Tn, float):
#         Tn = np.array([Tn])
#
#     # convert from kev to ev
#     Ti = Ti*1E3
#     Tn = Tn*1E3
#
#     an = np.array([-1.829079581680e+01,  2.169137615703e-01,  4.307131243894e-02,
#                    -5.754895093075e-04, -1.552077120204e-03, -1.876800283030e-04,
#                     1.125490270962e-04, -1.238982763007e-05,  4.163596197181e-07,
#                     1.640252721210e-01, -1.106722014459e-01,  8.948693624917e-03,
#                     6.062141761233e-03, -1.210431587568e-03, -4.052878751584e-05,
#                     2.875900435895e-05, -2.616998139678e-06,  7.558092849125e-08,
#                     3.364564509137e-02, -1.382158680424e-03, -1.209480567154e-02,
#                     1.075907881928e-03,  8.297212635856e-04, -1.907025662962e-04,
#                     1.338839628570e-05, -1.171762874107e-07, -1.328404104165e-08,
#                     9.530225559189e-03,  7.348786286628e-03, -3.675019470470e-04,
#                    -8.119301728339e-04,  1.361661816974e-04,  1.141663041636e-05,
#                    -4.340802793033e-06,  3.517971869029e-07, -9.170850253981e-09,
#                    -8.519413589968e-04, -6.343059502294e-04,  1.039643390686e-03,
#                     8.911036876068e-06, -1.008928628425e-04,  1.775681984457e-05,
#                    -7.003521917385e-07, -4.928692832866e-08,  3.208853883734e-09,
#                    -1.247583860943e-03, -1.919569450380e-04, -1.553840717902e-04,
#                     3.175388949811e-05,  1.080693990468e-05, -3.149286923815e-06,
#                     2.318308730487e-07,  1.756388998863e-10, -3.952740758950e-10,
#                     3.014307545716e-04,  4.075019351738e-05,  2.670827249272e-06,
#                    -4.515123641755e-06,  5.106059413591e-07,  3.105491554749e-08,
#                    -6.030983538280e-09, -1.446756795654e-10,  2.739558475782e-11,
#                    -2.499323170044e-05, -2.850044983009e-06,  7.695300597935e-07,
#                     2.187439283954e-07, -1.299275586093e-07,  2.274394089017e-08,
#                    -1.755944926274e-09,  7.143183138281e-11, -1.693040208927e-12,
#                     6.932627237765e-07,  6.966822400446e-08, -3.783302281524e-08,
#                    -2.911233951880e-09,  5.117133050290e-09, -1.130988250912e-09,
#                     1.005189187279e-10, -3.989884105603e-12,  6.388219930167e-14]).reshape((9, 9))
#
#     #scale with isotope mass
#     Ti = Ti / aion
#     Tn = Tn / aneut
#
#     #Make sure we are within limits of validity
#     Ti[Ti<1.0E-1] = 1.0E-1
#     Ti[Ti>2.0E4]  = 2.0E4
#     Tn[Tn<1.0E-1] = 1.0E-1
#     Tn[Tn>2.0E4]  = 2.0E4
#
#     logTi = np.log(Ti)
#     logTn = np.log(Tn)
#     svcxi = np.zeros(Ti.shape)
#     for i,(lTi,lTn) in enumerate(zip(logTi,logTn)):
#         for (ai,aj),coef in np.ndenumerate(an):
#             svcxi[i] = svcxi[i] + coef * lTi**aj * lTn**ai
#     return np.exp(svcxi)*1E-6


#############################################################################
#############################################################################


def calc_svione_degas():
    """Calculates the electron impact ionization reactivity using the degas cross sections

    accepts ne in m^-3 and Te in keV
    returns value in m^3/s

    """
    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    svion_data = np.zeros((15, 60))
    svion_data[0] = np.array([1.00000E-99, 8.23273E-79, 2.71317E-67, 2.66585E-55, 8.79947E-46, 3.16283E-38,
                        3.15323E-32, 1.83420E-27, 1.12534E-23, 1.16683E-20, 2.97480E-18, 2.51111E-16,
                        8.57971E-15, 1.50074E-13, 1.51826E-12, 9.93555E-12, 4.58972E-11, 1.60312E-10,
                        4.46589E-10, 1.03558E-09, 2.06751E-09, 3.65749E-09, 5.84839E-09, 8.59216E-09,
                        1.17700E-08, 1.52155E-08, 1.87465E-08, 2.21895E-08, 2.53935E-08, 2.82342E-08,
                        3.06141E-08, 3.24630E-08, 3.37401E-08, 3.44364E-08, 3.45743E-08, 3.42046E-08,
                        3.33990E-08, 3.22410E-08, 3.08169E-08, 2.86297E-08, 2.66209E-08, 2.46946E-08,
                        2.28604E-08, 2.11240E-08, 1.94798E-08, 1.79424E-08, 1.65048E-08, 1.51644E-08,
                        1.39180E-08, 1.27614E-08, 1.16905E-08, 1.07006E-08, 9.78692E-09, 8.94489E-09,
                        8.16984E-09, 7.45728E-09, 6.80287E-09, 6.20244E-09, 5.65204E-09, 5.14793E-09])
    svion_data[1] = np.array([1.00000E-99, 4.82970E-82, 7.83240E-67, 5.09204E-55, 1.56205E-45, 5.27263E-38,
                        4.97672E-32, 2.75568E-27, 1.61419E-23, 1.59996E-20, 3.90124E-18, 3.15201E-16,
                        1.03886E-14, 1.75120E-13, 1.71526E-12, 1.09243E-11, 4.93698E-11, 1.69495E-10,
                        4.65965E-10, 1.06969E-09, 2.11967E-09, 3.73201E-09, 5.94854E-09, 8.71687E-09,
                        1.19171E-08, 1.53810E-08, 1.89261E-08, 2.23789E-08, 2.55887E-08, 2.84317E-08,
                        3.08110E-08, 3.26572E-08, 3.39298E-08, 3.46203E-08, 3.47515E-08, 3.43747E-08,
                        3.35615E-08, 3.23958E-08, 3.09640E-08, 2.87693E-08, 2.67529E-08, 2.48194E-08,
                        2.29782E-08, 2.12350E-08, 1.95784E-08, 1.80330E-08, 1.65876E-08, 1.52399E-08,
                        1.39865E-08, 1.28234E-08, 1.17464E-08, 1.07508E-08, 9.83189E-09, 8.98502E-09,
                        8.20554E-09, 7.48893E-09, 6.83083E-09, 6.22706E-09, 5.67365E-09, 5.16683E-09])
    svion_data[2] = np.array([1.00000E-99, 8.24318E-79, 1.32124E-66, 1.08123E-54, 3.03359E-45, 9.48783E-38,
                        8.38562E-32, 4.38472E-27, 2.43932E-23, 2.30382E-20, 5.36106E-18, 4.13704E-16,
                        1.31130E-14, 2.12229E-13, 2.00320E-12, 1.23534E-11, 5.43439E-11, 1.82547E-10,
                        4.93352E-10, 1.11776E-09, 2.19295E-09, 3.83702E-09, 6.08955E-09, 8.89201E-09,
                        1.21234E-08, 1.56123E-08, 1.91769E-08, 2.26428E-08, 2.58601E-08, 2.87057E-08,
                        3.10837E-08, 3.29253E-08, 3.41911E-08, 3.48730E-08, 3.49946E-08, 3.46074E-08,
                        3.37834E-08, 3.26069E-08, 3.11642E-08, 2.89589E-08, 2.69323E-08, 2.49888E-08,
                        2.31381E-08, 2.13857E-08, 1.97125E-08, 1.81563E-08, 1.67008E-08, 1.53434E-08,
                        1.40809E-08, 1.29094E-08, 1.18244E-08, 1.08215E-08, 9.89581E-09, 9.04268E-09,
                        8.25745E-09, 7.53556E-09, 6.87263E-09, 6.26445E-09, 5.70702E-09, 5.19654E-09])
    svion_data[3] = np.array([1.00000E-99, 3.72430E-81, 3.97672E-66, 2.61445E-54, 6.61570E-45, 1.88911E-37,
                        1.54136E-31, 7.52212E-27, 3.93990E-23, 3.52433E-20, 7.79589E-18, 5.72904E-16,
                        1.74058E-14, 2.69512E-13, 2.44040E-12, 1.44945E-11, 6.17139E-11, 2.01711E-10,
                        5.33260E-10, 1.18741E-09, 2.29863E-09, 3.98879E-09, 6.29280E-09, 9.14480E-09,
                        1.24213E-08, 1.59453E-08, 1.95377E-08, 2.30214E-08, 2.62489E-08, 2.90976E-08,
                        3.14730E-08, 3.33074E-08, 3.45627E-08, 3.52317E-08, 3.53388E-08, 3.49360E-08,
                        3.40961E-08, 3.29034E-08, 3.14449E-08, 2.92239E-08, 2.71822E-08, 2.52242E-08,
                        2.33596E-08, 2.15940E-08, 1.98971E-08, 1.83259E-08, 1.68559E-08, 1.54850E-08,
                        1.42099E-08, 1.30266E-08, 1.19308E-08, 1.09178E-08, 9.98290E-09, 9.12131E-09,
                        8.32834E-09, 7.59939E-09, 6.93004E-09, 6.31601E-09, 5.75328E-09, 5.23799E-09])
    svion_data[4] = np.array([1.61095E-99, 8.35428E-79, 1.12534E-65, 7.35596E-54, 1.66349E-44, 4.28461E-37,
                        3.17867E-31, 1.42606E-26, 6.94611E-23, 5.82950E-20, 1.21751E-17, 8.47951E-16,
                        2.45830E-14, 3.62783E-13, 3.13735E-12, 1.78504E-11, 7.31077E-11, 2.31009E-10,
                        5.93697E-10, 1.29204E-09, 2.45657E-09, 4.21556E-09, 6.59497E-09, 9.52031E-09,
                        1.28626E-08, 1.64393E-08, 2.00730E-08, 2.35814E-08, 2.68237E-08, 2.96761E-08,
                        3.20469E-08, 3.38696E-08, 3.51084E-08, 3.57575E-08, 3.58423E-08, 3.54158E-08,
                        3.45517E-08, 3.33345E-08, 3.18518E-08, 2.96073E-08, 2.75428E-08, 2.55630E-08,
                        2.36776E-08, 2.18918E-08, 2.01603E-08, 1.85667E-08, 1.70757E-08, 1.56850E-08,
                        1.43915E-08, 1.31911E-08, 1.20795E-08, 1.10521E-08, 1.01040E-08, 9.23029E-09,
                        8.42632E-09, 7.68739E-09, 7.00899E-09, 6.38678E-09, 5.81666E-09, 5.29471E-09])
    svion_data[5] = np.array([5.79413E-99, 4.32978E-80, 4.04173E-65, 2.46015E-53, 4.99802E-44, 1.15206E-36,
                        7.64481E-31, 3.08827E-26, 1.37501E-22, 1.06796E-19, 2.08389E-17, 1.36484E-15,
                        3.75124E-14, 5.25264E-13, 4.31955E-12, 2.34238E-11, 9.17108E-11, 2.78194E-10,
                        6.89925E-10, 1.45705E-09, 2.70478E-09, 4.57133E-09, 7.06605E-09, 1.01029E-08,
                        1.35441E-08, 1.72022E-08, 2.08979E-08, 2.44437E-08, 2.77081E-08, 3.05660E-08,
                        3.29296E-08, 3.47307E-08, 3.59426E-08, 3.65595E-08, 3.66085E-08, 3.61443E-08,
                        3.52416E-08, 3.39855E-08, 3.24646E-08, 3.01831E-08, 2.80829E-08, 2.60690E-08,
                        2.41511E-08, 2.23325E-08, 2.05493E-08, 1.89216E-08, 1.73985E-08, 1.59778E-08,
                        1.46565E-08, 1.34305E-08, 1.22953E-08, 1.12462E-08, 1.02784E-08, 9.38681E-09,
                        8.56659E-09, 7.81294E-09, 7.12125E-09, 6.48707E-09, 5.90617E-09, 5.37454E-09])
    svion_data[6] = np.array([1.87859E-98, 1.40828E-79, 1.73719E-64, 9.86529E-53, 1.83383E-43, 3.80019E-36,
                        2.24241E-30, 8.04056E-26, 3.22285E-22, 2.28138E-19, 4.09736E-17, 2.49005E-15,
                        6.41230E-14, 8.45699E-13, 6.57101E-12, 3.37408E-11, 1.25350E-10, 3.61885E-10,
                        8.57877E-10, 1.74116E-09, 3.13144E-09, 5.17959E-09, 7.86421E-09, 1.10816E-08,
                        1.46801E-08, 1.84662E-08, 2.22562E-08, 2.58620E-08, 2.91585E-08, 3.20232E-08,
                        3.43737E-08, 3.61292E-08, 3.72930E-08, 3.78538E-08, 3.78414E-08, 3.73118E-08,
                        3.63438E-08, 3.50210E-08, 3.34358E-08, 3.10922E-08, 2.89327E-08, 2.68622E-08,
                        2.48908E-08, 2.30114E-08, 2.11508E-08, 1.94683E-08, 1.78938E-08, 1.64255E-08,
                        1.50601E-08, 1.37936E-08, 1.26214E-08, 1.15386E-08, 1.05401E-08, 9.62076E-09,
                        8.77547E-09, 7.99924E-09, 7.28723E-09, 6.63481E-09, 6.03757E-09, 5.49131E-09])
    svion_data[7] = np.array([4.76247E-98, 3.65341E-79, 8.32708E-64, 4.58586E-52, 8.07957E-43, 1.55101E-35,
                        8.29936E-30, 2.65754E-25, 9.50276E-22, 6.01804E-19, 9.72796E-17, 5.36604E-15,
                        1.27001E-13, 1.56219E-12, 1.13825E-11, 5.49938E-11, 1.92597E-10, 5.25151E-10,
                        1.17907E-09, 2.27577E-09, 3.93717E-09, 6.31833E-09, 9.34496E-09, 1.28826E-08,
                        1.67565E-08, 2.07620E-08, 2.47094E-08, 2.84152E-08, 3.17593E-08, 3.46284E-08,
                        3.69484E-08, 3.86217E-08, 3.96963E-08, 4.01536E-08, 4.00283E-08, 3.93679E-08,
                        3.82765E-08, 3.68250E-08, 3.51177E-08, 3.26571E-08, 3.03859E-08, 2.82097E-08,
                        2.61389E-08, 2.41295E-08, 2.21482E-08, 2.03684E-08, 1.87041E-08, 1.71530E-08,
                        1.57121E-08, 1.43768E-08, 1.31422E-08, 1.20031E-08, 1.09538E-08, 9.98875E-09,
                        9.10248E-09, 8.28954E-09, 7.54472E-09, 6.86300E-09, 6.23963E-09, 5.67011E-09])
    svion_data[8] = np.array([1.00000E-99, 5.61821E-78, 3.83436E-63, 2.11526E-51, 3.67921E-42, 6.81104E-35,
                        3.41408E-29, 9.98793E-25, 3.21535E-21, 1.82402E-18, 2.64770E-16, 1.32016E-14,
                        2.85639E-13, 3.27481E-12, 2.23582E-11, 1.01543E-10, 3.34763E-10, 8.59961E-10,
                        1.82096E-09, 3.32143E-09, 5.51275E-09, 8.51515E-09, 1.21694E-08, 1.62870E-08,
                        2.06542E-08, 2.50466E-08, 2.92679E-08, 3.31435E-08, 3.65625E-08, 3.94290E-08,
                        4.16845E-08, 4.32367E-08, 4.41594E-08, 4.44400E-08, 4.41218E-08, 4.32027E-08,
                        4.18850E-08, 4.01906E-08, 3.82539E-08, 3.55724E-08, 3.30892E-08, 3.07113E-08,
                        2.84498E-08, 2.61614E-08, 2.39701E-08, 2.20041E-08, 2.01677E-08, 1.84590E-08,
                        1.68743E-08, 1.54087E-08, 1.40567E-08, 1.28121E-08, 1.16684E-08, 1.06192E-08,
                        9.65804E-09, 8.77868E-09, 7.97504E-09, 7.24135E-09, 6.57213E-09, 5.96219E-09])
    svion_data[9] = np.array([1.14633E-97, 1.73862E-77, 1.44527E-62, 8.11467E-51, 1.42552E-41, 2.61941E-34,
                        1.27174E-28, 3.52827E-24, 1.06201E-20, 5.59217E-18, 7.50647E-16, 3.45384E-14,
                        6.90757E-13, 7.38136E-12, 4.70531E-11, 2.00060E-10, 6.19325E-10, 1.49840E-09,
                        2.99645E-09, 5.17518E-09, 8.25817E-09, 1.22735E-08, 1.69404E-08, 2.19923E-08,
                        2.71602E-08, 3.21915E-08, 3.68805E-08, 4.10643E-08, 4.46444E-08, 4.75492E-08,
                        4.97420E-08, 5.11611E-08, 5.18855E-08, 5.19235E-08, 5.13318E-08, 5.00178E-08,
                        4.83521E-08, 4.62746E-08, 4.39673E-08, 4.09210E-08, 3.80806E-08, 3.53564E-08,
                        3.27619E-08, 2.99359E-08, 2.73761E-08, 2.50684E-08, 2.29138E-08, 2.09113E-08,
                        1.90573E-08, 1.73466E-08, 1.57725E-08, 1.43276E-08, 1.30043E-08, 1.17944E-08,
                        1.06902E-08, 9.68365E-09, 8.76739E-09, 7.93419E-09, 7.17724E-09, 6.49014E-09])
    svion_data[10] = np.array([6.62317E-93, 5.28961E-77, 4.27658E-62, 2.45697E-50, 4.39889E-41, 8.16171E-34,
                         3.94672E-28, 1.07648E-23, 3.14271E-20, 1.57904E-17, 1.98519E-15, 8.42615E-14,
                         1.54413E-12, 1.50839E-11, 8.83838E-11, 3.48677E-10, 1.01215E-09, 2.31817E-09,
                         4.42230E-09, 7.32915E-09, 1.13559E-08, 1.64255E-08, 2.21454E-08, 2.81825E-08,
                         3.42211E-08, 3.99837E-08, 4.52550E-08, 4.98787E-08, 5.37636E-08, 5.68549E-08,
                         5.91325E-08, 6.05690E-08, 6.12316E-08, 6.11518E-08, 6.03980E-08, 5.88607E-08,
                         5.69511E-08, 5.45952E-08, 5.19895E-08, 4.86271E-08, 4.54567E-08, 4.23923E-08,
                         3.94510E-08, 3.59069E-08, 3.28849E-08, 3.01241E-08, 2.75291E-08, 2.51038E-08,
                         2.28480E-08, 2.07590E-08, 1.88319E-08, 1.70601E-08, 1.54361E-08, 1.39516E-08,
                         1.25979E-08, 1.13661E-08, 1.02473E-08, 9.23290E-09, 8.31453E-09, 7.48414E-09])
    svion_data[11] = np.array([1.35178E-97, 1.11383E-76, 9.48789E-62, 5.60697E-50, 1.03378E-40, 1.96630E-33,
                         9.68188E-28, 2.66194E-23, 7.67512E-20, 3.67843E-17, 4.25046E-15, 1.62518E-13,
                         2.68430E-12, 2.37645E-11, 1.28540E-10, 4.76520E-10, 1.31939E-09, 2.91458E-09,
                         5.40551E-09, 8.75819E-09, 1.33570E-08, 1.90609E-08, 2.54164E-08, 3.20581E-08,
                         3.86475E-08, 4.48952E-08, 5.05800E-08, 5.55483E-08, 5.97106E-08, 6.30194E-08,
                         6.54607E-08, 6.70303E-08, 6.77783E-08, 6.77503E-08, 6.70197E-08, 6.55577E-08,
                         6.36554E-08, 6.13113E-08, 5.86882E-08, 5.52898E-08, 5.20668E-08, 4.89334E-08,
                         4.59066E-08, 4.18891E-08, 3.86203E-08, 3.55932E-08, 3.27162E-08, 2.99976E-08,
                         2.74413E-08, 2.50485E-08, 2.28178E-08, 2.07458E-08, 1.88279E-08, 1.70582E-08,
                         1.54302E-08, 1.39367E-08, 1.25703E-08, 1.13232E-08, 1.01878E-08, 9.15631E-09])
    svion_data[12] = np.array([1.38886E-97, 1.72309E-76, 1.51393E-61, 9.17508E-50, 1.73972E-40, 3.41627E-33,
                         1.73068E-27, 4.85262E-23, 1.39073E-19, 6.32804E-17, 6.68092E-15, 2.31948E-13,
                         3.53317E-12, 2.92803E-11, 1.51144E-10, 5.42325E-10, 1.46777E-09, 3.18987E-09,
                         5.84524E-09, 9.38334E-09, 1.42194E-08, 2.01856E-08, 2.68050E-08, 3.37000E-08,
                         4.05242E-08, 4.69837E-08, 5.28553E-08, 5.79866E-08, 6.22883E-08, 6.57156E-08,
                         6.82565E-08, 6.99171E-08, 7.07385E-08, 7.07719E-08, 7.00928E-08, 6.87396E-08,
                         6.69029E-08, 6.46425E-08, 6.20913E-08, 5.87614E-08, 5.56053E-08, 5.25366E-08,
                         4.95716E-08, 4.53956E-08, 4.20963E-08, 3.90253E-08, 3.60915E-08, 3.33040E-08,
                         3.06676E-08, 2.81842E-08, 2.58534E-08, 2.36728E-08, 2.16388E-08, 1.97464E-08,
                         1.79904E-08, 1.63646E-08, 1.48627E-08, 1.34783E-08, 1.22048E-08, 1.10356E-08])
    svion_data[13] = np.array([1.41725E-97, 2.04412E-76, 1.90737E-61, 1.15967E-49, 2.22502E-40, 4.43725E-33,
                         2.29912E-27, 6.54464E-23, 1.86941E-19, 8.19511E-17, 8.16294E-15, 2.68496E-13,
                         3.93032E-12, 3.16279E-11, 1.60150E-10, 5.67430E-10, 1.52259E-09, 3.28943E-09,
                         6.00195E-09, 9.60515E-09, 1.45243E-08, 2.05839E-08, 2.72967E-08, 3.42822E-08,
                         4.11893E-08, 4.77241E-08, 5.36627E-08, 5.88534E-08, 6.32068E-08, 6.66790E-08,
                         6.92589E-08, 7.09565E-08, 7.18090E-08, 7.18700E-08, 7.12157E-08, 6.99139E-08,
                         6.81115E-08, 6.58955E-08, 6.33856E-08, 6.00975E-08, 5.69850E-08, 5.39615E-08,
                         5.10434E-08, 4.68274E-08, 4.35414E-08, 4.04798E-08, 3.75518E-08, 3.47664E-08,
                         3.21286E-08, 2.96403E-08, 2.73011E-08, 2.51087E-08, 2.30594E-08, 2.11485E-08,
                         1.93705E-08, 1.77195E-08, 1.61893E-08, 1.47733E-08, 1.34652E-08, 1.22586E-08])
    svion_data[14] = np.array([1.72232E-97, 2.43222E-76, 2.03219E-61, 1.23882E-49, 2.50993E-40, 4.98024E-33,
                         2.57731E-27, 7.37279E-23, 2.10281E-19, 9.05148E-17, 8.78487E-15, 2.82727E-13,
                         4.07609E-12, 3.24564E-11, 1.63242E-10, 5.75830E-10, 1.54043E-09, 3.32149E-09,
                         6.05218E-09, 9.67522E-09, 1.46209E-08, 2.07114E-08, 2.74539E-08, 3.44679E-08,
                         4.14067E-08, 4.79679E-08, 5.39334E-08, 5.91455E-08, 6.35178E-08, 6.70040E-08,
                         6.95961E-08, 7.13052E-08, 7.21676E-08, 7.22375E-08, 7.15913E-08, 7.03073E-08,
                         6.85171E-08, 6.63171E-08, 6.38224E-08, 6.05500E-08, 5.74545E-08, 5.44489E-08,
                         5.15497E-08, 4.73231E-08, 4.40452E-08, 4.09908E-08, 3.80692E-08, 3.52893E-08,
                         3.26561E-08, 3.01718E-08, 2.78357E-08, 2.56458E-08, 2.35982E-08, 2.16882E-08,
                         1.99104E-08, 1.82589E-08, 1.67273E-08, 1.53092E-08, 1.39982E-08, 1.27877E-08])

    # data are given in  cm^3/s. Convert to m^3/s
    svion_data = svion_data * 1E-6

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,  # log10(T in kev)
                                   10 + 0.5*(np.linspace(1, 15, 15)-1) + 6)   # log10(n in m^-3)

    # perform 2D interpolation
    #sigv_ion_e = griddata(np.column_stack((Te_vals.flatten(), ne_vals.flatten())),
    #                      svion_data.flatten(),
    #                      (np.asarray(Te), np.asarray(ne)),
    #                      method='linear',
    #                      rescale=True)

    sigv_ion_e = Rbf(Te_vals, ne_vals, svion_data)

    return sigv_ion_e


def calc_svioni_degas():
    """
    """
    # populate the 15x60 array to be used for interpolations
    # rows correspond to ion temperature, columns correspond to neutral temperature

    iion_data = np.zeros((48, 48))
    iion_data[0] = np.array([1.77756e-20, 1.90510e-20, 2.06419e-20, 2.24163e-20, 2.44683e-20, 2.68267e-20,
                           2.95246e-20, 3.25805e-20, 3.60820e-20, 4.00541e-20, 4.45524e-20, 4.96601e-20,
                           5.54103e-20, 6.18958e-20, 6.92027e-20, 7.74279e-20, 8.66804e-20, 9.70832e-20,
                           1.08774e-19, 1.21909e-19, 1.62430e-19, 1.11879e-15, 5.80689e-14, 1.81662e-13,
                           9.80896e-13, 1.03146e-11, 2.10150e-11, 3.37141e-11, 1.40000e-10, 3.08910e-10,
                           4.57076e-10, 6.88557e-10, 1.14692e-09, 2.08410e-09, 3.08290e-09, 4.10744e-09,
                           5.49035e-09, 7.57584e-09, 1.04521e-08, 1.38034e-08, 1.71306e-08, 2.01763e-08,
                           2.31246e-08, 2.70005e-08, 3.37111e-08, 3.74630e-08, 2.91758e-08, 2.76207e-08])
    iion_data[1] = np.array([1.87357e-20, 1.99446e-20, 2.14535e-20, 2.31606e-20, 2.51515e-20, 2.74539e-20,
                           3.01000e-20, 3.31272e-20, 3.65785e-20, 4.04846e-20, 4.49415e-20, 4.99886e-20,
                           5.57195e-20, 6.21714e-20, 6.94483e-20, 7.76467e-20, 8.68755e-20, 9.72571e-20,
                           1.08929e-19, 1.22047e-19, 1.74459e-19, 1.20419e-15, 5.82568e-14, 1.81677e-13,
                           1.00384e-12, 1.03154e-11, 2.10134e-11, 3.39069e-11, 1.40124e-10, 3.08867e-10,
                           4.57182e-10, 6.88755e-10, 1.14797e-09, 2.08401e-09, 3.08297e-09, 4.10782e-09,
                           5.49089e-09, 7.57631e-09, 1.04524e-08, 1.38035e-08, 1.71306e-08, 2.01766e-08,
                           2.31251e-08, 2.70008e-08, 3.37111e-08, 3.74624e-08, 2.91763e-08, 2.76210e-08])
    iion_data[2] = np.array([1.99662e-20, 2.10218e-20, 2.23782e-20, 2.40713e-20, 2.59867e-20, 2.82205e-20,
                           3.08038e-20, 3.37728e-20, 3.71693e-20, 4.10417e-20, 4.54454e-20, 5.04252e-20,
                           5.61089e-20, 6.25183e-20, 6.97574e-20, 7.79223e-20, 8.71211e-20, 9.74759e-20,
                           1.09124e-19, 1.22221e-19, 1.94421e-19, 1.31344e-15, 5.84926e-14, 1.81713e-13,
                           1.03259e-12, 1.03165e-11, 2.10113e-11, 3.41503e-11, 1.40279e-10, 3.08814e-10,
                           4.57315e-10, 6.89006e-10, 1.14927e-09, 2.08390e-09, 3.08306e-09, 4.10829e-09,
                           5.49158e-09, 7.57691e-09, 1.04526e-08, 1.38036e-08, 1.71307e-08, 2.01769e-08,
                           2.31257e-08, 2.70011e-08, 3.37111e-08, 3.74617e-08, 2.91768e-08, 2.76213e-08])
    iion_data[3] = np.array([2.13626e-20, 2.24025e-20, 2.36641e-20, 2.51870e-20, 2.70084e-20, 2.91575e-20,
                           3.16639e-20, 3.45624e-20, 3.78937e-20, 4.17047e-20, 4.60495e-20, 5.09906e-20,
                           5.65990e-20, 6.29552e-20, 7.01467e-20, 7.82691e-20, 8.74302e-20, 9.77514e-20,
                           1.09370e-19, 1.22439e-19, 2.28806e-19, 1.45363e-15, 5.87880e-14, 1.81787e-13,
                           1.06859e-12, 1.03181e-11, 2.10088e-11, 3.44579e-11, 1.40476e-10, 3.08748e-10,
                           4.57483e-10, 6.89326e-10, 1.15090e-09, 2.08377e-09, 3.08317e-09, 4.10889e-09,
                           5.49243e-09, 7.57767e-09, 1.04530e-08, 1.38037e-08, 1.71308e-08, 2.01773e-08,
                           2.31264e-08, 2.70016e-08, 3.37111e-08, 3.74608e-08, 2.91775e-08, 2.76217e-08])
    iion_data[4] = np.array([2.30139e-20, 2.39692e-20, 2.51360e-20, 2.65516e-20, 2.82603e-20, 3.03039e-20,
                           3.27153e-20, 3.55275e-20, 3.87797e-20, 4.25174e-20, 4.67934e-20, 5.16684e-20,
                           5.72123e-20, 6.35051e-20, 7.06369e-20, 7.87059e-20, 8.78194e-20, 9.80983e-20,
                           1.09679e-19, 1.22715e-19, 2.90359e-19, 1.63418e-15, 5.91577e-14, 1.81928e-13,
                           1.11358e-12, 1.03203e-11, 2.10058e-11, 3.48466e-11, 1.40725e-10, 3.08665e-10,
                           4.57694e-10, 6.89734e-10, 1.15294e-09, 2.08362e-09, 3.08331e-09, 4.10964e-09,
                           5.49352e-09, 7.57862e-09, 1.04534e-08, 1.38038e-08, 1.71310e-08, 2.01778e-08,
                           2.31273e-08, 2.70021e-08, 3.37111e-08, 3.74596e-08, 2.91784e-08, 2.76222e-08])
    iion_data[5] = np.array([2.49537e-20, 2.58220e-20, 2.68939e-20, 2.82031e-20, 2.97913e-20, 3.17085e-20,
                           3.40016e-20, 3.67071e-20, 3.98625e-20, 4.35115e-20, 4.77053e-20, 5.25030e-20,
                           5.79729e-20, 6.41933e-20, 7.12539e-20, 7.92559e-20, 8.83095e-20, 9.85350e-20,
                           1.10068e-19, 1.23062e-19, 4.04914e-19, 1.86759e-15, 5.96197e-14, 1.82191e-13,
                           1.16974e-12, 1.03236e-11, 2.10022e-11, 3.53380e-11, 1.41040e-10, 3.08561e-10,
                           4.57959e-10, 6.90256e-10, 1.15546e-09, 2.08343e-09, 3.08349e-09, 4.11059e-09,
                           5.49488e-09, 7.57981e-09, 1.04540e-08, 1.38039e-08, 1.71312e-08, 2.01785e-08,
                           2.31284e-08, 2.70028e-08, 3.37111e-08, 3.74582e-08, 2.91795e-08, 2.76229e-08])
    iion_data[6] = np.array([2.72177e-20, 2.79985e-20, 2.89727e-20, 3.01754e-20, 3.16444e-20, 3.34264e-20,
                           3.55776e-20, 3.81504e-20, 4.11861e-20, 4.47265e-20, 4.88207e-20, 5.35262e-20,
                           5.89094e-20, 6.50467e-20, 7.20261e-20, 7.99482e-20, 8.89266e-20, 9.90848e-20,
                           1.10558e-19, 1.23498e-19, 6.26456e-19, 2.17058e-15, 6.01956e-14, 1.82670e-13,
                           1.23966e-12, 1.03285e-11, 2.09982e-11, 3.59598e-11, 1.41440e-10, 3.08432e-10,
                           4.58293e-10, 6.90930e-10, 1.15859e-09, 2.08322e-09, 3.08371e-09, 4.11178e-09,
                           5.49659e-09, 7.58132e-09, 1.04547e-08, 1.38041e-08, 1.71314e-08, 2.01793e-08,
                           2.31298e-08, 2.70037e-08, 3.37111e-08, 3.74563e-08, 2.91809e-08, 2.76237e-08])
    iion_data[7] = np.array([2.98402e-20, 3.05388e-20, 3.14148e-20, 3.25080e-20, 3.38574e-20, 3.55056e-20,
                           3.75051e-20, 3.99187e-20, 4.28054e-20, 4.62115e-20, 5.01839e-20, 5.47778e-20,
                           6.00574e-20, 6.60974e-20, 7.29836e-20, 8.08146e-20, 8.97034e-20, 9.97773e-20,
                           1.11175e-19, 1.24048e-19, 1.07098e-18, 2.56542e-15, 6.09118e-14, 1.83533e-13,
                           1.32650e-12, 1.03355e-11, 2.09941e-11, 3.67464e-11, 1.41947e-10, 3.08272e-10,
                           4.58712e-10, 6.91802e-10, 1.16245e-09, 2.08299e-09, 3.08399e-09, 4.11328e-09,
                           5.49874e-09, 7.58321e-09, 1.04556e-08, 1.38043e-08, 1.71317e-08, 2.01803e-08,
                           2.31316e-08, 2.70048e-08, 3.37110e-08, 3.74540e-08, 2.91826e-08, 2.76247e-08])
    iion_data[8] = np.array([3.28571e-20, 3.34812e-20, 3.42651e-20, 3.52480e-20, 3.64745e-20, 3.79886e-20,
                           3.98379e-20, 4.20814e-20, 4.47895e-20, 4.80285e-20, 5.18502e-20, 5.63073e-20,
                           6.14616e-20, 6.73855e-20, 7.41625e-20, 8.18890e-20, 9.06754e-20, 1.00649e-19,
                           1.11952e-19, 1.24740e-19, 1.99382e-18, 3.08177e-15, 6.17991e-14, 1.85085e-13,
                           1.43399e-12, 1.03460e-11, 2.09908e-11, 3.77415e-11, 1.42589e-10, 3.08074e-10,
                           4.59239e-10, 6.92938e-10, 1.16719e-09, 2.08276e-09, 3.08433e-09, 4.11516e-09,
                           5.50145e-09, 7.58559e-09, 1.04568e-08, 1.38046e-08, 1.71320e-08, 2.01817e-08,
                           2.31338e-08, 2.70063e-08, 3.37110e-08, 3.74511e-08, 2.91848e-08, 2.76260e-08])
    iion_data[9] = np.array([3.63095e-20, 3.68662e-20, 3.75666e-20, 3.84461e-20, 3.95489e-20, 4.09251e-20,
                           4.26239e-20, 4.46989e-20, 4.72161e-20, 5.02547e-20, 5.38888e-20, 5.81769e-20,
                           6.31778e-20, 6.89611e-20, 7.56078e-20, 8.32117e-20, 9.18809e-20, 1.01740e-19,
                           1.12930e-19, 1.25612e-19, 3.96753e-18, 3.75879e-15, 6.28935e-14, 1.87868e-13,
                           1.56652e-12, 1.03614e-11, 2.09902e-11, 3.89992e-11, 1.43404e-10, 3.07831e-10,
                           4.59900e-10, 6.94431e-10, 1.17298e-09, 2.08254e-09, 3.08477e-09, 4.11753e-09,
                           5.50485e-09, 7.58858e-09, 1.04582e-08, 1.38050e-08, 1.71325e-08, 2.01834e-08,
                           2.31365e-08, 2.70081e-08, 3.37110e-08, 3.74474e-08, 2.91876e-08, 2.76276e-08])
    iion_data[10] = np.array([4.02437e-20, 4.07399e-20, 4.13646e-20, 4.21504e-20, 4.31372e-20, 4.43746e-20,
                            4.59187e-20, 4.78248e-20, 5.01529e-20, 5.29773e-20, 5.63867e-20, 6.04643e-20,
                            6.52755e-20, 7.08867e-20, 7.73756e-20, 8.48333e-20, 9.33651e-20, 1.03092e-19,
                            1.14154e-19, 1.26709e-19, 8.29118e-18, 4.64755e-15, 6.42356e-14, 1.92856e-13,
                            1.72911e-12, 1.03841e-11, 2.09965e-11, 4.05856e-11, 1.44436e-10, 3.07535e-10,
                            4.60730e-10, 6.96411e-10, 1.18002e-09, 2.08240e-09, 3.08531e-09, 4.12050e-09,
                            5.50914e-09, 7.59235e-09, 1.04600e-08, 1.38054e-08, 1.71331e-08, 2.01856e-08,
                            2.31400e-08, 2.70103e-08, 3.37109e-08, 3.74428e-08, 2.91911e-08, 2.76296e-08])
    iion_data[11] = np.array([4.47119e-20, 4.51542e-20, 4.57109e-20, 4.64118e-20, 4.72935e-20, 4.84007e-20,
                            4.97891e-20, 5.15216e-20, 5.36604e-20, 5.62725e-20, 5.94415e-20, 6.32669e-20,
                            6.78420e-20, 7.32404e-20, 7.95362e-20, 8.68169e-20, 9.51846e-20, 1.04757e-19,
                            1.15671e-19, 1.28101e-19, 1.79204e-17, 5.81333e-15, 6.58696e-14, 2.01769e-13,
                            1.92740e-12, 1.04174e-11, 2.10188e-11, 4.25804e-11, 1.45740e-10, 3.07180e-10,
                            4.61771e-10, 6.99059e-10, 1.18851e-09, 2.08240e-09, 3.08599e-09, 4.12425e-09,
                            5.51452e-09, 7.59708e-09, 1.04622e-08, 1.38060e-08, 1.71338e-08, 2.01884e-08,
                            2.31442e-08, 2.70132e-08, 3.37108e-08, 3.74370e-08, 2.91954e-08, 2.76322e-08])
    iion_data[12] = np.array([4.97735e-20, 5.01676e-20, 5.06638e-20, 5.12885e-20, 5.20750e-20, 5.30642e-20,
                            5.43065e-20, 5.58643e-20, 5.78082e-20, 6.02079e-20, 6.31388e-20, 6.66945e-20,
                            7.09866e-20, 7.61200e-20, 8.21770e-20, 8.92411e-20, 9.74102e-20, 1.06799e-19,
                            1.17540e-19, 1.30015e-19, 3.95243e-17, 7.33722e-15, 6.78429e-14, 2.17606e-13,
                            2.16746e-12, 1.04657e-11, 2.10750e-11, 4.50760e-11, 1.47381e-10, 3.06762e-10,
                            4.63077e-10, 7.02631e-10, 1.19867e-09, 2.08268e-09, 3.08683e-09, 4.12895e-09,
                            5.52128e-09, 7.60303e-09, 1.04651e-08, 1.38068e-08, 1.71347e-08, 2.01919e-08,
                            2.31496e-08, 2.70169e-08, 3.37107e-08, 3.74296e-08, 2.92009e-08, 2.76354e-08])
    iion_data[13] = np.array([5.54955e-20, 5.58467e-20, 5.62890e-20, 5.68457e-20, 5.75467e-20, 5.84291e-20,
                            5.95390e-20, 6.09329e-20, 6.26807e-20, 6.48619e-20, 6.75544e-20, 7.08429e-20,
                            7.48324e-20, 7.96483e-20, 8.54081e-20, 9.22041e-20, 1.00130e-19, 1.09296e-19,
                            1.19830e-19, 1.33926e-19, 8.78319e-17, 9.31615e-15, 7.02091e-14, 2.45437e-13,
                            2.45556e-12, 1.05348e-11, 2.12003e-11, 4.81760e-11, 1.49430e-10, 3.06286e-10,
                            4.64713e-10, 7.07479e-10, 1.21075e-09, 2.08345e-09, 3.08788e-09, 4.13485e-09,
                            5.52976e-09, 7.61050e-09, 1.04687e-08, 1.38077e-08, 1.71359e-08, 2.01965e-08,
                            2.31561e-08, 2.70216e-08, 3.37106e-08, 3.74204e-08, 2.92078e-08, 2.76395e-08])
    iion_data[14] = np.array([6.19538e-20, 6.22669e-20, 6.26611e-20, 6.31573e-20, 6.37820e-20, 6.45684e-20,
                            6.55585e-20, 6.68038e-20, 6.83678e-20, 7.03289e-20, 7.27763e-20, 7.57973e-20,
                            7.94871e-20, 8.39634e-20, 8.93668e-20, 9.58294e-20, 1.03455e-19, 1.12348e-19,
                            1.22632e-19, 1.51694e-19, 1.94194e-16, 1.18599e-14, 7.30543e-14, 2.93504e-13,
                            2.79771e-12, 1.06322e-11, 2.14603e-11, 5.19895e-11, 1.51967e-10, 3.05772e-10,
                            4.66769e-10, 7.14081e-10, 1.22498e-09, 2.08500e-09, 3.08921e-09, 4.14226e-09,
                            5.54041e-09, 7.61988e-09, 1.04732e-08, 1.38090e-08, 1.71375e-08, 2.02022e-08,
                            2.31642e-08, 2.70277e-08, 3.37104e-08, 3.74089e-08, 2.92165e-08, 2.76446e-08])
    iion_data[15] = np.array([6.92343e-20, 6.95134e-20, 6.98646e-20, 7.03069e-20, 7.08636e-20, 7.15645e-20,
                            7.24470e-20, 7.35578e-20, 7.49551e-20, 7.67100e-20, 7.89104e-20, 8.16563e-20,
                            8.50459e-20, 8.91859e-20, 9.42085e-20, 1.00271e-19, 1.07522e-19, 1.16078e-19,
                            1.26056e-19, 2.73949e-19, 4.21790e-16, 1.50838e-14, 7.66166e-14, 3.74467e-13,
                            3.19914e-12, 1.07667e-11, 2.19707e-11, 5.66233e-11, 1.55067e-10, 3.05268e-10,
                            4.69370e-10, 7.23053e-10, 1.24161e-09, 2.08773e-09, 3.09088e-09, 4.15155e-09,
                            5.55376e-09, 7.63165e-09, 1.04790e-08, 1.38105e-08, 1.71395e-08, 2.02095e-08,
                            2.31742e-08, 2.70354e-08, 3.37102e-08, 3.73943e-08, 2.92273e-08, 2.76510e-08])
    iion_data[16] = np.array([7.74335e-20, 7.76822e-20, 7.79953e-20, 7.83894e-20, 7.88856e-20, 7.95103e-20,
                            8.02967e-20, 8.12868e-20, 8.25332e-20, 8.41010e-20, 8.60700e-20, 8.85389e-20,
                            9.16199e-20, 9.54231e-20, 1.00068e-19, 1.05704e-19, 1.12506e-19, 1.20642e-19,
                            1.30242e-19, 1.12517e-18, 8.88771e-16, 1.90976e-14, 8.17313e-14, 5.06389e-13,
                            3.66368e-12, 1.09502e-11, 2.29208e-11, 6.21707e-11, 1.58806e-10, 3.04875e-10,
                            4.72719e-10, 7.35150e-10, 1.26089e-09, 2.09218e-09, 3.09305e-09, 4.16317e-09,
                            5.57047e-09, 7.64639e-09, 1.04862e-08, 1.38125e-08, 1.71422e-08, 2.02187e-08,
                            2.31864e-08, 2.70453e-08, 3.37098e-08, 3.73761e-08, 2.92409e-08, 2.76591e-08])
    iion_data[17] = np.array([8.66602e-20, 8.68818e-20, 8.71609e-20, 8.75121e-20, 8.79544e-20, 8.85111e-20,
                            8.92120e-20, 9.00944e-20, 9.12053e-20, 9.26038e-20, 9.43629e-20, 9.65721e-20,
                            9.93423e-20, 1.02799e-19, 1.07067e-19, 1.12278e-19, 1.18601e-19, 1.26234e-19,
                            1.35403e-19, 6.41799e-18, 1.79553e-15, 2.39901e-14, 9.12250e-14, 7.12649e-13,
                            4.19330e-12, 1.12045e-11, 2.45925e-11, 6.87004e-11, 1.63255e-10, 3.04778e-10,
                            4.77177e-10, 7.51226e-10, 1.28310e-09, 2.09897e-09, 3.09593e-09, 4.17769e-09,
                            5.59137e-09, 7.66486e-09, 1.04954e-08, 1.38151e-08, 1.71458e-08, 2.02301e-08,
                            2.32014e-08, 2.70582e-08, 3.37093e-08, 3.73531e-08, 2.92578e-08, 2.76693e-08])
    iion_data[18] = np.array([9.70368e-20, 9.72343e-20, 9.74830e-20, 9.77961e-20, 9.81902e-20, 9.86864e-20,
                            9.93111e-20, 1.00098e-19, 1.01088e-19, 1.02334e-19, 1.03903e-19, 1.05877e-19,
                            1.08356e-19, 1.11464e-19, 1.15343e-19, 1.20131e-19, 1.25978e-19, 1.33073e-19,
                            1.44071e-19, 3.44714e-17, 3.44218e-15, 2.98256e-14, 1.13421e-13, 1.01986e-12,
                            4.78805e-12, 1.15880e-11, 2.73584e-11, 7.62479e-11, 1.68497e-10, 3.05287e-10,
                            4.83403e-10, 7.72160e-10, 1.30856e-09, 2.10882e-09, 3.09994e-09, 4.19579e-09,
                            5.61744e-09, 7.68794e-09, 1.05070e-08, 1.38185e-08, 1.71508e-08, 2.02441e-08,
                            2.32197e-08, 2.70750e-08, 3.37085e-08, 3.73244e-08, 2.92790e-08, 2.76822e-08])
    iion_data[19] = np.array([1.08701e-19, 1.08877e-19, 1.09099e-19, 1.09378e-19, 1.09729e-19, 1.10171e-19,
                            1.10728e-19, 1.11429e-19, 1.12311e-19, 1.13422e-19, 1.14821e-19, 1.16581e-19,
                            1.18796e-19, 1.21577e-19, 1.25065e-19, 1.29417e-19, 1.34789e-19, 1.41350e-19,
                            2.33469e-19, 1.57584e-16, 6.21245e-15, 3.67766e-14, 1.68728e-13, 1.45333e-12,
                            5.44708e-12, 1.22637e-11, 3.16419e-11, 8.48166e-11, 1.74642e-10, 3.06889e-10,
                            4.92526e-10, 7.98753e-10, 1.33765e-09, 2.12247e-09, 3.10581e-09, 4.21829e-09,
                            5.64992e-09, 7.71674e-09, 1.05217e-08, 1.38230e-08, 1.71577e-08, 2.02614e-08,
                            2.32423e-08, 2.70967e-08, 3.37075e-08, 3.72883e-08, 2.93053e-08, 2.76983e-08])
    iion_data[20] = np.array([1.21808e-19, 1.21965e-19, 1.22162e-19, 1.22411e-19, 1.22724e-19, 1.23118e-19,
                            1.23614e-19, 1.24239e-19, 1.25025e-19, 1.26015e-19, 1.27262e-19, 1.28831e-19,
                            1.30806e-19, 1.33291e-19, 1.36412e-19, 1.40325e-19, 1.45208e-19, 1.51260e-19,
                            1.98749e-18, 5.96166e-16, 1.05047e-14, 4.63341e-14, 2.97344e-13, 2.03107e-12,
                            6.17546e-12, 1.36178e-11, 3.78340e-11, 9.43921e-11, 1.81856e-10, 3.10342e-10,
                            5.06223e-10, 8.31634e-10, 1.37091e-09, 2.14080e-09, 3.11485e-09, 4.24613e-09,
                            5.69026e-09, 7.75262e-09, 1.05405e-08, 1.38291e-08, 1.71672e-08, 2.02825e-08,
                            2.32700e-08, 2.71247e-08, 3.37062e-08, 3.72430e-08, 2.93379e-08, 2.77186e-08])
    iion_data[21] = np.array([1.36531e-19, 1.36670e-19, 1.36846e-19, 1.37068e-19, 1.37347e-19, 1.37698e-19,
                            1.38141e-19, 1.38697e-19, 1.39398e-19, 1.40281e-19, 1.41392e-19, 1.42790e-19,
                            1.44551e-19, 1.46767e-19, 1.49555e-19, 1.53056e-19, 1.57447e-19, 1.69750e-19,
                            2.47882e-17, 1.85189e-15, 1.66401e-14, 6.77047e-14, 5.60386e-13, 2.75895e-12,
                            7.02780e-12, 1.63600e-11, 4.61940e-11, 1.04974e-10, 1.90390e-10, 3.16946e-10,
                            5.26556e-10, 8.71239e-10, 1.40907e-09, 2.16485e-09, 3.12911e-09, 4.28045e-09,
                            5.74017e-09, 7.79719e-09, 1.05643e-08, 1.38377e-08, 1.71798e-08, 2.03081e-08,
                            2.33041e-08, 2.71606e-08, 3.37045e-08, 3.71864e-08, 2.93783e-08, 2.77442e-08])
    iion_data[22] = np.array([1.53065e-19, 1.53190e-19, 1.53347e-19, 1.53544e-19, 1.53793e-19, 1.54106e-19,
                            1.54500e-19, 1.54996e-19, 1.55621e-19, 1.56407e-19, 1.57397e-19, 1.58644e-19,
                            1.60213e-19, 1.62189e-19, 1.64675e-19, 1.67804e-19, 1.71748e-19, 7.65507e-19,
                            2.06535e-16, 4.74373e-15, 2.53229e-14, 1.36913e-13, 1.02527e-12, 3.63239e-12,
                            8.28872e-12, 2.14455e-11, 5.67838e-11, 1.16629e-10, 2.00675e-10, 3.29084e-10,
                            5.55562e-10, 9.17888e-10, 1.45316e-09, 2.19606e-09, 3.15147e-09, 4.32259e-09,
                            5.80166e-09, 7.85240e-09, 1.05947e-08, 1.38503e-08, 1.71962e-08, 2.03391e-08,
                            2.33463e-08, 2.72063e-08, 3.37025e-08, 3.71156e-08, 2.94281e-08, 2.77764e-08])
    iion_data[23] = np.array([1.71631e-19, 1.71742e-19, 1.71882e-19, 1.72058e-19, 1.72279e-19, 1.72559e-19,
                            1.72910e-19, 1.73352e-19, 1.73909e-19, 1.74610e-19, 1.75492e-19, 1.76603e-19,
                            1.78001e-19, 1.79762e-19, 1.81979e-19, 1.84769e-19, 2.04632e-19, 1.96127e-17,
                            1.10951e-15, 1.01928e-14, 4.46205e-14, 3.41952e-13, 1.73875e-12, 4.68472e-12,
                            1.08443e-11, 2.97327e-11, 6.94961e-11, 1.29565e-10, 2.13804e-10, 3.50516e-10,
                            5.94923e-10, 9.71972e-10, 1.50466e-09, 2.23636e-09, 3.18521e-09, 4.37435e-09,
                            5.87697e-09, 7.92056e-09, 1.06340e-08, 1.38684e-08, 1.72170e-08, 2.03768e-08,
                            2.33987e-08, 2.72641e-08, 3.37007e-08, 3.70273e-08, 2.94890e-08, 2.78170e-08])
    iion_data[24] = np.array([1.92474e-19, 1.92573e-19, 1.92698e-19, 1.92855e-19, 1.93052e-19, 1.93301e-19,
                            1.93614e-19, 1.94008e-19, 1.94504e-19, 1.95129e-19, 1.95915e-19, 1.96905e-19,
                            1.98152e-19, 1.99721e-19, 2.01696e-19, 2.05466e-19, 2.82148e-18, 2.72229e-16,
                            4.03455e-15, 2.03255e-14, 1.25692e-13, 8.11864e-13, 2.71598e-12, 6.31577e-12,
                            1.63239e-11, 4.16098e-11, 8.42086e-11, 1.44353e-10, 2.32916e-10, 3.85442e-10,
                            6.45968e-10, 1.03423e-09, 1.56561e-09, 2.28838e-09, 3.23375e-09, 4.43860e-09,
                            5.96850e-09, 8.00450e-09, 1.06856e-08, 1.38941e-08, 1.72430e-08, 2.04227e-08,
                            2.34637e-08, 2.73370e-08, 3.36993e-08, 3.69173e-08, 2.95630e-08, 2.78680e-08])
    iion_data[25] = np.array([2.15871e-19, 2.15959e-19, 2.16070e-19, 2.16210e-19, 2.16386e-19, 2.16608e-19,
                            2.16887e-19, 2.17238e-19, 2.17681e-19, 2.18237e-19, 2.18938e-19, 2.19821e-19,
                            2.20931e-19, 2.22330e-19, 2.24697e-19, 1.11268e-18, 9.98601e-17, 1.84898e-15,
                            1.11491e-14, 6.15567e-14, 4.20629e-13, 1.64694e-12, 4.11853e-12, 1.01091e-11,
                            2.63492e-11, 5.69366e-11, 1.01088e-10, 1.63199e-10, 2.64136e-10, 4.36702e-10,
                            7.09789e-10, 1.10602e-09, 1.63883e-09, 2.35565e-09, 3.30094e-09, 4.52018e-09,
                            6.07873e-09, 8.10813e-09, 1.07538e-08, 1.39293e-08, 1.72754e-08, 2.04791e-08,
                            2.35448e-08, 2.74283e-08, 3.36986e-08, 3.67811e-08, 2.96523e-08, 2.79323e-08])
    iion_data[26] = np.array([2.42133e-19, 2.42211e-19, 2.42310e-19, 2.42435e-19, 2.42592e-19, 2.42789e-19,
                            2.43038e-19, 2.43351e-19, 2.43745e-19, 2.44242e-19, 2.44866e-19, 2.45653e-19,
                            2.46643e-19, 2.50124e-19, 1.31308e-18, 7.44757e-17, 1.21524e-15, 7.66475e-15,
                            4.33092e-14, 2.81247e-13, 1.14068e-12, 3.00216e-12, 7.32270e-12, 1.88259e-11,
                            4.14907e-11, 7.53415e-11, 1.21812e-10, 1.92914e-10, 3.13871e-10, 5.05344e-10,
                            7.87788e-10, 1.18983e-09, 1.72818e-09, 2.44290e-09, 3.39181e-09, 4.62593e-09,
                            6.21064e-09, 8.23678e-09, 1.08429e-08, 1.39767e-08, 1.73164e-08, 2.05488e-08,
                            2.36459e-08, 2.75421e-08, 3.36981e-08, 3.66136e-08, 2.97588e-08, 2.80132e-08])
    iion_data[27] = np.array([2.71607e-19, 2.71677e-19, 2.71766e-19, 2.71877e-19, 2.72017e-19, 2.72193e-19,
                            2.72414e-19, 2.72693e-19, 2.73045e-19, 2.73487e-19, 2.74044e-19, 2.74832e-19,
                            3.17182e-19, 4.53498e-18, 1.24378e-16, 1.32570e-15, 7.85412e-15, 4.80182e-14,
                            2.70091e-13, 9.89996e-13, 2.59937e-12, 6.45518e-12, 1.59891e-11, 3.42503e-11,
                            6.14764e-11, 9.82759e-11, 1.53913e-10, 2.43666e-10, 3.85056e-10, 5.91563e-10,
                            8.83029e-10, 1.29022e-09, 1.83872e-09, 2.55631e-09, 3.51297e-09, 4.76357e-09,
                            6.36929e-09, 8.39577e-09, 1.09575e-08, 1.40400e-08, 1.73693e-08, 2.06354e-08,
                            2.37721e-08, 2.76834e-08, 3.36969e-08, 3.64102e-08, 2.98841e-08, 2.81153e-08])
    iion_data[28] = np.array([3.04686e-19, 3.04748e-19, 3.04827e-19, 3.04926e-19, 3.05051e-19, 3.05208e-19,
                            3.05405e-19, 3.05654e-19, 3.05967e-19, 3.06663e-19, 3.41878e-19, 2.00842e-18,
                            3.76004e-17, 4.07306e-16, 2.75661e-15, 1.65197e-14, 9.22618e-14, 3.85187e-13,
                            1.16696e-12, 2.98265e-12, 7.37733e-12, 1.68702e-11, 3.32105e-11, 5.66198e-11,
                            8.86685e-11, 1.36504e-10, 2.10547e-10, 3.21182e-10, 4.78287e-10, 6.97366e-10,
                            1.00028e-09, 1.41497e-09, 1.97689e-09, 2.70413e-09, 3.67358e-09, 4.94240e-09,
                            6.56251e-09, 8.58883e-09, 1.11025e-08, 1.41256e-08, 1.74389e-08, 2.07441e-08,
                            2.39291e-08, 2.78577e-08, 3.36938e-08, 3.61658e-08, 3.00294e-08, 2.82439e-08])
    iion_data[29] = np.array([3.41808e-19, 3.41863e-19, 3.41933e-19, 3.42022e-19, 3.42133e-19, 3.42301e-19,
                            3.43601e-19, 3.72556e-19, 9.10349e-19, 7.94099e-18, 7.31256e-17, 5.30261e-16,
                            3.23470e-15, 1.73670e-14, 7.82621e-14, 2.82216e-13, 8.31881e-13, 2.15421e-12,
                            5.17874e-12, 1.14157e-11, 2.22854e-11, 3.86325e-11, 6.14140e-11, 9.31951e-11,
                            1.39853e-10, 2.06859e-10, 2.99119e-10, 4.26172e-10, 5.94932e-10, 8.30210e-10,
                            1.14550e-09, 1.57581e-09, 2.15229e-09, 2.89939e-09, 3.88705e-09, 5.17107e-09,
                            6.79901e-09, 8.81844e-09, 1.12852e-08, 1.42424e-08, 1.75328e-08, 2.08813e-08,
                            2.41241e-08, 2.80709e-08, 3.36864e-08, 3.58754e-08, 3.01957e-08, 2.84060e-08])
    iion_data[30] = np.array([3.88034e-19, 4.09152e-19, 5.19167e-19, 1.10175e-18, 4.10171e-18, 1.90432e-17,
                            9.01784e-17, 4.08770e-16, 1.73380e-15, 6.78822e-15, 2.42603e-14, 7.84834e-14,
                            2.29232e-13, 6.07468e-13, 1.47450e-12, 3.30232e-12, 6.82701e-12, 1.29773e-11,
                            2.26848e-11, 3.68600e-11, 5.66712e-11, 8.37197e-11, 1.20320e-10, 1.69612e-10,
                            2.33580e-10, 3.14654e-10, 4.20941e-10, 5.60353e-10, 7.45397e-10, 9.98748e-10,
                            1.33403e-09, 1.79055e-09, 2.38392e-09, 3.15998e-09, 4.16587e-09, 5.45657e-09,
                            7.08586e-09, 9.08933e-09, 1.15151e-08, 1.44020e-08, 1.76631e-08, 2.10548e-08,
                            2.43651e-08, 2.83290e-08, 3.36711e-08, 3.55348e-08, 3.03843e-08, 2.86102e-08])
    iion_data[31] = np.array([1.57086e-14, 2.34844e-14, 3.65708e-14, 5.91047e-14, 9.86192e-14, 1.68765e-13,
                            2.93962e-13, 5.16976e-13, 9.10496e-13, 1.59342e-12, 2.75131e-12, 4.65826e-12,
                            7.69408e-12, 1.23489e-11, 1.92091e-11, 2.89289e-11, 4.22006e-11, 5.97402e-11,
                            8.22768e-11, 1.10540e-10, 1.45289e-10, 1.87400e-10, 2.37877e-10, 2.98065e-10,
                            3.71271e-10, 4.64495e-10, 5.85020e-10, 7.40138e-10, 9.48243e-10, 1.22960e-09,
                            1.59720e-09, 2.07956e-09, 2.69403e-09, 3.49929e-09, 4.51839e-09, 5.80748e-09,
                            7.43393e-09, 9.41478e-09, 1.18026e-08, 1.46225e-08, 1.78461e-08, 2.12742e-08,
                            2.46612e-08, 2.86367e-08, 3.36420e-08, 3.51428e-08, 3.05950e-08, 2.88676e-08])
    iion_data[32] = np.array([2.82089e-11, 2.97197e-11, 3.16108e-11, 3.39741e-11, 3.69210e-11, 4.05853e-11,
                            4.51253e-11, 5.07249e-11, 5.75924e-11, 6.59578e-11, 7.60662e-11, 8.81687e-11,
                            1.02512e-10, 1.19326e-10, 1.38820e-10, 1.61179e-10, 1.86579e-10, 2.15221e-10,
                            2.47389e-10, 2.83556e-10, 3.24534e-10, 3.71696e-10, 4.27274e-10, 4.94719e-10,
                            5.78967e-10, 6.86219e-10, 8.24249e-10, 1.00592e-09, 1.24829e-09, 1.56020e-09,
                            1.95368e-09, 2.46075e-09, 3.10417e-09, 3.92901e-09, 4.95419e-09, 6.23691e-09,
                            7.85445e-09, 9.81196e-09, 1.21640e-08, 1.49246e-08, 1.81011e-08, 2.15536e-08,
                            2.50215e-08, 2.89962e-08, 3.35910e-08, 3.47039e-08, 3.08274e-08, 2.91934e-08])
    iion_data[33] = np.array([4.62568e-10, 4.62887e-10, 4.63303e-10, 4.63848e-10, 4.64567e-10, 4.65517e-10,
                            4.66775e-10, 4.68440e-10, 4.70636e-10, 4.73515e-10, 4.77262e-10, 4.82088e-10,
                            4.88235e-10, 4.95981e-10, 5.05656e-10, 5.17655e-10, 5.32458e-10, 5.50669e-10,
                            5.73090e-10, 6.00843e-10, 6.35527e-10, 6.79396e-10, 7.35576e-10, 8.08301e-10,
                            9.03119e-10, 1.02709e-09, 1.18906e-09, 1.39951e-09, 1.66881e-09, 2.00591e-09,
                            2.42916e-09, 2.96821e-09, 3.63565e-09, 4.46432e-09, 5.48818e-09, 6.76618e-09,
                            8.36881e-09, 1.03039e-08, 1.26220e-08, 1.53309e-08, 1.84570e-08, 2.19126e-08,
                            2.54523e-08, 2.94051e-08, 3.35084e-08, 3.42303e-08, 3.10803e-08, 2.96038e-08])
    iion_data[34] = np.array([1.01567e-09, 1.01543e-09, 1.01514e-09, 1.01478e-09, 1.01433e-09, 1.01377e-09,
                            1.01308e-09, 1.01223e-09, 1.01122e-09, 1.01003e-09, 1.00868e-09, 1.00724e-09,
                            1.00585e-09, 1.00481e-09, 1.00456e-09, 1.00583e-09, 1.00959e-09, 1.01711e-09,
                            1.02998e-09, 1.05039e-09, 1.08126e-09, 1.12619e-09, 1.18917e-09, 1.27465e-09,
                            1.38793e-09, 1.53537e-09, 1.72416e-09, 1.96237e-09, 2.25867e-09, 2.62306e-09,
                            3.07305e-09, 3.62895e-09, 4.30241e-09, 5.12537e-09, 6.14865e-09, 7.42658e-09,
                            9.01062e-09, 1.09270e-08, 1.32138e-08, 1.58715e-08, 1.89451e-08, 2.23682e-08,
                            2.59560e-08, 2.98551e-08, 3.33844e-08, 3.37443e-08, 3.13544e-08, 3.01148e-08])
    iion_data[35] = np.array([1.45132e-09, 1.45202e-09, 1.45291e-09, 1.45402e-09, 1.45543e-09, 1.45718e-09,
                            1.45939e-09, 1.46216e-09, 1.46564e-09, 1.46998e-09, 1.47542e-09, 1.48220e-09,
                            1.49065e-09, 1.50114e-09, 1.51416e-09, 1.53032e-09, 1.55038e-09, 1.57543e-09,
                            1.60694e-09, 1.64707e-09, 1.69865e-09, 1.76484e-09, 1.84902e-09, 1.95551e-09,
                            2.09054e-09, 2.26156e-09, 2.47548e-09, 2.73841e-09, 3.05707e-09, 3.44144e-09,
                            3.90581e-09, 4.46619e-09, 5.14000e-09, 5.96133e-09, 6.98885e-09, 8.26009e-09,
                            9.82353e-09, 1.17159e-08, 1.39645e-08, 1.65757e-08, 1.95872e-08, 2.29404e-08,
                            2.65378e-08, 3.03328e-08, 3.32137e-08, 3.32800e-08, 3.16565e-08, 3.07374e-08])
    iion_data[36] = np.array([2.26671e-09, 2.26785e-09, 2.26928e-09, 2.27108e-09, 2.27334e-09, 2.27619e-09,
                            2.27977e-09, 2.28426e-09, 2.28991e-09, 2.29701e-09, 2.30590e-09, 2.31705e-09,
                            2.33101e-09, 2.34846e-09, 2.37021e-09, 2.39727e-09, 2.43082e-09, 2.47223e-09,
                            2.52309e-09, 2.58521e-09, 2.66066e-09, 2.75174e-09, 2.86154e-09, 2.99424e-09,
                            3.15391e-09, 3.34439e-09, 3.56932e-09, 3.83534e-09, 4.15323e-09, 4.53501e-09,
                            4.99454e-09, 5.54761e-09, 6.21610e-09, 7.03537e-09, 8.05751e-09, 9.31603e-09,
                            1.08588e-08, 1.27175e-08, 1.49214e-08, 1.74886e-08, 2.04103e-08, 2.36612e-08,
                            2.72100e-08, 3.08153e-08, 3.29981e-08, 3.28823e-08, 3.20011e-08, 3.14627e-08])
    iion_data[37] = np.array([3.85110e-09, 3.85218e-09, 3.85353e-09, 3.85523e-09, 3.85737e-09, 3.86006e-09,
                            3.86345e-09, 3.86770e-09, 3.87303e-09, 3.87973e-09, 3.88812e-09, 3.89864e-09,
                            3.91179e-09, 3.92821e-09, 3.94869e-09, 3.97416e-09, 4.00575e-09, 4.04480e-09,
                            4.09288e-09, 4.15179e-09, 4.22356e-09, 4.31043e-09, 4.41493e-09, 4.53965e-09,
                            4.68734e-09, 4.86263e-09, 5.07164e-09, 5.32005e-09, 5.61610e-09, 5.96874e-09,
                            6.39522e-09, 6.91861e-09, 7.56752e-09, 8.37804e-09, 9.39139e-09, 1.06421e-08,
                            1.21598e-08, 1.39797e-08, 1.61268e-08, 1.86277e-08, 2.14366e-08, 2.45432e-08,
                            2.79720e-08, 3.12623e-08, 3.27569e-08, 3.26071e-08, 3.24116e-08, 3.22473e-08])
    iion_data[38] = np.array([6.24946e-09, 6.24995e-09, 6.25057e-09, 6.25136e-09, 6.25235e-09, 6.25359e-09,
                            6.25516e-09, 6.25713e-09, 6.25962e-09, 6.26276e-09, 6.26671e-09, 6.27169e-09,
                            6.27797e-09, 6.28589e-09, 6.29589e-09, 6.30850e-09, 6.32443e-09, 6.34454e-09,
                            6.36994e-09, 6.40199e-09, 6.44245e-09, 6.49345e-09, 6.55764e-09, 6.63824e-09,
                            6.73929e-09, 6.86582e-09, 7.02317e-09, 7.21934e-09, 7.46781e-09, 7.78050e-09,
                            8.17478e-09, 8.67199e-09, 9.29982e-09, 1.00922e-08, 1.10831e-08, 1.23083e-08,
                            1.37866e-08, 1.55489e-08, 1.76170e-08, 2.00204e-08, 2.26923e-08, 2.56068e-08,
                            2.88381e-08, 3.16503e-08, 3.25474e-08, 3.25205e-08, 3.29143e-08, 3.29908e-08])
    iion_data[39] = np.array([8.97850e-09, 8.97857e-09, 8.97865e-09, 8.97876e-09, 8.97890e-09, 8.97907e-09,
                            8.97930e-09, 8.97959e-09, 8.97996e-09, 8.98044e-09, 8.98107e-09, 8.98189e-09,
                            8.98298e-09, 8.98443e-09, 8.98638e-09, 8.98904e-09, 8.99268e-09, 8.99774e-09,
                            9.00481e-09, 9.01478e-09, 9.02889e-09, 9.04893e-09, 9.07743e-09, 9.11783e-09,
                            9.17482e-09, 9.25457e-09, 9.36516e-09, 9.51726e-09, 9.72283e-09, 9.99706e-09,
                            1.03645e-08, 1.08460e-08, 1.14655e-08, 1.22470e-08, 1.32194e-08, 1.44120e-08,
                            1.58298e-08, 1.74957e-08, 1.94479e-08, 2.16955e-08, 2.41801e-08, 2.68847e-08,
                            2.97975e-08, 3.19787e-08, 3.24702e-08, 3.26984e-08, 3.35223e-08, 3.35200e-08])
    iion_data[40] = np.array([1.15832e-08, 1.15834e-08, 1.15836e-08, 1.15839e-08, 1.15843e-08, 1.15848e-08,
                            1.15855e-08, 1.15863e-08, 1.15873e-08, 1.15885e-08, 1.15901e-08, 1.15922e-08,
                            1.15947e-08, 1.15980e-08, 1.16021e-08, 1.16074e-08, 1.16142e-08, 1.16229e-08,
                            1.16342e-08, 1.16488e-08, 1.16680e-08, 1.16934e-08, 1.17272e-08, 1.17725e-08,
                            1.18339e-08, 1.19173e-08, 1.20310e-08, 1.21857e-08, 1.23947e-08, 1.26749e-08,
                            1.30437e-08, 1.35209e-08, 1.41352e-08, 1.49050e-08, 1.58477e-08, 1.69748e-08,
                            1.83014e-08, 1.98512e-08, 2.16436e-08, 2.36803e-08, 2.59295e-08, 2.83906e-08,
                            3.08202e-08, 3.22976e-08, 3.26438e-08, 3.31817e-08, 3.41689e-08, 3.35924e-08])
    iion_data[41] = np.array([1.43516e-08, 1.43522e-08, 1.43528e-08, 1.43537e-08, 1.43548e-08, 1.43562e-08,
                            1.43579e-08, 1.43601e-08, 1.43629e-08, 1.43664e-08, 1.43707e-08, 1.43762e-08,
                            1.43831e-08, 1.43918e-08, 1.44027e-08, 1.44165e-08, 1.44337e-08, 1.44554e-08,
                            1.44826e-08, 1.45167e-08, 1.45595e-08, 1.46131e-08, 1.46803e-08, 1.47644e-08,
                            1.48697e-08, 1.50015e-08, 1.51668e-08, 1.53740e-08, 1.56338e-08, 1.59587e-08,
                            1.63629e-08, 1.68615e-08, 1.74669e-08, 1.81920e-08, 1.90584e-08, 2.00764e-08,
                            2.12617e-08, 2.26238e-08, 2.41940e-08, 2.59817e-08, 2.79661e-08, 3.01112e-08,
                            3.19041e-08, 3.27370e-08, 3.31958e-08, 3.39654e-08, 3.46564e-08, 3.29946e-08])
    iion_data[42] = np.array([1.81771e-08, 1.81779e-08, 1.81789e-08, 1.81801e-08, 1.81816e-08, 1.81836e-08,
                            1.81860e-08, 1.81891e-08, 1.81930e-08, 1.81978e-08, 1.82039e-08, 1.82116e-08,
                            1.82213e-08, 1.82335e-08, 1.82487e-08, 1.82679e-08, 1.82919e-08, 1.83220e-08,
                            1.83597e-08, 1.84068e-08, 1.84655e-08, 1.85385e-08, 1.86290e-08, 1.87409e-08,
                            1.88784e-08, 1.90466e-08, 1.92508e-08, 1.94971e-08, 1.97916e-08, 2.01411e-08,
                            2.05522e-08, 2.10322e-08, 2.15890e-08, 2.22294e-08, 2.29574e-08, 2.37894e-08,
                            2.47516e-08, 2.58693e-08, 2.71632e-08, 2.86351e-08, 3.03142e-08, 3.19795e-08,
                            3.30383e-08, 3.35080e-08, 3.41756e-08, 3.48841e-08, 3.46165e-08, 3.17364e-08])
    iion_data[43] = np.array([2.38956e-08, 2.38963e-08, 2.38971e-08, 2.38982e-08, 2.38995e-08, 2.39011e-08,
                            2.39032e-08, 2.39058e-08, 2.39091e-08, 2.39133e-08, 2.39185e-08, 2.39250e-08,
                            2.39332e-08, 2.39436e-08, 2.39565e-08, 2.39727e-08, 2.39931e-08, 2.40185e-08,
                            2.40503e-08, 2.40899e-08, 2.41391e-08, 2.42001e-08, 2.42754e-08, 2.43679e-08,
                            2.44809e-08, 2.46179e-08, 2.47826e-08, 2.49786e-08, 2.52093e-08, 2.54775e-08,
                            2.57854e-08, 2.61353e-08, 2.65304e-08, 2.69781e-08, 2.74919e-08, 2.80878e-08,
                            2.87826e-08, 2.96108e-08, 3.05882e-08, 3.17376e-08, 3.29817e-08, 3.39283e-08,
                            3.43598e-08, 3.47620e-08, 3.54080e-08, 3.54876e-08, 3.37078e-08, 3.01026e-08])
    iion_data[44] = np.array([3.13495e-08, 3.13497e-08, 3.13498e-08, 3.13501e-08, 3.13503e-08, 3.13507e-08,
                            3.13511e-08, 3.13517e-08, 3.13524e-08, 3.13532e-08, 3.13543e-08, 3.13557e-08,
                            3.13574e-08, 3.13596e-08, 3.13623e-08, 3.13657e-08, 3.13700e-08, 3.13753e-08,
                            3.13820e-08, 3.13904e-08, 3.14009e-08, 3.14139e-08, 3.14301e-08, 3.14501e-08,
                            3.14749e-08, 3.15054e-08, 3.15430e-08, 3.15890e-08, 3.16455e-08, 3.17148e-08,
                            3.18002e-08, 3.19067e-08, 3.20419e-08, 3.22174e-08, 3.24513e-08, 3.27720e-08,
                            3.32090e-08, 3.37722e-08, 3.44548e-08, 3.52121e-08, 3.58116e-08, 3.60212e-08,
                            3.61043e-08, 3.63836e-08, 3.63813e-08, 3.50940e-08, 3.20476e-08, 2.84372e-08])
    iion_data[45] = np.array([3.85233e-08, 3.85229e-08, 3.85224e-08, 3.85218e-08, 3.85211e-08, 3.85201e-08,
                            3.85189e-08, 3.85174e-08, 3.85155e-08, 3.85131e-08, 3.85102e-08, 3.85064e-08,
                            3.85017e-08, 3.84958e-08, 3.84884e-08, 3.84792e-08, 3.84677e-08, 3.84533e-08,
                            3.84354e-08, 3.84132e-08, 3.83858e-08, 3.83522e-08, 3.83111e-08, 3.82614e-08,
                            3.82018e-08, 3.81313e-08, 3.80492e-08, 3.79558e-08, 3.78523e-08, 3.77424e-08,
                            3.76322e-08, 3.75320e-08, 3.74566e-08, 3.74260e-08, 3.74639e-08, 3.75905e-08,
                            3.78158e-08, 3.81373e-08, 3.84986e-08, 3.87109e-08, 3.86071e-08, 3.83482e-08,
                            3.81414e-08, 3.76855e-08, 3.62117e-08, 3.35109e-08, 3.02310e-08, 2.69297e-08])
    iion_data[46] = np.array([4.28593e-08, 4.28591e-08, 4.28587e-08, 4.28583e-08, 4.28577e-08, 4.28570e-08,
                            4.28561e-08, 4.28550e-08, 4.28536e-08, 4.28518e-08, 4.28496e-08, 4.28468e-08,
                            4.28433e-08, 4.28389e-08, 4.28334e-08, 4.28265e-08, 4.28179e-08, 4.28072e-08,
                            4.27938e-08, 4.27771e-08, 4.27565e-08, 4.27312e-08, 4.27002e-08, 4.26626e-08,
                            4.26174e-08, 4.25640e-08, 4.25019e-08, 4.24318e-08, 4.23554e-08, 4.22767e-08,
                            4.22026e-08, 4.21433e-08, 4.21121e-08, 4.21223e-08, 4.21817e-08, 4.22830e-08,
                            4.23935e-08, 4.24317e-08, 4.22661e-08, 4.18359e-08, 4.12502e-08, 4.05442e-08,
                            3.93876e-08, 3.73803e-08, 3.46114e-08, 3.14875e-08, 2.86687e-08, 2.57367e-08])
    iion_data[47] = np.array([4.49925e-08, 4.49928e-08, 4.49932e-08, 4.49936e-08, 4.49941e-08, 4.49948e-08,
                            4.49956e-08, 4.49967e-08, 4.49980e-08, 4.49997e-08, 4.50018e-08, 4.50045e-08,
                            4.50078e-08, 4.50120e-08, 4.50173e-08, 4.50240e-08, 4.50324e-08, 4.50429e-08,
                            4.50562e-08, 4.50728e-08, 4.50937e-08, 4.51198e-08, 4.51526e-08, 4.51936e-08,
                            4.52448e-08, 4.53087e-08, 4.53880e-08, 4.54859e-08, 4.56061e-08, 4.57518e-08,
                            4.59248e-08, 4.61232e-08, 4.63377e-08, 4.65453e-08, 4.67044e-08, 4.67483e-08,
                            4.65842e-08, 4.61255e-08, 4.53579e-08, 4.43105e-08, 4.29133e-08, 4.09698e-08,
                            3.83783e-08, 3.54225e-08, 3.24889e-08, 2.97256e-08, 2.74190e-08, 2.47924e-08])

    # data are given in  cm^3/s. Convert to m^3/s
    iion_data = iion_data * 1E-6

    Tn_vals, Ti_vals = np.meshgrid((np.linspace(1, 48, 48) - 1) / 10 - 3,  # log10(Tn in keV)
                                   (np.linspace(1, 48, 48) - 1) / 10 - 3)  # log10(Ti in keV)

    # perform 2D interpolation
    #sigv_ion_i = griddata(np.column_stack((Tn_vals.flatten(), Ti_vals.flatten())),
    #                      iion_data.flatten(),
    #                      (np.asarray(Tn), np.asarray(Ti)),
    #                      method='linear',
    #                      rescale=True)

    return Rbf(Tn_vals, Ti_vals, iion_data)


def calc_svrec_degas():
    """calculates the recombination cross section using degas data

    accepts ne in m^-3 and Te in keV

    """
    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    svrec_data = np.zeros((15, 60))
    svrec_data[0] = np.array([5.23562E-11, 2.56421E-11, 1.36798E-11, 7.91489E-12, 4.92355E-12, 3.25884E-12,
                            2.27112E-12, 1.65038E-12, 1.23991E-12, 9.56132E-13, 7.52235E-13, 6.00894E-13,
                            4.85427E-13, 3.95320E-13, 3.23710E-13, 2.65973E-13, 2.18903E-13, 1.80235E-13,
                            1.48263E-13, 1.21742E-13, 9.97055E-14, 8.13860E-14, 6.61766E-14, 5.35806E-14,
                            4.31833E-14, 3.46341E-14, 2.76382E-14, 2.19433E-14, 1.73330E-14, 1.36222E-14,
                            1.06529E-14, 8.29067E-15, 6.42229E-15, 4.95279E-15, 3.80330E-15, 2.90880E-15,
                            2.21619E-15, 1.68242E-15, 1.27288E-15, 9.59983E-16, 7.21847E-16, 5.41275E-16,
                            4.04819E-16, 3.02029E-16, 2.24829E-16, 1.67007E-16, 1.23811E-16, 9.16184E-17,
                            6.76794E-17, 4.99147E-17, 3.67574E-17, 2.70301E-17, 1.98505E-17, 1.45598E-17,
                            1.06668E-17, 7.80603E-18, 5.70660E-18, 4.16773E-18, 3.04103E-18, 2.21698E-18])
    svrec_data[1] = np.array([1.19483E-10, 5.22693E-11, 2.49652E-11, 1.30381E-11, 7.40515E-12, 4.53223E-12,
                            2.95744E-12, 2.03563E-12, 1.46337E-12, 1.08913E-12, 8.32990E-13, 6.50655E-13,
                            5.16415E-13, 4.14752E-13, 3.35944E-13, 2.73682E-13, 2.23754E-13, 1.83272E-13,
                            1.50150E-13, 1.22900E-13, 1.00405E-13, 8.17979E-14, 6.64123E-14, 5.37086E-14,
                            4.32476E-14, 3.46616E-14, 2.76454E-14, 2.19400E-14, 1.73250E-14, 1.36126E-14,
                            1.06434E-14, 8.28216E-15, 6.41503E-15, 4.94682E-15, 3.79850E-15, 2.90502E-15,
                            2.21325E-15, 1.68016E-15, 1.27117E-15, 9.58689E-16, 7.20878E-16, 5.40553E-16,
                            4.04285E-16, 3.01635E-16, 2.24540E-16, 1.66796E-16, 1.23658E-16, 9.15071E-17,
                            6.75989E-17, 4.98568E-17, 3.67159E-17, 2.70004E-17, 1.98294E-17, 1.45448E-17,
                            1.06561E-17, 7.79854E-18, 5.70133E-18, 4.16404E-18, 3.03846E-18, 2.21519E-18])
    svrec_data[2] = np.array([3.04509E-10, 1.20074E-10, 5.13310E-11, 2.40278E-11, 1.23189E-11, 6.88046E-12,
                            4.14747E-12, 2.66989E-12, 1.81578E-12, 1.29162E-12, 9.52439E-13, 7.22552E-13,
                            5.60335E-13, 4.41859E-13, 3.52780E-13, 2.84170E-13, 2.30287E-13, 1.87325E-13,
                            1.52648E-13, 1.24423E-13, 1.01319E-13, 8.23318E-14, 6.67143E-14, 5.38695E-14,
                            4.33259E-14, 3.46926E-14, 2.76507E-14, 2.19324E-14, 1.73119E-14, 1.35981E-14,
                            1.06296E-14, 8.26995E-15, 6.40474E-15, 4.93842E-15, 3.79181E-15, 2.89977E-15,
                            2.20919E-15, 1.67706E-15, 1.26882E-15, 9.56918E-16, 7.19553E-16, 5.39568E-16,
                            4.03555E-16, 3.01098E-16, 2.24145E-16, 1.66507E-16, 1.23447E-16, 9.13535E-17,
                            6.74874E-17, 4.97761E-17, 3.66576E-17, 2.69583E-17, 1.97992E-17, 1.45231E-17,
                            1.06406E-17, 7.78743E-18, 5.69341E-18, 4.15841E-18, 3.03446E-18, 2.21236E-18])
    svrec_data[3] = np.array([8.49060E-10, 3.07929E-10, 1.19057E-10, 4.99814E-11, 2.29598E-11, 1.15764E-11,
                            6.36225E-12, 3.77936E-12, 2.40078E-12, 1.61343E-12, 1.13563E-12, 8.29673E-13,
                            6.24257E-13, 4.80563E-13, 3.76446E-13, 2.98721E-13, 2.39248E-13, 1.92829E-13,
                            1.56011E-13, 1.26455E-13, 1.02527E-13, 8.30329E-14, 6.71062E-14, 5.40751E-14,
                            4.34228E-14, 3.47276E-14, 2.76528E-14, 2.19184E-14, 1.72914E-14, 1.35765E-14,
                            1.06094E-14, 8.25238E-15, 6.39009E-15, 4.92655E-15, 3.78239E-15, 2.89243E-15,
                            2.20354E-15, 1.67275E-15, 1.26556E-15, 9.54473E-16, 7.17730E-16, 5.38216E-16,
                            4.02557E-16, 3.00363E-16, 2.23607E-16, 1.66114E-16, 1.23160E-16, 9.11453E-17,
                            6.73365E-17, 4.96669E-17, 3.65787E-17, 2.69015E-17, 1.97582E-17, 1.44937E-17,
                            1.06195E-17, 7.77230E-18, 5.68258E-18, 4.15067E-18, 3.02893E-18, 2.20842E-18])
    svrec_data[4] = np.array([2.51294E-09, 8.61810E-10, 3.07765E-10, 1.17046E-10, 4.81415E-11, 2.18157E-11,
                            1.08271E-11, 5.86222E-12, 3.43283E-12, 2.15186E-12, 1.42894E-12, 9.95097E-13,
                            7.20098E-13, 5.37220E-13, 4.10418E-13, 3.19276E-13, 2.51737E-13, 2.00413E-13,
                            1.60594E-13, 1.29198E-13, 1.04141E-13, 8.39591E-14, 6.76164E-14, 5.43382E-14,
                            4.35422E-14, 3.47661E-14, 2.76488E-14, 2.18943E-14, 1.72598E-14, 1.35443E-14,
                            1.05799E-14, 8.22700E-15, 6.36907E-15, 4.90961E-15, 3.76901E-15, 2.88203E-15,
                            2.19555E-15, 1.66667E-15, 1.26098E-15, 9.51047E-16, 7.15182E-16, 5.36330E-16,
                            4.01167E-16, 2.99343E-16, 2.22861E-16, 1.65570E-16, 1.22765E-16, 9.08589E-17,
                            6.71293E-17, 4.95173E-17, 3.64709E-17, 2.68239E-17, 1.97025E-17, 1.44538E-17,
                            1.05909E-17, 7.75182E-18, 5.66795E-18, 4.14022E-18, 3.02148E-18, 2.20311E-18])
    svrec_data[5] = np.array([7.66327E-09, 2.55311E-09, 8.66270E-10, 3.05161E-10, 1.13729E-10, 4.64171E-11,
                            2.06645E-11, 1.00730E-11, 5.36784E-12, 3.09866E-12, 1.91780E-12, 1.25884E-12,
                            8.67441E-13, 6.21778E-13, 4.59922E-13, 3.48653E-13, 2.69302E-13, 2.10936E-13,
                            1.66877E-13, 1.32915E-13, 1.06301E-13, 8.51821E-14, 6.82782E-14, 5.46705E-14,
                            4.36849E-14, 3.48042E-14, 2.76325E-14, 2.18534E-14, 1.72108E-14, 1.34958E-14,
                            1.05363E-14, 8.18990E-15, 6.33857E-15, 4.88517E-15, 3.74979E-15, 2.86713E-15,
                            2.18414E-15, 1.65803E-15, 1.25448E-15, 9.46189E-16, 7.11574E-16, 5.33664E-16,
                            3.99207E-16, 2.97907E-16, 2.21813E-16, 1.64807E-16, 1.22211E-16, 9.04581E-17,
                            6.68398E-17, 4.93087E-17, 3.63209E-17, 2.67162E-17, 1.96253E-17, 1.43985E-17,
                            1.05513E-17, 7.72360E-18, 5.64782E-18, 4.12588E-18, 3.01127E-18, 2.19585E-18])
    svrec_data[6] = np.array([2.35526E-08, 7.76928E-09, 2.56820E-09, 8.60310E-10, 2.96996E-10, 1.10340E-10,
                            4.41495E-11, 1.92661E-11, 9.28160E-12, 4.89122E-12, 2.79070E-12, 1.70611E-12,
                            1.10638E-12, 7.53847E-13, 5.34848E-13, 3.91962E-13, 2.94623E-13, 2.25810E-13,
                            1.75600E-13, 1.37984E-13, 1.09194E-13, 8.67828E-14, 6.91185E-14, 5.50716E-14,
                            4.38386E-14, 3.48251E-14, 2.75875E-14, 2.17807E-14, 1.71313E-14, 1.34205E-14,
                            1.04698E-14, 8.13443E-15, 6.29343E-15, 4.84925E-15, 3.72171E-15, 2.84547E-15,
                            2.16761E-15, 1.64554E-15, 1.24511E-15, 9.39205E-16, 7.06397E-16, 5.29846E-16,
                            3.96403E-16, 2.95856E-16, 2.20317E-16, 1.63721E-16, 1.21424E-16, 8.98889E-17,
                            6.64294E-17, 4.90133E-17, 3.61087E-17, 2.65641E-17, 1.95164E-17, 1.43207E-17,
                            1.04958E-17, 7.68401E-18, 5.61963E-18, 4.10583E-18, 2.99702E-18, 2.18574E-18])
    svrec_data[7] = np.array([7.18560E-08, 2.37459E-08, 7.78945E-09, 2.54694E-09, 8.39929E-10, 2.91582E-10,
                            1.06702E-10, 4.19319E-11, 1.81855E-11, 8.64739E-12, 4.47779E-12, 2.50788E-12,
                            1.50716E-12, 9.64047E-13, 6.49039E-13, 4.55634E-13, 3.30723E-13, 2.46448E-13,
                            1.87395E-13, 1.44659E-13, 1.12888E-13, 8.87463E-14, 7.00865E-14, 5.54796E-14,
                            4.39423E-14, 3.47741E-14, 2.74669E-14, 2.16377E-14, 1.69906E-14, 1.32937E-14,
                            1.03614E-14, 8.04572E-15, 6.22226E-15, 4.79321E-15, 3.67823E-15, 2.81228E-15,
                            2.14245E-15, 1.62667E-15, 1.23104E-15, 9.28776E-16, 6.98708E-16, 5.24203E-16,
                            3.92278E-16, 2.92851E-16, 2.18135E-16, 1.62140E-16, 1.20282E-16, 8.90661E-17,
                            6.58375E-17, 4.85883E-17, 3.58041E-17, 2.63460E-17, 1.93606E-17, 1.42094E-17,
                            1.04165E-17, 7.62758E-18, 5.57951E-18, 4.07733E-18, 2.97680E-18, 2.17141E-18])
    svrec_data[8] = np.array([2.14888E-07, 7.17547E-08, 2.36556E-08, 7.69847E-09, 2.48750E-09, 8.26315E-10,
                            2.81038E-10, 1.00152E-10, 3.89393E-11, 1.65547E-11, 7.70001E-12, 3.90978E-12,
                            2.15684E-12, 1.28766E-12, 8.17223E-13, 5.45754E-13, 3.79913E-13, 2.73514E-13,
                            2.02243E-13, 1.52669E-13, 1.17055E-13, 9.07630E-14, 7.09188E-14, 5.56811E-14,
                            4.38280E-14, 3.45213E-14, 2.71715E-14, 2.13494E-14, 1.67323E-14, 1.30736E-14,
                            1.01799E-14, 7.89978E-15, 6.10686E-15, 4.70321E-15, 3.60883E-15, 2.75988E-15,
                            2.10292E-15, 1.59717E-15, 1.20913E-15, 9.12587E-16, 6.86804E-16, 5.15488E-16,
                            3.85924E-16, 2.88235E-16, 2.14793E-16, 1.59727E-16, 1.18545E-16, 8.78183E-17,
                            6.49434E-17, 4.79490E-17, 3.53478E-17, 2.60209E-17, 1.91293E-17, 1.40451E-17,
                            1.02999E-17, 7.54490E-18, 5.52095E-18, 4.03589E-18, 2.94750E-18, 2.15069E-18])
    svrec_data[9] = np.array([6.20599E-07, 2.11431E-07, 7.07279E-08, 2.32039E-08, 7.47208E-09, 2.42354E-09,
                            7.83075E-10, 2.58931E-10, 9.20545E-11, 3.55144E-11, 1.49398E-11, 6.85836E-12,
                            3.43706E-12, 1.89085E-12, 1.11290E-12, 6.95061E-13, 4.56783E-13, 3.13474E-13,
                            2.22962E-13, 1.63196E-13, 1.22152E-13, 9.29733E-14, 7.16187E-14, 5.56245E-14,
                            4.34261E-14, 3.39951E-14, 2.66354E-14, 2.08578E-14, 1.63072E-14, 1.27193E-14,
                            9.89226E-15, 7.67021E-15, 5.92643E-15, 4.56304E-15, 3.50095E-15, 2.67864E-15,
                            2.04171E-15, 1.55156E-15, 1.17528E-15, 8.87594E-16, 6.68444E-16, 5.02062E-16,
                            3.76144E-16, 2.81135E-16, 2.09655E-16, 1.56021E-16, 1.15878E-16, 8.59037E-17,
                            6.35720E-17, 4.69686E-17, 3.46482E-17, 2.55226E-17, 1.87749E-17, 1.37934E-17,
                            1.01215E-17, 7.41856E-18, 5.43161E-18, 3.97280E-18, 2.90299E-18, 2.11934E-18])
    svrec_data[10] = np.array([1.68084E-06, 5.94727E-07, 2.04616E-07, 6.84781E-08, 2.22914E-08, 7.21033E-09,
                             2.28280E-09, 7.27377E-10, 2.45127E-10, 8.79401E-11, 3.36822E-11, 1.38447E-11,
                             6.17606E-12, 3.04358E-12, 1.61878E-12, 9.26160E-13, 5.66110E-13, 3.66604E-13,
                             2.49165E-13, 1.76084E-13, 1.28362E-13, 9.57634E-14, 7.26464E-14, 5.57629E-14,
                             4.31414E-14, 3.35362E-14, 2.61330E-14, 2.03773E-14, 1.58785E-14, 1.23525E-14,
                             9.58708E-15, 7.42089E-15, 5.72616E-15, 4.40416E-15, 3.37619E-15, 2.58181E-15,
                             1.96708E-15, 1.49452E-15, 1.13198E-15, 8.54932E-16, 6.43962E-16, 4.83819E-16,
                             3.62626E-16, 2.71158E-16, 2.02333E-16, 1.50671E-16, 1.11986E-16, 8.30840E-17,
                             6.15363E-17, 4.55039E-17, 3.35977E-17, 2.47712E-17, 1.82389E-17, 1.34119E-17,
                             9.85052E-18, 7.22648E-18, 5.29566E-18, 3.87672E-18, 2.83518E-18, 2.07153E-18])
    svrec_data[11] = np.array([4.26644E-06, 1.57049E-06, 5.64006E-07, 1.94429E-07, 6.48335E-08, 2.12696E-08,
                             6.75409E-09, 2.12889E-09, 6.93217E-10, 2.31429E-10, 7.92709E-11, 2.84948E-11,
                             1.11125E-11, 4.83580E-12, 2.31651E-12, 1.21811E-12, 6.96709E-13, 4.28456E-13,
                             2.79815E-13, 1.91775E-13, 1.36789E-13, 1.00399E-13, 7.51934E-14, 5.71319E-14,
                             4.38383E-14, 3.38497E-14, 2.62308E-14, 2.03583E-14, 1.58007E-14, 1.22500E-14,
                             9.47927E-15, 7.31767E-15, 5.63305E-15, 4.32321E-15, 3.30761E-15, 2.52399E-15,
                             1.91958E-15, 1.45584E-15, 1.10092E-15, 8.30196E-16, 6.24461E-16, 4.68576E-16,
                             3.50801E-16, 2.62016E-16, 1.95329E-16, 1.45339E-16, 1.07951E-16, 8.00468E-17,
                             5.92629E-17, 4.38111E-17, 3.23434E-17, 2.38465E-17, 1.75602E-17, 1.29162E-17,
                             9.48990E-18, 6.96526E-18, 5.10719E-18, 3.74125E-18, 2.73815E-18, 2.00226E-18])
    svrec_data[12] = np.array([9.52006E-06, 3.75266E-06, 1.44886E-06, 5.21498E-07, 1.80055E-07, 6.08447E-08,
                             1.96868E-08, 6.22909E-09, 1.97452E-09, 6.10200E-10, 1.85444E-10, 5.85530E-11,
                             2.03072E-11, 7.97119E-12, 3.50663E-12, 1.71714E-12, 9.24936E-13, 5.40677E-13,
                             3.38421E-13, 2.23929E-13, 1.55958E-13, 1.12402E-13, 8.28984E-14, 6.21657E-14,
                             4.71671E-14, 3.60675E-14, 2.77139E-14, 2.13502E-14, 1.64623E-14, 1.26886E-14,
                             9.76730E-15, 7.50409E-15, 5.75153E-15, 4.39660E-15, 3.35145E-15, 2.54803E-15,
                             1.93153E-15, 1.46028E-15, 1.10104E-15, 8.27888E-16, 6.21073E-16, 4.64879E-16,
                             3.47222E-16, 2.58688E-16, 1.92420E-16, 1.42874E-16, 1.05907E-16, 7.83799E-17,
                             5.79217E-17, 4.27441E-17, 3.15027E-17, 2.31896E-17, 1.70509E-17, 1.25239E-17,
                             9.18977E-18, 6.73698E-18, 4.93455E-18, 3.61140E-18, 2.64100E-18, 1.92995E-18])
    svrec_data[13] = np.array([1.77710E-05, 7.72350E-06, 3.38849E-06, 1.29160E-06, 4.68588E-07, 1.64759E-07,
                             5.52418E-08, 1.78105E-08, 5.60216E-09, 1.64877E-09, 4.64186E-10, 1.35458E-10,
                             4.37594E-11, 1.60734E-11, 6.64369E-12, 3.06124E-12, 1.55215E-12, 8.55141E-13,
                             5.05819E-13, 3.17693E-13, 2.13257E-13, 1.49214E-13, 1.07133E-13, 7.84093E-14,
                             5.81956E-14, 4.36248E-14, 3.29243E-14, 2.49555E-14, 1.89609E-14, 1.44201E-14,
                             1.09653E-14, 8.33043E-15, 6.31927E-15, 4.78475E-15, 3.61537E-15, 2.72604E-15,
                             2.05093E-15, 1.53975E-15, 1.15356E-15, 8.61954E-16, 6.43069E-16, 4.78967E-16,
                             3.56154E-16, 2.64036E-16, 1.95612E-16, 1.44718E-16, 1.06915E-16, 7.88803E-17,
                             5.81227E-17, 4.27759E-17, 3.14452E-17, 2.30909E-17, 1.69388E-17, 1.24138E-17,
                             9.08936E-18, 6.64949E-18, 4.86062E-18, 3.55030E-18, 2.59136E-18, 1.89015E-18])
    svrec_data[14] = np.array([3.04680E-05, 1.43641E-05, 6.43600E-06, 2.71638E-06, 1.12353E-06, 4.16670E-07,
                             1.45733E-07, 4.87156E-08, 1.56243E-08, 4.57029E-09, 1.25906E-09, 3.58390E-10,
                             1.12861E-10, 4.03158E-11, 1.61463E-11, 7.17047E-12, 3.48326E-12, 1.82897E-12,
                             1.02671E-12, 6.10493E-13, 3.92896E-13, 2.65026E-13, 1.83619E-13, 1.29829E-13,
                             9.32299E-14, 6.77112E-14, 4.95903E-14, 3.65290E-14, 2.70132E-14, 2.00232E-14,
                             1.48599E-14, 1.10313E-14, 8.18631E-15, 6.07035E-15, 4.49678E-15, 3.32744E-15,
                             2.45955E-15, 1.81620E-15, 1.33990E-15, 9.85925E-16, 7.25643E-16, 5.33928E-16,
                             3.92707E-16, 2.87575E-16, 2.10977E-16, 1.54735E-16, 1.13420E-16, 8.30876E-17,
                             6.08306E-17, 4.45088E-17, 3.25469E-17, 2.37855E-17, 1.73723E-17, 1.26808E-17,
                             9.25092E-18, 6.74491E-18, 4.91503E-18, 3.57964E-18, 2.60569E-18, 1.89576E-18])

    # data are given in  cm^3/s. Convert to m^3/s
    svrec_data = svrec_data * 1E-6

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,  # log10(Te in keV)
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)  # log10(n in m^-3)
    # perform 2D interpolation
    #sigv_rec = griddata(np.column_stack((Te_vals.flatten(), ne_vals.flatten())),
    #                    svrec_data.flatten(),
    #                    (np.asarray(Te), np.asarray(ne)),
    #                    method='linear',
    #                    rescale=True)

    return Rbf(Te_vals, ne_vals, svrec_data)


def calc_svcx_degas():
    """
    """
    # populate the 15x60 array to be used for interpolations
    # rows correspond to ion temperature, columns correspond to neutral temperature

    cx_data = np.zeros((48, 48))
    cx_data[0] = np.array([1.10368e-08, 1.16403e-08, 1.24069e-08, 1.32213e-08, 1.41413e-08, 1.51684e-08,
                      1.63073e-08, 1.75477e-08, 1.89301e-08, 2.04397e-08, 2.20781e-08, 2.38607e-08,
                      2.57513e-08, 2.77570e-08, 2.98657e-08, 3.20660e-08, 3.43478e-08, 3.67036e-08,
                      3.91296e-08, 4.16270e-08, 4.42031e-08, 4.68725e-08, 4.96569e-08, 5.25851e-08,
                      5.56920e-08, 5.90165e-08, 6.25979e-08, 6.64719e-08, 7.06629e-08, 7.51763e-08,
                      7.99867e-08, 8.50253e-08, 9.01664e-08, 9.52169e-08, 9.99105e-08, 1.03914e-07,
                      1.06846e-07, 1.08316e-07, 1.07970e-07, 1.05545e-07, 1.00920e-07, 9.41528e-08,
                      8.55058e-08, 7.54616e-08, 6.47108e-08, 5.40877e-08, 4.44284e-08, 3.63711e-08])
    cx_data[1] = np.array([1.15095e-08, 1.20715e-08, 1.27900e-08, 1.35609e-08, 1.44466e-08, 1.54445e-08,
                      1.65565e-08, 1.77874e-08, 1.91426e-08, 2.06103e-08, 2.22253e-08, 2.39684e-08,
                      2.58537e-08, 2.78411e-08, 2.99354e-08, 3.21241e-08, 3.43963e-08, 3.67440e-08,
                      3.91633e-08, 4.16550e-08, 4.42264e-08, 4.68918e-08, 4.96729e-08, 5.25983e-08,
                      5.57030e-08, 5.90256e-08, 6.26055e-08, 6.64782e-08, 7.06682e-08, 7.51807e-08,
                      7.99903e-08, 8.50282e-08, 9.01687e-08, 9.52186e-08, 9.99117e-08, 1.03915e-07,
                      1.06847e-07, 1.08317e-07, 1.07970e-07, 1.05545e-07, 1.00920e-07, 9.41525e-08,
                      8.55056e-08, 7.54614e-08, 6.47107e-08, 5.40877e-08, 4.44284e-08, 3.63711e-08])
    cx_data[2] = np.array([1.21298e-08, 1.25834e-08, 1.31941e-08, 1.39736e-08, 1.48141e-08, 1.57756e-08,
                      1.68560e-08, 1.80572e-08, 1.93832e-08, 2.08380e-08, 2.24242e-08, 2.41244e-08,
                      2.59844e-08, 2.79482e-08, 3.00236e-08, 3.21972e-08, 3.44572e-08, 3.67949e-08,
                      3.92056e-08, 4.16902e-08, 4.42557e-08, 4.69161e-08, 4.96930e-08, 5.26151e-08,
                      5.57169e-08, 5.90371e-08, 6.26151e-08, 6.64861e-08, 7.06748e-08, 7.51862e-08,
                      7.99948e-08, 8.50318e-08, 9.01716e-08, 9.52208e-08, 9.99133e-08, 1.03916e-07,
                      1.06847e-07, 1.08317e-07, 1.07970e-07, 1.05545e-07, 1.00920e-07, 9.41521e-08,
                      8.55053e-08, 7.54612e-08, 6.47106e-08, 5.40877e-08, 4.44284e-08, 3.63710e-08])
    cx_data[3] = np.array([1.27956e-08, 1.32554e-08, 1.38123e-08, 1.44762e-08, 1.52599e-08, 1.61736e-08,
                      1.72146e-08, 1.83809e-08, 1.96740e-08, 2.10964e-08, 2.26501e-08, 2.43352e-08,
                      2.61491e-08, 2.80850e-08, 3.01356e-08, 3.22896e-08, 3.45339e-08, 3.68588e-08,
                      3.92589e-08, 4.17346e-08, 4.42926e-08, 4.69468e-08, 4.97185e-08, 5.26361e-08,
                      5.57343e-08, 5.90516e-08, 6.26271e-08, 6.64961e-08, 7.06831e-08, 7.51931e-08,
                      8.00005e-08, 8.50364e-08, 9.01752e-08, 9.52235e-08, 9.99152e-08, 1.03917e-07,
                      1.06848e-07, 1.08317e-07, 1.07970e-07, 1.05544e-07, 1.00919e-07, 9.41516e-08,
                      8.55049e-08, 7.54610e-08, 6.47105e-08, 5.40877e-08, 4.44284e-08, 3.63709e-08])
    cx_data[4] = np.array([1.35717e-08, 1.39807e-08, 1.44797e-08, 1.50826e-08, 1.58034e-08, 1.66554e-08,
                      1.76451e-08, 1.87681e-08, 2.00222e-08, 2.14078e-08, 2.29259e-08, 2.45756e-08,
                      2.63546e-08, 2.82582e-08, 3.02785e-08, 3.24066e-08, 3.46306e-08, 3.69392e-08,
                      3.93259e-08, 4.17904e-08, 4.43391e-08, 4.69854e-08, 4.97505e-08, 5.26627e-08,
                      5.57564e-08, 5.90698e-08, 6.26423e-08, 6.65087e-08, 7.06936e-08, 7.52017e-08,
                      8.00076e-08, 8.50422e-08, 9.01797e-08, 9.52270e-08, 9.99177e-08, 1.03918e-07,
                      1.06849e-07, 1.08317e-07, 1.07969e-07, 1.05544e-07, 1.00918e-07, 9.41510e-08,
                      8.55044e-08, 7.54607e-08, 6.47104e-08, 5.40876e-08, 4.44283e-08, 3.63708e-08])
    cx_data[5] = np.array([1.44632e-08, 1.48239e-08, 1.52694e-08, 1.58107e-08, 1.64628e-08, 1.72432e-08,
                      1.81656e-08, 1.92327e-08, 2.04387e-08, 2.17806e-08, 2.32576e-08, 2.48681e-08,
                      2.66089e-08, 2.84750e-08, 3.04599e-08, 3.25557e-08, 3.47529e-08, 3.70404e-08,
                      3.94102e-08, 4.18607e-08, 4.43976e-08, 4.70340e-08, 4.97909e-08, 5.26962e-08,
                      5.57842e-08, 5.90929e-08, 6.26614e-08, 6.65246e-08, 7.07068e-08, 7.52127e-08,
                      8.00166e-08, 8.50494e-08, 9.01855e-08, 9.52313e-08, 9.99207e-08, 1.03920e-07,
                      1.06850e-07, 1.08317e-07, 1.07969e-07, 1.05543e-07, 1.00918e-07, 9.41502e-08,
                      8.55038e-08, 7.54603e-08, 6.47102e-08, 5.40876e-08, 4.44283e-08, 3.63707e-08])
    cx_data[6] = np.array([1.54785e-08, 1.57925e-08, 1.61855e-08, 1.66694e-08, 1.72550e-08, 1.79587e-08,
                      1.88003e-08, 1.97936e-08, 2.09382e-08, 2.22265e-08, 2.36544e-08, 2.52193e-08,
                      2.69173e-08, 2.87424e-08, 3.06875e-08, 3.27453e-08, 3.49084e-08, 3.71682e-08,
                      3.95162e-08, 4.19490e-08, 4.44712e-08, 4.70953e-08, 4.98418e-08, 5.27385e-08,
                      5.58192e-08, 5.91220e-08, 6.26855e-08, 6.65447e-08, 7.07234e-08, 7.52265e-08,
                      8.00279e-08, 8.50586e-08, 9.01927e-08, 9.52368e-08, 9.99246e-08, 1.03923e-07,
                      1.06851e-07, 1.08317e-07, 1.07969e-07, 1.05542e-07, 1.00916e-07, 9.41492e-08,
                      8.55030e-08, 7.54599e-08, 6.47100e-08, 5.40875e-08, 4.44282e-08, 3.63705e-08])
    cx_data[7] = np.array([1.66261e-08, 1.68967e-08, 1.72374e-08, 1.76635e-08, 1.81867e-08, 1.88177e-08,
                      1.95739e-08, 2.04771e-08, 2.15406e-08, 2.27610e-08, 2.41290e-08, 2.56393e-08,
                      2.72871e-08, 2.90658e-08, 3.09674e-08, 3.29833e-08, 3.51062e-08, 3.73303e-08,
                      3.96498e-08, 4.20601e-08, 4.45638e-08, 4.71725e-08, 4.99061e-08, 5.27919e-08,
                      5.58635e-08, 5.91587e-08, 6.27160e-08, 6.65700e-08, 7.07444e-08, 7.52439e-08,
                      8.00422e-08, 8.50702e-08, 9.02018e-08, 9.52437e-08, 9.99295e-08, 1.03926e-07,
                      1.06853e-07, 1.08318e-07, 1.07968e-07, 1.05541e-07, 1.00915e-07, 9.41480e-08,
                      8.55021e-08, 7.54593e-08, 6.47097e-08, 5.40875e-08, 4.44282e-08, 3.63703e-08])
    cx_data[8] = np.array([1.79118e-08, 1.81445e-08, 1.84367e-08, 1.88040e-08, 1.92630e-08, 1.98253e-08,
                      2.05016e-08, 2.13099e-08, 2.22733e-08, 2.34048e-08, 2.46977e-08, 2.61415e-08,
                      2.77289e-08, 2.94530e-08, 3.13051e-08, 3.32752e-08, 3.53544e-08, 3.75366e-08,
                      3.98191e-08, 4.22000e-08, 4.46804e-08, 4.72698e-08, 4.99871e-08, 5.28592e-08,
                      5.59194e-08, 5.92051e-08, 6.27545e-08, 6.66019e-08, 7.07709e-08, 7.52658e-08,
                      8.00602e-08, 8.50847e-08, 9.02133e-08, 9.52523e-08, 9.99356e-08, 1.03930e-07,
                      1.06855e-07, 1.08318e-07, 1.07967e-07, 1.05540e-07, 1.00914e-07, 9.41465e-08,
                      8.55009e-08, 7.54585e-08, 6.47094e-08, 5.40874e-08, 4.44281e-08, 3.63700e-08])
    cx_data[9] = np.array([1.93419e-08, 1.95405e-08, 1.97899e-08, 2.01028e-08, 2.04958e-08, 2.09864e-08,
                      2.15864e-08, 2.23066e-08, 2.31652e-08, 2.41862e-08, 2.53818e-08, 2.67428e-08,
                      2.82569e-08, 2.99152e-08, 3.17086e-08, 3.36264e-08, 3.56581e-08, 3.77953e-08,
                      4.00344e-08, 4.23772e-08, 4.48273e-08, 4.73924e-08, 5.00894e-08, 5.29444e-08,
                      5.59902e-08, 5.92638e-08, 6.28032e-08, 6.66423e-08, 7.08044e-08, 7.52935e-08,
                      8.00829e-08, 8.51031e-08, 9.02277e-08, 9.52632e-08, 9.99434e-08, 1.03935e-07,
                      1.06857e-07, 1.08319e-07, 1.07967e-07, 1.05538e-07, 1.00912e-07, 9.41445e-08,
                      8.54993e-08, 7.54576e-08, 6.47090e-08, 5.40873e-08, 4.44280e-08, 3.63697e-08])
    cx_data[10] = np.array([2.09216e-08, 2.10892e-08, 2.12999e-08, 2.15647e-08, 2.18966e-08, 2.23133e-08,
                       2.28333e-08, 2.34686e-08, 2.42299e-08, 2.51357e-08, 2.62102e-08, 2.74649e-08,
                       2.88884e-08, 3.04669e-08, 3.21895e-08, 3.40453e-08, 3.60226e-08, 3.81110e-08,
                       4.03041e-08, 4.26023e-08, 4.50131e-08, 4.75469e-08, 5.02184e-08, 5.30520e-08,
                       5.60797e-08, 5.93382e-08, 6.28649e-08, 6.66935e-08, 7.08468e-08, 7.53284e-08,
                       8.01116e-08, 8.51262e-08, 9.02459e-08, 9.52770e-08, 9.99531e-08, 1.03941e-07,
                       1.06860e-07, 1.08320e-07, 1.07965e-07, 1.05536e-07, 1.00909e-07, 9.41420e-08,
                       8.54974e-08, 7.54564e-08, 6.47085e-08, 5.40871e-08, 4.44279e-08, 3.63692e-08])
    cx_data[11] = np.array([2.26537e-08, 2.27932e-08, 2.29688e-08, 2.31899e-08, 2.34678e-08, 2.38164e-08,
                       2.42540e-08, 2.48002e-08, 2.54671e-08, 2.62658e-08, 2.72145e-08, 2.83376e-08,
                       2.96458e-08, 3.11259e-08, 3.27626e-08, 3.45436e-08, 3.64563e-08, 3.84888e-08,
                       4.06324e-08, 4.28840e-08, 4.52492e-08, 4.77425e-08, 5.03814e-08, 5.31881e-08,
                       5.61931e-08, 5.94324e-08, 6.29431e-08, 6.67582e-08, 7.09004e-08, 7.53726e-08,
                       8.01477e-08, 8.51554e-08, 9.02689e-08, 9.52943e-08, 9.99653e-08, 1.03949e-07,
                       1.06864e-07, 1.08320e-07, 1.07964e-07, 1.05533e-07, 1.00906e-07, 9.41390e-08,
                       8.54950e-08, 7.54549e-08, 6.47078e-08, 5.40869e-08, 4.44277e-08, 3.63687e-08])
    cx_data[12] = np.array([2.45370e-08, 2.46514e-08, 2.47956e-08, 2.49772e-08, 2.52063e-08, 2.54947e-08,
                       2.58569e-08, 2.63120e-08, 2.68805e-08, 2.75747e-08, 2.84061e-08, 2.93927e-08,
                       3.05590e-08, 3.19149e-08, 3.34462e-08, 3.51361e-08, 3.69709e-08, 3.89373e-08,
                       4.10244e-08, 4.32262e-08, 4.55443e-08, 4.79909e-08, 5.05879e-08, 5.33604e-08,
                       5.63369e-08, 5.95520e-08, 6.30423e-08, 6.68404e-08, 7.09683e-08, 7.54286e-08,
                       8.01935e-08, 8.51922e-08, 9.02978e-08, 9.53161e-08, 9.99808e-08, 1.03958e-07,
                       1.06869e-07, 1.08322e-07, 1.07962e-07, 1.05530e-07, 1.00902e-07, 9.41351e-08,
                       8.54920e-08, 7.54530e-08, 6.47070e-08, 5.40867e-08, 4.44275e-08, 3.63680e-08])
    cx_data[13] = np.array([2.65657e-08, 2.66580e-08, 2.67744e-08, 2.69214e-08, 2.71069e-08, 2.73414e-08,
                       2.76372e-08, 2.80096e-08, 2.84785e-08, 2.90649e-08, 2.97820e-08, 3.06411e-08,
                       3.16605e-08, 3.28646e-08, 3.42633e-08, 3.58414e-08, 3.75812e-08, 3.94679e-08,
                       4.14884e-08, 4.36338e-08, 4.59022e-08, 4.83012e-08, 5.08503e-08, 5.35792e-08,
                       5.65194e-08, 5.97041e-08, 6.31685e-08, 6.69449e-08, 7.10545e-08, 7.54995e-08,
                       8.02513e-08, 8.52387e-08, 9.03343e-08, 9.53436e-08, 1.00000e-07, 1.03971e-07,
                       1.06875e-07, 1.08323e-07, 1.07960e-07, 1.05525e-07, 1.00896e-07, 9.41302e-08,
                       8.54882e-08, 7.54506e-08, 6.47059e-08, 5.40864e-08, 4.44272e-08, 3.63671e-08])
    cx_data[14] = np.array([2.87292e-08, 2.88025e-08, 2.88950e-08, 2.90120e-08, 2.91599e-08, 2.93470e-08,
                       2.95842e-08, 2.98844e-08, 3.02636e-08, 3.07423e-08, 3.13426e-08, 3.20780e-08,
                       3.29603e-08, 3.40078e-08, 3.52454e-08, 3.66834e-08, 3.83061e-08, 4.00953e-08,
                       4.20355e-08, 4.41149e-08, 4.63277e-08, 4.86773e-08, 5.11782e-08, 5.38574e-08,
                       5.67518e-08, 5.98977e-08, 6.33294e-08, 6.70779e-08, 7.11642e-08, 7.55895e-08,
                       8.03246e-08, 8.52975e-08, 9.03803e-08, 9.53783e-08, 1.00025e-07, 1.03986e-07,
                       1.06883e-07, 1.08325e-07, 1.07958e-07, 1.05520e-07, 1.00890e-07, 9.41240e-08,
                       8.54834e-08, 7.54477e-08, 6.47046e-08, 5.40860e-08, 4.44268e-08, 3.63660e-08])
    cx_data[15] = np.array([3.10125e-08, 3.10698e-08, 3.11423e-08, 3.12339e-08, 3.13500e-08, 3.14972e-08,
                       3.16840e-08, 3.19215e-08, 3.22234e-08, 3.26061e-08, 3.30914e-08, 3.37020e-08,
                       3.44520e-08, 3.53537e-08, 3.64258e-08, 3.76944e-08, 3.91705e-08, 4.08388e-08,
                       4.26805e-08, 4.46803e-08, 4.68286e-08, 4.91238e-08, 5.15754e-08, 5.42054e-08,
                       5.70479e-08, 6.01449e-08, 6.35348e-08, 6.72479e-08, 7.13041e-08, 7.57040e-08,
                       8.04176e-08, 8.53719e-08, 9.04385e-08, 9.54220e-08, 1.00056e-07, 1.04006e-07,
                       1.06893e-07, 1.08327e-07, 1.07954e-07, 1.05513e-07, 1.00882e-07, 9.41162e-08,
                       8.54773e-08, 7.54439e-08, 6.47029e-08, 5.40856e-08, 4.44264e-08, 3.63646e-08])
    cx_data[16] = np.array([3.33971e-08, 3.34414e-08, 3.34975e-08, 3.35685e-08, 3.36585e-08, 3.37727e-08,
                       3.39178e-08, 3.41028e-08, 3.43389e-08, 3.46403e-08, 3.50244e-08, 3.55138e-08,
                       3.61323e-08, 3.68946e-08, 3.78136e-08, 3.89089e-08, 4.02082e-08, 4.17245e-08,
                       4.34430e-08, 4.53447e-08, 4.74153e-08, 4.96481e-08, 5.20466e-08, 5.46272e-08,
                       5.74187e-08, 6.04608e-08, 6.37981e-08, 6.74656e-08, 7.14831e-08, 7.58501e-08,
                       8.05359e-08, 8.54664e-08, 9.05121e-08, 9.54772e-08, 1.00094e-07, 1.04030e-07,
                       1.06906e-07, 1.08330e-07, 1.07950e-07, 1.05505e-07, 1.00871e-07, 9.41064e-08,
                       8.54696e-08, 7.54391e-08, 6.47008e-08, 5.40850e-08, 4.44259e-08, 3.63629e-08])
    cx_data[17] = np.array([3.58630e-08, 3.58971e-08, 3.59402e-08, 3.59948e-08, 3.60641e-08, 3.61520e-08,
                       3.62638e-08, 3.64064e-08, 3.65887e-08, 3.68225e-08, 3.71224e-08, 3.75068e-08,
                       3.79993e-08, 3.86250e-08, 3.93994e-08, 4.03357e-08, 4.14553e-08, 4.27885e-08,
                       4.43508e-08, 4.61285e-08, 4.81027e-08, 5.02606e-08, 5.25990e-08, 5.51275e-08,
                       5.78689e-08, 6.08573e-08, 6.41355e-08, 6.77456e-08, 7.17130e-08, 7.60374e-08,
                       8.06868e-08, 8.55864e-08, 9.06054e-08, 9.55470e-08, 1.00143e-07, 1.04061e-07,
                       1.06921e-07, 1.08333e-07, 1.07945e-07, 1.05494e-07, 1.00858e-07, 9.40940e-08,
                       8.54600e-08, 7.54331e-08, 6.46981e-08, 5.40842e-08, 4.44252e-08, 3.63607e-08])
    cx_data[18] = np.array([3.83907e-08, 3.84170e-08, 3.84502e-08, 3.84923e-08, 3.85456e-08, 3.86132e-08,
                       3.86992e-08, 3.88089e-08, 3.89491e-08, 3.91291e-08, 3.93608e-08, 3.96598e-08,
                       4.00450e-08, 4.05415e-08, 4.11758e-08, 4.19643e-08, 4.29209e-08, 4.40692e-08,
                       4.54431e-08, 4.70617e-08, 4.89123e-08, 5.09763e-08, 5.32429e-08, 5.57134e-08,
                       5.84030e-08, 6.13394e-08, 6.45600e-08, 6.81055e-08, 7.20095e-08, 7.62782e-08,
                       8.08803e-08, 8.57395e-08, 9.07238e-08, 9.56353e-08, 1.00205e-07, 1.04100e-07,
                       1.06941e-07, 1.08338e-07, 1.07938e-07, 1.05480e-07, 1.00842e-07, 9.40785e-08,
                       8.54479e-08, 7.54256e-08, 6.46947e-08, 5.40832e-08, 4.44243e-08, 3.63579e-08])
    cx_data[19] = np.array([4.09640e-08, 4.09844e-08, 4.10103e-08, 4.10430e-08, 4.10845e-08, 4.11370e-08,
                       4.12038e-08, 4.12888e-08, 4.13973e-08, 4.15364e-08, 4.17155e-08, 4.19470e-08,
                       4.22469e-08, 4.26353e-08, 4.31391e-08, 4.37860e-08, 4.45936e-08, 4.55767e-08,
                       4.67616e-08, 4.81871e-08, 4.98765e-08, 5.18183e-08, 5.39943e-08, 5.63957e-08,
                       5.90284e-08, 6.19120e-08, 6.50773e-08, 6.85597e-08, 7.23920e-08, 7.65898e-08,
                       8.11296e-08, 8.59357e-08, 9.08747e-08, 9.57472e-08, 1.00284e-07, 1.04150e-07,
                       1.06966e-07, 1.08343e-07, 1.07929e-07, 1.05463e-07, 1.00821e-07, 9.40588e-08,
                       8.54325e-08, 7.54160e-08, 6.46904e-08, 5.40820e-08, 4.44232e-08, 3.63545e-08])
    cx_data[20] = np.array([4.35720e-08, 4.35883e-08, 4.36089e-08, 4.36349e-08, 4.36679e-08, 4.37095e-08,
                       4.37624e-08, 4.38296e-08, 4.39151e-08, 4.40245e-08, 4.41648e-08, 4.43457e-08,
                       4.45801e-08, 4.48848e-08, 4.52811e-08, 4.57975e-08, 4.64639e-08, 4.72987e-08,
                       4.83178e-08, 4.95510e-08, 5.10431e-08, 5.28222e-08, 5.48776e-08, 5.71913e-08,
                       5.97567e-08, 6.25830e-08, 6.56926e-08, 6.91144e-08, 7.28760e-08, 7.69931e-08,
                       8.14528e-08, 8.61886e-08, 9.10680e-08, 9.58897e-08, 1.00383e-07, 1.04212e-07,
                       1.06997e-07, 1.08350e-07, 1.07918e-07, 1.05441e-07, 1.00795e-07, 9.40341e-08,
                       8.54132e-08, 7.54040e-08, 6.46850e-08, 5.40804e-08, 4.44219e-08, 3.63501e-08])
    cx_data[21] = np.array([4.62125e-08, 4.62259e-08, 4.62428e-08, 4.62641e-08, 4.62911e-08, 4.63251e-08,
                       4.63683e-08, 4.64229e-08, 4.64924e-08, 4.65808e-08, 4.66937e-08, 4.68385e-08,
                       4.70253e-08, 4.72674e-08, 4.75826e-08, 4.79934e-08, 4.85303e-08, 4.92257e-08,
                       5.00989e-08, 5.11669e-08, 5.24637e-08, 5.40412e-08, 5.59323e-08, 5.81269e-08,
                       6.06061e-08, 6.33650e-08, 6.64144e-08, 6.97750e-08, 7.34682e-08, 7.75048e-08,
                       8.18725e-08, 8.65173e-08, 9.13172e-08, 9.60719e-08, 1.00508e-07, 1.04290e-07,
                       1.07036e-07, 1.08358e-07, 1.07904e-07, 1.05413e-07, 1.00762e-07, 9.40028e-08,
                       8.53888e-08, 7.53887e-08, 6.46780e-08, 5.40784e-08, 4.44201e-08, 3.63447e-08])
    cx_data[22] = np.array([4.88930e-08, 4.89044e-08, 4.89187e-08, 4.89369e-08, 4.89597e-08, 4.89886e-08,
                       4.90250e-08, 4.90711e-08, 4.91295e-08, 4.92036e-08, 4.92977e-08, 4.94177e-08,
                       4.95714e-08, 4.97692e-08, 5.00253e-08, 5.03583e-08, 5.07922e-08, 5.13599e-08,
                       5.20962e-08, 5.30217e-08, 5.41547e-08, 5.55336e-08, 5.72180e-08, 5.92459e-08,
                       6.16062e-08, 6.42782e-08, 6.72569e-08, 7.05512e-08, 7.41744e-08, 7.81317e-08,
                       8.24064e-08, 8.69456e-08, 9.16418e-08, 9.63071e-08, 1.00669e-07, 1.04389e-07,
                       1.07086e-07, 1.08369e-07, 1.07887e-07, 1.05379e-07, 1.00721e-07, 9.39634e-08,
                       8.53579e-08, 7.53693e-08, 6.46692e-08, 5.40758e-08, 4.44179e-08, 3.63379e-08])
    cx_data[23] = np.array([5.16327e-08, 5.16427e-08, 5.16554e-08, 5.16713e-08, 5.16913e-08, 5.17166e-08,
                       5.17485e-08, 5.17888e-08, 5.18396e-08, 5.19039e-08, 5.19853e-08, 5.20886e-08,
                       5.22198e-08, 5.23874e-08, 5.26024e-08, 5.28798e-08, 5.32394e-08, 5.37066e-08,
                       5.43172e-08, 5.51086e-08, 5.61026e-08, 5.73187e-08, 5.88004e-08, 6.06147e-08,
                       6.28042e-08, 6.53550e-08, 6.82428e-08, 7.14589e-08, 7.50052e-08, 7.88798e-08,
                       8.30607e-08, 8.74914e-08, 9.20663e-08, 9.66141e-08, 1.00877e-07, 1.04516e-07,
                       1.07148e-07, 1.08382e-07, 1.07864e-07, 1.05335e-07, 1.00668e-07, 9.39135e-08,
                       8.53188e-08, 7.53447e-08, 6.46579e-08, 5.40725e-08, 4.44151e-08, 3.63294e-08])
    cx_data[24] = np.array([5.44630e-08, 5.44720e-08, 5.44834e-08, 5.44978e-08, 5.45158e-08, 5.45386e-08,
                       5.45673e-08, 5.46034e-08, 5.46490e-08, 5.47065e-08, 5.47791e-08, 5.48709e-08,
                       5.49869e-08, 5.51340e-08, 5.53211e-08, 5.55600e-08, 5.58670e-08, 5.62628e-08,
                       5.67749e-08, 5.74415e-08, 5.83035e-08, 5.93833e-08, 6.07019e-08, 6.23073e-08,
                       6.42739e-08, 6.66469e-08, 6.94071e-08, 7.25233e-08, 7.59787e-08, 7.97606e-08,
                       8.38413e-08, 8.81601e-08, 9.26082e-08, 9.70175e-08, 1.01149e-07, 1.04680e-07,
                       1.07228e-07, 1.08398e-07, 1.07836e-07, 1.05279e-07, 1.00602e-07, 9.38505e-08,
                       8.52693e-08, 7.53135e-08, 6.46435e-08, 5.40681e-08, 4.44116e-08, 3.63188e-08])
    cx_data[25] = np.array([5.74268e-08, 5.74351e-08, 5.74455e-08, 5.74587e-08, 5.74753e-08, 5.74961e-08,
                       5.75224e-08, 5.75555e-08, 5.75971e-08, 5.76496e-08, 5.77157e-08, 5.77991e-08,
                       5.79042e-08, 5.80368e-08, 5.82045e-08, 5.84168e-08, 5.86868e-08, 5.90318e-08,
                       5.94740e-08, 6.00426e-08, 6.07789e-08, 6.17268e-08, 6.29096e-08, 6.43491e-08,
                       6.60971e-08, 6.82341e-08, 7.08052e-08, 7.37822e-08, 7.71222e-08, 8.07939e-08,
                       8.47603e-08, 8.89565e-08, 9.32712e-08, 9.75336e-08, 1.01509e-07, 1.04898e-07,
                       1.07333e-07, 1.08419e-07, 1.07799e-07, 1.05208e-07, 1.00518e-07, 9.37706e-08,
                       8.52065e-08, 7.52737e-08, 6.46249e-08, 5.40623e-08, 4.44071e-08, 3.63056e-08])
    cx_data[26] = np.array([6.05780e-08, 6.05857e-08, 6.05953e-08, 6.06075e-08, 6.06228e-08, 6.06420e-08,
                       6.06663e-08, 6.06968e-08, 6.07352e-08, 6.07835e-08, 6.08444e-08, 6.09210e-08,
                       6.10174e-08, 6.11387e-08, 6.12916e-08, 6.14842e-08, 6.17274e-08, 6.20351e-08,
                       6.24262e-08, 6.29244e-08, 6.35606e-08, 6.43791e-08, 6.54267e-08, 6.67273e-08,
                       6.83024e-08, 7.02064e-08, 7.25230e-08, 7.52936e-08, 7.84757e-08, 8.20088e-08,
                       8.58383e-08, 8.98924e-08, 9.40576e-08, 9.81627e-08, 1.01971e-07, 1.05189e-07,
                       1.07474e-07, 1.08449e-07, 1.07754e-07, 1.05119e-07, 1.00411e-07, 9.36694e-08,
                       8.51266e-08, 7.52228e-08, 6.46008e-08, 5.40546e-08, 4.44014e-08, 3.62892e-08])
    cx_data[27] = np.array([6.39790e-08, 6.39861e-08, 6.39950e-08, 6.40062e-08, 6.40203e-08, 6.40381e-08,
                       6.40604e-08, 6.40885e-08, 6.41239e-08, 6.41685e-08, 6.42245e-08, 6.42950e-08,
                       6.43836e-08, 6.44950e-08, 6.46351e-08, 6.48113e-08, 6.50327e-08, 6.53113e-08,
                       6.56625e-08, 6.61065e-08, 6.66686e-08, 6.73812e-08, 6.82914e-08, 6.94486e-08,
                       7.08760e-08, 7.25941e-08, 7.46570e-08, 7.71476e-08, 8.00983e-08, 8.34460e-08,
                       8.71052e-08, 9.09882e-08, 9.49779e-08, 9.89037e-08, 1.02531e-07, 1.05565e-07,
                       1.07670e-07, 1.08495e-07, 1.07699e-07, 1.05006e-07, 1.00276e-07, 9.35409e-08,
                       8.50249e-08, 7.51575e-08, 6.45695e-08, 5.40443e-08, 4.43940e-08, 3.62689e-08])
    cx_data[28] = np.array([6.76978e-08, 6.77042e-08, 6.77124e-08, 6.77226e-08, 6.77355e-08, 6.77518e-08,
                       6.77722e-08, 6.77979e-08, 6.78303e-08, 6.78710e-08, 6.79222e-08, 6.79866e-08,
                       6.80676e-08, 6.81693e-08, 6.82971e-08, 6.84576e-08, 6.86590e-08, 6.89118e-08,
                       6.92289e-08, 6.96271e-08, 7.01280e-08, 7.07586e-08, 7.15523e-08, 7.25583e-08,
                       7.38275e-08, 7.53813e-08, 7.72370e-08, 7.94449e-08, 8.20816e-08, 8.51637e-08,
                       8.86002e-08, 9.22723e-08, 9.60503e-08, 9.97636e-08, 1.03182e-07, 1.06014e-07,
                       1.07925e-07, 1.08571e-07, 1.07637e-07, 1.04865e-07, 1.00105e-07, 9.33775e-08,
                       8.48950e-08, 7.50735e-08, 6.45285e-08, 5.40304e-08, 4.43846e-08, 3.62438e-08])
    cx_data[29] = np.array([7.18032e-08, 7.18090e-08, 7.18163e-08, 7.18256e-08, 7.18372e-08, 7.18518e-08,
                       7.18702e-08, 7.18934e-08, 7.19226e-08, 7.19592e-08, 7.20054e-08, 7.20634e-08,
                       7.21363e-08, 7.22279e-08, 7.23429e-08, 7.24873e-08, 7.26685e-08, 7.28955e-08,
                       7.31799e-08, 7.35357e-08, 7.39811e-08, 7.45388e-08, 7.52367e-08, 7.61092e-08,
                       7.72059e-08, 7.85780e-08, 8.02430e-08, 8.22122e-08, 8.45271e-08, 8.72512e-08,
                       9.03776e-08, 9.37801e-08, 9.72992e-08, 1.00756e-07, 1.03926e-07, 1.06524e-07,
                       1.08222e-07, 1.08677e-07, 1.07578e-07, 1.04695e-07, 9.98899e-08, 9.31697e-08,
                       8.47287e-08, 7.49647e-08, 6.44744e-08, 5.40113e-08, 4.43723e-08, 3.62130e-08])
    cx_data[30] = np.array([7.63583e-08, 7.63634e-08, 7.63699e-08, 7.63780e-08, 7.63882e-08, 7.64011e-08,
                       7.64174e-08, 7.64378e-08, 7.64635e-08, 7.64958e-08, 7.65364e-08, 7.65875e-08,
                       7.66518e-08, 7.67325e-08, 7.68339e-08, 7.69611e-08, 7.71207e-08, 7.73207e-08,
                       7.75710e-08, 7.78839e-08, 7.82745e-08, 7.87617e-08, 7.93692e-08, 8.01251e-08,
                       8.10633e-08, 8.22325e-08, 8.36816e-08, 8.54215e-08, 8.74538e-08, 8.98056e-08,
                       9.25196e-08, 9.55577e-08, 9.87524e-08, 1.01899e-07, 1.04769e-07, 1.07088e-07,
                       1.08538e-07, 1.08783e-07, 1.07513e-07, 1.04498e-07, 9.96245e-08, 9.29061e-08,
                       8.45152e-08, 7.48233e-08, 6.44023e-08, 5.39847e-08, 4.43563e-08, 3.61755e-08])
    cx_data[31] = np.array([8.14106e-08, 8.14150e-08, 8.14205e-08, 8.14275e-08, 8.14362e-08, 8.14472e-08,
                       8.14611e-08, 8.14785e-08, 8.15005e-08, 8.15281e-08, 8.15628e-08, 8.16064e-08,
                       8.16613e-08, 8.17302e-08, 8.18168e-08, 8.19255e-08, 8.20619e-08, 8.22327e-08,
                       8.24466e-08, 8.27138e-08, 8.30473e-08, 8.34624e-08, 8.39785e-08, 8.46190e-08,
                       8.54114e-08, 8.63873e-08, 8.75922e-08, 8.90696e-08, 9.08207e-08, 9.28330e-08,
                       9.51128e-08, 9.76747e-08, 1.00444e-07, 1.03207e-07, 1.05717e-07, 1.07703e-07,
                       1.08858e-07, 1.08860e-07, 1.07405e-07, 1.04259e-07, 9.93062e-08, 9.25767e-08,
                       8.42413e-08, 7.46385e-08, 6.43054e-08, 5.39473e-08, 4.43350e-08, 3.61299e-08])
    cx_data[32] = np.array([8.69786e-08, 8.69822e-08, 8.69867e-08, 8.69924e-08, 8.69995e-08, 8.70085e-08,
                       8.70198e-08, 8.70340e-08, 8.70519e-08, 8.70745e-08, 8.71028e-08, 8.71384e-08,
                       8.71832e-08, 8.72395e-08, 8.73102e-08, 8.73989e-08, 8.75102e-08, 8.76498e-08,
                       8.78244e-08, 8.80427e-08, 8.83151e-08, 8.86541e-08, 8.90751e-08, 8.95963e-08,
                       9.02401e-08, 9.10311e-08, 9.19972e-08, 9.31774e-08, 9.46064e-08, 9.62718e-08,
                       9.81427e-08, 1.00200e-07, 1.02423e-07, 1.04698e-07, 1.06772e-07, 1.08360e-07,
                       1.09165e-07, 1.08880e-07, 1.07212e-07, 1.03931e-07, 9.89120e-08, 9.21738e-08,
                       8.38939e-08, 7.43963e-08, 6.41740e-08, 5.38938e-08, 4.43063e-08, 3.60750e-08])
    cx_data[33] = np.array([9.30323e-08, 9.30350e-08, 9.30385e-08, 9.30428e-08, 9.30482e-08, 9.30550e-08,
                       9.30636e-08, 9.30745e-08, 9.30881e-08, 9.31052e-08, 9.31268e-08, 9.31539e-08,
                       9.31880e-08, 9.32308e-08, 9.32846e-08, 9.33522e-08, 9.34369e-08, 9.35432e-08,
                       9.36762e-08, 9.38425e-08, 9.40499e-08, 9.43083e-08, 9.46290e-08, 9.50258e-08,
                       9.55149e-08, 9.61154e-08, 9.68475e-08, 9.77324e-08, 9.87999e-08, 1.00072e-07,
                       1.01519e-07, 1.03090e-07, 1.04734e-07, 1.06397e-07, 1.07928e-07, 1.09042e-07,
                       1.09435e-07, 1.08812e-07, 1.06896e-07, 1.03463e-07, 9.83865e-08, 9.16679e-08,
                       8.34609e-08, 7.40819e-08, 6.39949e-08, 5.38165e-08, 4.42669e-08, 3.60093e-08])
    cx_data[34] = np.array([9.94686e-08, 9.94704e-08, 9.94727e-08, 9.94757e-08, 9.94793e-08, 9.94839e-08,
                       9.94897e-08, 9.94970e-08, 9.95062e-08, 9.95177e-08, 9.95322e-08, 9.95505e-08,
                       9.95735e-08, 9.96023e-08, 9.96386e-08, 9.96841e-08, 9.97412e-08, 9.98128e-08,
                       9.99024e-08, 1.00014e-07, 1.00154e-07, 1.00329e-07, 1.00545e-07, 1.00812e-07,
                       1.01142e-07, 1.01545e-07, 1.02037e-07, 1.02631e-07, 1.03338e-07, 1.04176e-07,
                       1.05151e-07, 1.06217e-07, 1.07297e-07, 1.08313e-07, 1.09181e-07, 1.09718e-07,
                       1.09629e-07, 1.08614e-07, 1.06410e-07, 1.02804e-07, 9.76691e-08, 9.09956e-08,
                       8.29073e-08, 7.36800e-08, 6.37528e-08, 5.37041e-08, 4.42114e-08, 3.59308e-08])
    cx_data[35] = np.array([1.06081e-07, 1.06082e-07, 1.06083e-07, 1.06085e-07, 1.06086e-07, 1.06089e-07,
                       1.06092e-07, 1.06096e-07, 1.06100e-07, 1.06106e-07, 1.06114e-07, 1.06123e-07,
                       1.06135e-07, 1.06149e-07, 1.06168e-07, 1.06191e-07, 1.06220e-07, 1.06257e-07,
                       1.06303e-07, 1.06360e-07, 1.06432e-07, 1.06521e-07, 1.06631e-07, 1.06767e-07,
                       1.06935e-07, 1.07140e-07, 1.07389e-07, 1.07688e-07, 1.08042e-07, 1.08451e-07,
                       1.08920e-07, 1.09438e-07, 1.09939e-07, 1.10327e-07, 1.10501e-07, 1.10352e-07,
                       1.09690e-07, 1.08226e-07, 1.05694e-07, 1.01892e-07, 9.66966e-08, 9.00871e-08,
                       8.21623e-08, 7.31505e-08, 6.34305e-08, 5.35428e-08, 4.41319e-08, 3.58373e-08])
    cx_data[36] = np.array([1.12527e-07, 1.12527e-07, 1.12528e-07, 1.12528e-07, 1.12528e-07, 1.12528e-07,
                       1.12528e-07, 1.12529e-07, 1.12529e-07, 1.12529e-07, 1.12530e-07, 1.12531e-07,
                       1.12532e-07, 1.12533e-07, 1.12534e-07, 1.12536e-07, 1.12538e-07, 1.12541e-07,
                       1.12544e-07, 1.12549e-07, 1.12554e-07, 1.12560e-07, 1.12568e-07, 1.12577e-07,
                       1.12588e-07, 1.12600e-07, 1.12612e-07, 1.12624e-07, 1.12632e-07, 1.12632e-07,
                       1.12611e-07, 1.12561e-07, 1.12454e-07, 1.12213e-07, 1.11725e-07, 1.10877e-07,
                       1.09552e-07, 1.07568e-07, 1.04671e-07, 1.00654e-07, 9.53954e-08, 8.88680e-08,
                       8.11472e-08, 7.24166e-08, 6.29849e-08, 5.33158e-08, 4.40182e-08, 3.57251e-08])
    cx_data[37] = np.array([1.18304e-07, 1.18303e-07, 1.18302e-07, 1.18301e-07, 1.18299e-07, 1.18298e-07,
                       1.18295e-07, 1.18293e-07, 1.18289e-07, 1.18285e-07, 1.18279e-07, 1.18272e-07,
                       1.18263e-07, 1.18252e-07, 1.18238e-07, 1.18220e-07, 1.18198e-07, 1.18170e-07,
                       1.18135e-07, 1.18091e-07, 1.18036e-07, 1.17967e-07, 1.17881e-07, 1.17774e-07,
                       1.17641e-07, 1.17476e-07, 1.17271e-07, 1.17018e-07, 1.16707e-07, 1.16326e-07,
                       1.15858e-07, 1.15282e-07, 1.14576e-07, 1.13704e-07, 1.12579e-07, 1.11087e-07,
                       1.09111e-07, 1.06547e-07, 1.03242e-07, 9.89954e-08, 9.36774e-08, 8.72527e-08,
                       7.97762e-08, 7.13900e-08, 6.23336e-08, 5.29793e-08, 4.38569e-08, 3.55900e-08])
    cx_data[38] = np.array([1.22736e-07, 1.22735e-07, 1.22733e-07, 1.22731e-07, 1.22728e-07, 1.22725e-07,
                       1.22721e-07, 1.22715e-07, 1.22708e-07, 1.22700e-07, 1.22689e-07, 1.22676e-07,
                       1.22659e-07, 1.22637e-07, 1.22611e-07, 1.22577e-07, 1.22535e-07, 1.22481e-07,
                       1.22415e-07, 1.22331e-07, 1.22227e-07, 1.22096e-07, 1.21933e-07, 1.21729e-07,
                       1.21477e-07, 1.21164e-07, 1.20777e-07, 1.20303e-07, 1.19722e-07, 1.19016e-07,
                       1.18164e-07, 1.17140e-07, 1.15916e-07, 1.14464e-07, 1.12744e-07, 1.10670e-07,
                       1.08135e-07, 1.05038e-07, 1.01299e-07, 9.68099e-08, 9.14421e-08, 8.51439e-08,
                       7.79525e-08, 6.99735e-08, 6.13802e-08, 5.24487e-08, 4.36079e-08, 3.54263e-08])
    cx_data[39] = np.array([1.25014e-07, 1.25012e-07, 1.25010e-07, 1.25007e-07, 1.25003e-07, 1.24999e-07,
                       1.24993e-07, 1.24986e-07, 1.24977e-07, 1.24966e-07, 1.24952e-07, 1.24934e-07,
                       1.24911e-07, 1.24883e-07, 1.24847e-07, 1.24803e-07, 1.24746e-07, 1.24676e-07,
                       1.24587e-07, 1.24476e-07, 1.24337e-07, 1.24163e-07, 1.23946e-07, 1.23675e-07,
                       1.23339e-07, 1.22921e-07, 1.22405e-07, 1.21771e-07, 1.20996e-07, 1.20053e-07,
                       1.18916e-07, 1.17555e-07, 1.15941e-07, 1.14042e-07, 1.11831e-07, 1.09271e-07,
                       1.06289e-07, 1.02793e-07, 9.87065e-08, 9.39861e-08, 8.85827e-08, 8.24395e-08,
                       7.55721e-08, 6.80588e-08, 6.00155e-08, 5.16214e-08, 4.31896e-08, 3.52025e-08])
    cx_data[40] = np.array([1.24278e-07, 1.24276e-07, 1.24273e-07, 1.24270e-07, 1.24266e-07, 1.24261e-07,
                       1.24255e-07, 1.24247e-07, 1.24237e-07, 1.24225e-07, 1.24209e-07, 1.24189e-07,
                       1.24164e-07, 1.24133e-07, 1.24093e-07, 1.24043e-07, 1.23981e-07, 1.23903e-07,
                       1.23804e-07, 1.23681e-07, 1.23526e-07, 1.23332e-07, 1.23090e-07, 1.22788e-07,
                       1.22411e-07, 1.21944e-07, 1.21365e-07, 1.20652e-07, 1.19778e-07, 1.18712e-07,
                       1.17422e-07, 1.15875e-07, 1.14038e-07, 1.11880e-07, 1.09371e-07, 1.06491e-07,
                       1.03214e-07, 9.94825e-08, 9.52266e-08, 9.03997e-08, 8.50019e-08, 7.90451e-08,
                       7.25382e-08, 6.55375e-08, 5.81199e-08, 5.03784e-08, 4.24984e-08, 3.48465e-08])
    cx_data[41] = np.array([1.19784e-07, 1.19782e-07, 1.19780e-07, 1.19776e-07, 1.19773e-07, 1.19768e-07,
                       1.19762e-07, 1.19754e-07, 1.19744e-07, 1.19732e-07, 1.19717e-07, 1.19698e-07,
                       1.19674e-07, 1.19643e-07, 1.19605e-07, 1.19557e-07, 1.19496e-07, 1.19420e-07,
                       1.19324e-07, 1.19204e-07, 1.19054e-07, 1.18865e-07, 1.18629e-07, 1.18334e-07,
                       1.17966e-07, 1.17508e-07, 1.16939e-07, 1.16235e-07, 1.15369e-07, 1.14309e-07,
                       1.13019e-07, 1.11463e-07, 1.09605e-07, 1.07409e-07, 1.04847e-07, 1.01896e-07,
                       9.85438e-08, 9.47800e-08, 9.05629e-08, 8.58440e-08, 8.06061e-08, 7.48948e-08,
                       6.87814e-08, 6.23241e-08, 5.55820e-08, 4.85902e-08, 4.14102e-08, 3.42613e-08])
    cx_data[42] = np.array([1.11122e-07, 1.11120e-07, 1.11118e-07, 1.11115e-07, 1.11112e-07, 1.11108e-07,
                       1.11103e-07, 1.11096e-07, 1.11088e-07, 1.11078e-07, 1.11065e-07, 1.11048e-07,
                       1.11028e-07, 1.11002e-07, 1.10969e-07, 1.10928e-07, 1.10877e-07, 1.10812e-07,
                       1.10730e-07, 1.10628e-07, 1.10500e-07, 1.10338e-07, 1.10136e-07, 1.09883e-07,
                       1.09567e-07, 1.09172e-07, 1.08680e-07, 1.08068e-07, 1.07312e-07, 1.06381e-07,
                       1.05239e-07, 1.03851e-07, 1.02176e-07, 1.00177e-07, 9.78211e-08, 9.50814e-08,
                       9.19413e-08, 8.83993e-08, 8.44557e-08, 8.00812e-08, 7.52417e-08, 6.99459e-08,
                       6.42852e-08, 5.83863e-08, 5.23322e-08, 4.61432e-08, 3.97896e-08, 3.33258e-08])
    cx_data[43] = np.array([9.84401e-08, 9.84389e-08, 9.84374e-08, 9.84355e-08, 9.84331e-08, 9.84301e-08,
                       9.84263e-08, 9.84216e-08, 9.84156e-08, 9.84080e-08, 9.83985e-08, 9.83865e-08,
                       9.83715e-08, 9.83525e-08, 9.83286e-08, 9.82986e-08, 9.82608e-08, 9.82133e-08,
                       9.81535e-08, 9.80784e-08, 9.79839e-08, 9.78652e-08, 9.77161e-08, 9.75290e-08,
                       9.72943e-08, 9.70004e-08, 9.66327e-08, 9.61737e-08, 9.56018e-08, 9.48917e-08,
                       9.40133e-08, 9.29327e-08, 9.16121e-08, 9.00122e-08, 8.80947e-08, 8.58267e-08,
                       8.31838e-08, 8.01529e-08, 7.67377e-08, 7.29438e-08, 6.87451e-08, 6.41120e-08,
                       5.90748e-08, 5.37736e-08, 4.83850e-08, 4.29857e-08, 3.75218e-08, 3.19048e-08])
    cx_data[44] = np.array([8.25870e-08, 8.25863e-08, 8.25854e-08, 8.25843e-08, 8.25829e-08, 8.25811e-08,
                       8.25788e-08, 8.25760e-08, 8.25724e-08, 8.25679e-08, 8.25623e-08, 8.25551e-08,
                       8.25462e-08, 8.25349e-08, 8.25207e-08, 8.25028e-08, 8.24802e-08, 8.24519e-08,
                       8.24162e-08, 8.23712e-08, 8.23146e-08, 8.22434e-08, 8.21536e-08, 8.20407e-08,
                       8.18985e-08, 8.17194e-08, 8.14942e-08, 8.12108e-08, 8.08546e-08, 8.04074e-08,
                       7.98468e-08, 7.91459e-08, 7.82732e-08, 7.71926e-08, 7.58654e-08, 7.42526e-08,
                       7.23199e-08, 7.00406e-08, 6.74002e-08, 6.44016e-08, 6.10477e-08, 5.73044e-08,
                       5.31357e-08, 4.85920e-08, 4.38682e-08, 3.91824e-08, 3.45704e-08, 2.98852e-08])
    cx_data[45] = np.array([6.50586e-08, 6.50583e-08, 6.50580e-08, 6.50575e-08, 6.50569e-08, 6.50562e-08,
                       6.50553e-08, 6.50542e-08, 6.50527e-08, 6.50509e-08, 6.50487e-08, 6.50458e-08,
                       6.50422e-08, 6.50376e-08, 6.50319e-08, 6.50247e-08, 6.50156e-08, 6.50041e-08,
                       6.49896e-08, 6.49713e-08, 6.49482e-08, 6.49189e-08, 6.48820e-08, 6.48351e-08,
                       6.47757e-08, 6.47001e-08, 6.46039e-08, 6.44810e-08, 6.43239e-08, 6.41224e-08,
                       6.38633e-08, 6.35298e-08, 6.31001e-08, 6.25470e-08, 6.18377e-08, 6.09344e-08,
                       5.97967e-08, 5.83860e-08, 5.66696e-08, 5.46258e-08, 5.22512e-08, 4.95390e-08,
                       4.64361e-08, 4.28896e-08, 3.89722e-08, 3.49426e-08, 3.10445e-08, 2.72400e-08])
    cx_data[46] = np.array([4.77131e-08, 4.77131e-08, 4.77131e-08, 4.77131e-08, 4.77131e-08, 4.77131e-08,
                       4.77132e-08, 4.77132e-08, 4.77133e-08, 4.77133e-08, 4.77134e-08, 4.77135e-08,
                       4.77136e-08, 4.77137e-08, 4.77139e-08, 4.77141e-08, 4.77143e-08, 4.77145e-08,
                       4.77148e-08, 4.77152e-08, 4.77155e-08, 4.77157e-08, 4.77158e-08, 4.77156e-08,
                       4.77148e-08, 4.77130e-08, 4.77093e-08, 4.77027e-08, 4.76912e-08, 4.76719e-08,
                       4.76402e-08, 4.75891e-08, 4.75082e-08, 4.73826e-08, 4.71910e-08, 4.69051e-08,
                       4.64889e-08, 4.59004e-08, 4.50953e-08, 4.40319e-08, 4.26779e-08, 4.10208e-08,
                       3.90407e-08, 3.66552e-08, 3.37859e-08, 3.05334e-08, 2.72244e-08, 2.41049e-08])
    cx_data[47] = np.array([3.23089e-08, 3.23090e-08, 3.23092e-08, 3.23095e-08, 3.23098e-08, 3.23102e-08,
                       3.23107e-08, 3.23113e-08, 3.23121e-08, 3.23131e-08, 3.23143e-08, 3.23159e-08,
                       3.23178e-08, 3.23203e-08, 3.23234e-08, 3.23273e-08, 3.23322e-08, 3.23383e-08,
                       3.23461e-08, 3.23557e-08, 3.23679e-08, 3.23831e-08, 3.24020e-08, 3.24256e-08,
                       3.24550e-08, 3.24913e-08, 3.25361e-08, 3.25910e-08, 3.26578e-08, 3.27381e-08,
                       3.28334e-08, 3.29445e-08, 3.30703e-08, 3.32075e-08, 3.33484e-08, 3.34791e-08,
                       3.35771e-08, 3.36102e-08, 3.35363e-08, 3.33061e-08, 3.28682e-08, 3.21791e-08,
                       3.12191e-08, 2.99561e-08, 2.82707e-08, 2.60519e-08, 2.34337e-08, 2.08052e-08])

    # data are given in  cm^3/s. Convert to m^3/s
    cx_data = cx_data * 1E-6

    Tn_vals, Ti_vals = np.meshgrid((np.linspace(1, 48, 48) - 1) / 10 - 3,  # log10(Tn in keV)
                                   (np.linspace(1, 48, 48) - 1) / 10 - 3)  # log10(Ti in keV)
    # perform 2D interpolation
    #sv_cx = griddata(np.column_stack((Tn_vals.flatten(), Ti_vals.flatten())),
    #                 cx_data.flatten(),
    #                 (np.asarray(Tn), np.asarray(Ti)),
    #                 method='linear',
    #                 rescale=True)

    return Rbf(Tn_vals, Ti_vals, cx_data)

def calc_svrec(sv, n, T, xsec_rec='stacey_thomas'):
    """
        Calculate recombination cross section.

    :param n:
    :param T:
    :param xsec_rec:
    :return:
    """

    Te = T.e

    # reshape ne if necessary, i.e. when calculating face values
    # note, this shouldn't ever be necessary for this sv, but I could be wrong.
    if Te.ndim == 2:
        ne = np.repeat(n.e.reshape(-1, 1), Te.shape[1], axis=1)
    else:
        ne = n.e

    if xsec_rec == 'stacey_thomas':
        sv_rec = sv.rec_st(ne, Te)
    elif xsec_rec == 'degas':
        sv_rec = sv.rec_degas(ne, Te)
    return sv_rec


def calc_svcx(sv, T, Tn, en_grp, xsec_cx='degas'):
    """
        Calculate charge exchange cross sections

    :param T:
    :param Tn:
    :param en_grp:
    :param xsec_cx:
    :return:
    """
    # determine neutral temperature array to use based on specified group
    Tn = Tn.s if en_grp == 'slow' else Tn.t

    # reshape ne if necessary, i.e. when calculating face values
    if Tn.ndim == 2:
        Ti = np.repeat(T.i.reshape(-1, 1), Tn.shape[1], axis=1)
    else:
        Ti = T.i

    if xsec_cx == 'janev':
        sv_cx = sv.cx_janev(Ti, Tn)
    elif xsec_cx == 'stacey_thomas':
        sv_cx = sv.cx_st(Ti, Tn)
    elif xsec_cx == 'degas':
        sv_cx = sv.cx_degas(Ti, Tn)

    return sv_cx


def calc_svel(sv, T, Tn, en_grp, xsec_el='stacey_thomas'):
    """
        Calculates elastic scattering with ions cross sections

    :param T:
    :param Tn:
    :param en_grp:
    :param xsec_el:
    :return:
    """
    # determine neutral temperature array to use based on specified group
    Tn = Tn.s if en_grp == 'slow' else Tn.t

    # reshape ne if necessary, i.e. when calculating face values
    if Tn.ndim == 2:
        Ti = np.repeat(T.i.reshape(-1, 1), Tn.shape[1], axis=1)
    else:
        Ti = T.i

    if xsec_el == 'janev':
        print 'janev elastic scattering cross sections not available. Using Stacey-Thomas instead.'
        sv_el = sv.el_st(Ti, Tn)
    elif xsec_el == 'stacey_thomas':
        sv_el = sv.el_st(Ti, Tn)
    elif xsec_el == 'degas':
        print 'degas elastic scattering cross sections not available. Using Stacey-Thomas instead.'
        sv_el = sv.el_st(Ti, Tn)
    return sv_el


def calc_sveln(sv, Tn, en_grp, xsec_eln='stacey_thomas'):
    """
        Calculates elastic scattering with neutrals cross sections

    :param Tn:
    :param en_grp:
    :param xsec_eln:
    :return:
    """
    # determine neutral temperature array to use based on specified group
    Tn = Tn.s if en_grp == 'slow' else Tn.t

    if xsec_eln == 'janev':
        print 'janev elastic scattering cross sections not available. Using Stacey-Thomas instead..'
        sv_eln = sv.eln_st(Tn)
    elif xsec_eln == 'stacey_thomas':
        sv_eln = sv.eln_st(Tn)
    elif xsec_eln == 'degas':
        print 'degas elastic scattering cross sections not available. Using Stacey-Thomas instead..'
        sv_eln = sv.eln_st(Tn)
    return sv_eln

def calc_svione(sv, n, T, xsec_ione='degas'):
    """

    :param n:
    :param T:
    :param xsec_ione:
    :return:
    """

    Te = T.e

    # reshape ne if necessary, i.e. when calculating face values
    # note, this shouldn't ever be necessary for this sv, but I could be wrong.
    if Te.ndim == 2:
        ne = np.repeat(n.e.reshape(-1, 1), Te.shape[1], axis=1)
    else:
        ne = n.e

    if xsec_ione == 'janev':
        sv_ione = sv.ione_janev(Te)
    elif xsec_ione == 'stacey_thomas':
        sv_ione = sv.ione_st(ne, Te)
    elif xsec_ione == 'degas':
        sv_ione = sv.ione_degas(ne, Te)
    return sv_ione

#############################################################################
#############################################################################
# no idea what any of these are, but they're here if anyone figures it out
# and needs them... - MH

def calc_svnel_degas(ne, Te):
    """ Calculates cross section for "Neutral Electron Losses," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """


    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    nel_data = np.zeros((15, 60))
    nel_data[0] = np.array([4.87679E-78, 1.21210E-63, 3.22057E-52, 3.73566E-43, 5.78390E-36, 2.90433E-30,
                                9.57467E-26, 3.62929E-22, 2.46956E-19, 4.29975E-17, 2.53951E-15, 6.36583E-14,
                                8.08868E-13, 5.99444E-12, 2.89415E-11, 9.93357E-11, 2.59710E-10, 5.46356E-10,
                                9.66151E-10, 1.48712E-09, 2.15829E-09, 2.93300E-09, 3.71879E-09, 4.45745E-09,
                                5.10881E-09, 5.65019E-09, 6.07401E-09, 6.38372E-09, 6.58971E-09, 6.70578E-09,
                                6.74675E-09, 6.72690E-09, 6.65908E-09, 6.55439E-09, 6.42212E-09, 6.26985E-09,
                                6.10364E-09, 5.92831E-09, 5.74757E-09, 5.56431E-09, 5.38071E-09, 5.19841E-09,
                                5.01864E-09, 4.59139E-09, 4.30706E-09, 4.04605E-09, 3.79040E-09, 3.54215E-09,
                                3.30279E-09, 3.07341E-09, 2.85470E-09, 2.64710E-09, 2.45081E-09, 2.26585E-09,
                                2.09209E-09, 1.92928E-09, 1.77712E-09, 1.63522E-09, 1.50314E-09, 1.38044E-09])
    nel_data[1] = np.array([1.54216E-77, 3.83298E-63, 1.01843E-51, 1.18131E-42, 1.82902E-35, 9.18426E-30,
                                3.02776E-25, 1.14768E-21, 7.80939E-19, 1.35967E-16, 8.03001E-15, 2.01262E-13,
                                2.55665E-12, 1.89393E-11, 9.13890E-11, 3.13464E-10, 8.18945E-10, 1.72156E-09,
                                3.04222E-09, 4.67977E-09, 6.79003E-09, 9.22482E-09, 1.16925E-08, 1.40117E-08,
                                1.60564E-08, 1.77560E-08, 1.90866E-08, 2.00593E-08, 2.07067E-08, 2.10721E-08,
                                2.12020E-08, 2.11410E-08, 2.09295E-08, 2.06023E-08, 2.01884E-08, 1.97117E-08,
                                1.91912E-08, 1.86419E-08, 1.80755E-08, 1.75011E-08, 1.69255E-08, 1.63540E-08,
                                1.57902E-08, 1.44437E-08, 1.35530E-08, 1.27331E-08, 1.19299E-08, 1.11498E-08,
                                1.03976E-08, 9.67668E-09, 8.98922E-09, 8.33660E-09, 7.71945E-09, 7.13784E-09,
                                6.59137E-09, 6.07930E-09, 5.60064E-09, 5.15417E-09, 4.73858E-09, 4.35242E-09])
    nel_data[2] = np.array([4.87663E-77, 1.21207E-62, 3.22048E-51, 3.73557E-42, 5.78378E-35, 2.90428E-29,
                                9.57451E-25, 3.62923E-21, 2.46951E-18, 4.29946E-16, 2.53899E-14, 6.36238E-13,
                                8.07910E-12, 5.98134E-11, 2.88393E-10, 9.88261E-10, 2.57929E-09, 5.41665E-09,
                                9.56281E-09, 1.46977E-08, 2.13179E-08, 2.89506E-08, 3.66780E-08, 4.39401E-08,
                                5.03399E-08, 5.56613E-08, 5.98278E-08, 6.28759E-08, 6.49069E-08, 6.60564E-08,
                                6.64695E-08, 6.62857E-08, 6.56310E-08, 6.46139E-08, 6.33254E-08, 6.18395E-08,
                                6.02160E-08, 5.85017E-08, 5.67334E-08, 5.49392E-08, 5.31408E-08, 5.13542E-08,
                                4.95916E-08, 4.53528E-08, 4.25717E-08, 4.00015E-08, 3.74832E-08, 3.50370E-08,
                                3.26777E-08, 3.04159E-08, 2.82589E-08, 2.62108E-08, 2.42737E-08, 2.24478E-08,
                                2.07321E-08, 1.91241E-08, 1.76208E-08, 1.62184E-08, 1.49128E-08, 1.36995E-08])
    nel_data[3] = np.array([1.54200E-76, 3.83262E-62, 1.01834E-50, 1.18123E-41, 1.82890E-34, 9.18372E-29,
                                3.02761E-24, 1.14762E-20, 7.80893E-18, 1.35950E-15, 8.02726E-14, 2.01090E-12,
                                2.55202E-11, 1.88772E-10, 9.09128E-10, 3.11119E-09, 8.10835E-09, 1.70038E-08,
                                2.99795E-08, 4.60228E-08, 6.67218E-08, 9.05565E-08, 1.14655E-07, 1.37294E-07,
                                1.57234E-07, 1.73830E-07, 1.86821E-07, 1.96340E-07, 2.02692E-07, 2.06305E-07,
                                2.07626E-07, 2.07090E-07, 2.05086E-07, 2.01953E-07, 1.97972E-07, 1.93374E-07,
                                1.88343E-07, 1.83027E-07, 1.77539E-07, 1.71967E-07, 1.66379E-07, 1.60824E-07,
                                1.55342E-07, 1.42024E-07, 1.33387E-07, 1.25359E-07, 1.17491E-07, 1.09845E-07,
                                1.02469E-07, 9.53948E-08, 8.86462E-08, 8.22366E-08, 7.61727E-08, 7.04556E-08,
                                6.50817E-08, 6.00441E-08, 5.53334E-08, 5.09380E-08, 4.68450E-08, 4.30407E-08])
    nel_data[4] = np.array([4.87503E-76, 1.21171E-61, 3.21965E-50, 3.73471E-41, 5.78258E-34, 2.90374E-28,
                                9.57294E-24, 3.62869E-20, 2.46911E-17, 4.29830E-15, 2.53740E-13, 6.35319E-12,
                                8.05550E-11, 5.95064E-10, 2.86086E-09, 9.77085E-09, 2.54113E-08, 5.31803E-08,
                                9.35845E-08, 1.43424E-07, 2.07804E-07, 2.81783E-07, 3.56458E-07, 4.26555E-07,
                                4.88269E-07, 5.39668E-07, 5.79897E-07, 6.09482E-07, 6.29256E-07, 6.40589E-07,
                                6.44843E-07, 6.43371E-07, 6.37356E-07, 6.27842E-07, 6.15697E-07, 6.01625E-07,
                                5.86204E-07, 5.69883E-07, 5.53014E-07, 5.35870E-07, 5.18657E-07, 5.01535E-07,
                                4.84619E-07, 4.42913E-07, 4.16315E-07, 3.91389E-07, 3.66942E-07, 3.43172E-07,
                                3.20226E-07, 2.98210E-07, 2.77195E-07, 2.57227E-07, 2.38327E-07, 2.20500E-07,
                                2.03736E-07, 1.88015E-07, 1.73308E-07, 1.59580E-07, 1.46792E-07, 1.34902E-07])
    nel_data[5] = np.array([1.54041E-75, 3.82909E-61, 1.01751E-49, 1.18036E-40, 1.82771E-33, 9.17838E-28,
                                3.02604E-23, 1.14708E-19, 7.80523E-17, 1.35860E-14, 8.01704E-13, 2.00558E-11,
                                2.53915E-10, 1.87164E-09, 8.97377E-09, 3.05546E-08, 7.92119E-08, 1.65264E-07,
                                2.90008E-07, 4.43366E-07, 6.41832E-07, 8.69121E-07, 1.09810E-06, 1.31281E-06,
                                1.50177E-06, 1.65918E-06, 1.78245E-06, 1.87360E-06, 1.93465E-06, 1.97006E-06,
                                1.98387E-06, 1.98046E-06, 1.96303E-06, 1.93488E-06, 1.89864E-06, 1.85641E-06,
                                1.80999E-06, 1.76075E-06, 1.70975E-06, 1.65782E-06, 1.60559E-06, 1.55356E-06,
                                1.50208E-06, 1.37220E-06, 1.29140E-06, 1.21473E-06, 1.13946E-06, 1.06619E-06,
                                9.95403E-07, 9.27423E-07, 8.62481E-07, 8.00724E-07, 7.42226E-07, 6.87008E-07,
                                6.35048E-07, 5.86287E-07, 5.40643E-07, 4.98011E-07, 4.58276E-07, 4.21308E-07])
    nel_data[6] = np.array([4.85913E-75, 1.20819E-60, 3.21130E-49, 3.72607E-40, 5.77067E-33, 2.89841E-27,
                                9.55726E-23, 3.62332E-19, 2.46553E-16, 4.29055E-14, 2.52980E-12, 6.31797E-11,
                                7.97675E-10, 5.85728E-09, 2.79528E-08, 9.46885E-08, 2.44203E-07, 5.06986E-07,
                                8.85732E-07, 1.34898E-06, 1.95010E-06, 2.63487E-06, 3.32329E-06, 3.96820E-06,
                                4.53589E-06, 5.00915E-06, 5.38071E-06, 5.65710E-06, 5.84354E-06, 5.95361E-06,
                                5.99920E-06, 5.99651E-06, 5.95002E-06, 5.87118E-06, 5.76781E-06, 5.64605E-06,
                                5.51125E-06, 5.36778E-06, 5.21837E-06, 5.06564E-06, 4.91155E-06, 4.75757E-06,
                                4.60483E-06, 4.20537E-06, 3.96493E-06, 3.73301E-06, 3.50486E-06, 3.28240E-06,
                                3.06710E-06, 2.86001E-06, 2.66190E-06, 2.47323E-06, 2.29428E-06, 2.12515E-06,
                                1.96581E-06, 1.81611E-06, 1.67582E-06, 1.54466E-06, 1.42229E-06, 1.30833E-06])
    nel_data[7] = np.array([1.52464E-74, 3.79412E-60, 1.00922E-48, 1.17178E-39, 1.81586E-32, 9.12530E-27,
                                3.01041E-22, 1.14172E-18, 7.76947E-16, 1.35115E-13, 7.95074E-12, 1.97813E-10,
                                2.48338E-09, 1.80980E-08, 8.56111E-08, 2.87285E-07, 7.34052E-07, 1.51084E-06,
                                2.61947E-06, 3.96401E-06, 5.71297E-06, 7.69022E-06, 9.67329E-06, 1.15303E-05,
                                1.31679E-05, 1.45379E-05, 1.56209E-05, 1.64350E-05, 1.69944E-05, 1.73368E-05,
                                1.74951E-05, 1.75314E-05, 1.74330E-05, 1.72407E-05, 1.69765E-05, 1.66625E-05,
                                1.63047E-05, 1.59223E-05, 1.55180E-05, 1.51005E-05, 1.46759E-05, 1.42485E-05,
                                1.38216E-05, 1.26318E-05, 1.19421E-05, 1.12649E-05, 1.05956E-05, 9.94026E-06,
                                9.30365E-06, 8.68923E-06, 8.09954E-06, 7.53633E-06, 7.00069E-06, 6.49317E-06,
                                6.01389E-06, 5.56262E-06, 5.13887E-06, 4.74191E-06, 4.37087E-06, 4.02476E-06])
    nel_data[8] = np.array([4.70565E-74, 1.17405E-59, 3.13014E-48, 3.64189E-39, 5.65415E-32, 2.84612E-26,
                                9.40291E-22, 3.56989E-18, 2.42944E-15, 4.21600E-13, 2.46753E-11, 6.08354E-10,
                                7.54245E-09, 5.40739E-08, 2.51125E-07, 8.26793E-07, 2.07432E-06, 4.19956E-06,
                                7.17903E-06, 1.07398E-05, 1.53658E-05, 2.05527E-05, 2.57448E-05, 3.06174E-05,
                                3.49424E-05, 3.86005E-05, 4.15440E-05, 4.38133E-05, 4.54401E-05, 4.65133E-05,
                                4.71113E-05, 4.74224E-05, 4.73612E-05, 4.70470E-05, 4.65353E-05, 4.59721E-05,
                                4.52211E-05, 4.44213E-05, 4.35305E-05, 4.25849E-05, 4.16028E-05, 4.05954E-05,
                                3.95719E-05, 3.62922E-05, 3.44798E-05, 3.26635E-05, 3.08487E-05, 2.90544E-05,
                                2.72954E-05, 2.55836E-05, 2.39282E-05, 2.23359E-05, 2.08116E-05, 1.93585E-05,
                                1.79784E-05, 1.66720E-05, 1.54393E-05, 1.42792E-05, 1.31902E-05, 1.21703E-05])
    nel_data[9] = np.array([1.38310E-73, 3.47663E-59, 9.33179E-48, 1.09237E-38, 1.70528E-31, 8.62629E-26,
                                2.86222E-21, 1.08997E-17, 7.41641E-15, 1.27883E-12, 7.37327E-11, 1.77526E-09,
                                2.13461E-08, 1.47214E-07, 6.56064E-07, 2.07468E-06, 5.01853E-06, 9.84771E-06,
                                1.64114E-05, 2.40713E-05, 3.39651E-05, 4.49796E-05, 5.60090E-05, 6.64400E-05,
                                7.58445E-05, 8.39953E-05, 9.07920E-05, 9.62995E-05, 1.00556E-04, 1.03714E-04,
                                1.05906E-04, 1.07555E-04, 1.08381E-04, 1.08644E-04, 1.08449E-04, 1.08726E-04,
                                1.08171E-04, 1.07646E-04, 1.06772E-04, 1.05698E-04, 1.04470E-04, 1.03107E-04,
                                1.01625E-04, 9.41328E-05, 9.03740E-05, 8.64587E-05, 8.24268E-05, 7.83324E-05,
                                7.42218E-05, 7.01347E-05, 6.61040E-05, 6.21573E-05, 5.83170E-05, 5.46006E-05,
                                5.10219E-05, 4.75911E-05, 4.43151E-05, 4.11982E-05, 3.82425E-05, 3.54483E-05])
    nel_data[10] = np.array([3.57611E-73, 9.15374E-59, 2.49868E-47, 2.97073E-38, 4.70431E-31, 2.41108E-25,
                                8.09330E-21, 3.10734E-17, 2.11288E-14, 3.58059E-12, 1.98617E-10, 4.51856E-09,
                                5.08209E-08, 3.24678E-07, 1.34564E-06, 3.99279E-06, 9.16538E-06, 1.72570E-05,
                                2.78681E-05, 3.99411E-05, 5.54543E-05, 7.26510E-05, 8.99159E-05, 1.06418E-04,
                                1.21580E-04, 1.35096E-04, 1.46817E-04, 1.56826E-04, 1.65140E-04, 1.71949E-04,
                                1.77406E-04, 1.82220E-04, 1.85776E-04, 1.88492E-04, 1.90500E-04, 1.95020E-04,
                                1.97256E-04, 2.00155E-04, 2.02292E-04, 2.04095E-04, 2.05649E-04, 2.06950E-04,
                                2.07990E-04, 1.96401E-04, 1.92255E-04, 1.87466E-04, 1.82104E-04, 1.76259E-04,
                                1.70018E-04, 1.63461E-04, 1.56665E-04, 1.49702E-04, 1.42637E-04, 1.35531E-04,
                                1.28440E-04, 1.21413E-04, 1.14493E-04, 1.07720E-04, 1.01126E-04, 9.47368E-05])
    nel_data[11] = np.array([7.17234E-73, 1.89276E-58, 5.32204E-47, 6.51130E-38, 1.05987E-30, 5.57654E-25,
                                1.91757E-20, 7.49222E-17, 5.08655E-14, 8.30950E-12, 4.27673E-10, 8.83837E-09,
                                9.02480E-08, 5.24963E-07, 2.01677E-06, 5.64617E-06, 1.24165E-05, 2.26630E-05,
                                3.57945E-05, 5.05101E-05, 6.93900E-05, 9.02922E-05, 1.11331E-04, 1.31589E-04,
                                1.50430E-04, 1.67524E-04, 1.82705E-04, 1.96074E-04, 2.07626E-04, 2.17575E-04,
                                2.26077E-04, 2.34052E-04, 2.40613E-04, 2.46285E-04, 2.51208E-04, 2.61318E-04,
                                2.67800E-04, 2.76098E-04, 2.83537E-04, 2.90895E-04, 2.98336E-04, 3.05842E-04,
                                3.13389E-04, 3.01977E-04, 3.01863E-04, 3.00764E-04, 2.98706E-04, 2.95741E-04,
                                2.91922E-04, 2.87298E-04, 2.81923E-04, 2.75849E-04, 2.69133E-04, 2.61833E-04,
                                2.54010E-04, 2.45726E-04, 2.37047E-04, 2.28039E-04, 2.18770E-04, 2.09307E-04])
    nel_data[12] = np.array([1.05167E-72, 2.85763E-58, 8.28100E-47, 1.04496E-37, 1.75540E-30, 9.53530E-25,
                                3.38204E-20, 1.35292E-16, 9.16488E-14, 1.42657E-11, 6.72975E-10, 1.26640E-08,
                                1.19542E-07, 6.51925E-07, 2.39323E-06, 6.49372E-06, 1.39779E-05, 2.51423E-05,
                                3.93138E-05, 5.51014E-05, 7.53564E-05, 9.77782E-05, 1.20378E-04, 1.42210E-04,
                                1.62623E-04, 1.81283E-04, 1.98017E-04, 2.12940E-04, 2.26041E-04, 2.37541E-04,
                                2.47602E-04, 2.57247E-04, 2.65458E-04, 2.72813E-04, 2.79460E-04, 2.92920E-04,
                                3.02096E-04, 3.13922E-04, 3.25001E-04, 3.36353E-04, 3.48229E-04, 3.60650E-04,
                                3.73628E-04, 3.64328E-04, 3.68844E-04, 3.72516E-04, 3.75349E-04, 3.77372E-04,
                                3.78607E-04, 3.79071E-04, 3.78777E-04, 3.77733E-04, 3.75948E-04, 3.73429E-04,
                                3.70181E-04, 3.66213E-04, 3.61532E-04, 3.56150E-04, 3.50083E-04, 3.43348E-04])
    nel_data[13] = np.array([1.23357E-72, 3.40683E-58, 1.00475E-46, 1.29210E-37, 2.21508E-30, 1.22955E-24,
                                4.45884E-20, 1.81541E-16, 1.22751E-13, 1.84399E-11, 8.21551E-10, 1.46594E-08,
                                1.33075E-07, 7.05207E-07, 2.54096E-06, 6.81032E-06, 1.45446E-05, 2.60248E-05,
                                4.05521E-05, 5.66977E-05, 7.74202E-05, 1.00353E-04, 1.23487E-04, 1.45862E-04,
                                1.66832E-04, 1.86052E-04, 2.03350E-04, 2.18843E-04, 2.32518E-04, 2.44602E-04,
                                2.55256E-04, 2.65543E-04, 2.74398E-04, 2.82418E-04, 2.89755E-04, 3.04563E-04,
                                3.14847E-04, 3.28142E-04, 3.40770E-04, 3.53856E-04, 3.67699E-04, 3.82349E-04,
                                3.97853E-04, 3.89834E-04, 3.96746E-04, 4.02995E-04, 4.08594E-04, 4.13583E-04,
                                4.17990E-04, 4.21839E-04, 4.25150E-04, 4.27933E-04, 4.30198E-04, 4.31948E-04,
                                4.33185E-04, 4.33905E-04, 4.34102E-04, 4.33769E-04, 4.32896E-04, 4.31472E-04])
    nel_data[14] = np.array([1.30494E-72, 3.62728E-58, 1.07743E-46, 1.39655E-37, 2.41507E-30, 1.35344E-24,
                                4.95790E-20, 2.03522E-16, 1.37441E-13, 2.03018E-11, 8.82268E-10, 1.54038E-08,
                                1.37808E-07, 7.22474E-07, 2.58606E-06, 6.90400E-06, 1.47126E-05, 2.62836E-05,
                                4.09148E-05, 5.71767E-05, 7.80479E-05, 1.01139E-04, 1.24453E-04, 1.47014E-04,
                                1.68146E-04, 1.87549E-04, 2.05012E-04, 2.20691E-04, 2.34553E-04, 2.46843E-04,
                                2.57706E-04, 2.68217E-04, 2.77298E-04, 2.85552E-04, 2.93130E-04, 3.08403E-04,
                                3.19075E-04, 3.32884E-04, 3.46056E-04, 3.59756E-04, 3.74300E-04, 3.89750E-04,
                                4.06169E-04, 3.98651E-04, 4.06464E-04, 4.13697E-04, 4.20370E-04, 4.26530E-04,
                                4.32215E-04, 4.37460E-04, 4.42290E-04, 4.46730E-04, 4.50797E-04, 4.54507E-04,
                                4.57871E-04, 4.60898E-04, 4.63594E-04, 4.65960E-04, 4.67998E-04, 4.69705E-04])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, nel_data)


def calc_svcel_degas():
    """ Calculates cross section for "Continuum Electron Losses," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """



    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    cel_data = np.zeros((15, 60))
    cel_data[0] = np.array([1.12349E-11, 5.51216E-12, 2.94674E-12, 1.70906E-12, 1.06617E-12, 7.08039E-13,
                            4.95378E-13, 3.61651E-13, 2.73200E-13, 2.12056E-13, 1.68146E-13, 1.35586E-13,
                            1.10777E-13, 9.14484E-14, 7.61141E-14, 6.37695E-14, 5.37164E-14, 4.54578E-14,
                            3.86196E-14, 3.29270E-14, 2.81662E-14, 2.41680E-14, 2.07992E-14, 1.79513E-14,
                            1.55365E-14, 1.34830E-14, 1.17314E-14, 1.02327E-14, 8.94653E-15, 7.83930E-15,
                            6.88317E-15, 6.05501E-15, 5.33559E-15, 4.70888E-15, 4.16147E-15, 3.68217E-15,
                            3.26154E-15, 2.89165E-15, 2.56578E-15, 2.27821E-15, 2.02408E-15, 1.79921E-15,
                            1.60001E-15, 1.42339E-15, 1.26665E-15, 1.12747E-15, 1.00379E-15, 8.93851E-16,
                            7.96070E-16, 7.09075E-16, 6.31653E-16, 5.62734E-16, 5.01371E-16, 4.46725E-16,
                            3.98055E-16, 3.54702E-16, 3.16081E-16, 2.81673E-16, 2.51016E-16, 2.23700E-16])
    cel_data[0] = np.array([8.04787E-11, 3.52758E-11, 1.68863E-11, 8.84199E-12, 5.03733E-12, 3.09413E-12,
                            2.02756E-12, 1.40251E-12, 1.01413E-12, 7.59994E-13, 5.86027E-13, 4.62228E-13,
                            3.71155E-13, 3.02262E-13, 2.48928E-13, 2.06848E-13, 1.73133E-13, 1.45797E-13,
                            1.23398E-13, 1.04905E-13, 8.95413E-14, 7.67040E-14, 6.59318E-14, 5.68533E-14,
                            4.91744E-14, 4.26556E-14, 3.71027E-14, 3.23564E-14, 2.82860E-14, 2.47835E-14,
                            2.17601E-14, 1.91419E-14, 1.68677E-14, 1.48867E-14, 1.31565E-14, 1.16415E-14,
                            1.03119E-14, 9.14267E-15, 8.11253E-15, 7.20345E-15, 6.40003E-15, 5.68909E-15,
                            5.05930E-15, 4.50087E-15, 4.00530E-15, 3.56522E-15, 3.17417E-15, 2.82653E-15,
                            2.51734E-15, 2.24225E-15, 1.99744E-15, 1.77950E-15, 1.58546E-15, 1.41266E-15,
                            1.25875E-15, 1.12166E-15, 9.99531E-16, 8.90724E-16, 7.93779E-16, 7.07399E-16])
    cel_data[0] = np.array([6.42078E-10, 2.53755E-10, 1.08747E-10, 5.10491E-11, 2.62604E-11, 1.47241E-11,
                            8.91613E-12, 5.77045E-12, 3.94913E-12, 2.82987E-12, 2.10488E-12, 1.61323E-12,
                            1.26629E-12, 1.01300E-12, 8.22696E-13, 6.76243E-13, 5.61285E-13, 4.69608E-13,
                            3.95497E-13, 3.34957E-13, 2.85087E-13, 2.43689E-13, 2.09134E-13, 1.80127E-13,
                            1.55669E-13, 1.34955E-13, 1.17340E-13, 1.02304E-13, 8.94203E-14, 7.83416E-14,
                            6.87819E-14, 6.05058E-14, 5.33184E-14, 4.70580E-14, 4.15901E-14, 3.68023E-14,
                            3.26004E-14, 2.89050E-14, 2.56490E-14, 2.27755E-14, 2.02358E-14, 1.79884E-14,
                            1.59974E-14, 1.42318E-14, 1.26650E-14, 1.12736E-14, 1.00372E-14, 8.93794E-15,
                            7.96029E-15, 7.09045E-15, 6.31632E-15, 5.62718E-15, 5.01359E-15, 4.46717E-15,
                            3.98049E-15, 3.54698E-15, 3.16078E-15, 2.81670E-15, 2.51014E-15, 2.23699E-15])
    cel_data[0] = np.array([5.58430E-09, 2.03045E-09, 7.87243E-10, 3.31550E-10, 1.52873E-10, 7.74009E-11,
                            4.27498E-11, 2.55429E-11, 1.63364E-11, 1.10665E-11, 7.86206E-12, 5.80680E-12,
                            4.42536E-12, 3.45837E-12, 2.75755E-12, 2.23439E-12, 1.83404E-12, 1.52135E-12,
                            1.27287E-12, 1.07265E-12, 9.09509E-13, 7.75251E-13, 6.63934E-13, 5.70974E-13,
                            4.92911E-13, 4.26997E-13, 3.71077E-13, 3.23421E-13, 2.82636E-13, 2.47594E-13,
                            2.17373E-13, 1.91219E-13, 1.68510E-13, 1.48731E-13, 1.31457E-13, 1.16330E-13,
                            1.03054E-13, 9.13768E-14, 8.10875E-14, 7.20060E-14, 6.39790E-14, 5.68751E-14,
                            5.05813E-14, 4.50001E-14, 4.00467E-14, 3.56475E-14, 3.17383E-14, 2.82628E-14,
                            2.51716E-14, 2.24212E-14, 1.99734E-14, 1.77943E-14, 1.58541E-14, 1.41262E-14,
                            1.25873E-14, 1.12164E-14, 9.99518E-15, 8.90715E-15, 7.93773E-15, 7.07394E-15])
    cel_data[0] = np.array([5.13046E-08, 1.76458E-08, 6.32154E-09, 2.41284E-09, 9.96605E-10, 4.53674E-10,
                            2.26382E-10, 1.23358E-10, 7.27739E-11, 4.60153E-11, 3.08674E-11, 2.17507E-11,
                            1.59577E-11, 1.20974E-11, 9.41656E-12, 7.48728E-12, 6.05597E-12, 4.96660E-12,
                            4.11931E-12, 3.44842E-12, 2.90932E-12, 2.47064E-12, 2.11007E-12, 1.81101E-12,
                            1.56119E-12, 1.35109E-12, 1.17337E-12, 1.02225E-12, 8.93122E-13, 7.82295E-13,
                            6.86784E-13, 6.04162E-13, 5.32438E-13, 4.69977E-13, 4.15423E-13, 3.67651E-13,
                            3.25717E-13, 2.88831E-13, 2.56325E-13, 2.27631E-13, 2.02266E-13, 1.79815E-13,
                            1.59923E-13, 1.42281E-13, 1.26623E-13, 1.12716E-13, 1.00357E-13, 8.93688E-14,
                            7.95952E-14, 7.08989E-14, 6.31591E-14, 5.62689E-14, 5.01338E-14, 4.46702E-14,
                            3.98038E-14, 3.54690E-14, 3.16072E-14, 2.81666E-14, 2.51011E-14, 2.23696E-14])
    cel_data[0] = np.array([4.82362E-07, 1.61255E-07, 5.49142E-08, 1.94238E-08, 7.27309E-09, 2.98366E-09,
                            1.33641E-09, 6.56144E-10, 3.52497E-10, 2.05419E-10, 1.28555E-10, 8.54834E-11,
                            5.97969E-11, 4.36135E-11, 3.29157E-11, 2.55397E-11, 2.02651E-11, 1.63739E-11,
                            1.34257E-11, 1.11413E-11, 9.33772E-12, 7.89088E-12, 6.71496E-12, 5.74825E-12,
                            4.94615E-12, 4.27502E-12, 3.70954E-12, 3.23007E-12, 2.82118E-12, 2.47075E-12,
                            2.16901E-12, 1.90815E-12, 1.68176E-12, 1.48462E-12, 1.31244E-12, 1.16165E-12,
                            1.02927E-12, 9.12803E-13, 8.10147E-13, 7.19515E-13, 6.39384E-13, 5.68450E-13,
                            5.05591E-13, 4.49837E-13, 4.00347E-13, 3.56388E-13, 3.17320E-13, 2.82582E-13,
                            2.51683E-13, 2.24188E-13, 1.99717E-13, 1.77931E-13, 1.58532E-13, 1.41256E-13,
                            1.25868E-13, 1.12161E-13, 9.99494E-14, 8.90697E-14, 7.93760E-14, 7.07385E-14])
    cel_data[0] = np.array([4.52803E-06, 1.50015E-06, 4.98085E-07, 1.67638E-07, 5.81744E-08, 2.17374E-08,
                            8.75667E-09, 3.85210E-09, 1.87232E-09, 9.96968E-10, 5.75851E-10, 3.57165E-10,
                            2.35526E-10, 1.63589E-10, 1.18657E-10, 8.91877E-11, 6.90122E-11, 5.46765E-11,
                            4.41572E-11, 3.62224E-11, 3.00949E-11, 2.52675E-11, 2.14003E-11, 1.82572E-11,
                            1.56721E-11, 1.35236E-11, 1.17223E-11, 1.02004E-11, 8.90599E-12, 7.79850E-12,
                            6.84601E-12, 6.02320E-12, 5.30928E-12, 4.68769E-12, 4.14474E-12, 3.66915E-12,
                            3.25153E-12, 2.88403E-12, 2.56003E-12, 2.27390E-12, 2.02086E-12, 1.79682E-12,
                            1.59825E-12, 1.42209E-12, 1.26570E-12, 1.12677E-12, 1.00329E-12, 8.93484E-13,
                            7.95804E-13, 7.08883E-13, 6.31515E-13, 5.62634E-13, 5.01298E-13, 4.46673E-13,
                            3.98018E-13, 3.54675E-13, 3.16061E-13, 2.81659E-13, 2.51006E-13, 2.23693E-13])
    cel_data[0] = np.array([4.14989E-05, 1.38005E-05, 4.55521E-06, 1.49877E-06, 4.97484E-07, 1.73901E-07,
                            6.41350E-08, 2.54326E-08, 1.11412E-08, 5.36024E-09, 2.81461E-09, 1.60254E-09,
                            9.81634E-10, 6.41521E-10, 4.42701E-10, 3.19669E-10, 2.39587E-10, 1.85123E-10,
                            1.46634E-10, 1.18515E-10, 9.73725E-11, 8.10797E-11, 6.82579E-11, 5.79843E-11,
                            4.96268E-11, 4.27391E-11, 3.70000E-11, 3.21732E-11, 2.80804E-11, 2.45860E-11,
                            2.15845E-11, 1.89940E-11, 1.67468E-11, 1.47901E-11, 1.30806E-11, 1.15827E-11,
                            1.02669E-11, 9.10858E-12, 8.08688E-12, 7.18426E-12, 6.38576E-12, 5.67854E-12,
                            5.05153E-12, 4.49516E-12, 4.00112E-12, 3.56217E-12, 3.17195E-12, 2.82492E-12,
                            2.51617E-12, 2.24141E-12, 1.99683E-12, 1.77906E-12, 1.58514E-12, 1.41243E-12,
                            1.25859E-12, 1.12154E-12, 9.99447E-13, 8.90664E-13, 7.93736E-13, 7.07368E-13])
    cel_data[0] = np.array([3.64301E-04, 1.22696E-04, 4.07914E-05, 1.33862E-05, 4.36192E-06, 1.46169E-06,
                            5.01851E-07, 1.80746E-07, 7.11133E-08, 3.06557E-08, 1.44963E-08, 7.50654E-09,
                            4.23601E-09, 2.59210E-09, 1.69271E-09, 1.16779E-09, 8.43336E-10, 6.32609E-10,
                            4.89625E-10, 3.88787E-10, 3.15210E-10, 2.59921E-10, 2.17297E-10, 1.83702E-10,
                            1.56719E-10, 1.34694E-10, 1.16471E-10, 1.01221E-10, 8.83317E-11, 7.73496E-11,
                            6.79273E-11, 5.97993E-11, 5.27478E-11, 4.66059E-11, 4.12373E-11, 3.65314E-11,
                            3.23939E-11, 2.87491E-11, 2.55321E-11, 2.26883E-11, 2.01712E-11, 1.79406E-11,
                            1.59622E-11, 1.42061E-11, 1.26462E-11, 1.12599E-11, 1.00272E-11, 8.93073E-12,
                            7.95507E-12, 7.08668E-12, 6.31360E-12, 5.62523E-12, 5.01218E-12, 4.46616E-12,
                            3.97976E-12, 3.54645E-12, 3.16040E-12, 2.81644E-12, 2.50995E-12, 2.23685E-12])
    cel_data[0] = np.array([2.98250E-03, 1.03038E-03, 3.49226E-04, 1.15999E-04, 3.78000E-05, 1.24034E-05,
                            4.05493E-06, 1.35744E-06, 4.89118E-07, 1.91643E-07, 8.21267E-08, 3.85598E-08,
                            1.98465E-08, 1.12411E-08, 6.84665E-09, 4.44747E-09, 3.05509E-09, 2.20184E-09,
                            1.65222E-09, 1.28174E-09, 1.02152E-09, 8.32091E-10, 6.89757E-10, 5.79818E-10,
                            4.92871E-10, 4.22716E-10, 3.65146E-10, 3.17235E-10, 2.76889E-10, 2.42581E-10,
                            2.13172E-10, 1.87802E-10, 1.65783E-10, 1.46589E-10, 1.29794E-10, 1.15062E-10,
                            1.02091E-10, 9.06534E-11, 8.05466E-11, 7.16038E-11, 6.36813E-11, 5.66558E-11,
                            5.04203E-11, 4.48823E-11, 3.99608E-11, 3.55851E-11, 3.16930E-11, 2.82300E-11,
                            2.51479E-11, 2.24041E-11, 1.99611E-11, 1.77854E-11, 1.58477E-11, 1.41216E-11,
                            1.25840E-11, 1.12140E-11, 9.99348E-12, 8.90592E-12, 7.93685E-12, 7.07331E-12])
    cel_data[0] = np.array([2.05375E-02, 7.49175E-03, 2.65121E-03, 9.10679E-04, 3.03685E-04, 1.00458E-04,
                            3.24831E-05, 1.05621E-05, 3.63172E-06, 1.33066E-06, 5.21791E-07, 2.20456E-07,
                            1.01592E-07, 5.19312E-08, 2.88357E-08, 1.73323E-08, 1.11940E-08, 7.69709E-09,
                            5.57842E-09, 4.22030E-09, 3.30403E-09, 2.65827E-09, 2.18519E-09, 1.82685E-09,
                            1.54759E-09, 1.32470E-09, 1.14320E-09, 9.92953E-10, 8.66859E-10, 7.59846E-10,
                            6.68194E-10, 5.89126E-10, 5.20478E-10, 4.60589E-10, 4.08135E-10, 3.62077E-10,
                            3.21474E-10, 2.85630E-10, 2.53922E-10, 2.25837E-10, 2.00933E-10, 1.78829E-10,
                            1.59197E-10, 1.41748E-10, 1.26233E-10, 1.12432E-10, 1.00150E-10, 8.92187E-11,
                            7.94865E-11, 7.08204E-11, 6.31025E-11, 5.62281E-11, 5.01044E-11, 4.46490E-11,
                            3.97886E-11, 3.54580E-11, 3.15993E-11, 2.81610E-11, 2.50971E-11, 2.23667E-11])
    cel_data[0] = np.array([1.03912E-01, 4.06277E-02, 1.54470E-02, 5.61942E-03, 1.97117E-03, 6.78183E-04,
                            2.25209E-04, 7.40536E-05, 2.51129E-05, 8.72758E-06, 3.11615E-06, 1.17155E-06,
                            4.80291E-07, 2.20882E-07, 1.12607E-07, 6.34271E-08, 3.90649E-08, 2.59672E-08,
                            1.83779E-08, 1.36767E-08, 1.05867E-08, 8.45245E-09, 6.91286E-09, 5.76011E-09,
                            4.86952E-09, 4.16319E-09, 3.59066E-09, 3.11819E-09, 2.72250E-09, 2.38710E-09,
                            2.10002E-09, 1.85237E-09, 1.63733E-09, 1.44966E-09, 1.28519E-09, 1.14065E-09,
                            1.01317E-09, 9.00551E-10, 8.00870E-10, 7.12525E-10, 6.34141E-10, 5.64533E-10,
                            5.02676E-10, 4.47676E-10, 3.98749E-10, 3.55210E-10, 3.16454E-10, 2.81947E-10,
                            2.51218E-10, 2.23849E-10, 1.99470E-10, 1.77751E-10, 1.58401E-10, 1.41161E-10,
                            1.25799E-10, 1.12111E-10, 9.99134E-11, 8.90437E-11, 7.93572E-11, 7.07249E-11])
    cel_data[0] = np.array([3.39291E-01, 1.46233E-01, 6.15857E-02, 2.41195E-02, 9.03751E-03, 3.30540E-03,
                            1.15444E-03, 3.93335E-04, 1.34025E-04, 4.45070E-05, 1.45594E-05, 4.96915E-06,
                            1.87502E-06, 8.06655E-07, 3.92246E-07, 2.13819E-07, 1.28798E-07, 8.43266E-08,
                            5.90709E-08, 4.36523E-08, 3.36292E-08, 2.67644E-08, 2.18436E-08, 1.81767E-08,
                            1.53537E-08, 1.31206E-08, 1.13138E-08, 9.82472E-09, 8.57863E-09, 7.52293E-09,
                            6.61952E-09, 5.84018E-09, 5.16344E-09, 4.57271E-09, 4.05489E-09, 3.59952E-09,
                            3.19789E-09, 2.84295E-09, 2.52871E-09, 2.25013E-09, 2.00289E-09, 1.78328E-09,
                            1.58808E-09, 1.41447E-09, 1.26001E-09, 1.12253E-09, 1.00014E-09, 8.91144E-10,
                            7.94071E-10, 7.07601E-10, 6.30568E-10, 5.61936E-10, 5.00784E-10, 4.46295E-10,
                            3.97739E-10, 3.54470E-10, 3.15911E-10, 2.81549E-10, 2.50925E-10, 2.23633E-10])
    cel_data[0] = np.array([7.42406E-01, 3.58546E-01, 1.74613E-01, 7.37964E-02, 2.96478E-02, 1.15284E-02,
                                4.26912E-03, 1.51861E-03, 5.26892E-04, 1.71290E-04, 5.34789E-05, 1.74306E-05,
                                6.35417E-06, 2.66525E-06, 1.27494E-06, 6.87842E-07, 4.11445E-07, 2.68153E-07,
                                1.87269E-07, 1.38139E-07, 1.06280E-07, 8.45201E-08, 6.89424E-08, 5.73503E-08,
                                4.84334E-08, 4.13858E-08, 3.56872E-08, 3.09924E-08, 2.70646E-08, 2.37372E-08,
                                2.08899E-08, 1.84333E-08, 1.62998E-08, 1.44372E-08, 1.28042E-08, 1.13677E-08,
                                1.01006E-08, 8.98051E-09, 7.98870E-09, 7.10929E-09, 6.32870E-09, 5.63523E-09,
                                5.01875E-09, 4.47042E-09, 3.98249E-09, 3.54815E-09, 3.16143E-09, 2.81703E-09,
                                2.51027E-09, 2.23699E-09, 1.99352E-09, 1.77660E-09, 1.58330E-09, 1.41106E-09,
                                1.25757E-09, 1.12078E-09, 9.98878E-10, 8.90240E-10, 7.93420E-10, 7.07132E-10])
    cel_data[0] = np.array([1.34620E+00, 7.09801E-01, 3.55553E-01, 1.67698E-01, 7.74786E-02, 3.20814E-02,
                                1.25234E-02, 4.67239E-03, 1.67414E-03, 5.48642E-04, 1.70316E-04, 5.51586E-05,
                                2.00299E-05, 8.38709E-06, 4.01073E-06, 2.16390E-06, 1.29418E-06, 8.43779E-07,
                                5.89442E-07, 4.34833E-07, 3.34618E-07, 2.66188E-07, 2.17161E-07, 1.80678E-07,
                                1.52631E-07, 1.30448E-07, 1.12513E-07, 9.77303E-08, 8.53616E-08, 7.48810E-08,
                                6.59111E-08, 5.81708E-08, 5.14473E-08, 4.55761E-08, 4.04274E-08, 3.58970E-08,
                                3.19000E-08, 2.83661E-08, 2.52362E-08, 2.24605E-08, 1.99963E-08, 1.78067E-08,
                                1.58600E-08, 1.41281E-08, 1.25869E-08, 1.12148E-08, 9.99298E-09, 8.90477E-09,
                                7.93541E-09, 7.07180E-09, 6.30234E-09, 5.61671E-09, 5.00574E-09, 4.46128E-09,
                                3.97608E-09, 3.54366E-09, 3.15829E-09, 2.81484E-09, 2.50874E-09, 2.23593E-09])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60) - 1) / 10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, cel_data)


def calc_svn31_degas():
    """ Calculates cross section for "Neutral n=3 / n=1," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """

    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    n31_data = np.zeros((15, 60))
    n31_data[0] = np.array([4.72157E-89, 5.55328E-72, 1.96724E-58, 1.11253E-47, 3.77882E-39, 2.20902E-32,
                            5.12968E-27, 9.19486E-23, 2.15588E-19, 1.00486E-16, 1.29779E-14, 6.04445E-13,
                            1.25142E-11, 1.35986E-10, 8.85044E-10, 3.83187E-09, 1.19975E-08, 2.90294E-08,
                            5.72289E-08, 9.58646E-08, 1.41844E-07, 2.02884E-07, 2.68013E-07, 3.31782E-07,
                            3.90055E-07, 4.40135E-07, 4.80717E-07, 5.11618E-07, 5.33417E-07, 5.47127E-07,
                            5.53947E-07, 5.55096E-07, 5.51704E-07, 5.44771E-07, 5.35147E-07, 5.23533E-07,
                            5.10499E-07, 4.96497E-07, 4.81881E-07, 4.66927E-07, 4.51845E-07, 4.36795E-07,
                            4.21895E-07, 3.91802E-07, 3.67400E-07, 3.45752E-07, 3.24423E-07, 3.03611E-07,
                            2.83463E-07, 2.64088E-07, 2.45561E-07, 2.27930E-07, 2.11221E-07, 1.95445E-07,
                            1.80598E-07, 1.66666E-07, 1.53625E-07, 1.41447E-07, 1.30100E-07, 1.19547E-07])
    n31_data[1] = np.array([1.49491E-88, 1.75801E-71, 6.22697E-58, 3.52115E-47, 1.19587E-38, 6.99001E-32,
                            1.62293E-26, 2.90836E-22, 6.81640E-19, 3.17527E-16, 4.09772E-14, 1.90675E-12,
                            3.94365E-11, 4.28104E-10, 2.78354E-09, 1.20408E-08, 3.76698E-08, 9.10856E-08,
                            1.79469E-07, 3.00504E-07, 4.44516E-07, 6.35814E-07, 8.39802E-07, 1.03958E-06,
                            1.22216E-06, 1.37915E-06, 1.50643E-06, 1.60343E-06, 1.67196E-06, 1.71518E-06,
                            1.73682E-06, 1.74072E-06, 1.73037E-06, 1.70892E-06, 1.67903E-06, 1.64286E-06,
                            1.60224E-06, 1.55856E-06, 1.51294E-06, 1.46623E-06, 1.41911E-06, 1.37208E-06,
                            1.32549E-06, 1.23085E-06, 1.15460E-06, 1.08674E-06, 1.01987E-06, 9.54601E-07,
                            8.91404E-07, 8.30616E-07, 7.72476E-07, 7.17137E-07, 6.64684E-07, 6.15150E-07,
                            5.68522E-07, 5.24759E-07, 4.83788E-07, 4.45522E-07, 4.09858E-07, 3.76681E-07])
    n31_data[2] = np.array([4.74546E-88, 5.57832E-71, 1.97514E-57, 1.11650E-46, 3.79069E-38, 2.21496E-31,
                            5.14057E-26, 9.20692E-22, 2.15611E-18, 1.00328E-15, 1.29299E-13, 6.00717E-12,
                            1.24041E-10, 1.34442E-09, 8.72855E-09, 3.77070E-08, 1.17831E-07, 2.84644E-07,
                            5.60408E-07, 9.37788E-07, 1.38674E-06, 1.98348E-06, 2.61934E-06, 3.24243E-06,
                            3.81193E-06, 4.30215E-06, 4.69990E-06, 5.00360E-06, 5.21866E-06, 5.35491E-06,
                            5.42395E-06, 5.43780E-06, 5.40710E-06, 5.34168E-06, 5.24982E-06, 5.13819E-06,
                            5.01257E-06, 4.87729E-06, 4.73582E-06, 4.59087E-06, 4.44449E-06, 4.29826E-06,
                            4.15337E-06, 3.85641E-06, 3.61931E-06, 3.40735E-06, 3.19836E-06, 2.99431E-06,
                            2.79666E-06, 2.60647E-06, 2.42452E-06, 2.25127E-06, 2.08702E-06, 1.93186E-06,
                            1.78578E-06, 1.64863E-06, 1.52020E-06, 1.40023E-06, 1.28839E-06, 1.18433E-06])
    n31_data[3] = np.array([1.51868E-87, 1.78294E-70, 6.30569E-57, 3.56071E-46, 1.20771E-37, 7.04969E-31,
                            1.63422E-25, 2.92271E-21, 6.83174E-18, 3.17157E-15, 4.07639E-13, 1.88833E-11,
                            3.88771E-10, 4.20247E-09, 2.72174E-08, 1.17325E-07, 3.65966E-07, 8.82759E-07,
                            1.73597E-06, 2.90244E-06, 4.29039E-06, 6.13606E-06, 8.10132E-06, 1.00284E-05,
                            1.17905E-05, 1.33111E-05, 1.45460E-05, 1.54927E-05, 1.61658E-05, 1.65960E-05,
                            1.68185E-05, 1.68720E-05, 1.67864E-05, 1.65928E-05, 1.63168E-05, 1.59779E-05,
                            1.55955E-05, 1.51823E-05, 1.47493E-05, 1.43047E-05, 1.38550E-05, 1.34052E-05,
                            1.29589E-05, 1.20315E-05, 1.13005E-05, 1.06427E-05, 9.99358E-06, 9.35926E-06,
                            8.74440E-06, 8.15240E-06, 7.58566E-06, 7.04575E-06, 6.53360E-06, 6.04957E-06,
                            5.59363E-06, 5.16540E-06, 4.76425E-06, 4.38935E-06, 4.03974E-06, 3.71434E-06])
    n31_data[4] = np.array([4.97977E-87, 5.82448E-70, 2.05296E-56, 1.15566E-45, 3.90809E-37, 2.27427E-30,
                            5.25406E-25, 9.35875E-21, 2.17693E-17, 1.00492E-14, 1.28362E-12, 5.90826E-11,
                            1.20894E-09, 1.30009E-08, 8.38141E-08, 3.59880E-07, 1.11895E-06, 2.69229E-06,
                            5.28450E-06, 8.82372E-06, 1.30406E-05, 1.86480E-05, 2.46173E-05, 3.04773E-05,
                            3.58439E-05, 4.04932E-05, 4.42791E-05, 4.72101E-05, 4.93077E-05, 5.06728E-05,
                            5.14067E-05, 5.16456E-05, 5.14480E-05, 5.09182E-05, 5.01327E-05, 4.91401E-05,
                            4.80158E-05, 4.67918E-05, 4.55024E-05, 4.41730E-05, 4.28235E-05, 4.14693E-05,
                            4.01221E-05, 3.72544E-05, 3.50362E-05, 3.30201E-05, 3.10272E-05, 2.90767E-05,
                            2.71833E-05, 2.53580E-05, 2.36086E-05, 2.19401E-05, 2.03558E-05, 1.88571E-05,
                            1.74441E-05, 1.61159E-05, 1.48708E-05, 1.37063E-05, 1.26195E-05, 1.16074E-05])
    n31_data[5] = np.array([1.74270E-86, 2.01941E-69, 7.05648E-56, 3.94001E-45, 1.32175E-36, 7.62612E-30,
                            1.74454E-24, 3.07193E-20, 7.05017E-17, 3.20613E-14, 4.03154E-12, 1.82700E-10,
                            3.68511E-09, 3.91865E-08, 2.50191E-07, 1.06567E-06, 3.29233E-06, 7.88341E-06,
                            1.54206E-05, 2.56905E-05, 3.79870E-05, 5.43127E-05, 7.17138E-05, 8.88488E-05,
                            1.04611E-04, 1.18361E-04, 1.29644E-04, 1.38557E-04, 1.45030E-04, 1.49399E-04,
                            1.51927E-04, 1.53231E-04, 1.53116E-04, 1.52000E-04, 1.50103E-04, 1.47464E-04,
                            1.44458E-04, 1.41114E-04, 1.37542E-04, 1.33817E-04, 1.29999E-04, 1.26135E-04,
                            1.22264E-04, 1.13608E-04, 1.07099E-04, 1.01091E-04, 9.51279E-05, 8.92701E-05,
                            8.35652E-05, 7.80493E-05, 7.27484E-05, 6.76809E-05, 6.28583E-05, 5.82870E-05,
                            5.39691E-05, 4.99033E-05, 4.60856E-05, 4.25099E-05, 3.91684E-05, 3.60523E-05])
    n31_data[6] = np.array([6.94183E-86, 7.92309E-69, 2.72735E-55, 1.50003E-44, 4.95139E-36, 2.80241E-29,
                            6.25830E-24, 1.07040E-19, 2.37414E-16, 1.03995E-13, 1.25851E-11, 5.49617E-10,
                            1.07245E-08, 1.11294E-07, 6.96159E-07, 2.91634E-06, 8.89354E-06, 2.10895E-05,
                            4.09719E-05, 6.79612E-05, 1.00714E-04, 1.43921E-04, 1.90144E-04, 2.35971E-04,
                            2.78528E-04, 3.16155E-04, 3.47546E-04, 3.73168E-04, 3.92364E-04, 4.06126E-04,
                            4.15022E-04, 4.22631E-04, 4.25307E-04, 4.25188E-04, 4.22832E-04, 4.17455E-04,
                            4.11369E-04, 4.04113E-04, 3.96031E-04, 3.87309E-04, 3.78121E-04, 3.68610E-04,
                            3.58891E-04, 3.34389E-04, 3.16762E-04, 3.00104E-04, 2.83389E-04, 2.66814E-04,
                            2.50536E-04, 2.34679E-04, 2.19335E-04, 2.04576E-04, 1.90452E-04, 1.76995E-04,
                            1.64225E-04, 1.52149E-04, 1.40764E-04, 1.30063E-04, 1.20030E-04, 1.10645E-04])
    n31_data[7] = np.array([3.12631E-85, 3.54408E-68, 1.20943E-54, 6.57794E-44, 2.13620E-35, 1.17733E-28,
                            2.52352E-23, 4.08653E-19, 8.48037E-16, 3.45322E-13, 3.88360E-11, 1.58271E-09,
                            2.90678E-08, 2.89238E-07, 1.74742E-06, 7.11659E-06, 2.12220E-05, 4.94631E-05,
                            9.48704E-05, 1.55950E-04, 2.32021E-04, 3.30496E-04, 4.36129E-04, 5.41596E-04,
                            6.40617E-04, 7.29633E-04, 8.05478E-04, 8.69810E-04, 9.19933E-04, 9.58364E-04,
                            9.86021E-04, 1.02035E-03, 1.03876E-03, 1.05089E-03, 1.05791E-03, 1.05235E-03,
                            1.04783E-03, 1.03984E-03, 1.02934E-03, 1.01662E-03, 1.00207E-03, 9.86022E-04,
                            9.68737E-04, 9.08505E-04, 8.68560E-04, 8.29485E-04, 7.89314E-04, 7.48619E-04,
                            7.07879E-04, 6.67497E-04, 6.27801E-04, 5.89060E-04, 5.51486E-04, 5.15244E-04,
                            4.80457E-04, 4.47213E-04, 4.15569E-04, 3.85554E-04, 3.57177E-04, 3.30429E-04])
    n31_data[8] = np.array([1.34402E-84, 1.54458E-67, 5.33347E-54, 2.92599E-43, 9.50844E-35, 5.15217E-28,
                            1.05884E-22, 1.60752E-18, 3.07497E-15, 1.14604E-12, 1.18101E-10, 4.43411E-09,
                            7.57464E-08, 7.16522E-07, 4.14421E-06, 1.62594E-05, 4.69858E-05, 1.06709E-04,
                            2.00444E-04, 3.24165E-04, 4.82859E-04, 6.81679E-04, 8.94043E-04, 1.10619E-03,
                            1.30629E-03, 1.48796E-03, 1.64489E-03, 1.78162E-03, 1.89105E-03, 1.97873E-03,
                            2.04595E-03, 2.14881E-03, 2.21113E-03, 2.26291E-03, 2.30646E-03, 2.30639E-03,
                            2.31999E-03, 2.32649E-03, 2.32762E-03, 2.32377E-03, 2.31572E-03, 2.30401E-03,
                            2.28908E-03, 2.16352E-03, 2.09307E-03, 2.02130E-03, 1.94502E-03, 1.86543E-03,
                            1.78358E-03, 1.70038E-03, 1.61664E-03, 1.53304E-03, 1.45020E-03, 1.36861E-03,
                            1.28872E-03, 1.21090E-03, 1.13545E-03, 1.06262E-03, 9.92596E-04, 9.25524E-04])
    n31_data[9] = np.array([4.75401E-84, 5.58564E-67, 1.97114E-53, 1.10359E-42, 3.63844E-34, 1.96931E-27,
                            3.95044E-22, 5.73920E-18, 1.03594E-14, 3.61804E-12, 3.48271E-10, 1.21940E-08,
                            1.94557E-07, 1.73535E-06, 9.49220E-06, 3.53898E-05, 9.78145E-05, 2.13990E-04,
                            3.89932E-04, 6.15726E-04, 9.09986E-04, 1.26775E-03, 1.64753E-03, 2.02692E-03,
                            2.38678E-03, 2.71726E-03, 3.00761E-03, 3.26717E-03, 3.48135E-03, 3.66021E-03,
                            3.80503E-03, 4.04294E-03, 4.19928E-03, 4.34098E-03, 4.47235E-03, 4.50103E-03,
                            4.56976E-03, 4.62965E-03, 4.67739E-03, 4.71500E-03, 4.74422E-03, 4.76594E-03,
                            4.78093E-03, 4.54282E-03, 4.43896E-03, 4.32712E-03, 4.20370E-03, 4.07103E-03,
                            3.93117E-03, 3.78595E-03, 3.63696E-03, 3.48559E-03, 3.33305E-03, 3.18038E-03,
                            3.02847E-03, 2.87810E-03, 2.72993E-03, 2.58451E-03, 2.44233E-03, 2.30380E-03])
    n31_data[10] = np.array([1.32437E-83, 1.59705E-66, 5.78046E-53, 3.31567E-42, 1.11608E-33, 6.11314E-27,
                            1.22453E-21, 1.75406E-17, 3.08055E-14, 1.03017E-11, 9.32412E-10, 3.02481E-08,
                            4.44247E-07, 3.64081E-06, 1.84109E-05, 6.41237E-05, 1.67622E-04, 3.50921E-04,
                            6.18235E-04, 9.51988E-04, 1.38828E-03, 1.90995E-03, 2.46245E-03, 3.01685E-03,
                            3.54835E-03, 4.04494E-03, 4.49183E-03, 4.90373E-03, 5.25759E-03, 5.56770E-03,
                            5.83452E-03, 6.26920E-03, 6.58074E-03, 6.87909E-03, 7.17069E-03, 7.31948E-03,
                            7.52988E-03, 7.74704E-03, 7.94274E-03, 8.12483E-03, 8.29633E-03, 8.45745E-03,
                            8.60827E-03, 8.27208E-03, 8.19820E-03, 8.09907E-03, 7.97111E-03, 7.81777E-03,
                            7.64236E-03, 7.44803E-03, 7.23780E-03, 7.01450E-03, 6.78079E-03, 6.53914E-03,
                            6.29183E-03, 6.04091E-03, 5.78827E-03, 5.53555E-03, 5.28421E-03, 5.03551E-03])
    n31_data[11] = np.array([2.72542E-83, 3.39852E-66, 1.27128E-52, 7.52989E-42, 2.61228E-33, 1.46844E-26,
                            2.99847E-21, 4.33451E-17, 7.52583E-14, 2.40328E-11, 2.00182E-09, 5.85814E-08,
                            7.76711E-07, 5.77956E-06, 2.70374E-05, 8.87218E-05, 2.21940E-04, 4.50021E-04,
                            7.74829E-04, 1.17390E-03, 1.69570E-03, 2.31557E-03, 2.97218E-03, 3.63386E-03,
                            4.27344E-03, 4.87836E-03, 5.43187E-03, 5.95246E-03, 6.41180E-03, 6.82694E-03,
                            7.19766E-03, 7.79322E-03, 8.24363E-03, 8.68835E-03, 9.13569E-03, 9.44712E-03,
                            9.83012E-03, 1.02549E-02, 1.06601E-02, 1.10625E-02, 1.14679E-02, 1.18763E-02,
                            1.22873E-02, 1.19929E-02, 1.20947E-02, 1.21605E-02, 1.21846E-02, 1.21691E-02,
                            1.21157E-02, 1.20263E-02, 1.19028E-02, 1.17473E-02, 1.15618E-02, 1.13485E-02,
                            1.11097E-02, 1.08478E-02, 1.05653E-02, 1.02648E-02, 9.94887E-03, 9.62011E-03])
    n31_data[12] = np.array([4.02987E-83, 5.17936E-66, 1.99896E-52, 1.22259E-41, 4.38119E-33, 2.54178E-26,
                            5.34308E-21, 7.88499E-17, 1.36118E-13, 4.12808E-11, 3.14288E-09, 8.35611E-08,
                            1.02240E-06, 7.12696E-06, 3.18387E-05, 1.01221E-04, 2.47763E-04, 4.94985E-04,
                            8.43588E-04, 1.26928E-03, 1.82605E-03, 2.48616E-03, 3.18576E-03, 3.89229E-03,
                            4.57781E-03, 5.22974E-03, 5.83063E-03, 6.40069E-03, 6.90946E-03, 7.37522E-03,
                            7.79759E-03, 8.47193E-03, 8.99342E-03, 9.51487E-03, 1.00457E-02, 1.04561E-02,
                            1.09424E-02, 1.14969E-02, 1.20385E-02, 1.25912E-02, 1.31634E-02, 1.37570E-02,
                            1.43733E-02, 1.41708E-02, 1.44528E-02, 1.47054E-02, 1.49222E-02, 1.51043E-02,
                            1.52523E-02, 1.53670E-02, 1.54489E-02, 1.54982E-02, 1.55153E-02, 1.55004E-02,
                            1.54538E-02, 1.53756E-02, 1.52663E-02, 1.51261E-02, 1.49556E-02, 1.47556E-02])
    n31_data[13] = np.array([4.73952E-83, 6.19330E-66, 2.43351E-52, 1.51726E-41, 5.54927E-33, 3.28959E-26,
                            7.05994E-21, 1.05912E-16, 1.82170E-13, 5.32653E-11, 3.82531E-09, 9.63576E-08,
                            1.13322E-06, 7.67791E-06, 3.36702E-05, 1.05724E-04, 2.56819E-04, 5.10481E-04,
                            8.67091E-04, 1.30139E-03, 1.86980E-03, 2.54300E-03, 3.25702E-03, 3.97871E-03,
                            4.68045E-03, 5.34919E-03, 5.96726E-03, 6.55544E-03, 7.08255E-03, 7.56731E-03,
                            8.00925E-03, 8.71307E-03, 9.26167E-03, 9.81265E-03, 1.03759E-02, 1.08263E-02,
                            1.13545E-02, 1.19623E-02, 1.25612E-02, 1.31781E-02, 1.38232E-02, 1.44996E-02,
                            1.52098E-02, 1.50591E-02, 1.54321E-02, 1.57830E-02, 1.61056E-02, 1.64014E-02,
                            1.66716E-02, 1.69171E-02, 1.71387E-02, 1.73368E-02, 1.75119E-02, 1.76642E-02,
                            1.77937E-02, 1.79004E-02, 1.79841E-02, 1.80445E-02, 1.80813E-02, 1.80940E-02])
    n31_data[14] = np.array([5.01799E-83, 6.60031E-66, 2.61234E-52, 1.64198E-41, 6.05150E-33, 3.61758E-26,
                            7.84100E-21, 1.18520E-16, 2.03112E-13, 5.83687E-11, 4.09032E-09, 1.00716E-07,
                            1.16829E-06, 7.82800E-06, 3.41115E-05, 1.06754E-04, 2.58995E-04, 5.14143E-04,
                            8.72710E-04, 1.30967E-03, 1.88158E-03, 2.55846E-03, 3.27734E-03, 4.00441E-03,
                            4.71023E-03, 5.38435E-03, 6.00687E-03, 6.60076E-03, 7.13362E-03, 7.62526E-03,
                            8.07430E-03, 8.78828E-03, 9.34639E-03, 9.90769E-03, 1.04822E-02, 1.09467E-02,
                            1.14895E-02, 1.21160E-02, 1.27351E-02, 1.33746E-02, 1.40457E-02, 1.47517E-02,
                            1.54959E-02, 1.53652E-02, 1.57723E-02, 1.61604E-02, 1.65238E-02, 1.68643E-02,
                            1.71833E-02, 1.74822E-02, 1.77621E-02, 1.80240E-02, 1.82686E-02, 1.84966E-02,
                            1.87086E-02, 1.89049E-02, 1.90858E-02, 1.92516E-02, 1.94022E-02, 1.95377E-02])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, n31_data)


def calc_svc31_degas():
    """ Calculates cross section for "Continuum n=3 / n=1," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """

    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    c31_data = np.zeros((15, 60))
    c31_data[0] = np.array([1.22880E-09, 5.96713E-10, 3.15073E-10, 1.80040E-10, 1.10354E-10, 7.18020E-11,
                            4.90724E-11, 3.48876E-11, 2.55822E-11, 1.92070E-11, 1.46752E-11, 1.13548E-11,
                            8.86129E-12, 6.95280E-12, 5.47148E-12, 4.30986E-12, 3.39310E-12, 2.66732E-12,
                            2.09151E-12, 1.63516E-12, 1.27415E-12, 9.89251E-13, 7.65227E-13, 5.89712E-13,
                            4.52767E-13, 3.46360E-13, 2.64028E-13, 2.00586E-13, 1.51896E-13, 1.14673E-13,
                            8.63202E-14, 6.48003E-14, 4.85202E-14, 3.62426E-14, 2.70104E-14, 2.00872E-14,
                            1.49088E-14, 1.10448E-14, 8.16795E-15, 6.03057E-15, 4.44567E-15, 3.27258E-15,
                            2.40578E-15, 1.76632E-15, 1.29528E-15, 9.48789E-16, 6.94246E-16, 5.07482E-16,
                            3.70609E-16, 2.70409E-16, 1.97132E-16, 1.43596E-16, 1.04518E-16, 7.60196E-17,
                            5.52532E-17, 4.01330E-17, 2.91321E-17, 2.11338E-17, 1.53227E-17, 1.11033E-17])
    c31_data[1] = np.array([9.21089E-09, 3.99038E-09, 1.88499E-09, 9.71730E-10, 5.43476E-10, 3.26718E-10,
                            2.08830E-10, 1.40410E-10, 9.83396E-11, 7.11159E-11, 5.27066E-11, 3.97877E-11,
                            3.04374E-11, 2.35003E-11, 1.82542E-11, 1.42281E-11, 1.11063E-11, 8.67009E-12,
                            6.75982E-12, 5.26009E-12, 4.08305E-12, 3.15990E-12, 2.43789E-12, 1.87459E-12,
                            1.43666E-12, 1.09736E-12, 8.35456E-13, 6.34043E-13, 4.79716E-13, 3.61893E-13,
                            2.72249E-13, 2.04273E-13, 1.52887E-13, 1.14160E-13, 8.50548E-14, 6.32386E-14,
                            4.69268E-14, 3.47589E-14, 2.57021E-14, 1.89745E-14, 1.39868E-14, 1.02956E-14,
                            7.56843E-15, 5.55668E-15, 4.07486E-15, 2.98489E-15, 2.18418E-15, 1.59668E-15,
                            1.16611E-15, 8.50897E-16, 6.20369E-16, 4.51935E-16, 3.28983E-16, 2.39308E-16,
                            1.73958E-16, 1.26371E-16, 9.17438E-17, 6.65653E-17, 4.82693E-17, 3.49829E-17])
    c31_data[2] = np.array([7.94866E-08, 3.09013E-08, 1.30188E-08, 5.99741E-09, 3.01984E-09, 1.65305E-09,
                            9.73693E-10, 6.10651E-10, 4.03494E-10, 2.78034E-10, 1.98027E-10, 1.44695E-10,
                            1.07781E-10, 8.14250E-11, 6.21364E-11, 4.77370E-11, 3.68266E-11, 2.84727E-11,
                            2.20253E-11, 1.70281E-11, 1.31474E-11, 1.01297E-11, 7.78658E-12, 5.96895E-12,
                            4.56292E-12, 3.47791E-12, 2.64320E-12, 2.00304E-12, 1.51366E-12, 1.14074E-12,
                            8.57451E-13, 6.42917E-13, 4.80917E-13, 3.58931E-13, 2.67320E-13, 1.98691E-13,
                            1.47404E-13, 1.09162E-13, 8.07064E-14, 5.95748E-14, 4.39113E-14, 3.23211E-14,
                            2.37589E-14, 1.74435E-14, 1.27919E-14, 9.37046E-15, 6.85705E-15, 5.01289E-15,
                            3.66132E-15, 2.67182E-15, 1.94811E-15, 1.41932E-15, 1.03328E-15, 7.51708E-16,
                            5.46493E-16, 3.97044E-16, 2.88286E-16, 2.09196E-16, 1.51718E-16, 1.09973E-16])
    c31_data[3] = np.array([8.10718E-07, 2.86603E-07, 1.08050E-07, 4.42004E-08, 1.97587E-08, 9.69321E-09,
                            5.16911E-09, 2.97107E-09, 1.82295E-09, 1.17985E-09, 7.97366E-10, 5.57669E-10,
                            4.00553E-10, 2.93605E-10, 2.18533E-10, 1.64473E-10, 1.24752E-10, 9.51147E-11,
                            7.27370E-11, 5.57044E-11, 4.26737E-11, 3.26664E-11, 2.49754E-11, 1.90592E-11,
                            1.45153E-11, 1.10291E-11, 8.36035E-12, 6.32197E-12, 4.76890E-12, 3.58871E-12,
                            2.69424E-12, 2.01819E-12, 1.50846E-12, 1.12512E-12, 8.37527E-13, 6.22255E-13,
                            4.61494E-13, 3.41685E-13, 2.52575E-13, 1.86421E-13, 1.37398E-13, 1.01130E-13,
                            7.43402E-14, 5.45814E-14, 4.00287E-14, 2.93245E-14, 2.14608E-14, 1.56908E-14,
                            1.14616E-14, 8.36503E-15, 6.10005E-15, 4.44487E-15, 3.23639E-15, 2.35481E-15,
                            1.71221E-15, 1.24416E-15, 9.03507E-16, 6.55738E-16, 4.75647E-16, 3.44830E-16])
    c31_data[4] = np.array([1.00830E-05, 3.32356E-06, 1.13948E-06, 4.15227E-07, 1.63261E-07, 7.07967E-08,
                            3.35173E-08, 1.72630E-08, 9.63446E-09, 5.74327E-09, 3.61581E-09, 2.37973E-09,
                            1.62288E-09, 1.13822E-09, 8.16237E-10, 5.95419E-10, 4.39967E-10, 3.28199E-10,
                            2.46464E-10, 1.85920E-10, 1.40648E-10, 1.06547E-10, 8.07559E-11, 6.11819E-11,
                            4.63154E-11, 3.50144E-11, 2.64312E-11, 1.99181E-11, 1.49822E-11, 1.12482E-11,
                            8.42861E-12, 6.30462E-12, 4.70673E-12, 3.50742E-12, 2.60908E-12, 1.93737E-12,
                            1.43630E-12, 1.06314E-12, 7.85756E-13, 5.79915E-13, 4.27416E-13, 3.14612E-13,
                            2.31296E-13, 1.69845E-13, 1.24583E-13, 9.12869E-14, 6.68226E-14, 4.88682E-14,
                            3.57057E-14, 2.60661E-14, 1.90133E-14, 1.38581E-14, 1.00931E-14, 7.34582E-15,
                            5.34271E-15, 3.88331E-15, 2.82082E-15, 2.04782E-15, 1.48580E-15, 1.07745E-15])
    c31_data[5] = np.array([1.37049E-04, 4.41292E-05, 1.44100E-05, 4.85864E-06, 1.72186E-06, 6.65710E-07,
                            2.78612E-07, 1.26775E-07, 6.33420E-08, 3.41834E-08, 1.96774E-08, 1.19545E-08,
                            7.59326E-09, 5.00049E-09, 3.39412E-09, 2.36073E-09, 1.67427E-09, 1.20577E-09,
                            8.78720E-10, 6.46179E-10, 4.78388E-10, 3.55860E-10, 2.65609E-10, 1.98653E-10,
                            1.48762E-10, 1.11448E-10, 8.34923E-11, 6.25246E-11, 4.67866E-11, 3.49770E-11,
                            2.61186E-11, 1.94904E-11, 1.45221E-11, 1.08060E-11, 8.03020E-12, 5.95790E-12,
                            4.41501E-12, 3.26727E-12, 2.41477E-12, 1.78243E-12, 1.31405E-12, 9.67595E-13,
                            7.11664E-13, 5.22848E-13, 3.83720E-13, 2.81326E-13, 2.06053E-13, 1.50780E-13,
                            1.10235E-13, 8.05233E-14, 5.87717E-14, 4.28621E-14, 3.12358E-14, 2.27468E-14,
                            1.65535E-14, 1.20385E-14, 8.74945E-15, 6.35516E-15, 4.61339E-15, 3.34712E-15])
    c31_data[6] = np.array([1.57644E-03, 5.17160E-04, 1.69140E-04, 5.56930E-05, 1.87404E-05, 6.72829E-06,
                            2.57005E-06, 1.05659E-06, 4.78953E-07, 2.35601E-07, 1.24156E-07, 6.94133E-08,
                            4.08051E-08, 2.49992E-08, 1.58915E-08, 1.04188E-08, 7.00791E-09, 4.81392E-09,
                            3.36402E-09, 2.38370E-09, 1.70804E-09, 1.23475E-09, 8.98892E-10, 6.57896E-10,
                            4.83522E-10, 3.56440E-10, 2.63355E-10, 1.94905E-10, 1.44391E-10, 1.07039E-10,
                            7.93688E-11, 5.89520E-11, 4.37486E-11, 3.24574E-11, 2.40704E-11, 1.78278E-11,
                            1.32001E-11, 9.76545E-12, 7.21845E-12, 5.33090E-12, 3.93323E-12, 2.89923E-12,
                            2.13498E-12, 1.57067E-12, 1.15441E-12, 8.47651E-13, 6.21826E-13, 4.55744E-13,
                            3.33722E-13, 2.44156E-13, 1.78478E-13, 1.30359E-13, 9.51385E-14, 6.93804E-14,
                            5.05587E-14, 3.68168E-14, 2.67915E-14, 1.94833E-14, 1.41596E-14, 1.02844E-14])
    c31_data[7] = np.array([1.29414E-02, 4.41726E-03, 1.48932E-03, 4.97549E-04, 1.66369E-04, 5.80229E-05,
                            2.10666E-05, 8.09309E-06, 3.40132E-06, 1.54674E-06, 7.54293E-07, 3.91896E-07,
                            2.15254E-07, 1.23775E-07, 7.43317E-08, 4.63009E-08, 2.97309E-08, 1.95766E-08,
                            1.31596E-08, 8.99748E-09, 6.23814E-09, 4.37464E-09, 3.09675E-09, 2.20897E-09,
                            1.58565E-09, 1.14404E-09, 8.28902E-10, 6.02712E-10, 4.39455E-10, 3.21166E-10,
                            2.35147E-10, 1.72984E-10, 1.27265E-10, 9.37542E-11, 6.91438E-11, 5.09487E-11,
                            3.75963E-11, 2.77501E-11, 2.04860E-11, 1.51230E-11, 1.11621E-11, 8.23633E-12,
                            6.07504E-12, 4.47874E-12, 3.30000E-12, 2.42994E-12, 1.78804E-12, 1.31473E-12,
                            9.65964E-13, 7.09150E-13, 5.20187E-13, 3.81260E-13, 2.79204E-13, 2.04297E-13,
                            1.49363E-13, 1.09112E-13, 7.96447E-14, 5.80899E-14, 4.23364E-14, 3.08323E-14])
    c31_data[8] = np.array([7.09915E-02, 2.54411E-02, 8.96603E-03, 3.10441E-03, 1.06066E-03, 3.69645E-04,
                            1.30522E-04, 4.76788E-05, 1.87975E-05, 7.99304E-06, 3.66032E-06, 1.79938E-06,
                            9.42370E-07, 5.19773E-07, 3.01367E-07, 1.82112E-07, 1.13839E-07, 7.31531E-08,
                            4.80757E-08, 3.21774E-08, 2.18593E-08, 1.50311E-08, 1.04392E-08, 7.30962E-09,
                            5.15300E-09, 3.65305E-09, 2.60187E-09, 1.86073E-09, 1.33511E-09, 9.60752E-10,
                            6.93067E-10, 5.02826E-10, 3.65121E-10, 2.65741E-10, 1.93836E-10, 1.41274E-10,
                            1.03284E-10, 7.56138E-11, 5.54283E-11, 4.06770E-11, 2.98812E-11, 2.19696E-11,
                            1.61644E-11, 1.19005E-11, 8.76522E-12, 6.45811E-12, 4.75921E-12, 3.50750E-12,
                            2.58488E-12, 1.90464E-12, 1.40304E-12, 1.03315E-12, 7.60431E-13, 5.59396E-13,
                            4.11256E-13, 3.02142E-13, 2.21816E-13, 1.62720E-13, 1.19272E-13, 8.73536E-14])
    c31_data[9] = np.array([2.72666E-01, 1.02331E-01, 3.76277E-02, 1.35313E-02, 4.75784E-03, 1.67636E-03,
                            5.84444E-04, 2.06968E-04, 7.83906E-05, 3.20428E-05, 1.41836E-05, 6.78100E-06,
                            3.47119E-06, 1.87934E-06, 1.07297E-06, 6.39742E-07, 3.95085E-07, 2.51038E-07,
                            1.63226E-07, 1.08126E-07, 7.27122E-08, 4.94979E-08, 3.40317E-08, 2.35884E-08,
                            1.64588E-08, 1.15470E-08, 8.13787E-09, 5.75752E-09, 4.08637E-09, 2.90832E-09,
                            2.07477E-09, 1.48707E-09, 1.06709E-09, 7.67433E-10, 5.53137E-10, 3.98485E-10,
                            2.87949E-10, 2.08433E-10, 1.51102E-10, 1.09701E-10, 7.97586E-11, 5.80690E-11,
                            4.23340E-11, 3.09038E-11, 2.25867E-11, 1.65272E-11, 1.21066E-11, 8.87751E-12,
                            6.51590E-12, 4.78669E-12, 3.51913E-12, 2.58900E-12, 1.90582E-12, 1.40358E-12,
                            1.03407E-12, 7.62031E-13, 5.61632E-13, 4.13941E-13, 3.05057E-13, 2.24766E-13])
    c31_data[10] = np.array([8.22545E-01, 3.24432E-01, 1.24309E-01, 4.62657E-02, 1.67130E-02, 5.98163E-03,
                            2.08950E-03, 7.34765E-04, 2.75557E-04, 1.11671E-04, 4.90685E-05, 2.32796E-05,
                            1.18192E-05, 6.35489E-06, 3.60223E-06, 2.13253E-06, 1.30806E-06, 8.25860E-07,
                            5.33773E-07, 3.51582E-07, 2.35129E-07, 1.59193E-07, 1.08856E-07, 7.50350E-08,
                            5.20587E-08, 3.63090E-08, 2.54335E-08, 1.78795E-08, 1.26060E-08, 8.90996E-09,
                            6.31079E-09, 4.48413E-09, 3.19022E-09, 2.27367E-09, 1.62323E-09, 1.15927E-09,
                            8.29639E-10, 5.94691E-10, 4.26822E-10, 3.06746E-10, 2.20748E-10, 1.59073E-10,
                            1.14782E-10, 8.29417E-11, 6.00091E-11, 4.34739E-11, 3.15356E-11, 2.29048E-11,
                            1.66569E-11, 1.21281E-11, 8.84116E-12, 6.45254E-12, 4.71457E-12, 3.44847E-12,
                            2.52502E-12, 1.85070E-12, 1.35775E-12, 9.96995E-13, 7.32699E-13, 5.38876E-13])
    c31_data[11] = np.array([2.16525E+00, 8.92316E-01, 3.58603E-01, 1.38226E-01, 5.14668E-02, 1.88247E-02,
                            6.67227E-03, 2.37532E-03, 9.03093E-04, 3.70629E-04, 1.63790E-04, 7.75110E-05,
                            3.90869E-05, 2.08845E-05, 1.17630E-05, 6.92485E-06, 4.22780E-06, 2.65891E-06,
                            1.71283E-06, 1.12489E-06, 7.50256E-07, 5.06633E-07, 3.45542E-07, 2.37558E-07,
                            1.64366E-07, 1.14310E-07, 7.98272E-08, 5.59337E-08, 3.92981E-08, 2.76717E-08,
                            1.95206E-08, 1.37991E-08, 9.76607E-09, 6.92049E-09, 4.90977E-09, 3.48554E-09,
                            2.47736E-09, 1.76274E-09, 1.25536E-09, 8.94849E-10, 6.38486E-10, 4.56018E-10,
                            3.26029E-10, 2.33368E-10, 1.67214E-10, 1.19949E-10, 8.61452E-11, 6.19427E-11,
                            4.45949E-11, 3.21460E-11, 2.32018E-11, 1.67678E-11, 1.21336E-11, 8.79148E-12,
                            6.37804E-12, 4.63296E-12, 3.36950E-12, 2.45356E-12, 1.78869E-12, 1.30548E-12])
    c31_data[12] = np.array([4.88870E+00, 2.16049E+00, 9.34938E-01, 3.76958E-01, 1.45636E-01, 5.50373E-02,
                            2.00214E-02, 7.31364E-03, 2.86023E-03, 1.19931E-03, 5.33374E-04, 2.51350E-04,
                            1.25949E-04, 6.69915E-05, 3.76041E-05, 2.20846E-05, 1.34605E-05, 8.45474E-06,
                            5.44080E-06, 3.56995E-06, 2.37893E-06, 1.60506E-06, 1.09374E-06, 7.51232E-07,
                            5.19255E-07, 3.60728E-07, 2.51611E-07, 1.76069E-07, 1.23526E-07, 8.68442E-08,
                            6.11577E-08, 4.31344E-08, 3.04557E-08, 2.15247E-08, 1.52253E-08, 1.07760E-08,
                            7.63222E-09, 5.40905E-09, 3.83545E-09, 2.72106E-09, 1.93147E-09, 1.37173E-09,
                            9.74729E-10, 6.93104E-10, 4.93114E-10, 3.51058E-10, 2.50099E-10, 1.78307E-10,
                            1.27224E-10, 9.08530E-11, 6.49387E-11, 4.64609E-11, 3.32749E-11, 2.38569E-11,
                            1.71240E-11, 1.23058E-11, 8.85419E-12, 6.37881E-12, 4.60146E-12, 3.32374E-12])
    c31_data[13] = np.array([9.15993E+00, 4.46532E+00, 2.19690E+00, 9.38575E-01, 3.81294E-01, 1.50115E-01,
                            5.68247E-02, 2.14795E-02, 8.68494E-03, 3.72549E-03, 1.67124E-03, 7.88877E-04,
                            3.95622E-04, 2.10715E-04, 1.18442E-04, 6.96444E-05, 4.24890E-05, 2.67070E-05,
                            1.71948E-05, 1.12856E-05, 7.52150E-06, 5.07482E-06, 3.45786E-06, 2.37464E-06,
                            1.64099E-06, 1.13966E-06, 7.94643E-07, 5.55833E-07, 3.89776E-07, 2.73882E-07,
                            1.92758E-07, 1.35843E-07, 9.58322E-08, 6.76640E-08, 4.78087E-08, 3.37983E-08,
                            2.39054E-08, 1.69153E-08, 1.19734E-08, 8.47788E-09, 6.00459E-09, 4.25401E-09,
                            3.01458E-09, 2.13706E-09, 1.51530E-09, 1.07474E-09, 7.62502E-10, 5.41154E-10,
                            3.84197E-10, 2.72868E-10, 1.93880E-10, 1.37819E-10, 9.80179E-11, 6.97489E-11,
                            4.96625E-11, 3.53836E-11, 2.52280E-11, 1.80010E-11, 1.28550E-11, 9.18830E-12])
    c31_data[14] = np.array([1.57231E+01, 8.31557E+00, 4.17893E+00, 1.97722E+00, 9.15943E-01, 3.80513E-01,
                            1.50522E-01, 5.94129E-02, 2.49763E-02, 1.10109E-02, 5.02519E-03, 2.40069E-03,
                            1.21522E-03, 6.52153E-04, 3.68709E-04, 2.17755E-04, 1.33281E-04, 8.39733E-05,
                            5.41562E-05, 3.55873E-05, 2.37376E-05, 1.60251E-05, 1.09232E-05, 7.50309E-06,
                            5.18563E-06, 3.60158E-06, 2.51121E-06, 1.75642E-06, 1.23156E-06, 8.65250E-07,
                            6.08858E-07, 4.28974E-07, 3.02545E-07, 2.13550E-07, 1.50831E-07, 1.06588E-07,
                            7.53549E-08, 5.32920E-08, 3.76995E-08, 2.66754E-08, 1.88788E-08, 1.33632E-08,
                            9.46047E-09, 6.69916E-09, 4.74417E-09, 3.36013E-09, 2.38018E-09, 1.68626E-09,
                            1.19481E-09, 8.46714E-10, 6.00130E-10, 4.25432E-10, 3.01646E-10, 2.13923E-10,
                            1.51746E-10, 1.07668E-10, 7.64156E-11, 5.42517E-11, 3.85297E-11, 2.73744E-11])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, c31_data)


def calc_svn21_degas():
    """ Calculates cross section for "Neutral n=2 / n=1," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """

    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    n21_data = np.zeros((15, 60))
    n21_data[0] = np.array([8.46420E-76, 2.10374E-61, 5.58965E-50, 6.48365E-41, 1.00386E-33, 5.04074E-28,
                            1.66168E-23, 6.29701E-20, 4.28132E-17, 7.43919E-15, 4.37615E-13, 1.08966E-11,
                            1.37128E-10, 1.00381E-09, 4.77792E-09, 1.61519E-08, 4.15912E-08, 8.62387E-08,
                            1.50492E-07, 2.28926E-07, 3.31018E-07, 4.46452E-07, 5.62111E-07, 6.69822E-07,
                            7.63971E-07, 8.41529E-07, 9.01643E-07, 9.45015E-07, 9.73291E-07, 9.88568E-07,
                            9.93047E-07, 9.88823E-07, 9.77768E-07, 9.61488E-07, 9.41323E-07, 9.18359E-07,
                            8.93466E-07, 8.67328E-07, 8.40477E-07, 8.13320E-07, 7.86168E-07, 7.59252E-07,
                            7.32744E-07, 6.66474E-07, 6.26529E-07, 5.88188E-07, 5.50699E-07, 5.14348E-07,
                            4.79346E-07, 4.45840E-07, 4.13927E-07, 3.83663E-07, 3.55072E-07, 3.28151E-07,
                            3.02878E-07, 2.79215E-07, 2.57113E-07, 2.36512E-07, 2.17348E-07, 1.99553E-07])
    n21_data[0] = np.array([2.67659E-75, 6.65255E-61, 1.76759E-49, 2.05030E-40, 3.17446E-33, 1.59401E-27,
                            5.25467E-23, 1.99128E-19, 1.35387E-16, 2.35245E-14, 1.38382E-12, 3.44558E-11,
                            4.33571E-10, 3.17341E-09, 1.51020E-08, 5.10412E-08, 1.31398E-07, 2.72382E-07,
                            4.75204E-07, 7.22710E-07, 1.04491E-06, 1.40915E-06, 1.77399E-06, 2.11373E-06,
                            2.41067E-06, 2.65528E-06, 2.84488E-06, 2.98169E-06, 3.07091E-06, 3.11914E-06,
                            3.13333E-06, 3.12008E-06, 3.08528E-06, 3.03401E-06, 2.97048E-06, 2.89813E-06,
                            2.81968E-06, 2.73730E-06, 2.65266E-06, 2.56705E-06, 2.48146E-06, 2.39660E-06,
                            2.31302E-06, 2.10370E-06, 1.97782E-06, 1.85686E-06, 1.73858E-06, 1.62388E-06,
                            1.51344E-06, 1.40772E-06, 1.30701E-06, 1.21151E-06, 1.12128E-06, 1.03631E-06,
                            9.56547E-07, 8.81859E-07, 8.12092E-07, 7.47062E-07, 6.86566E-07, 6.30389E-07])
    n21_data[0] = np.array([8.46393E-75, 2.10368E-60, 5.58951E-49, 6.48350E-40, 1.00384E-32, 5.04065E-27,
                            1.66165E-22, 6.29691E-19, 4.28125E-16, 7.43895E-14, 4.37581E-12, 1.08946E-10,
                            1.37074E-09, 1.00307E-08, 4.77225E-08, 1.61238E-07, 4.14936E-07, 8.59826E-07,
                            1.49954E-06, 2.27984E-06, 3.29581E-06, 4.44396E-06, 5.59354E-06, 6.66399E-06,
                            7.59941E-06, 8.37010E-06, 8.96748E-06, 9.39867E-06, 9.67999E-06, 9.83226E-06,
                            9.87733E-06, 9.83597E-06, 9.72676E-06, 9.56564E-06, 9.36588E-06, 9.13830E-06,
                            8.89148E-06, 8.63224E-06, 8.36585E-06, 8.09637E-06, 7.82688E-06, 7.55968E-06,
                            7.29648E-06, 6.63562E-06, 6.23943E-06, 5.85812E-06, 5.48524E-06, 5.12364E-06,
                            4.77542E-06, 4.44204E-06, 4.12447E-06, 3.82328E-06, 3.53870E-06, 3.27073E-06,
                            3.01913E-06, 2.78353E-06, 2.56345E-06, 2.35829E-06, 2.16744E-06, 1.99019E-06])
    n21_data[0] = np.array([2.67632E-74, 6.65194E-60, 1.76745E-48, 2.05015E-39, 3.17426E-32, 1.59392E-26,
                            5.25440E-22, 1.99118E-18, 1.35380E-15, 2.35227E-13, 1.38360E-11, 3.44440E-10,
                            4.33279E-09, 3.16963E-08, 1.50736E-07, 5.09032E-07, 1.30925E-06, 2.71155E-06,
                            4.72652E-06, 7.18263E-06, 1.03815E-05, 1.39946E-05, 1.76103E-05, 2.09767E-05,
                            2.39179E-05, 2.63419E-05, 2.82208E-05, 2.95779E-05, 3.04640E-05, 3.09447E-05,
                            3.10885E-05, 3.09608E-05, 3.06197E-05, 3.01153E-05, 2.94893E-05, 2.87758E-05,
                            2.80015E-05, 2.71880E-05, 2.63518E-05, 2.55056E-05, 2.46592E-05, 2.38198E-05,
                            2.29927E-05, 2.09081E-05, 1.96639E-05, 1.84637E-05, 1.72899E-05, 1.61514E-05,
                            1.50549E-05, 1.40049E-05, 1.30047E-05, 1.20559E-05, 1.11593E-05, 1.03150E-05,
                            9.52216E-06, 8.77968E-06, 8.08602E-06, 7.43937E-06, 6.83772E-06, 6.27894E-06])
    n21_data[0] = np.array([8.46116E-74, 2.10306E-59, 5.58806E-48, 6.48200E-39, 1.00363E-31, 5.03972E-26,
                            1.66138E-21, 6.29594E-18, 4.28057E-15, 7.43738E-13, 4.37421E-11, 1.08870E-09,
                            1.36900E-08, 1.00094E-07, 4.75675E-07, 1.60504E-06, 4.12465E-06, 8.53510E-06,
                            1.48657E-05, 2.25745E-05, 3.26189E-05, 4.39541E-05, 5.52901E-05, 6.58412E-05,
                            7.50585E-05, 8.26579E-05, 8.85489E-05, 9.28112E-05, 9.55972E-05, 9.71154E-05,
                            9.75788E-05, 9.71926E-05, 9.61373E-05, 9.45699E-05, 9.26206E-05, 9.03975E-05,
                            8.79817E-05, 8.54423E-05, 8.28302E-05, 8.01855E-05, 7.75388E-05, 7.49128E-05,
                            7.23245E-05, 6.57598E-05, 6.18674E-05, 5.81001E-05, 5.44144E-05, 5.08387E-05,
                            4.73937E-05, 4.40944E-05, 4.09504E-05, 3.79675E-05, 3.51483E-05, 3.24927E-05,
                            2.99986E-05, 2.76625E-05, 2.54796E-05, 2.34443E-05, 2.15504E-05, 1.97912E-05])
    n21_data[0] = np.array([2.67355E-73, 6.64581E-59, 1.76600E-47, 2.04865E-38, 3.17218E-31, 1.59299E-25,
                            5.25165E-21, 1.99022E-17, 1.35313E-14, 2.35084E-12, 1.38231E-10, 3.43891E-09,
                            4.32118E-08, 3.15613E-07, 1.49794E-06, 5.04696E-06, 1.29500E-05, 2.67576E-05,
                            4.65409E-05, 7.05917E-05, 1.01948E-04, 1.37283E-04, 1.72593E-04, 2.05447E-04,
                            2.34149E-04, 2.57824E-04, 2.76190E-04, 2.89519E-04, 2.98253E-04, 3.03053E-04,
                            3.04574E-04, 3.03472E-04, 3.00276E-04, 2.95482E-04, 2.89494E-04, 2.82662E-04,
                            2.75212E-04, 2.67375E-04, 2.59299E-04, 2.51113E-04, 2.42913E-04, 2.34769E-04,
                            2.26735E-04, 2.06140E-04, 1.94048E-04, 1.82285E-04, 1.70769E-04, 1.59591E-04,
                            1.48816E-04, 1.38491E-04, 1.28648E-04, 1.19305E-04, 1.10470E-04, 1.02146E-04,
                            9.43247E-05, 8.69964E-05, 8.01466E-05, 7.37579E-05, 6.78111E-05, 6.22858E-05])
    n21_data[0] = np.array([8.43356E-73, 2.09695E-58, 5.57357E-47, 6.46701E-38, 1.00156E-30, 5.03045E-25,
                            1.65864E-20, 6.28636E-17, 4.27401E-14, 7.42378E-12, 4.36270E-10, 1.08419E-08,
                            1.36010E-07, 9.91123E-07, 4.69101E-06, 1.57574E-05, 4.03078E-05, 8.30431E-05,
                            1.44066E-04, 2.18033E-04, 3.14528E-04, 4.23008E-04, 5.31321E-04, 6.32096E-04,
                            7.20204E-04, 7.92984E-04, 8.49602E-04, 8.90904E-04, 9.18177E-04, 9.33446E-04,
                            9.38683E-04, 9.36095E-04, 9.26950E-04, 9.12869E-04, 8.95081E-04, 8.74817E-04,
                            8.52489E-04, 8.28970E-04, 8.04624E-04, 7.79869E-04, 7.55008E-04, 7.30261E-04,
                            7.05796E-04, 6.41816E-04, 6.04764E-04, 5.68466E-04, 5.32879E-04, 4.98288E-04,
                            4.64903E-04, 4.32877E-04, 4.02313E-04, 3.73274E-04, 3.45792E-04, 3.19873E-04,
                            2.95503E-04, 2.72651E-04, 2.51277E-04, 2.31328E-04, 2.12748E-04, 1.95475E-04])
    n21_data[0] = np.array([2.64619E-72, 6.58512E-58, 1.75161E-46, 2.03376E-37, 3.15161E-30, 1.58377E-24,
                            5.22437E-20, 1.98069E-16, 1.34663E-13, 2.33749E-11, 1.37135E-09, 3.39802E-08,
                            4.24498E-07, 3.07584E-06, 1.44622E-05, 4.82384E-05, 1.22541E-04, 2.50845E-04,
                            4.32737E-04, 6.51881E-04, 9.37689E-04, 1.25783E-03, 1.57720E-03, 1.87461E-03,
                            2.13537E-03, 2.35180E-03, 2.52152E-03, 2.64686E-03, 2.73141E-03, 2.78090E-03,
                            2.80092E-03, 2.79895E-03, 2.77687E-03, 2.73996E-03, 2.69178E-03, 2.63786E-03,
                            2.57612E-03, 2.51103E-03, 2.44267E-03, 2.37255E-03, 2.30163E-03, 2.23058E-03,
                            2.15991E-03, 1.96662E-03, 1.85672E-03, 1.74807E-03, 1.64111E-03, 1.53677E-03,
                            1.43574E-03, 1.33853E-03, 1.24550E-03, 1.15690E-03, 1.07285E-03, 9.93418E-04,
                            9.18586E-04, 8.48293E-04, 7.82433E-04, 7.20874E-04, 6.63457E-04, 6.10007E-04])
    n21_data[0] = np.array([8.16718E-72, 2.03769E-57, 5.43271E-46, 6.32091E-37, 9.81339E-30, 4.93965E-24,
                            1.63175E-19, 6.19248E-16, 4.20988E-13, 7.29225E-11, 4.25665E-09, 1.04603E-07,
                            1.29217E-06, 9.22441E-06, 4.26516E-05, 1.39820E-04, 3.49348E-04, 7.04559E-04,
                            1.20018E-03, 1.78971E-03, 2.55602E-03, 3.41031E-03, 4.26222E-03, 5.05852E-03,
                            5.76246E-03, 6.35457E-03, 6.82838E-03, 7.18909E-03, 7.44479E-03, 7.60918E-03,
                            7.69574E-03, 7.72591E-03, 7.69981E-03, 7.63226E-03, 7.53234E-03, 7.43346E-03,
                            7.29892E-03, 7.15777E-03, 7.00195E-03, 6.83778E-03, 6.66826E-03, 6.49519E-03,
                            6.32007E-03, 5.77885E-03, 5.48100E-03, 5.18193E-03, 4.88426E-03, 4.59097E-03,
                            4.30440E-03, 4.02641E-03, 3.75837E-03, 3.50132E-03, 3.25594E-03, 3.02267E-03,
                            2.80173E-03, 2.59316E-03, 2.39685E-03, 2.21259E-03, 2.04006E-03, 1.87888E-03])
    n21_data[0] = np.array([2.40052E-71, 6.03407E-57, 1.61964E-45, 1.89594E-36, 2.95969E-29, 1.49715E-23,
                            4.96686E-19, 1.89053E-15, 1.28495E-12, 2.21164E-10, 1.27205E-08, 3.05415E-07,
                            3.66153E-06, 2.51624E-05, 1.11735E-04, 3.52102E-04, 8.48885E-04, 1.66061E-03,
                            2.75963E-03, 4.03739E-03, 5.68707E-03, 7.51649E-03, 9.34320E-03, 1.10658E-02,
                            1.26147E-02, 1.39521E-02, 1.50637E-02, 1.59574E-02, 1.66447E-02, 1.71490E-02,
                            1.74933E-02, 1.77251E-02, 1.78307E-02, 1.78421E-02, 1.77764E-02, 1.78161E-02,
                            1.77022E-02, 1.75962E-02, 1.74320E-02, 1.72355E-02, 1.70142E-02, 1.67712E-02,
                            1.65092E-02, 1.52627E-02, 1.46345E-02, 1.39802E-02, 1.33084E-02, 1.26280E-02,
                            1.19465E-02, 1.12704E-02, 1.06051E-02, 9.95493E-03, 9.32356E-03, 8.71381E-03,
                            8.12783E-03, 7.56719E-03, 7.03297E-03, 6.52577E-03, 6.04584E-03, 5.59311E-03])
    n21_data[0] = np.array([6.20673E-71, 1.58873E-56, 4.33674E-45, 5.15603E-36, 8.16483E-29, 4.18456E-23,
                            1.40442E-18, 5.38935E-15, 3.66046E-12, 6.19199E-10, 3.42670E-08, 7.77551E-07,
                            8.72168E-06, 5.55377E-05, 2.29419E-04, 6.78550E-04, 1.55288E-03, 2.91562E-03,
                            4.69636E-03, 6.71547E-03, 9.30805E-03, 1.21730E-02, 1.50424E-02, 1.77786E-02,
                            2.02872E-02, 2.25174E-02, 2.44475E-02, 2.60866E-02, 2.74452E-02, 2.85519E-02,
                            2.94336E-02, 3.01697E-02, 3.07130E-02, 3.11133E-02, 3.13923E-02, 3.21395E-02,
                            3.24766E-02, 3.29280E-02, 3.32517E-02, 3.35203E-02, 3.37484E-02, 3.39349E-02,
                            3.40784E-02, 3.21433E-02, 3.14404E-02, 3.06311E-02, 2.97291E-02, 2.87494E-02,
                            2.77060E-02, 2.66124E-02, 2.54811E-02, 2.43238E-02, 2.31514E-02, 2.19739E-02,
                            2.08003E-02, 1.96388E-02, 1.84964E-02, 1.73795E-02, 1.62933E-02, 1.52422E-02])
    n21_data[0] = np.array([1.24484E-70, 3.28509E-56, 9.23700E-45, 1.13011E-35, 1.83951E-28, 9.67839E-23,
                            3.32751E-18, 1.29942E-14, 8.81194E-12, 1.43695E-09, 7.37870E-08, 1.52103E-06,
                            1.54908E-05, 8.98218E-05, 3.43968E-04, 9.59983E-04, 2.10491E-03, 3.83149E-03,
                            6.03661E-03, 8.49949E-03, 1.16569E-02, 1.51426E-02, 1.86432E-02, 2.20065E-02,
                            2.51289E-02, 2.79551E-02, 3.04609E-02, 3.26576E-02, 3.45536E-02, 3.61803E-02,
                            3.75657E-02, 3.88138E-02, 3.98461E-02, 4.07251E-02, 4.14736E-02, 4.31528E-02,
                            4.41862E-02, 4.55264E-02, 4.67211E-02, 4.79024E-02, 4.90973E-02, 5.03028E-02,
                            5.15146E-02, 4.96000E-02, 4.95555E-02, 4.93478E-02, 4.89833E-02, 4.84708E-02,
                            4.78189E-02, 4.70359E-02, 4.61305E-02, 4.51115E-02, 4.39880E-02, 4.27698E-02,
                            4.14668E-02, 4.00894E-02, 3.86482E-02, 3.71544E-02, 3.56188E-02, 3.40527E-02])
    n21_data[0] = np.array([1.82530E-70, 4.95974E-56, 1.43726E-44, 1.81365E-35, 3.04669E-28, 1.65490E-22,
                            5.86876E-18, 2.34644E-14, 1.58771E-11, 2.46695E-09, 1.16111E-07, 2.17949E-06,
                            2.05204E-05, 1.11557E-04, 4.08231E-04, 1.10428E-03, 2.37009E-03, 4.25166E-03,
                            6.63188E-03, 9.27475E-03, 1.26629E-02, 1.64030E-02, 2.01647E-02, 2.37910E-02,
                            2.71757E-02, 3.02626E-02, 3.30273E-02, 3.54820E-02, 3.76351E-02, 3.95192E-02,
                            4.11628E-02, 4.26828E-02, 4.39848E-02, 4.51382E-02, 4.61664E-02, 4.84038E-02,
                            4.98805E-02, 5.18028E-02, 5.35973E-02, 5.54366E-02, 5.73622E-02, 5.93773E-02,
                            6.14838E-02, 5.99141E-02, 6.06310E-02, 6.12074E-02, 6.16467E-02, 6.19535E-02,
                            6.21316E-02, 6.21836E-02, 6.21117E-02, 6.19175E-02, 6.16023E-02, 6.11671E-02,
                            6.06130E-02, 5.99412E-02, 5.91532E-02, 5.82508E-02, 5.72365E-02, 5.61134E-02])
    n21_data[0] = np.array([2.14100E-70, 5.91293E-56, 1.74386E-44, 2.24259E-35, 3.84451E-28, 2.13395E-22,
                            7.73729E-18, 3.14856E-14, 2.12653E-11, 3.18882E-09, 1.41748E-07, 2.52298E-06,
                            2.28446E-05, 1.20682E-04, 4.33463E-04, 1.15822E-03, 2.46640E-03, 4.40134E-03,
                            6.84149E-03, 9.54454E-03, 1.30112E-02, 1.68371E-02, 2.06882E-02, 2.44053E-02,
                            2.78829E-02, 3.10632E-02, 3.39216E-02, 3.64710E-02, 3.87196E-02, 4.07005E-02,
                            4.24425E-02, 4.40671E-02, 4.54745E-02, 4.67365E-02, 4.78768E-02, 5.03387E-02,
                            5.19979E-02, 5.41629E-02, 5.62128E-02, 5.83380E-02, 6.05880E-02, 6.29706E-02,
                            6.54935E-02, 6.41339E-02, 6.52454E-02, 6.62460E-02, 6.71406E-02, 6.79354E-02,
                            6.86353E-02, 6.92443E-02, 6.97653E-02, 7.02004E-02, 7.05509E-02, 7.08176E-02,
                            7.10004E-02, 7.10989E-02, 7.11122E-02, 7.10389E-02, 7.08775E-02, 7.06260E-02])
    n21_data[0] = np.array([2.26487E-70, 6.29554E-56, 1.87001E-44, 2.42387E-35, 4.19162E-28, 2.34896E-22,
                            8.60330E-18, 3.52979E-14, 2.38103E-11, 3.51084E-09, 1.52228E-07, 2.65124E-06,
                            2.36584E-05, 1.23646E-04, 4.41194E-04, 1.17425E-03, 2.49508E-03, 4.44544E-03,
                            6.90321E-03, 9.62583E-03, 1.31174E-02, 1.69701E-02, 2.08512E-02, 2.45995E-02,
                            2.81043E-02, 3.13150E-02, 3.42013E-02, 3.67816E-02, 3.90612E-02, 4.10761E-02,
                            4.28527E-02, 4.45140E-02, 4.59584E-02, 4.72584E-02, 4.84381E-02, 5.09774E-02,
                            5.27005E-02, 5.49502E-02, 5.70900E-02, 5.93164E-02, 6.16819E-02, 6.41966E-02,
                            6.68704E-02, 6.55930E-02, 6.68529E-02, 6.80155E-02, 6.90869E-02, 7.00745E-02,
                            7.09849E-02, 7.18234E-02, 7.25946E-02, 7.33021E-02, 7.39492E-02, 7.45382E-02,
                            7.50709E-02, 7.55488E-02, 7.59728E-02, 7.63433E-02, 7.66603E-02, 7.69235E-02])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, n21_data)


def calc_svc21_degas():
    """ Calculates cross section for "Continuum n=2 / n=1," whatever that is.

    accepts ne in m^-3 and Te in eV
    results in erg/s, i.e. units of power.
    """

    # populate the 15x60 array to be used for interpolations
    # rows correspond to ne, columns correspond to Te

    c21_data = np.zeros((15, 60))
    c21_data[0] = np.array([5.76221E-10, 2.81357E-10, 1.49512E-10, 8.60758E-11, 5.32204E-11, 3.49745E-11,
                            2.41739E-11, 1.74037E-11, 1.29394E-11, 9.86245E-12, 7.65927E-12, 6.03060E-12,
                            4.79418E-12, 3.83537E-12, 3.07951E-12, 2.47629E-12, 1.99083E-12, 1.59836E-12,
                            1.28000E-12, 1.02174E-12, 8.12527E-13, 6.43457E-13, 5.07335E-13, 3.98197E-13,
                            3.11114E-13, 2.41976E-13, 1.87370E-13, 1.44464E-13, 1.10923E-13, 8.48339E-14,
                            6.46376E-14, 4.90747E-14, 3.71344E-14, 2.80108E-14, 2.10665E-14, 1.58000E-14,
                            1.18195E-14, 8.82036E-15, 6.56739E-15, 4.87958E-15, 3.61838E-15, 2.67821E-15,
                            1.97891E-15, 1.45984E-15, 1.07530E-15, 7.90928E-16, 5.80989E-16, 4.26242E-16,
                            3.12345E-16, 2.28630E-16, 1.67178E-16, 1.22124E-16, 8.91283E-17, 6.49905E-17,
                            4.73504E-17, 3.44710E-17, 2.50761E-17, 1.82287E-17, 1.32421E-17, 9.61349E-18])
    c21_data[0] = np.array([4.19643E-09, 1.83068E-09, 8.71321E-10, 4.53008E-10, 2.55835E-10, 1.55500E-10,
                            1.00637E-10, 6.86115E-11, 4.87907E-11, 3.58728E-11, 2.70647E-11, 2.08225E-11,
                            1.62514E-11, 1.28124E-11, 1.01687E-11, 8.10188E-12, 6.46611E-12, 5.16120E-12,
                            4.11399E-12, 3.27164E-12, 2.59398E-12, 2.04922E-12, 1.61258E-12, 1.26368E-12,
                            9.86067E-13, 7.66149E-13, 5.92759E-13, 4.56713E-13, 3.50483E-13, 2.67928E-13,
                            2.04068E-13, 1.54889E-13, 1.17174E-13, 8.83684E-14, 6.64500E-14, 4.98316E-14,
                            3.72736E-14, 2.78136E-14, 2.07080E-14, 1.53854E-14, 1.14085E-14, 8.44408E-15,
                            6.23924E-15, 4.60271E-15, 3.39034E-15, 2.49380E-15, 1.83191E-15, 1.34403E-15,
                            9.84931E-16, 7.20984E-16, 5.27224E-16, 3.85160E-16, 2.81116E-16, 2.04998E-16,
                            1.49367E-16, 1.08748E-16, 7.91153E-17, 5.75167E-17, 4.17864E-17, 3.03386E-17])
    c21_data[0] = np.array([3.42472E-08, 1.34650E-08, 5.73668E-09, 2.67399E-09, 1.36364E-09, 7.56632E-10,
                            4.52424E-10, 2.88458E-10, 1.94001E-10, 1.36252E-10, 9.90408E-11, 7.39436E-11,
                            5.63366E-11, 4.35679E-11, 3.40511E-11, 2.68004E-11, 2.11822E-11, 1.67766E-11,
                            1.32901E-11, 1.05166E-11, 8.30519E-12, 6.53989E-12, 5.13309E-12, 4.01397E-12,
                            3.12685E-12, 2.42613E-12, 1.87497E-12, 1.44334E-12, 1.10681E-12, 8.45607E-13,
                            6.43750E-13, 4.88419E-13, 3.69377E-13, 2.78501E-13, 2.09381E-13, 1.56993E-13,
                            1.17415E-13, 8.76073E-14, 6.52217E-14, 4.84555E-14, 3.59293E-14, 2.65929E-14,
                            1.96491E-14, 1.44954E-14, 1.06774E-14, 7.85404E-15, 5.76965E-15, 4.23320E-15,
                            3.10230E-15, 2.27103E-15, 1.66079E-15, 1.21334E-15, 8.85633E-16, 6.45870E-16,
                            4.70629E-16, 3.42668E-16, 2.49314E-16, 1.81265E-16, 1.31701E-16, 9.56283E-17])
    c21_data[0] = np.array([3.07782E-07, 1.11228E-07, 4.28393E-08, 1.79036E-08, 8.17961E-09, 4.09782E-09,
                            2.23427E-09, 1.31448E-09, 8.25532E-10, 5.47486E-10, 3.79573E-10, 2.72640E-10,
                            2.01307E-10, 1.51793E-10, 1.16249E-10, 9.00176E-11, 7.02259E-11, 5.50415E-11,
                            4.32405E-11, 3.39886E-11, 2.66974E-11, 2.09319E-11, 1.63718E-11, 1.27659E-11,
                            9.92163E-12, 7.68373E-12, 5.92917E-12, 4.55863E-12, 3.49227E-12, 2.66596E-12,
                            2.02825E-12, 1.53806E-12, 1.16271E-12, 8.76365E-13, 6.58697E-13, 4.93792E-13,
                            3.69253E-13, 2.75483E-13, 2.05076E-13, 1.52351E-13, 1.12965E-13, 8.36102E-14,
                            6.17793E-14, 4.55765E-14, 3.35733E-14, 2.46969E-14, 1.81436E-14, 1.33128E-14,
                            9.75693E-15, 7.14304E-15, 5.22405E-15, 3.81689E-15, 2.78620E-15, 2.03207E-15,
                            1.48084E-15, 1.07830E-15, 7.84602E-16, 5.70497E-16, 4.14540E-16, 3.01025E-16])
    c21_data[0] = np.array([2.97602E-06, 1.01533E-06, 3.60655E-07, 1.36367E-07, 5.57232E-08, 2.50735E-08,
                            1.23388E-08, 6.61330E-09, 3.82687E-09, 2.36563E-09, 1.54580E-09, 1.05698E-09,
                            7.49446E-10, 5.46735E-10, 4.07620E-10, 3.08859E-10, 2.36764E-10, 1.82962E-10,
                            1.42108E-10, 1.10683E-10, 8.62978E-11, 6.72593E-11, 5.23531E-11, 4.06631E-11,
                            3.15031E-11, 2.43341E-11, 1.87380E-11, 1.43822E-11, 1.10028E-11, 8.39009E-12,
                            6.37746E-12, 4.83274E-12, 3.65131E-12, 2.75089E-12, 2.06694E-12, 1.54911E-12,
                            1.15820E-12, 8.63976E-13, 6.43119E-13, 4.77758E-13, 3.54247E-13, 2.62201E-13,
                            1.93749E-13, 1.42944E-13, 1.05307E-13, 7.74723E-14, 5.69211E-14, 4.17705E-14,
                            3.06172E-14, 2.24176E-14, 1.63972E-14, 1.19820E-14, 8.74761E-15, 6.38078E-15,
                            4.65052E-15, 3.38680E-15, 2.46466E-15, 1.79233E-15, 1.30253E-15, 9.45975E-16])
    c21_data[0] = np.array([3.06179E-05, 1.01040E-05, 3.39650E-06, 1.18528E-06, 4.37432E-07, 1.76755E-07,
                            7.78221E-08, 3.74632E-08, 1.96923E-08, 1.11907E-08, 6.80219E-09, 4.37431E-09,
                            2.94603E-09, 2.05967E-09, 1.48266E-09, 1.09156E-09, 8.17283E-10, 6.19529E-10,
                            4.73713E-10, 3.64293E-10, 2.81111E-10, 2.17266E-10, 1.67967E-10, 1.29744E-10,
                            1.00067E-10, 7.70134E-11, 5.91266E-11, 4.52732E-11, 3.45676E-11, 2.63181E-11,
                            1.99797E-11, 1.51258E-11, 1.14193E-11, 8.59827E-12, 6.45767E-12, 4.83843E-12,
                            3.61674E-12, 2.69768E-12, 2.00798E-12, 1.49170E-12, 1.10613E-12, 8.18795E-13,
                            6.05113E-13, 4.46510E-13, 3.29000E-13, 2.42085E-13, 1.77903E-13, 1.30578E-13,
                            9.57329E-14, 7.01105E-14, 5.12933E-14, 3.74903E-14, 2.73766E-14, 1.99739E-14,
                            1.45610E-14, 1.06066E-14, 7.72041E-15, 5.61561E-15, 4.08188E-15, 2.96514E-15])
    c21_data[0] = np.array([3.39649E-04, 1.10070E-04, 3.57654E-05, 1.17806E-05, 3.99886E-06, 1.46076E-06,
                            5.74239E-07, 2.45910E-07, 1.16212E-07, 5.99761E-08, 3.34340E-08, 1.99186E-08,
                            1.25570E-08, 8.30550E-09, 5.70546E-09, 4.03894E-09, 2.92660E-09, 2.15887E-09,
                            1.61401E-09, 1.21842E-09, 9.26006E-10, 7.06858E-10, 5.40956E-10, 4.14435E-10,
                            3.17510E-10, 2.43046E-10, 1.85783E-10, 1.41753E-10, 1.07926E-10, 8.19827E-11,
                            6.21252E-11, 4.69725E-11, 3.54259E-11, 2.66538E-11, 2.00074E-11, 1.49869E-11,
                            1.12009E-11, 8.35463E-12, 6.21919E-12, 4.62089E-12, 3.42726E-12, 2.53769E-12,
                            1.87603E-12, 1.38481E-12, 1.02075E-12, 7.51391E-13, 5.52407E-13, 4.05630E-13,
                            2.97513E-13, 2.17978E-13, 1.59543E-13, 1.16660E-13, 8.52245E-14, 6.22055E-14,
                            4.53663E-14, 3.30593E-14, 2.40729E-14, 1.75167E-14, 1.27373E-14, 9.25597E-15])
    c21_data[0] = np.array([4.05713E-03, 1.30956E-03, 4.19507E-04, 1.33913E-04, 4.30914E-05, 1.45871E-05,
                            5.19850E-06, 1.98603E-06, 8.36565E-07, 3.85526E-07, 1.92909E-07, 1.04070E-07,
                            6.01028E-08, 3.69766E-08, 2.38624E-08, 1.60100E-08, 1.10791E-08, 7.85839E-09,
                            5.68318E-09, 4.17216E-09, 3.09768E-09, 2.31922E-09, 1.74678E-09, 1.32086E-09,
                            1.00123E-09, 7.59844E-10, 5.76800E-10, 4.37663E-10, 3.31751E-10, 2.51125E-10,
                            1.89778E-10, 1.43225E-10, 1.07863E-10, 8.10744E-11, 6.08202E-11, 4.55649E-11,
                            3.40583E-11, 2.54164E-11, 1.89310E-11, 1.40757E-11, 1.04482E-11, 7.74305E-12,
                            5.72947E-12, 4.23341E-12, 3.12354E-12, 2.30155E-12, 1.69372E-12, 1.24490E-12,
                            9.13956E-13, 6.70254E-13, 4.91020E-13, 3.59359E-13, 2.62752E-13, 1.91943E-13,
                            1.40097E-13, 1.02171E-13, 7.44548E-14, 5.42168E-14, 3.94519E-14, 2.86885E-14])
    c21_data[0] = np.array([4.69329E-02, 1.54304E-02, 5.00098E-03, 1.59742E-03, 5.05682E-04, 1.64227E-04,
                            5.44574E-05, 1.88551E-05, 7.09727E-06, 2.90680E-06, 1.29467E-06, 6.25422E-07,
                            3.26700E-07, 1.85160E-07, 1.10897E-07, 6.95351E-08, 4.52626E-08, 3.03959E-08,
                            2.09466E-08, 1.47442E-08, 1.05565E-08, 7.66282E-09, 5.62293E-09, 4.16053E-09,
                            3.09767E-09, 2.31666E-09, 1.73786E-09, 1.30619E-09, 9.82695E-10, 7.39523E-10,
                            5.56362E-10, 4.18542E-10, 3.14471E-10, 2.36005E-10, 1.76889E-10, 1.32675E-10,
                            9.92445E-11, 7.41831E-11, 5.53496E-11, 4.12343E-11, 3.06734E-11, 2.27839E-11,
                            1.68993E-11, 1.25180E-11, 9.25924E-12, 6.83966E-12, 5.04580E-12, 3.71777E-12,
                            2.73596E-12, 2.01109E-12, 1.47662E-12, 1.08303E-12, 7.93537E-13, 5.80850E-13,
                            4.24766E-13, 3.10344E-13, 2.26548E-13, 1.65241E-13, 1.20429E-13, 8.77032E-14])
    c21_data[0] = np.array([4.56247E-01, 1.55750E-01, 5.20996E-02, 1.70564E-02, 5.46877E-03, 1.76179E-03,
                            5.63720E-04, 1.83880E-04, 6.41800E-05, 2.41440E-05, 9.81506E-06, 4.30736E-06,
                            2.04266E-06, 1.06074E-06, 5.82779E-07, 3.36342E-07, 2.02520E-07, 1.26577E-07,
                            8.17244E-08, 5.42594E-08, 3.68729E-08, 2.55609E-08, 1.80161E-08, 1.28727E-08,
                            9.30031E-09, 6.77937E-09, 4.97663E-09, 3.67341E-09, 2.72267E-09, 2.02418E-09,
                            1.50809E-09, 1.12592E-09, 8.41072E-10, 6.28544E-10, 4.69738E-10, 3.52817E-10,
                            2.64111E-10, 1.97952E-10, 1.48161E-10, 1.10796E-10, 8.27817E-11, 6.17881E-11,
                            4.60674E-11, 3.43142E-11, 2.55240E-11, 1.89615E-11, 1.40681E-11, 1.04240E-11,
                            7.71381E-12, 5.70093E-12, 4.20797E-12, 3.10213E-12, 2.28414E-12, 1.67986E-12,
                            1.23404E-12, 9.05531E-13, 6.63767E-13, 4.86052E-13, 3.55567E-13, 2.59865E-13])
    c21_data[0] = np.array([3.38043E+00, 1.22760E+00, 4.32248E-01, 1.47635E-01, 4.89128E-02, 1.60574E-02,
                            5.14413E-03, 1.65262E-03, 5.58930E-04, 1.99898E-04, 7.56147E-05, 3.03213E-05,
                            1.30153E-05, 6.10769E-06, 3.04404E-06, 1.60689E-06, 8.93726E-07, 5.20841E-07,
                            3.16115E-07, 1.98624E-07, 1.28406E-07, 8.50615E-08, 5.75120E-08, 3.95508E-08,
                            2.75848E-08, 1.94653E-08, 1.38695E-08, 9.96245E-09, 7.20371E-09, 5.23761E-09,
                            3.82531E-09, 2.80566E-09, 2.06359E-09, 1.52155E-09, 1.12412E-09, 8.38086E-10,
                            6.23322E-10, 4.65478E-10, 3.47700E-10, 2.59964E-10, 1.94531E-10, 1.45649E-10,
                            1.09082E-10, 8.17411E-11, 6.12222E-11, 4.58371E-11, 3.42992E-11, 2.56470E-11,
                            1.91608E-11, 1.43009E-11, 1.06619E-11, 7.93954E-12, 5.90494E-12, 4.38604E-12,
                            3.25352E-12, 2.41017E-12, 1.78300E-12, 1.31725E-12, 9.71859E-13, 7.16085E-13])
    c21_data[0] = np.array([1.75514E+01, 6.85194E+00, 2.60072E+00, 9.44261E-01, 3.30471E-01, 1.13386E-01,
                            3.75186E-02, 1.22729E-02, 4.12678E-03, 1.41308E-03, 4.91822E-04, 1.77423E-04,
                            6.84079E-05, 2.89866E-05, 1.32687E-05, 6.54148E-06, 3.44547E-06, 1.92125E-06,
                            1.12391E-06, 6.84063E-07, 4.29736E-07, 2.77232E-07, 1.82794E-07, 1.22687E-07,
                            8.35502E-08, 5.75799E-08, 4.00739E-08, 2.81187E-08, 1.98646E-08, 1.41140E-08,
                            1.00769E-08, 7.22638E-09, 5.20029E-09, 3.75404E-09, 2.71748E-09, 1.98315E-09,
                            1.44703E-09, 1.06096E-09, 7.79446E-10, 5.74110E-10, 4.23976E-10, 3.13874E-10,
                            2.32896E-10, 1.73312E-10, 1.29156E-10, 9.64161E-11, 7.20840E-11, 5.39611E-11,
                            4.04361E-11, 3.03250E-11, 2.27547E-11, 1.70795E-11, 1.28208E-11, 9.62262E-12,
                            7.21973E-12, 5.41392E-12, 4.05686E-12, 3.03729E-12, 2.27163E-12, 1.69703E-12])
    c21_data[0] = np.array([5.77965E+01, 2.48978E+01, 1.04798E+01, 4.10160E+00, 1.53564E+00, 5.61076E-01,
                            1.95667E-01, 6.64878E-02, 2.25319E-02, 7.39906E-03, 2.36942E-03, 7.79574E-04,
                            2.77860E-04, 1.10356E-04, 4.82117E-05, 2.29838E-05, 1.18120E-05, 6.46417E-06,
                            3.72495E-06, 2.23847E-06, 1.39025E-06, 8.87374E-07, 5.79084E-07, 3.84686E-07,
                            2.59233E-07, 1.76721E-07, 1.21602E-07, 8.43114E-08, 5.88196E-08, 4.12448E-08,
                            2.90433E-08, 2.05257E-08, 1.45491E-08, 1.03393E-08, 7.36412E-09, 5.26849E-09,
                            3.77269E-09, 2.71000E-09, 1.95028E-09, 1.40662E-09, 1.01685E-09, 7.36812E-10,
                            5.35167E-10, 3.90026E-10, 2.84788E-10, 2.08476E-10, 1.53003E-10, 1.12578E-10,
                            8.30425E-11, 6.14074E-11, 4.55179E-11, 3.38174E-11, 2.51791E-11, 1.87854E-11,
                            1.40412E-11, 1.05125E-11, 7.88210E-12, 5.91714E-12, 4.44648E-12, 3.34389E-12])
    c21_data[0] = np.array([1.26809E+02, 6.12325E+01, 2.98148E+01, 1.25976E+01, 5.05954E+00, 1.96650E+00,
                            7.27625E-01, 2.58354E-01, 8.92411E-02, 2.87242E-02, 8.79175E-03, 2.76681E-03,
                            9.54211E-04, 3.69828E-04, 1.59019E-04, 7.50098E-05, 3.82647E-05, 2.08251E-05,
                            1.19472E-05, 7.15232E-06, 4.42655E-06, 2.81588E-06, 1.83141E-06, 1.21239E-06,
                            8.14035E-07, 5.52782E-07, 3.78788E-07, 2.61450E-07, 1.81516E-07, 1.26613E-07,
                            8.86533E-08, 6.22674E-08, 4.38455E-08, 3.09385E-08, 2.18691E-08, 1.54914E-08,
                            1.09858E-08, 7.80297E-09, 5.54878E-09, 3.95064E-09, 2.81630E-09, 2.01020E-09,
                            1.43670E-09, 1.02928E-09, 7.38021E-10, 5.30020E-10, 3.81282E-10, 2.74774E-10,
                            1.98392E-10, 1.43529E-10, 1.04055E-10, 7.56043E-11, 5.50585E-11, 4.01915E-11,
                            2.94108E-11, 2.15757E-11, 1.58680E-11, 1.16999E-11, 8.64854E-12, 6.40893E-12])
    c21_data[0] = np.array([2.30140E+02, 1.21337E+02, 6.07757E+01, 2.86623E+01, 1.32403E+01, 5.48092E+00,
                            2.13829E+00, 7.96525E-01, 2.84229E-01, 9.22597E-02, 2.80915E-02, 8.78887E-03,
                            3.02156E-03, 1.16925E-03, 5.02684E-04, 2.37204E-04, 1.21057E-04, 6.59032E-05,
                            3.78113E-05, 2.26327E-05, 1.40023E-05, 8.90256E-06, 5.78602E-06, 3.82707E-06,
                            2.56705E-06, 1.74120E-06, 1.19160E-06, 8.21291E-07, 5.69282E-07, 3.96393E-07,
                            2.77013E-07, 1.94147E-07, 1.36390E-07, 9.59958E-08, 6.76684E-08, 4.77574E-08,
                            3.37425E-08, 2.38621E-08, 1.68883E-08, 1.19610E-08, 8.47680E-09, 6.01123E-09,
                            4.26536E-09, 3.03157E-09, 2.15468E-09, 1.53254E-09, 1.09085E-09, 7.77091E-10,
                            5.54057E-10, 3.95407E-10, 2.82471E-10, 2.02015E-10, 1.44648E-10, 1.03707E-10,
                            7.44579E-11, 5.35396E-11, 3.85612E-11, 2.78220E-11, 2.01114E-11, 1.45668E-11])

    Te_vals, ne_vals = np.meshgrid(-1.2 + (np.linspace(1, 60, 60)-1)/10 - 3,
                                   10 + 0.5 * (np.linspace(1, 15, 15) - 1) + 6)

    return Rbf(Te_vals, ne_vals, c21_data)


if __name__ == "__main__":
    ne = np.random.uniform(1E19, 3E19, size=(5, 5))
    Te = np.random.uniform(1.0, 3.0, size=(5, 5))
    Ti = np.random.uniform(1.0, 3.0, size=(5, 5))
    Tn = np.random.uniform(0.005, 0.5, size=(5, 5))

    sv = calc_xsec()
    output = sv.ione_degas(1E15, 1E-1)

    print output
    #plt.imshow(output)
    #plt.colorbar()


    #lt.show()
