import glog
import networkx as nx
from typing import List, Dict

import ascend.protos.api.api_pb2 as api_pb2
from ascend.sdk.client import Client
from ascend.sdk.definitions import (Component, Dataflow, ReadConnector, Transform, WriteConnector,
                                    ComponentGroup, ComponentIdToUuidMap, ComponentUuidType,
                                    DataFeed, DataFeedConnector, DataService)
from openapi_client.exceptions import ApiException


class DataServiceApplier:
  """DataServiceApplier is a utility class that accepts a DataService definition
  and 'applies' it - ensuring that the DataService is created if it does not
  already exist, and applying any configuration changes needed to match the supplied
  definition.
  """
  def __init__(self, client: Client):
    self.client = client

  def apply(self, data_service: DataService, delete=True, dry_run=False):
    """
    Create or update `data_service`.
    :param data_service: DataService definition
    :param delete: If set to True - delete any Dataflow that is not part of `data_service`.
    At the Dataflow level, remove any components that are not part of the Dataflow definition.
    :param dry_run: If set to True - skips any create or update operations
    :return:
    """
    glog.info(f"Apply DataService: {data_service.id}")
    exists = False
    try:
      self.client.get_data_service(data_service.id)
      exists = True
    except ApiException as e:
      if e.status != 404:
        raise e

    if exists:
      glog.info(f"Update DataService: {data_service.id}")
      if not dry_run:
        self.client.update_data_service(data_service.id, data_service.to_proto())
    else:
      if not dry_run:
        glog.info(f"Create DataService: {data_service.id}")
        self.client.create_data_service(data_service.to_proto())

    for df in data_service.dataflows:
      DataflowApplier(self.client).apply(data_service.id, df, delete, dry_run)

    try:
      if delete:
        expected = [dataflow.id for dataflow in data_service.dataflows]
        # TODO - does not yet account for dependencies between dataflows
        for df in self.client.list_dataflows(data_service.id).data:
          if df.id not in expected:
            glog.info(f"Delete Dataflow: (ds={data_service.id} df={df.id})")
            if not dry_run:
              self.client.delete_dataflow(data_service.id, df.id)
    except ApiException as e:
      # tolerate 404s during dry runs, since they're likely to happen
      if e.status != 404 or not dry_run:
        raise e


class DataflowApplier:
  """DataflowApplier is a utility class that accepts a Dataflow definition
  and 'applies' it - ensuring that a Dataflow is created if it does not already
  exist, binding it to the DataService identified by `data_service_id`,
  and applying any configuration changes needed to match the supplied definition.
  """
  def __init__(self, client: Client):
    self.client = client

  def apply(self, data_service_id: str, dataflow: Dataflow, delete=True, dry_run=False):
    """Accepts a Dataflow definition, and ensures that it is created, bound to the DataService
    identified by `data_service_id`, and has all of the constituent elements included in the
    Dataflow definition - components (read connectors, transforms, write connectors), component
    groups, data feeds, and data feed connectors. The specified DataService must already exist.

    :param data_service_id: DataService id
    :param dataflow: Dataflow definition
    :param delete: if set to True - delete any components, data feeds, data feed connectors,
    or groups not defined in `dataflow`
    :param dry_run: If set to True - skips any create or update operations
    :return:
    """
    self._apply_dataflow(data_service_id, dataflow, dry_run)

    id_map: ComponentIdToUuidMap = {}
    for dfc in dataflow.data_feed_connectors:
      resource = DataFeedConnectorApplier(self.client).apply(data_service_id, dataflow.id, dfc,
                                                             dry_run)
      id_map[dfc.id] = ComponentUuidType(type=dfc.legacy_type(), uuid=resource.uuid)

    id_to_component = self._build_component_id_map(dataflow.components)
    g = nx.DiGraph()
    for component in dataflow.components:
      g.add_node(component.id)
      for dep in component.dependencies():
        g.add_edge(component.id, dep)

    for component_id in reversed(list(nx.topological_sort(g))):
      component = id_to_component[component_id]
      applier = ComponentApplier(self.client, id_map, dry_run)
      resource = applier.apply(data_service_id, dataflow.id, component)
      id_map[component.id] = ComponentUuidType(type=component.legacy_type(), uuid=resource.uuid)

    for df in dataflow.data_feeds:
      resource = DataFeedApplier(self.client, id_map).apply(data_service_id, dataflow.id, df,
                                                            dry_run)
      id_map[df.id] = ComponentUuidType(type=df.legacy_type(), uuid=resource.uuid)

    for group in dataflow.groups:
      # validate groups are non-overlapping ?
      GroupApplier(self.client, id_map).apply(data_service_id, dataflow.id, group, dry_run)

    try:
      if delete:
        self._sweep(data_service_id, dataflow, dry_run)
    except ApiException as e:
      # tolerate 404s during dry runs, since they're likely to happen
      if e.status != 404 or not dry_run:
        raise e

  def _build_component_id_map(self, components: List[Component]) -> Dict[str, Component]:
    id_to_component: Dict[str, Component] = {}
    for component in components:
      if not component.id:
        raise ValueError(f"empty component id")
      elif component.id in id_to_component:
        raise ValueError(f"duplicate component id {component.id} in component list")
      else:
        id_to_component[component.id] = component
    return id_to_component

  def _apply_dataflow(self, data_service_id: str, dataflow: Dataflow, dry_run=False):
    """ Create a dataflow if it does not already exist, otherwise update it.
    """
    glog.info(f"Apply Dataflow: (ds={data_service_id} df={dataflow.id})")
    exists = False
    try:
      self.client.get_dataflow(data_service_id, dataflow.id)
      exists = True
    except ApiException as e:
      if e.status != 404:
        raise e

    if exists:
      glog.info(f"Update Dataflow: (ds={data_service_id} df={dataflow.id})")
      if not dry_run:
        self.client.update_dataflow(data_service_id, dataflow.id, dataflow.to_proto())
    else:
      glog.info(f"Create Dataflow: (ds={data_service_id} df={dataflow.id})")
      if not dry_run:
        self.client.create_dataflow(data_service_id, dataflow.to_proto())

  def _sweep(self, data_service_id: str, dataflow: Dataflow, dry_run=False):
    """ Delete any components, data feeds, data feed connectors, or component groups
    that are not part of `dataflow`
    """
    expected_groups = [group.id for group in dataflow.groups]
    for group in self.client.list_component_groups(data_service_id, dataflow.id).data:
      if group.id not in expected_groups:
        glog.info(f"Delete ComponentGroup: (ds={data_service_id} df={dataflow.id} {group.id})")
        if not dry_run:
          self.client.delete_component_group(data_service_id, dataflow.id, group.id)

    expected_data_feeds = [data_feed.id for data_feed in dataflow.data_feeds]
    for data_feed in self.client.list_data_feeds(data_service_id, dataflow.id).data:
      if data_feed.id not in expected_data_feeds:
        glog.info(f"Delete DataFeed: (ds={data_service_id} df={dataflow.id} {data_feed.id})")
        if not dry_run:
          self.client.delete_data_feed(data_service_id, dataflow.id, data_feed.id)

    uuid_to_id: Dict[str, str] = {}
    id_to_type: Dict[str, str] = {}
    components = []
    for component in self.client.list_dataflow_components(data_service_id, dataflow.id).data:
      if component.type not in ["source", "view", "sink"]:
        continue
      components.append(component)
      uuid_to_id[component.uuid] = component.id
      id_to_type[component.id] = component.type

    g = nx.DiGraph()
    for component in components:
      g.add_node(component.id)
      if component.type == "view":
        for input in component.inputs:
          g.add_edge(component.id, uuid_to_id[input.uuid])
      elif component.type == "sink":
        g.add_edge(component.id, uuid_to_id[component.inputUUID])

    expected_components = [component.id for component in dataflow.components]
    for component_id in list(nx.topological_sort(g)):
      if component_id not in expected_components:
        if id_to_type[component_id] == "source":
          glog.info(f"Delete ReadConnector: (ds={data_service_id} df={dataflow.id} {component_id})")
          if not dry_run:
            self.client.delete_read_connector(data_service_id, dataflow.id, component_id)
        elif id_to_type[component_id] == "view":
          glog.info(f"Delete Transform: (ds={data_service_id} df={dataflow.id} {component_id})")
          if not dry_run:
            self.client.delete_transform(data_service_id, dataflow.id, component_id)
        elif id_to_type[component_id] == "sink":
          glog.info(
              f"Delete WriteConnector: (ds={data_service_id} df={dataflow.id} {component_id})")
          if not dry_run:
            self.client.delete_write_connector(data_service_id, dataflow.id, component_id)

    expected_dfcs = [dfc.id for dfc in dataflow.data_feed_connectors]
    for dfc in self.client.list_data_feed_connectors(data_service_id, dataflow.id).data:
      if dfc.id not in expected_dfcs:
        glog.info(f"Delete DataFeedConnector: (ds={data_service_id} df={dataflow.id} {dfc.id})")
        if not dry_run:
          self.client.delete_data_feed_connector(data_service_id, dataflow.id, dfc.id)


class GroupApplier:
  def __init__(self, client: Client, id_map: ComponentIdToUuidMap):
    self.client = client
    self.id_map = id_map

  def apply(self, data_service_id, dataflow_id, group: ComponentGroup, dry_run=False):
    glog.info(f"Apply ComponentGroup: (ds={data_service_id} df={dataflow_id} {group.id})")
    exists = False
    try:
      self.client.get_component_group(data_service_id, dataflow_id, group.id)
      exists = True
    except ApiException as e:
      if e.status != 404:
        raise e

    if exists:
      glog.info(f"Update ComponentGroup: (ds={data_service_id} df={dataflow_id} {group.id})")
      if not dry_run:
        return self.client.update_component_group(data_service_id, dataflow_id, group.id,
                                                  group.to_proto(self.id_map)).data
      else:
        return api_pb2.ComponentGroup()
    else:
      glog.info(f"Create ComponentGroup: (ds={data_service_id} df={dataflow_id} {group.id})")
      if not dry_run:
        return self.client.create_component_group(data_service_id, dataflow_id,
                                                  group.to_proto(self.id_map)).data
      else:
        return api_pb2.ComponentGroup()

  @staticmethod
  def build(client: Client, data_service_id: str, dataflow_id: str) -> 'GroupApplier':
    id_map = _component_id_to_uuid_map(client, data_service_id, dataflow_id)
    return GroupApplier(client, id_map)


class DataFeedApplier:
  def __init__(self, client: Client, id_map: ComponentIdToUuidMap):
    self.client = client
    self.id_map = id_map

  def apply(self, data_service_id, dataflow_id, data_feed: DataFeed, dry_run=False):
    glog.info(f"Apply DataFeed: (ds={data_service_id} df={dataflow_id} {data_feed.id})")
    exists = False
    try:
      self.client.get_data_feed(data_service_id, dataflow_id, data_feed.id)
      exists = True
    except ApiException as e:
      if e.status != 404:
        raise e

    ds_uuid_to_id = {ds.uuid: ds.id for ds in self.client.list_data_services().data}
    roles = self.client.list_data_service_roles().data
    ds_to_role_map = {
        ds_uuid_to_id[role.org_id]: role.uuid
        for role in roles if role.id == 'Everyone'
    }

    if exists:
      glog.info(f"Update DataFeed: (ds={data_service_id} df={dataflow_id} {data_feed.id})")
      if not dry_run:
        return self.client.update_data_feed(
            data_service_id, dataflow_id, data_feed.id,
            data_feed.to_proto(data_service_id, self.id_map, ds_to_role_map)).data
      else:
        return api_pb2.DataFeed()
    else:
      glog.info(f"Create DataFeed: (ds={data_service_id} df={dataflow_id} {data_feed.id})")
      if not dry_run:
        return self.client.create_data_feed(
            data_service_id, dataflow_id,
            data_feed.to_proto(data_service_id, self.id_map, ds_to_role_map)).data
      else:
        return api_pb2.DataFeed()

  @staticmethod
  def build(client: Client, data_service_id: str, dataflow_id: str) -> 'DataFeedApplier':
    id_map = _component_id_to_uuid_map(client, data_service_id, dataflow_id)
    return DataFeedApplier(client, id_map)


class DataFeedConnectorApplier:
  def __init__(self, client: Client):
    self.client = client

  def apply(self,
            data_service_id,
            dataflow_id,
            data_feed_connector: DataFeedConnector,
            dry_run=False):
    try:
      data_feed = self.client.get_data_feed(data_feed_connector.input_data_service_id,
                                            data_feed_connector.input_dataflow_id,
                                            data_feed_connector.input_data_feed_id).data
    except ApiException as e:
      if e.status == 404 and dry_run and \
        data_feed_connector.input_data_service_id == data_service_id:  # noqa: E121
        # with dry_run it is possible we haven't created the host DataService yet
        data_feed = api_pb2.DataFeed()
      else:
        raise e

    dfc_repr = f"(ds={data_service_id} df={dataflow_id} {data_feed_connector.id})"
    glog.info(f"Apply DataFeedConnector: {dfc_repr}")
    exists = False
    try:
      self.client.get_data_feed_connector(data_service_id, dataflow_id, data_feed_connector.id)
      exists = True
    except ApiException as e:
      if e.status != 404:
        raise e

    if exists:
      glog.info(f"Update DataFeedConnector: {dfc_repr}")
      if not dry_run:
        return self.client.update_data_feed_connector(
            data_service_id, dataflow_id, data_feed_connector.id,
            data_feed_connector.to_proto(data_feed.uuid)).data
      else:
        return api_pb2.DataFeedConnector()
    else:
      glog.info(f"Create DataFeedConnector: {dfc_repr}")
      if not dry_run:
        return self.client.create_data_feed_connector(data_service_id, dataflow_id,
                                                      data_feed_connector.to_proto(
                                                          data_feed.uuid)).data
      else:
        return api_pb2.DataFeedConnector()


class ComponentApplier:
  def __init__(self, client: Client, id_map: ComponentIdToUuidMap, dry_run=False):
    self.client = client
    self.id_map = id_map
    self.dry_run = dry_run

  def _component_exists(self, data_service_id, dataflow_id, component) -> bool:
    try:
      if isinstance(component, ReadConnector):
        self.client.get_read_connector(data_service_id, dataflow_id, component.id)
      elif isinstance(component, WriteConnector):
        self.client.get_write_connector(data_service_id, dataflow_id, component.id)
      elif isinstance(component, Transform):
        self.client.get_transform(data_service_id, dataflow_id, component.id)
      return True
    except ApiException as e:
      if e.status == 404:
        return False
      else:
        raise e

  def _create_component(self, data_service_id, dataflow_id, component: Component):
    proto = component.to_proto(self.id_map)
    if isinstance(component, ReadConnector):
      glog.info(f"Create ReadConnector: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.create_read_connector(data_service_id, dataflow_id, proto).data
      else:
        return api_pb2.ReadConnector()
    elif isinstance(component, WriteConnector):
      glog.info(f"Create WriteConnector: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.create_write_connector(data_service_id, dataflow_id, proto).data
      else:
        return api_pb2.WriteConnector()
    elif isinstance(component, Transform):
      glog.info(f"Create Transform: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.create_transform(data_service_id, dataflow_id, proto).data
      else:
        return api_pb2.Transform()

  def _update_component(self, data_service_id, dataflow_id, component: Component):
    proto = component.to_proto(self.id_map)
    if isinstance(component, ReadConnector):
      glog.info(f"Update ReadConnector: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.update_read_connector(data_service_id, dataflow_id, component.id,
                                                 proto).data
      else:
        return api_pb2.ReadConnector()
    elif isinstance(component, WriteConnector):
      glog.info(f"Update WriteConnector: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.update_write_connector(data_service_id, dataflow_id, component.id,
                                                  proto).data
      else:
        return api_pb2.WriteConnector()
    elif isinstance(component, Transform):
      glog.info(f"Update Transform: (ds={data_service_id} df={dataflow_id} {component.id})")
      if not self.dry_run:
        return self.client.update_transform(data_service_id, dataflow_id, component.id, proto).data
      else:
        return api_pb2.Transform()

  def apply(self, data_service_id, dataflow_id, component: Component):
    """
    Applies the provided component and
    :param data_service_id:
    :param dataflow_id:
    :param component:
    :return:
    """
    glog.info(f"Apply Component: (ds={data_service_id} df={dataflow_id} {component.id})")
    if self._component_exists(data_service_id, dataflow_id, component):
      return self._update_component(data_service_id, dataflow_id, component)
    else:
      return self._create_component(data_service_id, dataflow_id, component)

  @staticmethod
  def build(client: Client, data_service_id: str, dataflow_id: str) -> 'ComponentApplier':
    id_map = _component_id_to_uuid_map(client, data_service_id, dataflow_id)
    return ComponentApplier(client, id_map)


def _component_id_to_uuid_map(client: Client, data_service_id: str,
                              dataflow_id: str) -> ComponentIdToUuidMap:
  id_map: ComponentIdToUuidMap = {}
  components = client.list_dataflow_components(data_service_id, dataflow_id).data
  for c in components:
    id_map[c.id] = ComponentUuidType(type=c.type, uuid=c.uuid)

  return id_map
