from typing import Dict

import ascend.protos.api.api_pb2 as api_pb2
from ascend.sdk.client import Client
from ascend.sdk.definitions import (Dataflow, ReadConnector, Transform, WriteConnector,
                                    ComponentGroup, ComponentUuidToIdMap, DataFeed,
                                    DataFeedConnector, DataService)


def data_service_from_proto(client: Client, proto: api_pb2.DataService) -> 'DataService':
  """
  Build a DataService definition from the API DataService proto representation.
  This is a utility function that helps bridge the declarative and imperative forms
  of the SDK by transforming a DataService proto message to a DataService definition
  that can be 'applied'.

  :param client: SDK client instance
  :param proto: DataService proto object
  :return: DataService definition
  """
  dataflow_protos = client.list_dataflows(proto.id).data
  return DataService(
      id=proto.id,
      name=proto.name,
      description=proto.description,
      dataflows=[
          dataflow_from_proto(client, proto.id, dataflow_proto)
          for dataflow_proto in dataflow_protos
      ],
  )


def dataflow_from_proto(client: Client, data_service_id: str,
                        proto: api_pb2.Dataflow) -> 'Dataflow':
  """
  Build a Dataflow definition from the API Dataflow proto representation.

  :param client: SDK client instance
  :param data_service_id: DataService id
  :param proto: Dataflow proto object
  :return: Dataflow definition
  """
  component_protos = client.list_dataflow_components(data_service_id, proto.id, deep=True).data

  uuid_map: ComponentUuidToIdMap = {}
  for c in component_protos:
    uuid_map[c.uuid] = c.id

  read_connectors = [
      ReadConnector.from_proto(client.get_read_connector(data_service_id, proto.id, c.id).data)
      if c.source.container.HasField('immediate') else ReadConnector.from_proto(c)
      for c in component_protos if c.type == "source"
  ]
  transforms = [Transform.from_proto(c, uuid_map) for c in component_protos if c.type == "view"]
  write_connectors = [
      WriteConnector.from_proto(c, uuid_map) for c in component_protos if c.type == "sink"
  ]

  role_to_ds_map = _role_to_ds_map(client)
  data_feeds = [
      DataFeed.from_proto(c, data_service_id, role_to_ds_map, uuid_map) for c in component_protos
      if c.type == "pub"
  ]

  data_feed_connectors = []
  groups = []
  for c in component_protos:
    if c.type == "sub":
      df = client.get_data_feed_for_data_feed_connector(data_service_id, proto.id, c.id).data
      data_feed_connectors.append(
          DataFeedConnector(id=c.id,
                            name=c.name,
                            description=c.description,
                            input_data_service_id=df.organization.id,
                            input_dataflow_id=df.project.id,
                            input_data_feed_id=df.id))
    elif c.type == "group":
      component_ids = [uuid_map[input.uuid] for input in c.content]
      groups.append(
          ComponentGroup(id=c.id,
                         name=c.name,
                         description=c.description,
                         component_ids=component_ids))

  return Dataflow(
      id=proto.id,
      name=proto.name,
      description=proto.description,
      components=read_connectors + transforms + write_connectors,
      data_feeds=data_feeds,
      data_feed_connectors=data_feed_connectors,
      groups=groups,
  )


def _role_to_ds_map(client: Client) -> Dict[str, str]:
  ds_uuid_to_id = {ds.uuid: ds.id for ds in client.list_data_services().data}
  roles = client.list_data_service_roles().data
  return {role.uuid: ds_uuid_to_id[role.org_id] for role in roles if role.id == 'Everyone'}
