"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Standard attributes
 * Specified following the OpenID Connect spec
 * @see https://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
 */
var UserPoolAttribute;
(function (UserPoolAttribute) {
    /**
     * End-User's preferred postal address.
     */
    UserPoolAttribute["ADDRESS"] = "address";
    /**
     * End-User's birthday, represented as an ISO 8601:2004 [ISO8601‑2004] YYYY-MM-DD format.
     * The year MAY be 0000, indicating that it is omitted.
     * To represent only the year, YYYY format is allowed.
     */
    UserPoolAttribute["BIRTHDATE"] = "birthdate";
    /**
     * End-User's preferred e-mail address.
     * Its value MUST conform to the RFC 5322 [RFC5322] addr-spec syntax.
     */
    UserPoolAttribute["EMAIL"] = "email";
    /**
     * Surname(s) or last name(s) of the End-User.
     * Note that in some cultures, people can have multiple family names or no family name;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["FAMILY_NAME"] = "family_name";
    /**
     * End-User's gender.
     */
    UserPoolAttribute["GENDER"] = "gender";
    /**
     * Given name(s) or first name(s) of the End-User.
     * Note that in some cultures, people can have multiple given names;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["GIVEN_NAME"] = "given_name";
    /**
     * End-User's locale, represented as a BCP47 [RFC5646] language tag.
     * This is typically an ISO 639-1 Alpha-2 [ISO639‑1] language code in lowercase
     * and an ISO 3166-1 Alpha-2 [ISO3166‑1] country code in uppercase, separated by a dash.
     * For example, en-US or fr-CA.
     */
    UserPoolAttribute["LOCALE"] = "locale";
    /**
     * Middle name(s) of the End-User.
     * Note that in some cultures, people can have multiple middle names;
     * all can be present, with the names being separated by space characters.
     * Also note that in some cultures, middle names are not used.
     */
    UserPoolAttribute["MIDDLE_NAME"] = "middle_name";
    /**
     * End-User's full name in displayable form including all name parts,
     * possibly including titles and suffixes, ordered according to the End-User's locale and preferences.
     */
    UserPoolAttribute["NAME"] = "name";
    /**
     * Casual name of the End-User that may or may not be the same as the given_name.
     * For instance, a nickname value of Mike might be returned alongside a given_name value of Michael.
     */
    UserPoolAttribute["NICKNAME"] = "nickname";
    /**
     * End-User's preferred telephone number.
     * E.164 [E.164] is RECOMMENDED as the format of this Claim, for example, +1 (425) 555-1212 or +56 (2) 687 2400.
     * If the phone number contains an extension, it is RECOMMENDED that the extension be represented using the
     * RFC 3966 [RFC3966] extension syntax, for example, +1 (604) 555-1234;ext=5678.
     */
    UserPoolAttribute["PHONE_NUMBER"] = "phone_number";
    /**
     * URL of the End-User's profile picture.
     * This URL MUST refer to an image file (for example, a PNG, JPEG, or GIF image file),
     * rather than to a Web page containing an image.
     * Note that this URL SHOULD specifically reference a profile photo of the End-User
     * suitable for displaying when describing the End-User, rather than an arbitrary photo taken by the End-User
     */
    UserPoolAttribute["PICTURE"] = "picture";
    /**
     * Shorthand name by which the End-User wishes to be referred to.
     */
    UserPoolAttribute["PREFERRED_USERNAME"] = "preferred_username";
    /**
     * URL of the End-User's profile page. The contents of this Web page SHOULD be about the End-User.
     */
    UserPoolAttribute["PROFILE"] = "profile";
    /**
     * The End-User's time zone
     */
    UserPoolAttribute["TIMEZONE"] = "timezone";
    /**
     * Time the End-User's information was last updated.
     * Its value is a JSON number representing the number of seconds from 1970-01-01T0:0:0Z
     * as measured in UTC until the date/time.
     */
    UserPoolAttribute["UPDATED_AT"] = "updated_at";
    /**
     * URL of the End-User's Web page or blog.
     * This Web page SHOULD contain information published by the End-User or an organization that the End-User is affiliated with.
     */
    UserPoolAttribute["WEBSITE"] = "website";
})(UserPoolAttribute = exports.UserPoolAttribute || (exports.UserPoolAttribute = {}));
/**
 * Methods of user sign-in
 */
var SignInType;
(function (SignInType) {
    /**
     * End-user will sign in with a username, with optional aliases
     */
    SignInType[SignInType["USERNAME"] = 0] = "USERNAME";
    /**
     * End-user will sign in using an email address
     */
    SignInType[SignInType["EMAIL"] = 1] = "EMAIL";
    /**
     * End-user will sign in using a phone number
     */
    SignInType[SignInType["PHONE"] = 2] = "PHONE";
    /**
     * End-user will sign in using either an email address or phone number
     */
    SignInType[SignInType["EMAIL_OR_PHONE"] = 3] = "EMAIL_OR_PHONE";
})(SignInType = exports.SignInType || (exports.SignInType = {}));
/**
 * Define a Cognito User Pool
 */
class UserPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userPoolName,
        });
        this.triggers = {};
        let aliasAttributes;
        let usernameAttributes;
        if (props.usernameAliasAttributes != null && props.signInType !== SignInType.USERNAME) {
            throw new Error(`'usernameAliasAttributes' can only be set with a signInType of 'USERNAME'`);
        }
        if (props.usernameAliasAttributes
            && !props.usernameAliasAttributes.every(a => {
                return a === UserPoolAttribute.EMAIL || a === UserPoolAttribute.PHONE_NUMBER || a === UserPoolAttribute.PREFERRED_USERNAME;
            })) {
            throw new Error(`'usernameAliasAttributes' can only include EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME`);
        }
        if (props.autoVerifiedAttributes
            && !props.autoVerifiedAttributes.every(a => a === UserPoolAttribute.EMAIL || a === UserPoolAttribute.PHONE_NUMBER)) {
            throw new Error(`'autoVerifiedAttributes' can only include EMAIL or PHONE_NUMBER`);
        }
        switch (props.signInType) {
            case SignInType.USERNAME:
                aliasAttributes = props.usernameAliasAttributes;
                break;
            case SignInType.EMAIL:
                usernameAttributes = [UserPoolAttribute.EMAIL];
                break;
            case SignInType.PHONE:
                usernameAttributes = [UserPoolAttribute.PHONE_NUMBER];
                break;
            case SignInType.EMAIL_OR_PHONE:
                usernameAttributes = [UserPoolAttribute.EMAIL, UserPoolAttribute.PHONE_NUMBER];
                break;
            default:
                aliasAttributes = props.usernameAliasAttributes;
                break;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: this.physicalName,
            usernameAttributes,
            aliasAttributes,
            autoVerifiedAttributes: props.autoVerifiedAttributes,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => this.triggers })
        });
        this.userPoolId = this.getResourceNameAttribute(userPool.ref);
        this.userPoolArn = this.getResourceArnAttribute(userPool.attrArn, {
            service: 'cognito',
            resource: 'userpool',
            resourceName: this.physicalName,
        });
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported user pool properties
     */
    static fromUserPoolAttributes(scope, id, attrs) {
        /**
         * Define a user pool which has been declared in another stack
         */
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolId = attrs.userPoolId;
                this.userPoolArn = attrs.userPoolArn;
                this.userPoolProviderName = attrs.userPoolProviderName;
                this.userPoolProviderUrl = attrs.userPoolProviderUrl;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Attach 'Create Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addCreateAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'CreateAuthChallenge');
        this.triggers = { ...this.triggers, createAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Custom Message' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     * @param fn the lambda function to attach
     */
    addCustomMessageTrigger(fn) {
        this.addLambdaPermission(fn, 'CustomMessage');
        this.triggers = { ...this.triggers, customMessage: fn.functionArn };
    }
    /**
     * Attach 'Define Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addDefineAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'DefineAuthChallenge');
        this.triggers = { ...this.triggers, defineAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Post Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     * @param fn the lambda function to attach
     */
    addPostAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostAuthentication');
        this.triggers = { ...this.triggers, postAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Post Confirmation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     * @param fn the lambda function to attach
     */
    addPostConfirmationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostConfirmation');
        this.triggers = { ...this.triggers, postConfirmation: fn.functionArn };
    }
    /**
     * Attach 'Pre Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     * @param fn the lambda function to attach
     */
    addPreAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreAuthentication');
        this.triggers = { ...this.triggers, preAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Pre Sign Up' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     * @param fn the lambda function to attach
     */
    addPreSignUpTrigger(fn) {
        this.addLambdaPermission(fn, 'PreSignUp');
        this.triggers = { ...this.triggers, preSignUp: fn.functionArn };
    }
    /**
     * Attach 'Pre Token Generation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
     * @param fn the lambda function to attach
     */
    addPreTokenGenerationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreTokenGeneration');
        this.triggers = { ...this.triggers, preTokenGeneration: fn.functionArn };
    }
    /**
     * Attach 'User Migration' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
     * @param fn the lambda function to attach
     */
    addUserMigrationTrigger(fn) {
        this.addLambdaPermission(fn, 'UserMigration');
        this.triggers = { ...this.triggers, userMigration: fn.functionArn };
    }
    /**
     * Attach 'Verify Auth Challenge Response' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     * @param fn the lambda function to attach
     */
    addVerifyAuthChallengeResponseTrigger(fn) {
        this.addLambdaPermission(fn, 'VerifyAuthChallengeResponse');
        this.triggers = { ...this.triggers, verifyAuthChallengeResponse: fn.functionArn };
    }
    addLambdaPermission(fn, name) {
        const normalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${normalize}Cognito`, {
            principal: new iam.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn
        });
    }
}
exports.UserPool = UserPool;
//# sourceMappingURL=data:application/json;base64,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