"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const cognito = require("../lib");
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool'
        }));
        test.done();
    },
    'support tags'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const pool = new cognito.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        cdk.Tag.add(pool, "PoolTag", "PoolParty");
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: "PoolParty",
            }
        }));
        test.done();
    },
    'lambda triggers are defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new cognito.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        pool.addCustomMessageTrigger(fn);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
                CustomMessage: stack.resolve(fn.functionArn)
            }
        }));
        test.done();
    },
    'on* API correctly appends triggers'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const createAuthChallengeLambdaFn = new lambda.Function(stack, 'createAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const customMessageLambdaFn = new lambda.Function(stack, 'customMessageLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const defineAuthChallengeLambdaFn = new lambda.Function(stack, 'defineAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postAuthenticationLambdaFn = new lambda.Function(stack, 'postAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postConfirmationLambdaFn = new lambda.Function(stack, 'postConfirmationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preAuthenticationLambdaFn = new lambda.Function(stack, 'preAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preSignUpLambdaFn = new lambda.Function(stack, 'preSignUpLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preTokenGenerationLambdaFn = new lambda.Function(stack, 'preTokenGenerationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const userMigrationLambdaFn = new lambda.Function(stack, 'userMigrationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const verifyAuthChallengeResponseLambdaFn = new lambda.Function(stack, 'verifyAuthChallengeResponseLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new cognito.UserPool(stack, 'Pool', {});
        pool.addCreateAuthChallengeTrigger(createAuthChallengeLambdaFn);
        pool.addCustomMessageTrigger(customMessageLambdaFn);
        pool.addDefineAuthChallengeTrigger(defineAuthChallengeLambdaFn);
        pool.addPostAuthenticationTrigger(postAuthenticationLambdaFn);
        pool.addPostConfirmationTrigger(postConfirmationLambdaFn);
        pool.addPreAuthenticationTrigger(preAuthenticationLambdaFn);
        pool.addPreSignUpTrigger(preSignUpLambdaFn);
        pool.addPreTokenGenerationTrigger(preTokenGenerationLambdaFn);
        pool.addUserMigrationTrigger(userMigrationLambdaFn);
        pool.addVerifyAuthChallengeResponseTrigger(verifyAuthChallengeResponseLambdaFn);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallengeLambdaFn.functionArn),
                CustomMessage: stack.resolve(customMessageLambdaFn.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallengeLambdaFn.functionArn),
                PostAuthentication: stack.resolve(postAuthenticationLambdaFn.functionArn),
                PostConfirmation: stack.resolve(postConfirmationLambdaFn.functionArn),
                PreAuthentication: stack.resolve(preAuthenticationLambdaFn.functionArn),
                PreSignUp: stack.resolve(preSignUpLambdaFn.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGenerationLambdaFn.functionArn),
                UserMigration: stack.resolve(userMigrationLambdaFn.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponseLambdaFn.functionArn)
            }
        }));
        test.done();
    },
    'lambdas are given cognito service grant'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Permission', {
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com'
        }));
        test.done();
    },
    'set sign in type'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            signInType: cognito.SignInType.EMAIL,
            autoVerifiedAttributes: [cognito.UserPoolAttribute.EMAIL]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email'],
            AutoVerifiedAttributes: ['email']
        }));
        test.done();
    },
    'usernameAliasAttributes require signInType of USERNAME'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.EMAIL,
                usernameAliasAttributes: [cognito.UserPoolAttribute.PREFERRED_USERNAME]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'usernameAliasAttributes' can only be set with a signInType of 'USERNAME'/);
        test.done();
    },
    'usernameAliasAttributes must be one or more of EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.USERNAME,
                usernameAliasAttributes: [cognito.UserPoolAttribute.GIVEN_NAME]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'usernameAliasAttributes' can only include EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME/);
        test.done();
    },
    'autoVerifiedAttributes must be one or more of EMAIL or PHONE_NUMBER'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.EMAIL,
                autoVerifiedAttributes: [cognito.UserPoolAttribute.EMAIL, cognito.UserPoolAttribute.GENDER]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'autoVerifiedAttributes' can only include EMAIL or PHONE_NUMBER/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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