"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster {
    constructor(props) {
        this.props = props;
        this.visibleToAllUsers = (this.props.visibleToAllUsers !== undefined) ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        this._serviceRole = this.props.serviceRole;
        this._clusterRole = this.props.clusterRole;
        this._autoScalingRole = this.props.autoScalingRole;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call CreateCluster.`);
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    bind(task) {
        var _a, _b;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = this._serviceRole || this.createServiceRole(task);
        this._clusterRole = this._clusterRole || this.createClusterRole(task);
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole(task);
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            policyStatements: this.createPolicyStatements(task, this._serviceRole, this._clusterRole, this._autoScalingRole),
            parameters: {
                Instances: EmrCreateCluster.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(EmrCreateCluster.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(EmrCreateCluster.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(EmrCreateCluster.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: cdk.numberToCloudFormation(this.props.ebsRootVolumeSize),
                KerberosAttributes: (this.props.kerberosAttributes === undefined) ?
                    this.props.kerberosAttributes :
                    EmrCreateCluster.KerberosAttributesPropertyToJson(this.props.kerberosAttributes),
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_b = this.props.scaleDownBehavior) === null || _b === void 0 ? void 0 : _b.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                Tags: cdk.listMapper(cdk.cfnTagToCloudFormation)(this.props.tags),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            },
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(task, serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:RunJobFlow',
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows',
                ],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [
                serviceRole.roleArn,
                clusterRole.roleArn,
            ],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    })],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole(task) {
        return new iam.Role(task, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole'),
            ],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole(task) {
        const role = new iam.Role(task, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(task, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole(task) {
        var _a;
        const role = new iam.Role(task, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole'),
            ],
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
            ],
            actions: [
                'sts:AssumeRole',
            ],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Render the EbsBlockDeviceConfigProperty as JSON
     *
     * @param property
     */
    function EbsBlockDeviceConfigPropertyToJson(property) {
        var _a;
        return {
            VolumeSpecification: {
                Iops: cdk.numberToCloudFormation(property.volumeSpecification.iops),
                SizeInGB: cdk.numberToCloudFormation(property.volumeSpecification.sizeInGB),
                VolumeType: cdk.stringToCloudFormation((_a = property.volumeSpecification.volumeType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            },
            VolumesPerInstance: cdk.numberToCloudFormation(property.volumesPerInstance),
        };
    }
    EmrCreateCluster.EbsBlockDeviceConfigPropertyToJson = EbsBlockDeviceConfigPropertyToJson;
    /**
     * Render the EbsConfigurationProperty to JSON
     *
     * @param property
     */
    function EbsConfigurationPropertyToJson(property) {
        return {
            EbsBlockDeviceConfigs: cdk.listMapper(EbsBlockDeviceConfigPropertyToJson)(property.ebsBlockDeviceConfigs),
            EbsOptimized: cdk.booleanToCloudFormation(property.ebsOptimized),
        };
    }
    EmrCreateCluster.EbsConfigurationPropertyToJson = EbsConfigurationPropertyToJson;
    /**
     * Render the InstanceTypeConfigProperty to JSON]
     *
     * @param property
     */
    function InstanceTypeConfigPropertyToJson(property) {
        var _a;
        return {
            BidPrice: cdk.stringToCloudFormation(property.bidPrice),
            BidPriceAsPercentageOfOnDemandPrice: cdk.numberToCloudFormation(property.bidPriceAsPercentageOfOnDemandPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceType: cdk.stringToCloudFormation((_a = property.instanceType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            WeightedCapacity: cdk.numberToCloudFormation(property.weightedCapacity),
        };
    }
    EmrCreateCluster.InstanceTypeConfigPropertyToJson = InstanceTypeConfigPropertyToJson;
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Render the InstanceFleetProvisioningSpecificationsProperty to JSON
     *
     * @param property
     */
    function InstanceFleetProvisioningSpecificationsPropertyToJson(property) {
        var _a;
        return {
            SpotSpecification: {
                BlockDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.blockDurationMinutes),
                TimeoutAction: cdk.stringToCloudFormation((_a = property.spotSpecification.timeoutAction) === null || _a === void 0 ? void 0 : _a.valueOf()),
                TimeoutDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.timeoutDurationMinutes),
            },
        };
    }
    EmrCreateCluster.InstanceFleetProvisioningSpecificationsPropertyToJson = InstanceFleetProvisioningSpecificationsPropertyToJson;
    /**
     * Render the InstanceFleetConfigProperty as JSON
     *
     * @param property
     */
    function InstanceFleetConfigPropertyToJson(property) {
        var _a;
        return {
            InstanceFleetType: cdk.stringToCloudFormation((_a = property.instanceFleetType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceTypeConfigs: cdk.listMapper(InstanceTypeConfigPropertyToJson)(property.instanceTypeConfigs),
            LaunchSpecifications: (property.launchSpecifications === undefined) ?
                property.launchSpecifications :
                InstanceFleetProvisioningSpecificationsPropertyToJson(property.launchSpecifications),
            Name: cdk.stringToCloudFormation(property.name),
            TargetOnDemandCapacity: cdk.numberToCloudFormation(property.targetOnDemandCapacity),
            TargetSpotCapacity: cdk.numberToCloudFormation(property.targetSpotCapacity),
        };
    }
    EmrCreateCluster.InstanceFleetConfigPropertyToJson = InstanceFleetConfigPropertyToJson;
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * Render the MetricDimensionProperty as JSON
     *
     * @param property
     */
    function MetricDimensionPropertyToJson(property) {
        return {
            Key: cdk.stringToCloudFormation(property.key),
            Value: cdk.stringToCloudFormation(property.value),
        };
    }
    EmrCreateCluster.MetricDimensionPropertyToJson = MetricDimensionPropertyToJson;
    /**
     * Render the ScalingTriggerProperty to JSON
     *
     * @param property
     */
    function ScalingTriggerPropertyToJson(property) {
        var _a, _b, _c;
        return {
            CloudWatchAlarmDefinition: {
                ComparisonOperator: cdk.stringToCloudFormation((_a = property.cloudWatchAlarmDefinition.comparisonOperator) === null || _a === void 0 ? void 0 : _a.valueOf()),
                Dimensions: cdk.listMapper(MetricDimensionPropertyToJson)(property.cloudWatchAlarmDefinition.dimensions),
                EvaluationPeriods: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.evalutionPeriods),
                MetricName: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.metricName),
                Namespace: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.namespace),
                Period: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.period.toSeconds()),
                Statistic: cdk.stringToCloudFormation((_b = property.cloudWatchAlarmDefinition.statistic) === null || _b === void 0 ? void 0 : _b.valueOf()),
                Threshold: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.threshold),
                Unit: cdk.stringToCloudFormation((_c = property.cloudWatchAlarmDefinition.unit) === null || _c === void 0 ? void 0 : _c.valueOf()),
            },
        };
    }
    EmrCreateCluster.ScalingTriggerPropertyToJson = ScalingTriggerPropertyToJson;
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
    /**
     * Render the ScalingActionPropety to JSON
     *
     * @param property
     */
    function ScalingActionPropertyToJson(property) {
        var _a;
        return {
            Market: cdk.stringToCloudFormation((_a = property.market) === null || _a === void 0 ? void 0 : _a.valueOf()),
            SimpleScalingPolicyConfiguration: {
                AdjustmentType: cdk.stringToCloudFormation(property.simpleScalingPolicyConfiguration.adjustmentType),
                CoolDown: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.coolDown),
                ScalingAdjustment: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.scalingAdjustment),
            },
        };
    }
    EmrCreateCluster.ScalingActionPropertyToJson = ScalingActionPropertyToJson;
    /**
     * Render the ScalingRuleProperty to JSON
     *
     * @param property
     */
    function ScalingRulePropertyToJson(property) {
        return {
            Action: ScalingActionPropertyToJson(property.action),
            Description: cdk.stringToCloudFormation(property.description),
            Name: cdk.stringToCloudFormation(property.name),
            Trigger: ScalingTriggerPropertyToJson(property.trigger),
        };
    }
    EmrCreateCluster.ScalingRulePropertyToJson = ScalingRulePropertyToJson;
    /**
     * Render the AutoScalingPolicyProperty to JSON
     *
     * @param property
     */
    function AutoScalingPolicyPropertyToJson(property) {
        return {
            Constraints: {
                MaxCapacity: cdk.numberToCloudFormation(property.constraints.maxCapacity),
                MinCapacity: cdk.numberToCloudFormation(property.constraints.minCapacity),
            },
            Rules: cdk.listMapper(ScalingRulePropertyToJson)(property.rules),
        };
    }
    EmrCreateCluster.AutoScalingPolicyPropertyToJson = AutoScalingPolicyPropertyToJson;
    /**
     * Render the InstanceGroupConfigProperty to JSON
     *
     * @param property
     */
    function InstanceGroupConfigPropertyToJson(property) {
        var _a, _b;
        return {
            AutoScalingPolicy: (property.autoScalingPolicy === undefined) ?
                property.autoScalingPolicy :
                AutoScalingPolicyPropertyToJson(property.autoScalingPolicy),
            BidPrice: cdk.numberToCloudFormation(property.bidPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceRole: cdk.stringToCloudFormation((_a = property.instanceRole) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceType: cdk.stringToCloudFormation(property.instanceType),
            Market: cdk.stringToCloudFormation((_b = property.market) === null || _b === void 0 ? void 0 : _b.valueOf()),
            Name: cdk.stringToCloudFormation(property.name),
        };
    }
    EmrCreateCluster.InstanceGroupConfigPropertyToJson = InstanceGroupConfigPropertyToJson;
    /**
     * Render the PlacementTypeProperty to JSON
     *
     * @param property
     */
    function PlacementTypePropertyToJson(property) {
        return {
            AvailabilityZone: cdk.stringToCloudFormation(property.availabilityZone),
            AvailabilityZones: cdk.listMapper(cdk.stringToCloudFormation)(property.availabilityZones),
        };
    }
    EmrCreateCluster.PlacementTypePropertyToJson = PlacementTypePropertyToJson;
    /**
     * Render the InstancesConfigProperty to JSON
     *
     * @param property
     */
    function InstancesConfigPropertyToJson(property) {
        return {
            AdditionalMasterSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalMasterSecurityGroups),
            AdditionalSlaveSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalSlaveSecurityGroups),
            Ec2KeyName: cdk.stringToCloudFormation(property.ec2KeyName),
            Ec2SubnetId: cdk.stringToCloudFormation(property.ec2SubnetId),
            Ec2SubnetIds: cdk.listMapper(cdk.stringToCloudFormation)(property.ec2SubnetIds),
            EmrManagedMasterSecurityGroup: cdk.stringToCloudFormation(property.emrManagedMasterSecurityGroup),
            EmrManagedSlaveSecurityGroup: cdk.stringToCloudFormation(property.emrManagedSlaveSecurityGroup),
            HadoopVersion: cdk.stringToCloudFormation(property.hadoopVersion),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceFleets: cdk.listMapper(InstanceFleetConfigPropertyToJson)(property.instanceFleets),
            InstanceGroups: cdk.listMapper(InstanceGroupConfigPropertyToJson)(property.instanceGroups),
            KeepJobFlowAliveWhenNoSteps: true,
            MasterInstanceType: cdk.stringToCloudFormation(property.masterInstanceType),
            Placement: (property.placement === undefined) ?
                property.placement :
                PlacementTypePropertyToJson(property.placement),
            ServiceAccessSecurityGroup: cdk.stringToCloudFormation(property.serviceAccessSecurityGroup),
            SlaveInstanceType: cdk.stringToCloudFormation(property.slaveInstanceType),
            TerminationProtected: cdk.booleanToCloudFormation(property.terminationProtected),
        };
    }
    EmrCreateCluster.InstancesConfigPropertyToJson = InstancesConfigPropertyToJson;
    /**
     * Render the ApplicationConfigProperty as JSON
     *
     * @param property
     */
    function ApplicationConfigPropertyToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            Args: cdk.listMapper(cdk.stringToCloudFormation)(property.args),
            Version: cdk.stringToCloudFormation(property.version),
            AdditionalInfo: cdk.objectToCloudFormation(property.additionalInfo),
        };
    }
    EmrCreateCluster.ApplicationConfigPropertyToJson = ApplicationConfigPropertyToJson;
    /**
     * Render the BootstrapActionProperty as JSON
     *
     * @param property
     */
    function BootstrapActionConfigToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            ScriptBootstrapAction: {
                Path: cdk.stringToCloudFormation(property.scriptBootstrapAction.path),
                Args: cdk.listMapper(cdk.stringToCloudFormation)(property.scriptBootstrapAction.args),
            },
        };
    }
    EmrCreateCluster.BootstrapActionConfigToJson = BootstrapActionConfigToJson;
    /**
     * Render the ConfigurationProperty as JSON
     *
     * @param property
     */
    function ConfigurationPropertyToJson(property) {
        return {
            Classification: cdk.stringToCloudFormation(property.classification),
            Properties: cdk.objectToCloudFormation(property.properties),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
        };
    }
    EmrCreateCluster.ConfigurationPropertyToJson = ConfigurationPropertyToJson;
    /**
     * Render the KerberosAttributesProperty as JSON
     *
     * @param property
     */
    function KerberosAttributesPropertyToJson(property) {
        return {
            ADDomainJoinPassword: cdk.stringToCloudFormation(property.adDomainJoinPassword),
            ADDomainJoinUser: cdk.stringToCloudFormation(property.adDomainJoinUser),
            CrossRealmTrustPrincipalPassword: cdk.stringToCloudFormation(property.crossRealmTrustPrincipalPassword),
            KdcAdminPassword: cdk.stringToCloudFormation(property.kdcAdminPassword),
            Realm: cdk.stringToCloudFormation(property.realm),
        };
    }
    EmrCreateCluster.KerberosAttributesPropertyToJson = KerberosAttributesPropertyToJson;
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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