import { Project, Task } from "projen";
import { NodePackageManager } from "projen/lib/javascript";
import { TypeScriptProject, TypeScriptProjectOptions } from "projen/lib/typescript";
import { Nx } from "./nx-types";
import { SyncpackConfig } from "./syncpack-options";
/**
 * Execute command to run based on package manager configured.
 *
 * @param packageManager package manager being used.
 * @param args args to append.
 */
export declare function buildExecutableCommand(packageManager: NodePackageManager, ...args: string[]): string;
/**
 * Workspace configurations.
 *
 * @see https://classic.yarnpkg.com/lang/en/docs/workspaces/
 */
export interface WorkspaceConfig {
    /**
     * List of package globs to exclude from hoisting in the workspace.
     *
     * @see https://classic.yarnpkg.com/blog/2018/02/15/nohoist/
     */
    readonly noHoist?: string[];
    /**
     * Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies".
     *
     * @default false
     */
    readonly disableNoHoistBundled?: boolean;
    /**
     * List of additional package globs to include in the workspace.
     *
     * All packages which are parented by the monorepo are automatically added to the workspace, but you can use this
     * property to specify any additional paths to packages which may not be managed by projen.
     */
    readonly additionalPackages?: string[];
}
/**
 * Configuration for Monorepo Upgrade Deps task.
 */
export interface MonorepoUpgradeDepsOptions {
    /**
     * Name of the task to create.
     *
     * @default upgrade-deps
     */
    readonly taskName?: string;
    /**
     * Syncpack configuration.
     *
     * No merging is performed and as such a complete syncpackConfig is required if supplied.
     *
     * @default SyncpackConfig.DEFAULT_CONFIG
     */
    readonly syncpackConfig?: SyncpackConfig;
}
/**
 * Configuration options for the NxMonorepoProject.
 */
export interface NxMonorepoProjectOptions extends TypeScriptProjectOptions {
    /**
     * Configuration for NX.
     */
    readonly nxConfig?: Nx.WorkspaceConfig;
    /**
     * Configuration for workspace.
     */
    readonly workspaceConfig?: WorkspaceConfig;
    /**
     * Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies.
     *
     * @default true
     */
    readonly monorepoUpgradeDeps?: boolean;
    /**
     * Monorepo Upgrade Deps options.
     *
     * This is only used if monorepoUpgradeDeps is true.
     *
     * @default undefined
     */
    readonly monorepoUpgradeDepsOptions?: MonorepoUpgradeDepsOptions;
}
/**
 *
 * @see https://nx.dev/packages/nx/documents/run-many#options
 */
export interface NxRunManyOptions {
    /** Task to run for affected projects */
    readonly target: string;
    /** This is the configuration to use when performing tasks on projects */
    readonly configuration?: string;
    /** Exclude certain projects from being processed */
    readonly exclude?: string;
    /**
     * Do not stop command execution after the first failed task.
     */
    readonly noBail?: boolean;
    /**
     * Defines how Nx emits outputs tasks logs
     * @default "stream"
     */
    readonly outputStyle?: "dynamic" | "static" | "stream" | "stream-without-prefixes";
    /**
     * Max number of parallel processes
     * @default 3
     */
    readonly parallel?: number;
    /** Project to run as list project names and/or patterns. */
    readonly projects?: string[];
    /** This is the name of the tasks runner configuration in nx.json */
    readonly runner?: string;
    /** Rerun the tasks even when the results are available in the cache. */
    readonly skipCache?: boolean;
    /** Ignore cycles in the task graph */
    readonly ignoreCycles?: boolean;
    /** Prints additional information about the commands (e.g. stack traces). */
    readonly verbose?: boolean;
}
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
export declare class NxMonorepoProject extends TypeScriptProject {
    private readonly implicitDependencies;
    private readonly nxConfig?;
    private readonly workspaceConfig?;
    private readonly workspacePackages;
    private readonly nxJson;
    constructor(options: NxMonorepoProjectOptions);
    /**
     * Helper to format `npx nx run-many ...` style command.
     * @param options
     */
    formatNxRunManyCommand(options: NxRunManyOptions): string;
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param force - Force unlocking task (eg: build task is locked)
     * @returns - The task that was overridden
     * @internal
     */
    protected _overrideNxBuildTask(task: Task | string | undefined, options: NxRunManyOptions, force?: boolean): Task | undefined;
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name: string, options: NxRunManyOptions): Task;
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent: Project, dependee: Project): void;
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs: string[]): void;
    private get instantiationOrderSubProjects();
    get subProjects(): Project[];
    /**
     * @inheritDoc
     */
    synth(): void;
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    private validateSubProjects;
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    private synthesizeNonNodePackageJson;
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    private updateWorkspace;
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    private wirePythonDependencies;
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    private copyStepIntoTask;
}
