#
# Copyright (c) 2021 salesforce.com, inc.
# All rights reserved.
# SPDX-License-Identifier: BSD-3-Clause
# For full license text, see the LICENSE file in the repo root or https://opensource.org/licenses/BSD-3-Clause
#
from abc import ABC
from collections import Iterator
from typing import Tuple, Optional, Any

from merlion.models.base import ModelBase
from merlion.models.forecast.base import ForecasterBase
from merlion.utils import TimeSeries


class LayerMixIn(ModelBase, ABC):
    """
    Base Interface for Implemented Layers

    This abstract class contains all of the methods that Layers should implement. Ideally, these would be generated by
    an existing mix-in.
    """

    def generate_theta(self, train_data: TimeSeries) -> Iterator:
        """
        :param train_data: Training data to use for generation of hyperparameters :math:`\theta`

        Returns an iterator of hyperparameter candidates for consideration with th underlying model.
        """
        raise NotImplementedError

    def evaluate_theta(
        self, thetas: Iterator, train_data: TimeSeries, train_config=None
    ) -> Tuple[Any, Optional[ForecasterBase], Optional[Tuple[TimeSeries, Optional[TimeSeries]]]]:
        """
        :param thetas: Iterator of the hyperparameter candidates
        :param train_data: Training data
        :param train_config: Training configuration

        Return the optimal hyperparameter, as well as optionally a model and result of the training procedure.
        """
        raise NotImplementedError

    def set_theta(self, model, theta, train_data: TimeSeries = None):
        """
        :param model: Underlying base model to which the new theta is applied
        :param theta: Hyperparameter to apply
        :param train_data: Training data (Optional)

        Sets the hyperparameter to the provided ``model``. This is used to apply the :math:`\theta` to the model, since
        this behavior is custom to every model. Oftentimes in internal implementations, ``model`` is the optimal model.
        """
        raise NotImplementedError
