import numpy as np

'''
Lebedev quadrature nodes for oder 53, 83 and 131

NOTE: The nodes and weights are currently stored as variables for ease with packaging, 
    but can be moved to text values for clarity

    To make it easier to generate the entire set of nodes and weights, they are separated by theta values
    rather than node class
        For each B node, 2 values are stored
        For each C node, 2 values are stored
        For each D node, 3 values are stored
        The entire set of nodes are generated by rotating the stored nodes around phi
'''

q53 = [ ('A1', 0.000143829419053), ('A3', 0.001125772288287), \
    ('C', 0.000494802934195, [0.7853981633974483, 1.5278138441597446], [0.060749030949022884, 1.5278534941727189]), \
    ('C', 0.000735799010913, [0.7853981633974483, 1.464869622457061], [0.14924767362180266, 1.465458946141118]), \
    ('C', 0.00088891327713, [0.7853981633974483, 1.3921013767391794], [0.25008972118398765, 1.3948880328922042]), \
    ('C', 0.000988834783892, [0.7853981633974483, 1.3121840021380575], [0.35799078039612386, 1.3204233354692447]), \
    ('C', 0.001053299681709, [0.7853981633974483, 1.2254391629113082], [0.47066274863042046, 1.2443735978067951]), \
    ('C', 0.001092778807015, [0.7853981633974483, 1.1311092630816586], [0.5870167849631627, 1.1683699063550894]), \
    ('C', 0.001114389394063, [0.7853981633974483, 1.0277390347536057], [0.7065858572901342, 1.0938340687015582]), \
    ('C', 0.001123724788052, [0.7853981633974483, 0.9133197976907225], [0.8292880366025691, 1.0222394480312291]), \
    ('C', 0.001125239325244, [0.6413186957303229, 0.7853981633974483], [0.8952586114613129, 1.0850608336014413]), \
    ('C', 0.001126153271816, [0.47885672752538766, 0.7853981633974483], [0.8449179190207093, 1.2189691285785837]), \
    ('C', 0.001130286931124, [0.29760788580744874, 0.7853981633974483], [0.8078680071347109, 1.3572222159425091]), \
    ('C', 0.001134986534364, [0.10120605958438532, 0.7853981633974483], [0.7879632020899496, 1.4991105440203898]), \
    ('B', 0.000682336792711, [0.0, 0.0], [0.12398609597449904, 1.4468102308203972]), \
    ('B', 0.000945415816045, [0.0, 0.0], [0.29849041771429907, 1.2723059090805973]), \
    ('B', 0.001074429975386, [0.0, 0.0], [0.4890363219001108, 1.0817600048947857]), \
    ('B', 0.001129300086569, [0.0, 0.0], [0.6860119025945851, 0.8847844242003113]), \
    ('D', 0.00084368845009, [1.2844717187109742, 1.509748732042571, 1.3660937679633063], [0.21313436183952542, 1.3664645581270842, 1.51102024856936]), \
    ('D', 0.001075255720449, [1.2803416901970945, 1.4152034540995053, 1.0875222928699522], [0.5010920920050308, 1.0925058149246718, 1.4327825599403752]), \
    ('D', 0.001108577236864, [0.9776871891143827, 1.1675933110934318, 1.0065826028974167], [0.6518838602885556, 1.0436519864358127, 1.224834520862354]), \
    ('D', 0.000956647532378, [1.1468354819331308, 1.4378678947221721, 1.2827632205245156], [0.3142768651838654, 1.2851677234370784, 1.443283539203218]), \
    ('D', 0.001080663250717, [1.014542082067774, 1.2673637084143072, 1.104284105302142], [0.5352454525044077, 1.122804805502996, 1.2981176846358145]), \
    ('D', 0.001126797131196, [1.3476343348058228, 1.3877829580880248, 0.8866402095306907], [0.696484527185878, 0.8948744895484233, 1.428336302598283]), \
    ('D', 0.001022568715358, [1.3959649614525338, 1.4982038892706024, 1.180246028807435], [0.39598751445278896, 1.1811735167431068, 1.503653042702088]), \
    ('D', 0.001108960267713, [1.4435718389475798, 1.4860707722813518, 0.9846798261234677], [0.5898618141028261, 0.9863345944081674, 1.5001602202818463]), \
    ('D', 0.001122790653436, [1.1382535686381714, 1.214336501366638, 0.8920997418384345], [0.7263242469589386, 0.9235876602494371, 1.2886800913690697]), \
    ('D', 0.001032401847117, [1.0668212152243122, 1.3570797872576532, 1.1958660060621467], [0.4223739182051457, 1.2036422565880658, 1.3715221801665984]), \
    ('D', 0.001107249382284, [1.1987249014408132, 1.3211748825540521, 0.9921233839168557], [0.6115683799553363, 1.0064464173809158, 1.3605194293521365]), \
    ('D', 0.00112178004852, [0.9503657220266917, 1.0558353467087092, 0.9009876553171728], [0.7718926482064131, 0.9673302475304114, 1.1532572117670186]) \
]

q83 = [ ('A1', 3.9226162707e-05), ('A2', 0.000470383175085), ('A3', 0.000467820280128), \
    ('C', 0.000143783222898, [0.7853981633974483, 1.5478880696670096], [0.03239150308108666, 1.5478940782943034]), \
    ('C', 0.000230357249358, [0.7853981633974483, 1.5128763345085945], [0.08181988712167648, 1.5129732436592165]), \
    ('C', 0.000293311075245, [0.7853981633974483, 1.4715190178813398], [0.1399416632744466, 1.472004671779117]), \
    ('C', 0.000340290599836, [0.7853981633974483, 1.4259961520819113], [0.2033707232897262, 1.4274907520431568]), \
    ('C', 0.000375913846687, [0.7853981633974485, 1.37722887996987], [0.27042586330905755, 1.380756696790698]), \
    ('C', 0.00040306384479, [0.7853981633974483, 1.3255852705884101], [0.3401472883411553, 1.3326422899916492]), \
    ('C', 0.000423659143224, [0.7853981633974485, 1.2711299669984422], [0.41194919995054524, 1.2837476636470748]), \
    ('C', 0.000439052265695, [0.7853981633974483, 1.213733953708799], [0.48546443044318155, 1.2345461570975749]), \
    ('C', 0.000450252346663, [0.7853981633974483, 1.1531280330560942], [0.5604652952593183, 1.1854456932589208]), \
    ('C', 0.000458057772778, [0.7853981633974485, 1.0889302104166294], [0.6368199315520892, 1.1368273394106678]), \
    ('C', 0.000463139161662, [0.7853981633974483, 1.0206606311355504], [0.7144669663381061, 1.0890732474827065]), \
    ('C', 0.00046609289537, [0.7853981633974485, 0.947752069132732], [0.7934001594958044, 1.0425899854623069]), \
    ('C', 0.000467475180794, [0.7853981633974483, 0.8695626029718091], [0.8736582608612342, 0.997830529591237]), \
    ('C', 0.000467641490393, [0.6945555548407819, 0.7853981633974483], [0.9156616750983065, 1.0384770374251961]), \
    ('C', 0.000467408649235, [0.5964008076543235, 0.7853981633974483], [0.8795921714811326, 1.1232513615809576]), \
    ('C', 0.000467492853948, [0.4905012468974965, 0.7853981633974485], [0.8479609818146427, 1.209732602644781]), \
    ('C', 0.000468074897969, [0.3768260219808742, 0.7853981633974483], [0.8217389797567912, 1.2979468039133677]), \
    ('C', 0.000469044980639, [0.25600458431491085, 0.7853981633974485], [0.8019618809928429, 1.3877836169245485]), \
    ('C', 0.000469987707586, [0.12956731378494293, 0.7853981633974483], [0.7896068315075573, 1.4789216292680016]), \
    ('B', 0.000209994228107, [0.0, 0.0], [0.06749155779696747, 1.5033047689979289]), \
    ('B', 0.000317226915071, [0.0, 0.0], [0.16866703436844993, 1.4021292924264468]), \
    ('B', 0.000383205135855, [0.0, 0.0], [0.2831232424524513, 1.2876730843424453]), \
    ('B', 0.000425219381815, [0.0, 0.0], [0.4044643882204781, 1.1663319385744184]), \
    ('B', 0.000451380796375, [0.0, 0.0], [0.5296866164276711, 1.0411097103672255]), \
    ('B', 0.000465779746911, [0.0, 0.0], [0.6570586147261469, 0.9137377120687498]), \
    ('D', 0.000273336280052, [0.28552278478943144, 1.5372537457925648, 1.4569795864095834], [0.11857509702215088, 1.4570430572712532, 1.537470609681566]), \
    ('D', 0.000323548536846, [0.4223004237655866, 1.4972970791518627, 1.4083711983745744], [0.17775522714760858, 1.4088020807490815, 1.498261063127842]), \
    ('D', 0.000362490872601, [0.5014191803397784, 1.4527400038626896, 1.3577017094004171], [0.24192224055375727, 1.359140917425484, 1.4553863867078463]), \
    ('D', 0.000392554007071, [0.5528406467075646, 1.4044194857749304, 1.3050709799353444], [0.30952157777571815, 1.3085733256427445, 1.4101572280886208]), \
    ('D', 0.000415612978112, [0.5889469348378037, 1.352668392657542, 1.2504241272847294], [0.37966851034406807, 1.2575232516866157, 1.3634436631366842]), \
    ('D', 0.000433064498462, [0.6157212263044634, 1.2975195177024463, 1.1935824593776465], [0.45182612766531105, 1.2063532776035815, 1.315882009861121]), \
    ('D', 0.000445967772592, [0.6363904863631938, 1.2388031234953782, 1.134268234314837], [0.5256612789131551, 1.1553932838286778, 1.2679805157564064]), \
    ('D', 0.000455159300446, [0.6528402674297308, 1.1761962275633266, 1.0721193638874666], [0.6009715720038169, 1.1049634157855215, 1.2201805261526368]), \
    ('D', 0.000461334146275, [0.6662401069510652, 1.1092482874033933, 1.0066975985424804], [0.6776453176556749, 1.0553933942656466, 1.172896458325957]), \
    ('D', 0.000465101961827, [0.6773482754674274, 1.0373959578060827, 0.9374939708060844], [0.755638328147006, 1.0070403913669377, 1.12654557272392]), \
    ('D', 0.00046702495361, [0.6866718174567418, 0.9599754579564165, 0.8639354886282324], [0.8349596682772986, 0.9603071776796986, 1.081573267260856]), \
    ('D', 0.000354955557644, [0.17319622430817022, 1.5310639589866286, 1.3473624326208797], [0.22671318351301734, 1.3475329686278446, 1.5320506127424134]), \
    ('D', 0.000385610824525, [0.2869034969249369, 1.4865782960086822, 1.2921221746064222], [0.2899083865573624, 1.2930597675418436, 1.4898128740202983]), \
    ('D', 0.000409862284576, [0.3666296100119042, 1.438000725806461, 1.2360136380356255], [0.35670268782920417, 1.2387484307511374, 1.445293906612033]), \
    ('D', 0.000428632860427, [0.42545950074999817, 1.3855566953839382, 1.1786767383416332], [0.42618379368229736, 1.1847333227233605, 1.3993313471352267]), \
    ('D', 0.000442780219899, [0.4706212783934438, 1.3291928362642662, 1.1197390779748229], [0.4977917662968449, 1.1311965178413608, 1.3525565231860692]), \
    ('D', 0.000453047351149, [0.5063768581182908, 1.2686547165128108, 1.0587968380817283], [0.5711783515164267, 1.0783539045302768, 1.305485759975292]), \
    ('D', 0.00046008054757, [0.5353769874262576, 1.2035262872991621, 0.9954054118808735], [0.6461326480688495, 1.0264550456732284, 1.2585762431270904]), \
    ('D', 0.000464459905996, [0.5593412434850699, 1.1332502260855466, 0.9290743174949707], [0.7225395334435659, 0.9757898488229266, 1.2122643884895092]), \
    ('D', 0.000466727445571, [0.5794206110746619, 1.057140396904753, 0.8592656741943057], [0.8003552711737016, 0.9266996700433338, 1.1669957465794643]), \
    ('D', 0.000406936051802, [0.1246929944514357, 1.5261738606092032, 1.2285746770470467], [0.3446930676355144, 1.2288893844480697, 1.5287583117563806]), \
    ('D', 0.000426044281992, [0.21782366814646884, 1.4771430310484712, 1.1694594983559636], [0.41000775295101166, 1.171036449983246, 1.484546332472184]), \
    ('D', 0.000440867850803, [0.2896954739585901, 1.423807259254898, 1.109792245652285], [0.47817775954113606, 1.114097625094979, 1.4389647076136611]), \
    ('D', 0.000451874811555, [0.3467152939185042, 1.3660062002471536, 1.0490695726537846], [0.5486350371663319, 1.0581451760091638, 1.392636360582717]), \
    ('D', 0.000459556487538, [0.39299377007371356, 1.303374788222269, 0.9867929630324785], [0.6210174420373237, 1.003320042386053, 1.346082173840589]), \
    ('D', 0.000464398877432, [0.4312497103392795, 1.2353724483148918, 0.9224509132763913], [0.6951020695739003, 0.9498248246709314, 1.2997725391138326]), \
    ('D', 0.000466882749165, [0.4633276271949143, 1.1612971897844546, 0.8555076877715875], [0.7707657750012719, 0.8979256987124614, 1.2541645941750355]), \
    ('D', 0.000440054182374, [0.09789691921161021, 1.521432374007353, 1.1047189017263608], [0.4680069846669097, 1.105208053199772, 1.5266904042887262]), \
    ('D', 0.000451451289019, [0.17638822981361618, 1.4672883242055235, 1.0431172514426508], [0.5345081705733076, 1.0454488116320224, 1.4812867509037024]), \
    ('D', 0.000459619862735, [0.2405304129276186, 1.4080735691888384, 0.9809470711861328], [0.6034255892551799, 0.9870783925906188, 1.4352005905389456]), \
    ('D', 0.00046486590168, [0.2938120112955257, 1.343283117012832, 0.9176464093206581], [0.6744026659869607, 0.9302020023418196, 1.388958875040808]), \
    ('D', 0.000467550201716, [0.33867024052989114, 1.2722139196078799, 0.8526542200739708], [0.7472154356526481, 0.8750015898298612, 1.3430473324610688]), \
    ('D', 0.000459849447646, [0.08102617205018207, 1.516148755763779, 0.9779832410855134], [0.594336749814104, 0.9786752813060116, 1.5254590626290043]), \
    ('D', 0.000465491695515, [0.1489773513491236, 1.4557624301250913, 0.9147214011021235], [0.6614664589232284, 0.9179238999549367, 1.479494650935372]), \
    ('D', 0.000468470977951, [0.20662810572099238, 1.3889501377507252, 0.8507245479199286], [0.7307449077812772, 0.8589573819517202, 1.4334353575224716]), \
    ('D', 0.000469144553911, [0.06948714856385513, 1.5097001114975335, 0.8497487095533518], [0.7222458989391731, 0.8506746432841582, 1.5248811805657625]) \
]

q131 = [('A1', 9.735347946e-06), ('A2', 0.00019075812418), ('A3', 0.000190105954674), \
    ('C', 3.9264245389e-05, [0.7853981633974485, 1.5589716080645184], [0.016721898276516087, 1.5589724346674132]), \
    ('C', 6.6679054673e-05, [0.7853981633974483, 1.54015571589333], [0.043318816851264395, 1.5401700891975842]), \
    ('C', 8.868891315e-05, [0.7853981633974485, 1.517397149727933], [0.07544622473625402, 1.5174731204683403]), \
    ('C', 0.000106630600096, [0.7853981633974485, 1.4919903063712159], [0.11121896971875246, 1.4922338810754578]), \
    ('C', 0.000121450674334, [0.7853981633974485, 1.4646025897395305], [0.14962112971671765, 1.4651963566630974]), \
    ('C', 0.000133805468164, [0.7853981633974485, 1.4356198935538746], [0.19002070111693736, 1.4368382654087468]), \
    ('C', 0.000144167702363, [0.7853981633974483, 1.4052724128227794], [0.2319940331069521, 1.4074944900582882]), \
    ('C', 0.000152888020083, [0.7853981633974483, 1.3736935977271814], [0.2752439903085378, 1.3774145586684607]), \
    ('C', 0.000160233062377, [0.7853981633974483, 1.3409519164606218], [0.3195559178224702, 1.3467936634191788]), \
    ('C', 0.000166410265345, [0.7853981633974485, 1.3070694432688825], [0.3647717440592329, 1.315791074298251]), \
    ('C', 0.000171584585401, [0.7853981633974485, 1.2720333162602593], [0.410773800800993, 1.284541867674585]), \
    ('C', 0.000175890100013, [0.7853981633974485, 1.2358030369006023], [0.4574742604348293, 1.2531648475086097]), \
    ('C', 0.000179438248526, [0.7853981633974483, 1.198315214995792], [0.5048080093477555, 1.2217681902040156]), \
    ('C', 0.000182323810676, [0.7853981633974485, 1.1594866933520822], [0.5527277175375845, 1.1904536863469675]), \
    ('C', 0.000184629325296, [0.7853981633974483, 1.119216638448027], [0.60120036055839, 1.1593201070964045]), \
    ('C', 0.000186428407932, [0.7853981633974483, 1.0773879988953163], [0.6502047257565111, 1.1284660307840553]), \
    ('C', 0.000187788269463, [0.7853981633974485, 1.0338686402180965], [0.6997295936598458, 1.0979923528262998]), \
    ('C', 0.000188771632185, [0.7853981633974485, 0.9885124284513214], [0.74977237792139, 1.068004632494899]), \
    ('C', 0.000189438163818, [0.7853981633974483, 0.9411605401941001], [0.8003380591758863, 1.0386153832616587]), \
    ('C', 0.000189845489953, [0.7853981633974485, 0.8916433160130134], [0.8514382728630202, 1.0099463767834254]), \
    ('C', 0.000190049792958, [0.7853981633974483, 0.83978304452945], [0.903090415187696, 0.9821309946823449]), \
    ('C', 0.000190067150192, [0.7283094832440615, 0.7853981633974483], [0.9296669825952297, 1.0081424157080632]), \
    ('C', 0.000189983755553, [0.6683491615917139, 0.7853981633974483], [0.9053643318376159, 1.06159488315965]), \
    ('C', 0.000189901411316, [0.6053732308002467, 0.7853981633974483], [0.8826110666759377, 1.115699986579379]), \
    ('C', 0.000189858125771, [0.5392784285306611, 0.7853981633974485], [0.8616303897462702, 1.1704788551497591]), \
    ('C', 0.00018988047561, [0.4700237379927424, 0.7853981633974483], [0.842665197806536, 1.2259427230862325]), \
    ('C', 0.000189979361043, [0.3976561938489771, 0.7853981633974483], [0.8259742078953178, 1.282086458602126]), \
    ('C', 0.000190146455484, [0.3223389032537266, 0.7853981633974485], [0.8118241352885889, 1.3388810005729874]), \
    ('C', 0.000190353324626, [0.24437679048821093, 0.7853981633974483], [0.8004768948496634, 1.396265734004105]), \
    ('C', 0.000190555615846, [0.16423276533383752, 0.7853981633974483], [0.7921715892823792, 1.454142771779016]), \
    ('C', 0.000190703715566, [0.08252529088338643, 0.7853981633974485], [0.7871027020996212, 1.5123758528046696]), \
    ('B', 5.9929978442e-05, [0.0, 0.0], [0.035532179427368187, 1.535264147367527]), \
    ('B', 9.7490593825e-05, [0.0, 0.0], [0.09163997602850539, 1.4791563507663912]), \
    ('B', 0.00012416808046, [0.0, 0.0], [0.15726727357196196, 1.4135290532229348]), \
    ('B', 0.00014376261543, [0.0, 0.0], [0.22853079239748592, 1.3422655343974108]), \
    ('B', 0.000158420005479, [0.0, 0.0], [0.30346035257245174, 1.2673359742224448]), \
    ('B', 0.000169443655098, [0.0, 0.0], [0.3808915167430048, 1.1899048100518919]), \
    ('B', 0.000177661701402, [0.0, 0.0], [0.4600684594383822, 1.1107278673565142]), \
    ('B', 0.000183613243444, [0.0, 0.0], [0.5404637627373676, 1.030332564057529]), \
    ('B', 0.000187649472708, [0.0, 0.0], [0.6216845658039938, 0.9491117609909027]), \
    ('B', 0.000189990653534, [0.0, 0.0], [0.7034183231763038, 0.8673780036185929]), \
    ('D', 8.1432528208e-05, [0.28528089728965045, 1.552883774738187, 1.5097891029448185], [0.06357005366123927, 1.5097988657970174, 1.5529170914896095]), \
    ('D', 9.9988598909e-05, [0.42176866068784985, 1.5310925978927172, 1.4824930311281053], [0.09673439409905298, 1.4825622612615263, 1.5312471301200548]), \
    ('D', 0.000115619940307, [0.5005666001870127, 1.5065328912210598, 1.4536973443039403], [0.1332932951508847, 1.4539368613360937, 1.5069717853308484]), \
    ('D', 0.000128763209264, [0.5516471573557709, 1.4798384296541829, 1.4236494709241767], [0.17230814418587248, 1.4242490600688076, 1.4808160476535617]), \
    ('D', 0.000139837864337, [0.5874029436433695, 1.4513900961611523, 1.3925101563971825], [0.21319639381983396, 1.3937531826868128, 1.4532652757623719]), \
    ('D', 0.000149187646842, [0.6138283006176445, 1.421418878983217, 1.3603736367737282], [0.25556815399885396, 1.362649462388686, 1.4246665200525563]), \
    ('D', 0.000157085567918, [0.6341620537909199, 1.3900594033177736, 1.3272849660148256], [0.2991501039439629, 1.3311000448035588, 1.3952811910581586]), \
    ('D', 0.00016374839481, [0.650302982167086, 1.3573801920540185, 1.2932521889904653], [0.3437448477848595, 1.2992405274149346, 1.365315079228693]), \
    ('D', 0.000169350056663, [0.6634359981251888, 1.3234018078572505, 1.2582545950940849], [0.38920714635412124, 1.2671881156711942, 1.3349367363716949]), \
    ('D', 0.000174032276939, [0.6743378354312282, 1.2881081812752508, 1.2222482763064413], [0.4354291158359261, 1.2350475022226581, 1.3042894193919732]), \
    ('D', 0.000177912663728, [0.683538255619441, 1.251453876310185, 1.185169862422972], [0.4823305807349093, 1.202915253077608, 1.2734992702179382]), \
    ('D', 0.000181090810884, [0.6914105037220578, 1.2133688292334217, 1.1469390251858882], [0.529852569688363, 1.170883222450201, 1.2426812057441285]), \
    ('D', 0.00018365291326, [0.6982247152962389, 1.1737614784397898, 1.1074601595181532], [0.5779528164390484, 1.1390413439480767, 1.211943379135568]), \
    ('D', 0.000185675284178, [0.7041808155310658, 1.1325208761470529, 1.0666235393667354], [0.6266025847511608, 1.1074800349266574, 1.1813907422299006]), \
    ('D', 0.000187227056661, [0.7094295229470299, 1.0895181998354677, 1.0243061855250286], [0.6757843877666295, 1.076292380246558, 1.1511280500151928]), \
    ('D', 0.000188372264559, [0.7140861714951358, 1.0446079980936671, 0.9803726592136832], [0.7254903157900778, 1.0455762154571198, 1.1212625354734114]), \
    ('D', 0.000189171432453, [0.7182400435030037, 0.9976294810535477, 0.9346760007281719], [0.775720768448725, 1.0154361970497185, 1.0919064117292525]), \
    ('D', 0.000189682748045, [0.7219608092662064, 0.9484081853815253, 0.8870590640196887], [0.8264834310841535, 0.9859859212898239, 1.06317930867928]), \
    ('D', 0.000189962841706, [0.7253030504044619, 0.8967584015462655, 0.8373565550006887], [0.8777923529934422, 0.9573501267829395, 1.035210710231188]), \
    ('D', 0.0001123301829, [0.17266241865648854, 1.549267966133977, 1.4479557086339208], [0.12467571016551615, 1.4479838886269978, 1.549430141945543]), \
    ('D', 0.000125369882671, [0.2856999389043986, 1.5250259063276514, 1.4161113438201995], [0.16110982060532605, 1.4162707749024686, 1.5255716505876191]), \
    ('D', 0.000136626611768, [0.36469292356814437, 1.4986156709516387, 1.383616068559134], [0.20001944041062508, 1.3840922038276988, 1.4998721991682349]), \
    ('D', 0.000146273685611, [0.4227677120224608, 1.470375228449944, 1.3504693560028185], [0.24081890351977522, 1.3515440378815786, 1.4727870975960229]), \
    ('D', 0.000154507646669, [0.46718316454381953, 1.4405117091709478, 1.3166590674765635], [0.28311458762132174, 1.3187224469746683, 1.4446514055339477]), \
    ('D', 0.000161509628081, [0.5022293799903303, 1.4091443691801957, 1.2821572253867255], [0.32663092128437565, 1.2857184808469748, 1.4157226296460328]), \
    ('D', 0.000167436663974, [0.5305868458998894, 1.3763301141424902, 1.2469200649374022], [0.3711694750360288, 1.2526175949250764, 1.3862064926073776]), \
    ('D', 0.000172422500244, [0.554009226621664, 1.3420793107857993, 1.2108891787909295], [0.4165847626807598, 1.219500784939292, 1.3562733886580156]), \
    ('D', 0.000176581082299, [0.5736885179132251, 1.306365866394875, 1.173992724139619], [0.46276914245823575, 1.1864461833132791, 1.3260694244809177]), \
    ('D', 0.000180010412601, [0.5904612289166222, 1.2691337702884764, 1.1361463973272359], [0.509642994799035, 1.153530735732767, 1.2957241771981889]), \
    ('D', 0.000182796043733, [0.6049303578115357, 1.2303013831585663, 1.0972541385256736], [0.55714812151689, 1.120831851907304, 1.2653564019352286]), \
    ('D', 0.000185014030072, [0.6175404979149882, 1.1897642772299972, 1.0572086217453984], [0.6052431953592655, 1.0884290251903883, 1.2350784368933654]), \
    ('D', 0.000186733350739, [0.6286257039864557, 1.1473971690049443, 1.0158916232799677], [0.6539005570438571, 1.0564054513294068, 1.204999780081923]), \
    ('D', 0.000188017868864, [0.6384409166642313, 1.1030553505222216, 0.973174385391849], [0.7031039160573992, 1.0248496871596215, 1.1752301503952676]), \
    ('D', 0.000188927892565, [0.6471831256419169, 1.0565759675781288, 0.9289181183629428], [0.7528466586283982, 0.9938573901396459, 1.1458822460577331]), \
    ('D', 0.000189521383251, [0.6550059355509634, 1.007779491516958, 0.8829748210862775], [0.803130549143475, 0.9635331738649967, 1.1170743477811444]), \
    ('D', 0.00018985482774, [0.6620297754245246, 0.956471774786418, 0.8351886531633449], [0.853964654012843, 0.9339926031223869, 1.0889328653883972]), \
    ('D', 0.000134910593594, [0.12381742946312652, 1.5466383302211417, 1.3790325136090689], [0.1932067962321366, 1.3790871072801147, 1.5470809884936143]), \
    ('D', 0.000144406006837, [0.2158839912277757, 1.5203108453351415, 1.3443422542420214], [0.231647204454199, 1.344621038611322, 1.5215977010194737]), \
    ('D', 0.000152679739093, [0.2866058251193055, 1.4921046776329483, 1.3093627056354324], [0.2720211091038408, 1.3101355323968054, 1.4947681141807112]), \
    ('D', 0.000159820877141, [0.34246307929301384, 1.4621974908503266, 1.2740093897161358], [0.3139359629026187, 1.275657733708712, 1.4669104584963188]), \
    ('D', 0.000165935436862, [0.387626505145403, 1.4306877861310008, 1.2381999450091916], [0.35711258120972883, 1.2412271704344024, 1.4382739115054768]), \
    ('D', 0.000171127991095, [0.42487223376730787, 1.3976160058168112, 1.2018495597693406], [0.40134818046377285, 1.2068907695392799, 1.4090604636992246]), \
    ('D', 0.00017549527256, [0.4561064202418243, 1.3629779629326553, 1.1648685951579143], [0.4464933766681586, 1.172700570888163, 1.3794394209006224]), \
    ('D', 0.00017912478508, [0.4826746121203363, 1.3267335310223791, 1.1271612693506858], [0.49243737927669706, 1.1387128883133693, 1.349557426416046]), \
    ('D', 0.000182095430088, [0.505549949823486, 1.288812314705263, 1.0886248546093091], [0.5390981709107115, 1.104988282947561, 1.3195456791664273]), \
    ('D', 0.000184478852455, [0.5254512032391345, 1.2491173596599185, 1.0491491346702801], [0.5864158266615432, 1.0715920281838782, 1.2895253614549205]), \
    ('D', 0.000186340948171, [0.5429189324032369, 1.20752759487903, 1.0086160240773405], [0.6343478714751047, 1.0385949069530898, 1.259611913593006]), \
    ('D', 0.00018774330088, [0.5583659079061154, 1.1638995010910391, 0.966899339421262], [0.6828659926886614, 1.006074266326304, 1.2299185711291327]), \
    ('D', 0.000188744454371, [0.5721112635774959, 1.118068402148222, 0.9238647728347347], [0.7319536656827956, 0.9741152989184014, 1.20055944531049]), \
    ('D', 0.000189400982938, [0.584404098664016, 1.069849748471518, 0.8793701705976901], [0.7816043904317572, 0.9428125428626676, 1.1716523410793587]), \
    ('D', 0.000189768334504, [0.5954400711994067, 1.0190407881499786, 0.8332662760174524], [0.8318203156742462, 0.9122715999345323, 1.1433214473045534]), \
    ('D', 0.000151732703747, [0.09665608555276219, 1.544473602865386, 1.3056422758605846], [0.2663396290374038, 1.3057298881473507, 1.5453931228081539]), \
    ('D', 0.000158774055748, [0.1736982211176252, 1.5162303894173659, 1.269018311874361], [0.3061077092288124, 1.2694407420357738, 1.5186916833801642]), \
    ('D', 0.000164909338227, [0.23630873199145053, 1.4862137110583216, 1.23226504970528], [0.34745744138422013, 1.2333853637636416, 1.4909933646902418]), \
    ('D', 0.000170191521619, [0.28808212925697174, 1.45449941975948, 1.1952630632425676], [0.39011333655504815, 1.1975699792154266, 1.4625383908779033]), \
    ('D', 0.000174684775314, [0.33155413355988045, 1.4211092350502634, 1.1578973044693799], [0.43387175696232544, 1.162014654710549, 1.4335242304707025]), \
    ('D', 0.000178455551201, [0.36854755202718026, 1.3860219105664247, 1.120053777850822], [0.4785808375181585, 1.1267501646682418, 1.4041183171577007]), \
    ('D', 0.000181568756211, [0.40039725953090355, 1.349180555258331, 1.0816172009573128], [0.5241269564372015, 1.0918162312557165, 1.3744671256984977]), \
    ('D', 0.000184086437066, [0.4280978739906751, 1.3104974094883595, 1.0424693102966598], [0.5704254617692405, 1.0572607024803709, 1.3447028795873717]), \
    ('D', 0.000186067678539, [0.4524021377476342, 1.2698569316335488, 1.0024876049454454], [0.6174142092727561, 1.0231393914796745, 1.31494871004482]), \
    ('D', 0.000187569058374, [0.4738875687313485, 1.2271177911420048, 0.9615444217464061], [0.6650489897676893, 0.989516408574628, 1.285322804471092]), \
    ('D', 0.000188645323635, [0.4930023819988209, 1.182114224087966, 0.919506311590908], [0.7133002437506023, 0.9564648881723246, 1.2559419073721458]), \
    ('D', 0.000189350112333, [0.5100976349114337, 1.134657149807458, 0.8762337487381324], [0.7621506560402752, 0.9240680564122605, 1.22692442511583]), \
    ('D', 0.000189736618452, [0.5254500517522904, 1.0845354531341633, 0.8315812647151984], [0.8115933403298019, 0.8924206104054287, 1.1983933113284246]), \
    ('D', 0.000164390881515, [0.07942311186836311, 1.5425118437134293, 1.2292570093063981], [0.3425369680260213, 1.229383245225742, 1.5441447524208605]), \
    ('D', 0.000169630035091, [0.14557661999624696, 1.5123809567852233, 1.1912523342127797], [0.38321756146885916, 1.191839451691642, 1.5165296953767056]), \
    ('D', 0.000174155310384, [0.2013885231417953, 1.4804552886981464, 1.1531791319167313], [0.4252382095603029, 1.154691961355802, 1.4881824577209477]), \
    ('D', 0.000178001528239, [0.24903261796457504, 1.4467359350400875, 1.1149014796274102], [0.46839970074070003, 1.117944134140616, 1.4592959169258013]), \
    ('D', 0.000181211678708, [0.2901379857017055, 1.4111823859459156, 1.0762874145224177], [0.5125510769745851, 1.0816126654316527, 1.4300355458531526]), \
    ('D', 0.000183832315809, [0.32593911470891007, 1.3737184375489095, 1.037206516049897], [0.5575779942850255, 1.0457258508595373, 1.4005475730229606]), \
    ('D', 0.000185911311984, [0.3573821432395365, 1.3342360131813709, 0.9975279574283168], [0.603394441486093, 1.0103226205134352, 1.370965197395697]), \
    ('D', 0.000187496922022, [0.3852000261056879, 1.2925975583247686, 0.9571189574545905], [0.6499367991102107, 0.9754521981048231, 1.3414134931638477]), \
    ('D', 0.000188637561268, [0.40996601063754107, 1.248637514161606, 0.9158435892931192], [0.6971595367222118, 0.9411742762068068, 1.31201344670209]), \
    ('D', 0.000189381957581, [0.43213199055998164, 1.2021632950008492, 0.8735619439544509], [0.7450320603112583, 0.9075596342513634, 1.2828854373148877]), \
    ('D', 0.000189779474826, [0.4520562562099885, 1.1529561815323695, 0.8301296945295876], [0.7935363604312111, 0.874691153245738, 1.2541523847189315]), \
    ('D', 0.000173896392658, [0.06755547353493631, 1.5405911398668541, 1.150796587766159], [0.420850623302345, 1.1509664320588064, 1.5432149069953887]), \
    ('D', 0.000177744235987, [0.1255656446761502, 1.5084906273030798, 1.1117758883130644], [0.4621675542166031, 1.1125469105418553, 1.514925896343309]), \
    ('D', 0.000181001081507, [0.1758315971478302, 1.4744729282846745, 1.0726932611561621], [0.5046522011268062, 1.0746408791358795, 1.4861174436848785]), \
    ('D', 0.000183692031825, [0.21975346428222917, 1.438475644556984, 1.0334041829887086], [0.5481580996274673, 1.037257060458142, 1.456952994087191]), \
    ('D', 0.000185848947321, [0.25842538312345836, 1.4004002650584244, 0.9937670187128818], [0.5925728314832822, 1.0004170020936405, 1.427578540203268]), \
    ('D', 0.00018750793425, [0.29270784258129445, 1.3601149230685505, 0.9536411901021968], [0.6378109464823318, 0.96415434382935, 1.3981283369064235]), \
    ('D', 0.00018870802391, [0.3232819651604262, 1.3174559806436212, 0.9128856150046349], [0.6838087342082747, 0.928514594398557, 1.368729548265241]), \
    ('D', 0.000189490575218, [0.3506900107307, 1.272228862952002, 0.8713574510646377], [0.7305203448580754, 0.8935553805415252, 1.339506176891]), \
    ('D', 0.00018989910612, [0.3753655832625733, 1.2242085453729108, 0.828911187812836], [0.7779148856961775, 0.8593471360919289, 1.310582536063838]), \
    ('D', 0.000180906501646, [0.058912127758296566, 1.5385929419539042, 1.0708825766333525], [0.5006445449864412, 1.0711007231535026, 1.5425316215526859]), \
    ('D', 0.00018362971216, [0.11064674419581752, 1.5043437078314195, 1.0311031260229666], [0.54240112975283, 1.0320766860598933, 1.513766704578511]), \
    ('D', 0.000185842691624, [0.15637972572648412, 1.4679610948870088, 0.9912367164920839], [0.5851991839305968, 0.9936613058709521, 1.4846627368934764]), \
    ('D', 0.000187565410113, [0.19705499901654258, 1.4293205720957654, 0.9511338120764631], [0.6289306821295411, 0.9558733219607902, 1.4553653440623975]), \
    ('D', 0.000188824075183, [0.23343206884014583, 1.3882621687492853, 0.9106465879164434], [0.6735127337945295, 0.9187443267249282, 1.4260098283051452]), \
    ('D', 0.000189649738387, [0.2661237528969422, 1.3445910844127709, 0.8696275058329479], [0.7188831339751454, 0.8823188242915053, 1.3967255496360458]), \
    ('D', 0.000190077553022, [0.2956256444583723, 1.2980777054957955, 0.8279281145910752], [0.7649969883820036, 0.8466547040325073, 1.3676397453355753]), \
    ('D', 0.000185852504148, [0.052354203178164936, 1.536415304506939, 0.9899638886286268], [0.5814615824448407, 0.9902350341143454, 1.542050170375683]), \
    ('D', 0.000187624869008, [0.09912510093568072, 1.4997402083129125, 0.9496187203628887], [0.6235084054866246, 0.950814010310329, 1.5129810019093293]), \
    ('D', 0.000188940443906, [0.14111309990862095, 1.4606160233319878, 0.9091423042508114], [0.666502524843189, 0.9120885175541108, 1.483733757293008]), \
    ('D', 0.000189816853927, [0.17897489989919244, 1.418849378994623, 0.8683817135316415], [0.7103653438059226, 0.8740908198858901, 1.4544446048289765]), \
    ('D', 0.000190277994066, [0.21324905613015105, 1.3742079761014951, 0.827184611954067], [0.7550376770339869, 0.8368668302404735, 1.4252450668752643]), \
    ('D', 0.000189012564173, [0.04721844302802548, 1.5339564539800772, 0.9083862717504001], [0.6629509471710564, 0.9087154003228725, 1.5417426633033782]), \
    ('D', 0.00018994346378, [0.08997265710953405, 1.4944671446274873, 0.8676295668072698], [0.7051665794239899, 0.8690675011500809, 1.512525264138776]), \
    ('D', 0.000190452085683, [0.12882242002665842, 1.4520953569378034, 0.8266854191783667], [0.7482584375378902, 0.8302031182301477, 1.4832808075024158]), \
    ('D', 0.000190553449873, [0.043090374834379094, 1.5311012100198818, 0.8264346362389648], [0.7448244839219743, 0.8268273262435484, 1.541592714558998]) \
]

def loadPoints(order):
    if order == 53:
        node = q53
    elif order == 83:
        node = q83
    else:
        node = q131

    phi = []
    theta = []
    weights = []
    for i in range(len(node)):
        #A1 - (1, 0, 0) and corresponding points on sphere
        if node[i][0] == 'A1':
            phi = np.concatenate((phi, [0, np.pi/2, np.pi, 3*np.pi/2, 0, 0]))
            theta = np.concatenate((theta, [np.pi/2, np.pi/2, np.pi/2, np.pi/2, 0, np.pi]))
            w = [node[i][1] for n in range(6)]
            weights = np.concatenate((weights, w))

        #A2 - (sqrt(2), sqrt(2), 0)
        elif node[i][0] == 'A2':
            s1 = np.array([1, 3, 5, 7])
            s2 = np.array([1, 2, 3, 4])
            phi = np.concatenate((phi, s1*np.pi/4, s2*np.pi/2, s2*np.pi/2))

            s1 = np.array([1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3])
            theta = np.concatenate((theta, s1*np.pi/4))

            w = [node[i][1] for n in range(12)]
            weights = np.concatenate((weights, w))

        #A3 - (sqrt(3), sqrt(3), sqrt(3))
        elif node[i][0] == 'A3':
            s1 = np.array([1, 3, 5, 7])
            phi = np.concatenate((phi, s1*np.pi/4, s1*np.pi/4))

            s1 = np.array([1, 1, 1, 1, 3, 3, 3, 3])
            theta = np.concatenate((theta, s1*np.pi/4))

            w = [node[i][1] for n in range(8)]
            weights = np.concatenate((weights, w))

        #B - (0, i, j)
        elif node[i][0] == 'B':
            p = np.array(node[i][2])
            t = np.array(node[i][3])

            #Rotate along phi every 90 degrees and add points for theta = 90
            op = np.array([0, np.pi/2, np.pi, 3*np.pi/2])
            ot = np.array([0, 0, 0, 0])
            phi = np.concatenate((phi, p[0]+op, p[1]+op, p[0]+op, \
                                                p[1]+op, t[0]+op, t[1]+op))
            theta = np.concatenate((theta, t[0]+ot, t[1]+ot, np.pi-t[0]+ot, \
                                                np.pi-t[1]+ot, np.pi/2+ot, np.pi/2+ot))

            w = [node[i][1] for n in range(24)]
            weights = np.concatenate((weights, w))

        #C - (i, i, j)
        elif node[i][0] == 'C':
            p = np.array(node[i][2])
            t = np.array(node[i][3])

            #Upper and lower points
            u = np.argmin(t)
            l = np.argmax(t)

            #Rotate points along phi every 90 degrees
            #For point at lower z coord (higher theta), repeat for phi = 90-phi
            op = np.array([0, np.pi/2, np.pi, 3*np.pi/2])
            ot = np.array([0, 0, 0, 0])
            phi = np.concatenate((phi, p[u]+op, p[l]+op, (np.pi/2-p[1])+op, \
                                                p[u]+op, p[l]+op, (np.pi/2-p[1])+op))
            theta = np.concatenate((theta, t[u]+ot, t[l]+ot, t[l]+ot, \
                                                np.pi-t[u]+ot, np.pi-t[l]+ot, np.pi-t[l]+ot))

            w = [node[i][1] for n in range(24)]
            weights = np.concatenate((weights, w))

        #D - (i, j, k)
        elif node[i][0] == 'D':
            p = np.array(node[i][2])
            t = np.array(node[i][3])

            u = np.argmin(t)
            l = np.delete([0, 1, 2], u)

            #Rotate points along phi every 90 degrees
            #For the lower 2 points (higher theta), repeat for phi = 90-phi
            op = np.array([0, np.pi/2, np.pi, 3*np.pi/2])
            ot = np.array([0, 0, 0, 0])
            phi = np.concatenate((phi, p[u]+op, p[l[0]]+op, p[l[1]]+op, \
                                                np.pi/2-p[u]+op, np.pi/2-p[l[0]]+op, np.pi/2-p[l[1]]+op, \
                                                p[u]+op, p[l[0]]+op, p[l[1]]+op, \
                                                np.pi/2-p[u]+op, np.pi/2-p[l[0]]+op, np.pi/2-p[l[1]]+op))
            theta = np.concatenate((theta, t[u]+ot, t[l[0]]+ot, t[l[1]]+ot, \
                                                t[u]+ot, t[l[0]]+ot, t[l[1]]+ot, \
                                                np.pi-t[u]+ot, np.pi-t[l[0]]+ot, np.pi-t[l[1]]+ot, \
                                                np.pi-t[u]+ot, np.pi-t[l[0]]+ot, np.pi-t[l[1]]+ot))

            w = [node[i][1] for n in range(48)]
            weights = np.concatenate((weights, w))

    return phi, theta, weights


            
