from dataclasses import dataclass
from pollination_dsl.function import Inputs, Outputs, Function, command


@dataclass
class PmvMap(Function):
    """Get CSV files with maps of PMV comfort from EnergyPlus and Radiance results."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus and contains '
        'hourly or sub-hourly thermal comfort results.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    enclosure_info = Inputs.file(
        description='A JSON file containing information about the radiant '
        'enclosure that sensor points belong to.', path='enclosure_info.json',
        extensions=['json']
    )

    epw = Inputs.file(
        description='Weather file used to estimate conditions for any outdoor '
        'sensors and to compute sun positions.', path='weather.epw', extensions=['epw']
    )

    total_irradiance = Inputs.file(
        description='A Radiance .ill containing total irradiance for each sensor in '
        'the enclosure-info.', path='total.ill', extensions=['ill', 'irr']
    )

    direct_irradiance = Inputs.file(
        description='A Radiance .ill containing direct irradiance for each sensor in '
        'the enclosure-info.', path='direct.ill', extensions=['ill', 'irr']
    )

    ref_irradiance = Inputs.file(
        description='A Radiance .ill containing ground-reflected irradiance for each '
        'sensor in the enclosure-info.', path='ref.ill', extensions=['ill', 'irr']
    )

    sun_up_hours = Inputs.file(
        description='A sun-up-hours.txt file output by Radiance and aligns with the '
        'input irradiance files.', path='sun-up-hours.txt'
    )

    air_speed = Inputs.str(
        description='A single number for air speed in m/s or a string of a JSON array '
        'with numbers that align with the result-sql reporting period. This '
        'will be used for all indoor comfort evaluation.', default='0.1'
    )

    met_rate = Inputs.str(
        description='A single number for metabolic rate in met or a string of a '
        'JSON array with numbers that align with the result-sql reporting period.',
        default='1.1'
    )

    clo_value = Inputs.str(
        description='A single number for clothing level in clo or a string of a JSON '
        'array with numbers that align with the result-sql reporting period.',
        default='0.7'
    )

    solarcal_par = Inputs.str(
        description='A SolarCalParameter string to customize the assumptions of '
        'the SolarCal model.', default='SolarCalParameter: [posture:seated] '
        '[sharp:135] [absorptivity:0.7] [emissivity:0.95]'
    )

    comfort_par = Inputs.str(
        description='A PMVParameter string to customize the assumptions of '
        'the PMV comfort model.', default='PMVParameter: [ppd_threshold:10]'
    )

    @command
    def run_pmv_map(self):
        return 'ladybug-comfort map pmv result.sql enclosure_info.json ' \
            'weather.epw --total-irradiance total.ill --direct-irradiance direct.ill ' \
            '--ref-irradiance ref.ill --sun-up-hours sun-up-hours.txt ' \
            '--air-speed {{self.air_speed}} --met-rate {{self.met_rate}} ' \
            '--clo-value {{self.clo_value}} --solarcal-par {{self.solarcal_par}} ' \
            '--comfort-par {{self.comfort_par}} --folder output'

    result_folder = Outputs.folder(
        description='Folder containing all of the output CSV files.', path='output'
    )

    temperature_map = Outputs.file(
        description='CSV file containing a map of Standard Effective Temperature '
        '(SET) for each sensor and step of the analysis.', path='output/temperature.csv'
    )

    condition_map = Outputs.file(
        description='CSV file containing a map of comfort conditions for each '
        'sensor and step of the analysis. -1 indicates unacceptably cold conditions. '
        '+1 indicates unacceptably hot conditions. 0 indicates neutral (comfortable) '
        'conditions.', path='output/condition.csv'
    )

    pmv_map = Outputs.file(
        description='CSV file containing the Predicted Mean Vote (PMV) for each '
        'sensor and step of the analysis. This can be used to understand not just '
        'whether conditions are acceptable but how uncomfortably hot or cold they are.',
        path='output/condition_intensity.csv'
    )


@dataclass
class AdaptiveMap(Function):
    """Get CSV files with maps of Adaptive comfort from EnergyPlus and Radiance results."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus and contains '
        'hourly or sub-hourly thermal comfort results.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    enclosure_info = Inputs.file(
        description='A JSON file containing information about the radiant '
        'enclosure that sensor points belong to.', path='enclosure_info.json',
        extensions=['json']
    )

    epw = Inputs.file(
        description='Weather file used to estimate conditions for any outdoor '
        'sensors and to provide prevailing outdoor temperature for the adaptive '
        'comfort model.', path='weather.epw', extensions=['epw']
    )

    total_irradiance = Inputs.file(
        description='A Radiance .ill containing total irradiance for each sensor in '
        'the enclosure-info.', path='total.ill', extensions=['ill', 'irr']
    )

    direct_irradiance = Inputs.file(
        description='A Radiance .ill containing direct irradiance for each sensor in '
        'the enclosure-info.', path='direct.ill', extensions=['ill', 'irr']
    )

    ref_irradiance = Inputs.file(
        description='A Radiance .ill containing ground-reflected irradiance for each '
        'sensor in the enclosure-info.', path='ref.ill', extensions=['ill', 'irr']
    )

    sun_up_hours = Inputs.file(
        description='A sun-up-hours.txt file output by Radiance and aligns with the '
        'input irradiance files.', path='sun-up-hours.txt'
    )

    air_speed = Inputs.str(
        description='A single number for air speed in m/s or a string of a JSON array '
        'with numbers that align with the result-sql reporting period. This '
        'will be used for all indoor comfort evaluation.', default='0.1'
    )

    solarcal_par = Inputs.str(
        description='A SolarCalParameter string to customize the assumptions of '
        'the SolarCal model.', default='SolarCalParameter: [posture:seated] '
        '[sharp:135] [absorptivity:0.7] [emissivity:0.95]'
    )

    comfort_par = Inputs.str(
        description='An AdaptiveParameter string to customize the assumptions of '
        'the Adaptive comfort model.', default='AdaptiveParameter: [standard:ASHRAE-55]'
    )

    @command
    def run_adaptive_map(self):
        return 'ladybug-comfort map adaptive result.sql enclosure_info.json ' \
            'weather.epw --total-irradiance total.ill --direct-irradiance direct.ill ' \
            '--ref-irradiance ref.ill --sun-up-hours sun-up-hours.txt ' \
            '--air-speed {{self.air_speed}} --solarcal-par {{self.solarcal_par}} ' \
            '--comfort-par {{self.comfort_par}} --folder output'

    result_folder = Outputs.folder(
        description='Folder containing all of the output CSV files.', path='output'
    )

    temperature_map = Outputs.file(
        description='CSV file containing a map of Operative Temperature for each '
        'sensor and step of the analysis.', path='output/temperature.csv'
    )

    condition_map = Outputs.file(
        description='CSV file containing a map of comfort conditions for each '
        'sensor and step of the analysis. -1 indicates unacceptably cold conditions. '
        '+1 indicates unacceptably hot conditions. 0 indicates neutral (comfortable) '
        'conditions.', path='output/condition.csv'
    )

    deg_from_neutral_map = Outputs.file(
        description='CSV file containing a map of the degrees Celsius from the '
        'adaptive comfort neutral temperature for each sensor and step of the '
        'analysis. This can be used to understand not just whether conditions are '
        'acceptable but how uncomfortably hot or cold they are.',
        path='output/condition_intensity.csv'
    )


@dataclass
class UtciMap(Function):
    """Get CSV files with maps of UTCI comfort from EnergyPlus and Radiance results."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus and contains '
        'hourly or sub-hourly thermal comfort results.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    enclosure_info = Inputs.file(
        description='A JSON file containing information about the radiant '
        'enclosure that sensor points belong to.', path='enclosure_info.json',
        extensions=['json']
    )

    epw = Inputs.file(
        description='Weather file used to estimate conditions for any outdoor '
        'sensors and to compute sun positions.', path='weather.epw', extensions=['epw']
    )

    total_irradiance = Inputs.file(
        description='A Radiance .ill containing total irradiance for each sensor in '
        'the enclosure-info.', path='total.ill', extensions=['ill', 'irr']
    )

    direct_irradiance = Inputs.file(
        description='A Radiance .ill containing direct irradiance for each sensor in '
        'the enclosure-info.', path='direct.ill', extensions=['ill', 'irr']
    )

    ref_irradiance = Inputs.file(
        description='A Radiance .ill containing ground-reflected irradiance for each '
        'sensor in the enclosure-info.', path='ref.ill', extensions=['ill', 'irr']
    )

    sun_up_hours = Inputs.file(
        description='A sun-up-hours.txt file output by Radiance and aligns with the '
        'input irradiance files.', path='sun-up-hours.txt'
    )

    wind_speed = Inputs.str(
        description='A single number for meteorological wind speed in m/s or a string '
        'of a JSON array with numbers that align with the result-sql reporting period. '
        'This will be used for all indoor comfort evaluation while the EPW wind speed '
        'will be used for the outdoors.', default='0.5'
    )

    solarcal_par = Inputs.str(
        description='A SolarCalParameter string to customize the assumptions of '
        'the SolarCal model.', default='SolarCalParameter: [posture:standing] '
        '[sharp:135] [absorptivity:0.7] [emissivity:0.95]'
    )

    comfort_par = Inputs.str(
        description='A UTCIParameter string to customize the assumptions of '
        'the UTCI comfort model.', default='UTCIParameter: [cold:9] [heat:26]'
    )

    @command
    def run_utci_map(self):
        return 'ladybug-comfort map utci result.sql enclosure_info.json ' \
            'weather.epw --total-irradiance total.ill --direct-irradiance direct.ill ' \
            '--ref-irradiance ref.ill --sun-up-hours sun-up-hours.txt ' \
            '--wind-speed {{self.wind_speed}} --solarcal-par {{self.solarcal_par}} ' \
            '--comfort-par {{self.comfort_par}} --folder output'

    result_folder = Outputs.folder(
        description='Folder containing all of the output CSV files.', path='output'
    )

    temperature_map = Outputs.file(
        description='CSV file containing a map of Universal Thermal Climate Index '
        '(UTCI) temperatures for each sensor and step of the analysis.',
        path='output/temperature.csv'
    )

    condition_map = Outputs.file(
        description='CSV file containing a map of comfort conditions for each '
        'sensor and step of the analysis. -1 indicates unacceptably cold conditions. '
        '+1 indicates unacceptably hot conditions. 0 indicates neutral (comfortable) '
        'conditions.', path='output/condition.csv'
    )

    category_map = Outputs.file(
        description='CSV file containing a map of the heat/cold stress categories '
        'for each sensor and step of the analysis. -5 indicates extreme cold stress. '
        '+5 indicates extreme heat stress. 0 indicates no thermal stress. '
        'This can be used to understand not just whether conditions are '
        'acceptable but how uncomfortably hot or cold they are.',
        path='output/condition_intensity.csv'
    )
