"""Base plugin definition."""
from abc import ABC, abstractmethod
from typing import List, Optional

from .anchor_utils_mixin import AnchorUtilsMixin
from .connection_callback_strategy import (
    ConnectionCallbackStrategy,
    UpdateOnlyConnectionCallbackStrategy,
    WorkflowRunConnectionCallbackStrategy,
)
from .connection_interface import ConnectionInterface
from .engine_proxy import EngineProxy
from .events import ConnectionEvents, PluginEvents
from .observable_mixin import ObservableMixin
from .tool_config import ToolConfiguration
from .workflow_config import WorkflowConfiguration


class BasePlugin(ABC, AnchorUtilsMixin, ObservableMixin):
    """Base plugin to inherit from."""

    def __init__(self, tool_id: int, alteryx_engine, output_anchor_mgr):
        AnchorUtilsMixin.__init__(self)
        ObservableMixin.__init__(self)

        self.tool_id = tool_id
        self.engine = EngineProxy(alteryx_engine, tool_id)

        self.tool_config = ToolConfiguration(self.tool_name, output_anchor_mgr)

        # These properties get assigned in pi_init
        self.input_anchors = None
        self.output_anchors = None
        self.workflow_config = None

    def pi_init(self, workflow_config_xml_string: str) -> None:
        """Plugin initialization from the engine."""
        self.input_anchors = self.tool_config.build_input_anchors()
        self.output_anchors = self.tool_config.build_output_anchors()
        self.workflow_config = WorkflowConfiguration(workflow_config_xml_string)

    def pi_add_incoming_connection(
        self, anchor_name: str, connection_name: str
    ) -> ConnectionInterface:
        """Add incoming connection to the tool from the engine."""
        anchor = [a for a in self.input_anchors if a.name == anchor_name][0]

        connection = ConnectionInterface(self, connection_name)
        anchor.connections.append(connection)
        self._subscribe_to_connection(connection)
        return connection

    def _subscribe_to_connection(self, connection: ConnectionInterface):
        """Subscribe to events of interest generated by a connection."""
        connection.subscribe(
            ConnectionEvents.CONNECTION_INITIALIZED,
            self.callback_strategy.connection_initialized_callback,
        )
        connection.subscribe(
            ConnectionEvents.RECORD_RECEIVED,
            self.callback_strategy.single_record_received_callback,
        )
        connection.subscribe(
            ConnectionEvents.CONNECTION_CLOSED,
            self.callback_strategy.connection_closed_callback,
        )
        connection.subscribe(
            ConnectionEvents.PROGRESS_UPDATE,
            self.callback_strategy.update_progress_callback,
        )

    def pi_add_outgoing_connection(self, anchor_name: str) -> bool:
        """Register an outgoing connection from this tool."""
        anchor = [a for a in self.output_anchors if a.name == anchor_name][0]
        anchor.num_connections += 1
        return True

    def pi_push_all_records(self, n_record_limit: int) -> bool:
        """Push all records when no inputs are connected."""
        if len(self.required_input_anchors) == 0:
            success = self.initialize_plugin()
            if success and self.engine.update_only_mode:
                self.on_complete()
            self.close_output_anchors()

            return success

        self.engine.error(self.engine.xmsg("Missing Incoming Connection(s)."))
        return False

    def pi_close(self, b_has_errors: bool) -> None:
        """pi_close is useless. Never use it."""
        pass

    def _run_plugin_initialization(self):
        """Run initialize plugin code."""
        self.notify_topic(PluginEvents.PLUGIN_INITIALIZED, self.initialize_plugin())

    @property
    def callback_strategy(self) -> ConnectionCallbackStrategy:
        """Generate the callback strategy for the tool"""
        return (
            UpdateOnlyConnectionCallbackStrategy(self)
            if self.engine.update_only_mode
            else WorkflowRunConnectionCallbackStrategy(self)
        )

    """All properties below this point can/should be overridden for custom tools."""

    @property
    @abstractmethod
    def tool_name(self) -> str:
        """Get the tool name."""
        pass

    @property
    @abstractmethod
    def record_batch_size(self) -> Optional[int]:
        """Get the record batch size."""
        pass

    def initialize_plugin(self) -> bool:
        """Initialize plugin."""
        pass

    @abstractmethod
    def process_records(self) -> None:
        """Process records in batches."""
        pass

    @abstractmethod
    def on_complete(self) -> None:
        """Finalizer for plugin."""
        pass
