# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['globus_action_provider_tools', 'globus_action_provider_tools.flask']

package_data = \
{'': ['*']}

install_requires = \
['dogpile.cache>=0.7.1,<0.8.0',
 'globus-nexus-client>=0.2.8,<0.3.0',
 'globus-sdk>=1.7.1,<2.0.0',
 'openapi-core>=0.13,<0.14']

setup_kwargs = {
    'name': 'globus-action-provider-tools',
    'version': '0.7.2',
    'description': 'Tools to help developers build services that implement the Action Provider specification.',
    'long_description': 'Action Provider Tools Introduction\n==================================\n\nThis is an experimental and unsupported toolkit to help developers build Action Providers for use in Globus Automate including for invocation via Globus Flows.\n\nAs this is experimental, no support is implied or provided for any sort of use of this package. It is published for ease of distribution among those planning to use it for its intended, experimental, purpose.\n\nThis README is intended to introduce the concepts, requirements and overview of this toolkit to be used when implementing an Action for use with Globus Automate. The three main sections are 1) an introduction to the interface required of all Actions, 2) A description of how to prepare your service to use Globus Auth, and 3) A description of the content of this toolkit and how it can be used in a Python implementation. This overview does not provide any ;guidance on how to host or operate the Action so that it is accessible via the internet.\n\n\nActions Overview\n================\nThe fundamental purpose of the Globus Automate platform is to tie together multiple operations or units of work into a coordinated orchestration. We refer to each of these operations as an *Action*. In particular, the *Flows* service provides a means of coordinating multiple actions across potentially long periods of time to perform some aggregate function larger than any single Action provides. The *Triggers* service ties *Events*, or occurrences within a managed environment, to Actions such that each occurrence of the Event automatically invokes the Action associated with it.\n\nIn both the Flows and the Triggers cases, the Actions require a uniform interface for invocation, monitoring and possibly termination so that new Actions may be introduced without requiring customization or re-implementation of the invoking services. We refer to the service endpoints which can be invoked in this manner as *Action Providers* and the uniform interface for interacting with the Action Providers as the *Action Provider Interface*. We provide here an overview of the *Action Provider Interface* as a guide for use when implementing an *Action Provider*. \n\nThe Action Provider Interface\n-----------------------------\n\nThe Action Provider Interface is a RESTful model for starting, monitoring, canceling and removing state associated with the invocation of an Action. Following the REST resource life-cycle pattern, each Action invocation returns an identifier representing the invocation (an *Action Instance*). This identifier is used to monitor the progress of the Action Instance via further REST calls until its completion, or it may be used to request cancellation of the instance.\n\nBecause the interface is intended to support arbitrary Action types, we recognize that some Action instances may be long-running (asynchronous) such as the execution of a computational job. Other Actions may be short-running (synchronous), able to return their final result directly in response to their invocation request as is the case in typical RESTful models. The Action Life-cycle described below specifically supports these execution modes as well as handling failures and Actions which may be, temporarily, unable to make progress.\n\nAction Life-cycle\n^^^^^^^^^^^^^^^^^\n\nThe Life-cycle of an Action defines the set of states that the Action may be in, and how it can transition among the states. The states are defined as follows:\n\n*  ``ACTIVE``: The Action is executing and is making progress toward completion\n\n*  ``INACTIVE``: The Action is paused in its execution and is not making progress toward completion. Out-of-band (i.e. not via the Action Provider Interface) measures may be required to allow the Action to proceed.\n\n*  ``SUCCEEDED``: The Action reached a completion state which was considered "normal" or not due to failure or other unrecoverable error. \n\n*  ``FAILED``: The Action is in a completion state which is "not normal" such as due to an error condition which is not considered recoverable in any manner. \n\n*  *"Released"*: The Action Provider has removed the record of the existence of the Action. Further attempts to interact with the Action will be errors as if the Action had never existed. All resources associated with the Action may have been deleted or removed. This is not a true state in the sense that the state can be observed, but ultimately all Actions will be released and unavailable for further operations. Any subsequent references to the Action, e.g. via the API methods described below, will behave as if the Action never existed.\n\nUpon initial creation of an Action (see operations below), the Action may be in any of the first four states. If it is in an ``ACTIVE`` or ``INACTIVE`` state, the Action is considered "asynchronous" and further queries to get the state of the Action may return updated information. If the Action is in a the ``SUCCEEDED`` or ``FAILED`` states, the Action is synchronous, all information about the Action is returned on the creation operation and no changes to the state are possible.\n\nAn asynchronous Action may change state between ``ACTIVE`` and ``INACTIVE`` during its life time, and may update further details about its progress while in either of these states. When a completed state of ``SUCCEEDED`` or ``FAILED`` is reached, the Action state cannot be updated further. The Action Provider is, however, required to maintain this final state for some period of time so that the client of the Action may retrieve the completion state. Upon completion, the client may request that the Action be "released" or the Action Provider may do so on its own after the required time-out occurs. To save server resources, it is preferred that the client release the Action when it has reliably retrieved and processed the final state.\n\nAction Provider Interface and Document Types\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nThe primary purpose of the Action Provider Interface is to securely support and report Actions progressing through the life-cycle described above. The document types supporting this are the initial Action invocation *Action Request* document, and the *Action Status* document which contains the life-cycle status described above along with additional detailed status information specific to the type of Action being executed.\n\n.. note:: In the below we describe URL paths where operations can be performed. We assume that all of these share a common "Base URL" which we don\'t name in this document. The Base URL may be at any place in the URL path namespace desired by the Action Provider, and so may be used in conjunction with any other service URLs it may support.\n\n.. note:: For brevity and clear presentation, in the descriptions of document types in the following sections, we present the key concepts, but do not enumerate every option or field on the documents. Refer to the toolkit components, including the OpenAPI format specification (as described in the toolkit section), for a complete definition.\n\nStarting an Action: The Action Request Document and the ``POST /run`` Method\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nStarting an Action is performed by making a REST ``POST`` request to the path ``/run`` containing an Action Request document. The request document contains the following fields:\n\n*  ``request_id``: A client-generated Identifier for this request. A user may re-invoke the ``/run`` method with the same ``request_id`` any number of times, but the Action must only be initiated once. In this way, the user may re-issue the request in case it cannot be determined if a request was successfully initiated for example due to network failure.\n\n*  ``body``: An Action Provider-specific object which provides the input for the Action to be performed. The ``body`` must conform to the input specification for the Action Provider being invoked, and thus the client must understand the requirements of the Action Provider when providing the value of the ``body``. Thus, the Action Provider must provide documentation on the format for the ``body`` property.\n\n*  ``manage_by`` and ``monitor_by``: Each of these is a list of principal values in `URN format <https://docs.globus.org/api/search/#principal_Urns>`_, and they allow the user invoking the Action to delegate some capability over the Action to other principals. ``manage_by`` defines the principals who are allowed to attempt to change the execution of the Action (see operations ``/cancel`` and ``/release`` below) while it is running. ``monitor_by`` defines principals which are allowed to see the state of the Action before its state has been destroyed in a release operation. In both cases, the Globus Auth identity associated with the ``/run`` operation is implicitly part of both the ``manage_by`` and ``monitor_by`` sets. That is, the invoking user need not include their own identity into these lists.\n\nAny request to the ``/run`` method which contains an Action Request which adheres to the input schema will return an Action Status document as described in the next section. \n\nMonitoring and Completing an Action: The Action Status Document and Management Methods\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nAll information about an Action is contained in the Action Status document which is returned on almost all operations related to an Action (the exception is the log operation which is optional and is described briefly below). Notable fields of the Action Status document include:\n\n*  ``action_id``: The unique identifier for this particular action. The ``action_id`` may be any string, and it should be treated as an opaque value (that is, having no semantic or implied meaning) by the client. The client will first learn of an Action\'s ``action_id`` in the Action Status returned by the ``/run`` method.\n\n*  ``status`` and ``display_status``: These provide the description of the state of the Action. ``status`` is the specific life-cycle value described above. ``display_status`` is an optional field the Action Provider may supply which gives a short text description of the status using language which is specific to the Action.\n\n*  ``details``: The Action Provider-specific state, particularly the completion state, of the Action are returned in the ``details`` field. In the completion states, the ``details`` can be considered the "result" or the "return value" of the Action. For a ``SUCCEEDED`` status value, it is the successful return value and for ``FAILED`` it is the error result. The exact content ``details`` is always specific to the Action Provider, so must be documented by the Action Provider to describe its interpretation to clients.\n\n*  ``monitor_by`` and ``manage_by``: As in the Action Request.\n\n*  ``start_time`` and ``completion_time``: Represent the time the Action was first received by the ``/run`` operation and the time the Action Provider determined that the Action reached a completed state (``SUCCEEDED`` or ``FAILED``) respectively. Action Providers are not required to continuously monitor the progress of Actions, so the ``completion_time`` noted may be different than the actual completion of the activity running as the Action.  These values **may** be the same in the case of a synchronous operation, but ``completion_time`` must never be before ``start_time``.\n\n*  ``release_after``: As stated above, Action state is automatically removed from the Action Provider after some interval after it reaches a completion state. The ``release_after`` is a time duration, in seconds, which states how long after completion the Action will automatically be released. A typical value would be 30-days, but Action Providers may define their own policy which is to be exposed in the Action Status.\n\nIn addition to the ``/run`` method described above, the Action Status is the "universal" return value from operations on a Action. We describe the operations on Actions next. Each uses the ``action_id`` as part of the URL path much like other RESTful resources do with their ids, and none of them requires any input body. \n\n*  ``GET /<action_id>/status``: This is a read-only operation for retrieving the most recent state of the Action. It is commonly used to poll the state of an Action while awaiting it entering a completion state. Use of this API call requires that the user authenticates with a principal value which is in the ``monitor_by`` list established when the Action was started.\n\n*  ``POST /<action_id>/cancel``: Cancellation provides an advisory or hint to the Action Provider that the user does not want the Action to continue execution. The Action Provider is not required to insure immediate completion or that the cancel operation truly causes the Action to terminate in any manner other than it would have without the cancel request. Thus, the Action Status returned from the cancel operation may contain a non-completion state. If the Action is already in a completed state, the Action Provider may treat the request much as a ``/status`` request to simply return the current status. Use of this API call requires that the user authenticates with a principal value which is in the ``manage_by`` list established when the Action was started. \n\n*  ``POST /<action_id>/release``: As described in the section on life-cycle, the very last step of the life-cycle is for the Action state to be removed from the Action Provider. A user can specify that it has retrieved final state or is no longer interested in the state using the ``/release`` operation which returns the final state. If the Action is not already in a completion state, ``/release`` will return an error as this operation does not attempt to stop execution (that is what ``/cancel`` does). The Action Status document returned from ``/release`` will be the last record of the Action present at the Action Provider. After the call to ``/release`` the ``action_id`` is no longer valid, and use in any other calls will return an error, most likely an HTTP status 404 indicating not found.\n\nDetailed Execution History: logging\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nSome Actions, particularly those that are long running, may have associated with them a list or log of activities or sub-events which occurred during the Action\'s life. This detailed log is typically larger, more complex, or more fine-grain than the snapshot of the status returned by the ``/status`` method. Not all Action Providers or Actions are suitable for logging, so support is considered optional and will be advertised by the Action Provider in its description (see below). The request to retrieve the log takes the form ``GET /<action_id>/log?<filters,pagination>``. The filters and pagination query parameters are used to limit (e.g. based on start time) which log records to retrieve and the pagination parameter is used to scroll through a long set of log records across multiple requests. Each record in the log contains the following properties:\n\n*  ``time``: A timestamp representing the time this log record occurred.\n\n*  ``code``: A short Action Provider-specific description of the type of the log record.\n\n*  ``description``: A textual description of the purpose, cause, or information on the log record.\n\n*  ``details`` (optional): An object providing additional and structured Action Provider-specific representation of the log record.\n\n\nAction Provider Introspection (``GET /``)\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nThe Automate platform is intended to help users both find and make use of the variety of Action Providers which may be available on the network. The primary means of accomplishing this assistance is by making Action Providers, the services which implement the Action Provider Interface, self-describing via an *Introspection* interface. Accessing the introspection method is performed simply via a ``GET /``. That is, the HTTP ``GET`` method on the Base URL. The returned JSON document contains the following fields:\n\n*  ``api_version``: A version string defining the version of the Action Provider Interface supported by the Action Provider. The version described in this document and currently the only version available will have value ``"1.0"``.\n\n*  ``title``, ``subtitle``, ``description``, ``keywords``: Each of these provide human-readable text which helps a user discover the purpose of the Action Provider.\n\n*  ``visible_to`` and ``runnable_by``: Access to the action provider is limited by and published through these properties. Each contains a list of principal values in URN format. ``visible_to`` controls who can retrieve the information via introspection (this operation) and may contain the string ``"public"`` indicating that all users, even those who present no credentials, may access the information. The ``runnable_by`` property enumerates who can use the ``/run`` method to start an Action at this provider. It allows the string ``"all_authenticated_users"`` indicating that any user who presents valid credentials via a Bearer token may start an Action at the provider.\n\n*  ``synchronous`` and ``log_supported``: These are boolean values which simply describe capabilities and modes for the Action Provider. If ``synchronous`` is true, a user calling ``/run`` can assume that the returned status will always be in a completed (``SUCCEEDED`` or ``FAILED``) state and there will never be a need to poll using the ``/status`` method (use of ``/release`` is still permitted and encouraged to remove the status from the Action Provider). As indicated in the discussion of the ``/log`` method, support for it is optional, and the ``log_supported`` flag provides an indication to users whether they can make use of ``/log`` for fine grained monitoring of an Action.\n\n*  ``input_schema``: The ``input_schema`` value provides a complete schema description for the ``body`` property of the Action Request understood by this Action Provider. The schema is provided in `JSON Schema <https://json-schema.org/>`_ format.\n\n\nSetting Up an Action Provider in Globus Auth\n============================================\n\nThe Action Provider Interface makes use of and is bound closely with authentication via the `Globus Auth <https://globus.org/>`_ system. To authenticate RESTful requests using Globus Auth, a service must register as a "resource server". This is a multi-step process involving use of both the Globus Auth developer portal, and the Globus Auth API for configuring various access control state. To help with this process, we provide a step-by-step guide to using Globus Auth for this purpose:\n\n1. Register a new App on `<https://developers.globus.org>`_ using a browser. After insuring that you are logged in to the developer portal in a browser at this URL, perform the following steps:\n\n   - Click Add another project\n\n     - Provide a name, contact email and select which of your own Globus Auth linked identities are permitted to administer the project. In future interactions with the Globus Developer Portal to manipulate the resource server, you will be required to login with this identity.\n\n   - Find your new, empty project, and select Add drop down and "new app"\n\n     - Provide a name for the specific app within the project. This will be a common name displayed to users when they make use of the Action Provider.\n\n     - When creating a resource server, the other fields on the app creation page are not used.\n\n       - "Redirects" is not used, but a value must be provided. You can use a URL associated with your service or a placeholder value like "https://localhost".\n\n       - "Scopes" are not relevant so any entered values and make no difference, so should be left blank. The "Privacy Policy" and "Terms and Conditions" may be displayed to users making use of your action provider, but they are not required.\n\n    - Make note of the "Client Id" in the expanded description of your app. This value will be used elsewhere in the creation of the service and is often referenced as ``client_id``.\n\n    - In the section "Client Secrets" click "Create a new secret"\n\n        - Provide a name which is meaningful to you. It will not be displayed to other users.\n\n        - Make note of the generated secret. Like the ``client_id`` this will be used later in development. Be sure **not to lose it** as it can only be displayed once. However, new client secrets can be created and old ones deleted at any time should the need for a replacement secret arise.\n\n2. Use the Globus Auth REST API to introspect your Action Provider Resource Server and create required Scopes.\n\n.. note:: In the examples below, we will use the command line tool ``curl`` to perform the HTTP operations as it is very widely available. However, other tools and clients exist for interacting with REST and HTTP services, so you may need to translate the ``curl`` commands to your preferred tools.\n\n   - Introspect the Globus Auth client to see the same settings you setup in the developer portal:\n.. code-block:: BASH\n\n    curl --user <client_id>:<client_secret> https://auth.globus.org/v2/api/clients/<client_id>\n\n\nNote the use of the ``<client_id>`` and ``<client_secret>`` values generated during your registration on the Globus Developer Portal. A successful return from this command is a JSON representation of the Globus Auth client similar to:\n\n.. code-block:: JSON\n\n    {\n      "client": {\n        "scopes": [],\n        "redirect_uris": [\n          "https://localhost"\n        ],\n        "name": "My Action Provider",\n        "links": {\n          "privacy_policy": null,\n          "terms_and_conditions": null\n        },\n        "grant_types": [\n          "authorization_code",\n          "client_credentials",\n          "refresh_token",\n          "urn:globus:auth:grant_type:dependent_token"\n        ],\n        "fqdns": [],\n        "visibility": "private",\n        "project": "a47b9014-9250-4e21-9de5-b4aac81d464b",\n        "required_idp": null,\n        "preselect_idp": null,\n        "id": "8e98ba5a-21a9-4bef-ab6a-0fcdbed36405",\n        "public_client": false,\n        "parent_client": null\n      }\n    }\n\n\nOf note are the fields ``fqdns`` and ``scopes``. ``fqdns`` stands for "Fully Qualified Domain Names" and these are DNS host names which have been associated with your client. These are not required, but if an ``fqdn`` is registered, then when ``scopes`` are created (see below), they will reflect the ``fqdn`` and will be easier to read and understand than the default scope representation (which will also be generated) which are simply opaque UUIDs. We briefly describe registering an FQDN below.\n\n``scopes`` are created to identify operations on the Action Provider. Typically, an Action Provide defines just one scope and it is exposed to users in the Action Provider\'s introspection (``GET /``) information in the field ``globus_auth_scope``. After the discussion of (optional) registration an ``fqdn`` we discuss required Scope creation below.\n\n    - *Registering an FQDN for your Action Provider*:\n\n      - Create a TXT record in DNS for your hostname containing the <client_id> generated\n\n      - This may take some time to propagate through DNS. You can shorten this time by setting the TTL value on the DNS record to a smaller value (such as 60 seconds)\n\n    - Use the Globus Auth API to register the FQDN which it will do by insuring that the TXT record is in DNS for the client_id:\n\n.. code-block:: BASH\n\n    curl -XPOST --user <client_id>:<client_secret> https://auth.globus.org/v2/api/clients/<client_id>/fqdns -d \'{"fqdn": "<your FQDN string>"}\'\n\n\n     - *Creating a Scope*: Creation of a scope is required as the scope will be used in authenticating REST calls on the Action Provider.\n       - Start by creating a "scope definition" JSON document in the following format replacing the ``name``, ``description`` and optionally the ``scope_suffix``:\n  \n.. code-block:: JSON\n\n    {\n      "scope": {\n        "name": "Action Provider Operations",\n        "description": "All Operations on My Action Provider",\n        "scope_suffix": "action_all",\n        "dependent_scopes": [],\n        "advertised": true,\n        "allow_refresh_tokens": true\n      }\n    }\n\n\nThe ``name`` and ``description`` fields are purely informative and will be presented to other users who use the Globus Auth API to lookup the scope. The ``scope_suffix`` will be placed at the end of the generated "scope string" which is a URL identifier for the scope. It provides the context for the operations this scope covers among all operations your service provides. For Action Providers, we commonly use ``action_all`` to indicate all operations defined by the Action Provider API, but any string is acceptable.\n\n``dependent_scopes`` define scopes of other Globus Auth resource servers that your Action Provider will invoke to perform its work. For example, if your Action Provider uses Globus Transfer to first move some data to compute upon, the scope for the Globus Transfer service would be placed in the ``dependent_scopes`` list. In the most common case in which the Action Provider does *not* rely on other services, this list should be empty. Enumerating scopes which may be useful, such as the example Globus Transfer used here, or how authentication is performed using dependent scopes is beyond the scope of this document. We encourage you to consult the `Globus Auth Documentation <https://docs.globus.org/api/auth/>`_ for more information.\n\nThe ``advertised`` property indicates whether the scope will be visible to all users who do scope look ups on Globus Auth. You may select either ``true`` or ``false`` for this depending on your own policy. ``allow_refresh_tokens`` should generally be set to ``true``, indicating that a client of the Action Provider who has authenticated the user via Globus Auth is allowed to refresh that authentication without further interactions from the user. Especially in the case where an Action may be long running and is monitored by an automated system like Globus Flows, it is important that token refresh is permitted.\n\nWith the scope creation JSON document complete, the scope will be created in Globus Auth with the following REST interaction via the ``curl`` command:\n\n.. code-block:: BASH\n\n    curl --user <client_id>:<client_secret> -H Content-Type: application/json -XPOST     https://auth.globus.org/v2/api/clients/<client_id>/scopes -d \'<Insert Scope creation document from above>\'\n\n\nThis should return the definition of the new scope matching the values provided in your scope creation document. As an example:\n\n.. code-block:: JSON\n\n    {\n      "scopes": [\n        {\n          "dependent_scopes": [],\n          "description": "<your description>",\n          "allows_refresh_token": true,\n          "client": "<client_id>",\n          "advertised": true,\n          "scope_string": "https://auth.globus.org/scopes/<client_id>/action_all",\n          "id": "<A UUID for this scope>",\n          "name": "<your scope name>"\n        }\n      ]\n    }\n\n\nThe returned ``scope_string``, which always takes the form of a URL, will be the value exposed to users who wish to authenticate with Globus Auth to use your Action Provider. It will be part of the Action Provider description document, returned on the Action Provider Introspection operation (``GET /``) with the key ``globus_auth_scope``.\n\nNote that the returned value is an *array* of scopes. That is, more than one scope definition may be generated from the single scope creation request. This happens in the case where an FQDN has been registered for your ``client_id``. In this case, a similar scope definition will also be generated, but the ``scope_string`` will contain the FQDN value(s). The ``scope_string`` values may be used interchangeably both by users requesting authentication to the Action Provider and in the ``globus_auth_scope`` value of the Action Provider Description. \n\n\nUsing the Toolkit\n==================\n\nThis toolkit provides the following components:\n\n1. Authentication helpers that make it easier to validate Globus Auth tokens and determine if a given request should be authorized\n\n2. An `OpenAPI v3 specification <http://spec.openapis.org/oas/v3.0.2>`_ and associated helpers that can be used to validate incoming requests and verify the responses your Action Provider generates. This document also defines the interface which must be supported by your REST API to have it function as an Action Provider.\n\n3. Simple bindings for the document types "Action Request" and "Action Status" to Python NamedTuple representations and helper functions for serializing and deserializing these structures to/from JSON.\n\n4. Helper methods for binding the REST API calls defined by the Action Interface to a Flask application. These helpers will perform the Authentication and Validation steps (as provided by components 1 and 2) and communicate with an Action Provider implementation using the structures defined in 3. For those users building an Action Provider using Flask, this provides a simplified method of getting the REST API implemented and removing common requirements so the focus can be on the logic of the Action provided.\n\n\nInstallation\n------------\n\nInstallation is via PyPi using, for example:\n\n.. code-block:: BASH\n\n    pip install globus-action-provider-tools\n\n\nAuthentication\n---------------\n\nThe authentication helpers can be used in your action provider as follows:\n\n.. code-block:: python\n\n    from globus_action_provider_tools.authentication import TokenChecker\n    # You will need to register a client and scope(s) in Globus Auth\n    # Then initialize a TokenChecker instance for your provider:\n    checker = TokenChecker(\n        client_id=\'YOUR_CLIENT_ID\',\n        client_secret=\'YOUR_CLIENT_SECRET\',\n        expected_scopes=[\'https://auth.globus.org/scopes/YOUR_SCOPES_HERE\'],\n        expected_audience=\'YOUR_CLIENT_NAME\',\n    )\n\n\n(expected_audience should be unnecessary in the near future)\n\nWhen a request comes in, use your TokenChecker to validate the access token from the HTTP Authorization header.\n\n.. code-block:: python\n\n    access_token = request.headers[\'Authorization\'].replace(\'Bearer \', \'\')\n    auth_state = checker.check_token(access_token)\n\n\nThe AuthState has several properties and methods that will make it easier for you to decide whether or not to allow a request to proceed:\n\n.. code-block:: python\n\n    # This user\'s Globus identities:\n    auth_state.identities\n    # frozenset({\'urn:globus:auth:identity:9d437146-f150-42c2-be88-9d625d9e7cf9\',\n    #           \'urn:globus:auth:identity:c38f015b-8ad9-4004-9160-754b309b5b33\',\n    #           \'urn:globus:auth:identity:ffb5652b-d418-4849-9b57-556656706970\'})\n    # Groups this user is a member of:\n    auth_state.groups\n    # frozenset({\'urn:globus:groups:id:606dbaa9-3d57-44b8-a33e-422a9de0c712\',\n    #           \'urn:globus:groups:id:d2ff42bc-c708-460f-9e9b-b535c3776bdd\'})\n\n\nYou\'ll notice that both groups and identities are represented as strings that unambiguously signal what type of entity they represent. This makes it easy to merge the two sets without conflict, for situations where you\'d like to work with a single set containing all authentications:\n\n.. code-block:: python\n\n    all_principals = auth_state.identities.union(auth_state.groups)\n\n\nThe AuthState object also offers a helper method, ``check_authorization()`` that is designed to help you test whether a request should be authorized:\n\n.. code-block:: python\n\n    resource_allows = [\'urn:globus:auth:identity:c38f015b-8ad9-4004-9160-754b309b5b33\']\n    auth_state.check_authorization(resource_allows)\n    # True\n\n\nThis method also accepts two special string values, ``\'public\'`` and ``\'all_authenticated_users\'``, together with keyword arguments that enable their use:\n\n.. code-block:: python\n\n    resource_allows = [\'public\']\n    auth_state.check_authorization(resource_allows, allow_public=True)\n    # True\n    resource_allows = [\'all_authenticated_users\']\n    auth_state.check_authorization(resource_allows, allow_all_authenticated_users=True)\n    # True\n\n\nCaching\n^^^^^^^\n\nTo avoid excessively taxing Globus Auth, the ``AuthState`` will, by default, cache identities and group memberships for 30 seconds.\n\nThe cache is initialized when you first instantiate your ``TokenChecker()``.  You should only need to create one TokenChecker instance for your application, and then you can re-use it to check each new token. If you do try to make multiple instances, you may get an exception:\n\n> ``dogpile.cache.exception.RegionAlreadyConfigured: This region is already configured``\n\nbecause it\'s trying to re-initialize a cache that\'s already been set up.\n\n\nValidation\n----------\n\nThere is an OpenAPI v3 specification for the Action Provider API available as described above. You can use any tools that accept OpenAPI v3 to validate requests to and responses from your service, but this toolkit offers some pre-configured validators using the openapi-core library.\n\n.. code-block:: python\n\n    from globus_action_provider_tools.validation import request_validator, response_validator\n    # Validating a request\n    result = request_validator.validate(request)\n    # Or a response:\n    result = response_validator.validate(request, response)\n    # raise errors if invalid\n    result.raise_for_errors()\n    # or get list of errors\n    errors = result.errors\n\n\nNote that the ``request`` and ``response`` objects passed to the validators must conform to the `BaseOpenAPI <https://github.com/p1c2u/openapi-core/blob/master/openapi_core/wrappers/base.py>`_ request and response interfaces. If you\'re using Flask, you can use the `provided wrappers <https://github.com/p1c2u/openapi-core/blob/master/openapi_core/wrappers/flask.py#L10>`_, but if you\'re using a different web framework, you\'ll need to write your own small wrapper class.\n\nData Types\n----------\n\nThe toolkit provides some simple bindings for the document types defined by the Action Provider Interface to type-annotated Python3 `NamedTuples <https://docs.python.org/3/library/typing.html#typing.NamedTuple>`_. This can provide a convenient way to manipulate these document types within an Action Provider implementation. There is also a Python JSON Encoder provided which can be used with the built-in Python json package to properly encode these data types into JSON.\n\n.. code-block:: python\n\n    from globus_action_provider_tools.data_types import (\n        ActionProviderDescription,\n        ActionProviderJsonEncoder,\n        ActionRequest,\n        ActionStatus,\n        ActionStatusValue,\n    )\n    status = ActionStatus(\n        action_id=str(uuid.uuid4()),\n        status=ActionStatusValue.SUCCEEDED,\n        creator_id=caller_id,\n        monitor_by=request.monitor_by,\n        manage_by=request.manage_by,\n        start_time=str(datetime.datetime.now().isoformat()),\n        completion_time=str(datetime.datetime.now().isoformat()),\n        release_after=60 * 60 * 24 * 30,  # 30-days in seconds\n        display_status=ActionStatusValue.SUCCEEDED.name,\n        details=result_details,\n    )\n    json_string = json.dumps(action_status, cls=ActionProviderJsonEncoder)\n\n\nFlask Helper\n------------\n\nAs Action Providers are HTTP-servers, a common approach to building them is the use of the `Flask <https://palletsprojects.com/p/flask/>`_ framework. To aid in developing Flask-based Action Providers, helper methods are provided which encapsulate much of the other functionality in the framework: authentication, validation and serialization for easy use in a Flask-based application. Rather than defining each of the Action Provider Interface routes in the Flask application, helpers are provided which declare the necessary routes to Flask, perform the serialization, validation and authentication on the request, and pass only those requests which have satisfied these conditions on to a user-defined implementation of the routes.\n\nTo use the helpers, you must define functions corresponding to the various methods of the Action Provider interface (``run``, ``status``, ``release``, ``cancel``), and must provide the Action Provider introspection information in an instance of the ``ActionProviderDescription`` ``NamedTuple`` class defined in the tookit\'s ``data_types`` package. The application must also provide a Flask ``blueprint`` object to which the toolkit can attach the new routes. It is recommended that the ``blueprint`` be created with a ``url_prefix`` so that the Action Provider Interface routes are rooted at a distinct root path in the application\'s URL namespace.\n\nA brief example of setting up the flask helper is provided immediately below. A more complete example showing implementation of all the required functions is provided in the Appendix. It is appropriate to use the skeleton in the Appendix as a starting point for any new Action Providers which are developed.\n\n.. code-block:: python\n                \n    from globus_action_provider_tools.data_types import (\n        ActionProviderDescription,\n        ActionRequest,\n        ActionStatus,\n        ActionStatusValue,\n    )\n    from globus_action_provider_tools.flask import (\n        ActionStatusReturn,\n        add_action_routes_to_blueprint,\n    )\n    action_blueprint = Blueprint("action", __name__, url_prefix="/action")\n    provider_description = ActionProviderDescription(\n        globus_auth_scope="<scope created in Globus Auth>",\n        title="My Action Provider",\n        admin_contact="support@example.com",\n        synchronous=True,\n        input_schema={}, # JSON Schema representation of the input on the request\n        log_supported=False\n    )\n    add_action_routes_to_blueprint(\n        action_blueprint,\n        CLIENT_ID,\n        CLIENT_SECRET,\n        CLIENT_NAME,\n        provider_description,\n        action_run,\n        action_status,\n        action_cancel,\n        action_release,\n    )\n\n\nIn this example, the values ``CLIENT_ID``, ``CLIENT_SECRET`` and ``CLIENT_NAME`` are as defined in Globus Auth as described above. The values ``action_run``, ``action_status``, ``action_cancel`` and ``action_release`` are all **functions** which will be called by the framework when the corresponding HTTP requests are called. Where appropriate, these functions are implemented in terms of the toolkit\'s data types so the need for JSON serialization and deserialization is greatly reduced from the application code. The example in the Appendix demonstrates how these functions can be implemented.\n\nAppendix: Example Action Provider\n=================================\n\n\n.. code-block:: python\n\n    skeleton_blueprint = Blueprint("skeleton", __name__, url_prefix="/skeleton")\n    \n    input_schema = {\n        "$id": "https://automate.globus.org/skeleton_action_provider.input.schema.json",\n        "$schema": "http://json-schema.org/draft-07/schema#",\n        "title": "Skeleton Action Provider Input Schema",\n        "type": "object",\n        "properties": {"input_string": {"type": "string"}},\n        "additionalProperties": False,\n        "required": ["input_string"],\n    }\n    \n    provider_description = ActionProviderDescription(  globus_auth_scope="https://auth.globus.org/scopes/16e16447-209a-4825-ae19-25e279d91642/action_all_with_groups",\n        title="skeleton_action_provider",\n        admin_contact="support@globus.org",\n        synchronous=True,\n        input_schema=input_schema,\n        log_supported=False,  # This provider doesn\'t implement the log callback\n    )\n\n    # A simulated database mapping input user action requests identifiers to a previously\n    # seen request id and the corresponding action id\n    _fake_request_db: Dict[str, Tuple[ActionRequest, str]] = {}\n    \n    _fake_action_db: Dict[str, ActionStatus] = {}\n    \n    def _authorize_action_access(\n        action_id: str, auth_state: AuthState, policy_type: str = "manage_by") -> ActionStatus:\n        status = _fake_action_db.get(action_id)\n        if status is None:\n            raise NotFound(f"No Action with id {action_id}")\n        if policy_type == "manage_by":\n            policy_group = status.manage_by\n        elif policy_type == "monitor_by":\n            policy_group = status.monitor_by\n        if policy_group is None:\n            policy_group = []\n        allowed_set = set([status.creator_id] + [status.monitor_by])\n        if auth_state.check_authorization(allowed_set, allow_all_authenticated_users=True):\n            return status\n        else:\n            raise NotFound(f"No Action with id {action_id}")\n\n    def action_run(request: ActionRequest, auth: AuthState) -> ActionStatusReturn:\n        caller_id = auth.effective_identity\n        request_id = request.request_id\n        full_request_id = f"{caller_id}:{request_id}"\n        prev_request = _fake_request_db.get(full_request_id)\n        if prev_request is not None:\n            if prev_request[0] == request:\n                return action_status(prev_request[1], auth)\n            else:\n                raise Conflict(\n                    f"Request with id {request_id} already present with different parameters "\n                )\n        # Local processing would happen here\n        result_details = {\n            # This is safe because the input has been validated against the input schema\n            "you_said": request.body["input_string"]\n        }\n        status = ActionStatus(\n            action_id=str(uuid.uuid4()),\n            status=ActionStatusValue.SUCCEEDED,\n            creator_id=caller_id,\n            monitor_by=request.monitor_by,\n            manage_by=request.manage_by,\n            start_time=str(datetime.datetime.now().isoformat()),\n            completion_time=str(datetime.datetime.now().isoformat()),\n            release_after=request.release_after or "P30D",\n            display_status=ActionStatusValue.SUCCEEDED.name,\n            details=result_details,\n        )\n        _fake_request_db[full_request_id] = (request, status.action_id)\n        _fake_action_db[status.action_id] = status\n        return status\n    \n    def action_status(action_id: str, auth: AuthState) -> ActionStatusReturn:\n        return _authorize_action_access(action_id, auth, "monitor_by"), 200\n    \n    def action_cancel(action_id: str, auth: AuthState) -> ActionStatusReturn:\n        status = _authorize_action_access(action_id, auth, "manage_by")\n        if status.status in (ActionStatusValue.SUCCEEDED, ActionStatusValue.FAILED):\n            return status\n        # Process Action cancellation\n        status.status = ActionStatusValue.FAILED\n        status.display_status = "Canceled by user request"\n        return status\n    \n    def action_release(action_id: str, auth: AuthState) -> ActionStatusReturn:\n        status = _authorize_action_access(action_id, auth, "manage_by")\n        if status.status not in (ActionStatusValue.SUCCEEDED, ActionStatusValue.FAILED):\n            raise Conflict("Action is not complete")\n        _fake_action_db.pop(action_id)\n        # Both fake and badly inefficient\n        remove_req_id: Optional[str] = None\n        for req_id, req_and_action_id in _fake_request_db.items():\n            if req_and_action_id[1] == action_id:\n                remove_req_id = req_id\n                break\n        if remove_req_id is not None:\n            _fake_request_db.pop(remove_req_id)\n        return status, 200\n    \n    def main():\n        app = Flask(__name__)\n        app.url_map.strict_slashes = False\n        add_action_routes_to_blueprint(\n            skeleton_blueprint,\n            CLIENT_ID,\n            CLIENT_SECRET,\n            CLIENT_NAME,\n            provider_description,\n            action_run,\n            action_status,\n            action_cancel,\n            action_release,\n        )\n        app.register_blueprint(skeleton_blueprint)\n        app.run(debug=True)\n    \n    if __name__ == "__main__":\n        main()\n\n',
    'author': 'Jim Pruyne',
    'author_email': 'pruyne@globus.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
