"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 *
 * @stability stable
 */
class Deployment extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.string({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     *
     * @stability stable
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "@aws-cdk/aws-apigateway.Deployment", version: "1.127.0" };
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.uncachedString({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveW1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMsd0NBQTJFO0FBRTNFLGlFQUF1RDtBQUV2RCx1Q0FBd0U7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQWtCeEUsTUFBYSxVQUFXLFNBQVEsZUFBUTs7OztJQU90QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHdCQUF3QixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDN0QsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLE9BQU8sRUFBRSxLQUFLLENBQUMsR0FBRztTQUNuQixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLG9CQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDeEQ7UUFFRCxJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDckIsSUFBSSxDQUFDLFlBQVksR0FBRyxXQUFJLENBQUMsTUFBTSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUV0RSxJQUFJLEtBQUssQ0FBQyxHQUFHLFlBQVkscUJBQVcsRUFBRTtZQUNwQyxLQUFLLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25DO0lBQ0gsQ0FBQzs7Ozs7Ozs7OztJQUdNLGNBQWMsQ0FBQyxJQUFTO1FBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxvQkFBb0IsQ0FBQyxNQUFjO1FBQ3hDLDBFQUEwRTtRQUMxRSw0RUFBNEU7UUFDNUUsdUVBQXVFO1FBQ3ZFLHFFQUFxRTtRQUNyRSxvREFBb0Q7UUFDcEQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUEyQixDQUFDLENBQUM7SUFDbkUsQ0FBQzs7QUF2REgsZ0NBd0RDOzs7QUFPRCxNQUFNLHdCQUF5QixTQUFRLG9DQUFhO0lBS2xELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBb0M7UUFDaEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsU0FBUyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUztTQUNuQyxDQUFDLENBQUM7UUFSWSxtQkFBYyxHQUFHLElBQUksS0FBSyxFQUFPLENBQUM7UUFVakQsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxJQUFJLENBQUMsaUJBQWlCLENBQUMsV0FBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUM1RixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksY0FBYyxDQUFDLElBQWE7UUFDakMsNEVBQTRFO1FBQzVFLHdFQUF3RTtRQUN4RSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQztTQUM5RTtRQUVELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFTyxrQkFBa0I7UUFDeEIsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUV0QyxJQUFJLElBQUksQ0FBQyxHQUFHLFlBQVksaUJBQU8sSUFBSSxJQUFJLENBQUMsR0FBRyxZQUFZLHFCQUFXLEVBQUUsRUFBRSxvQ0FBb0M7WUFFeEcsdUdBQXVHO1lBQ3ZHLE1BQU0sWUFBWSxHQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQW9CLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztZQUM3RSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7U0FDN0M7UUFFRCxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUM7UUFFakMsNEVBQTRFO1FBQzVFLDRDQUE0QztRQUM1QyxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ25CLE1BQU0sR0FBRyxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqRixHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMxQjtRQUVELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgeyBMYXp5LCBSZW1vdmFsUG9saWN5LCBSZXNvdXJjZSwgQ2ZuUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuRGVwbG95bWVudCB9IGZyb20gJy4vYXBpZ2F0ZXdheS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgTWV0aG9kIH0gZnJvbSAnLi9tZXRob2QnO1xuaW1wb3J0IHsgSVJlc3RBcGksIFJlc3RBcGksIFNwZWNSZXN0QXBpLCBSZXN0QXBpQmFzZSB9IGZyb20gJy4vcmVzdGFwaSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgYXMgQ29yZUNvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5leHBvcnQgaW50ZXJmYWNlIERlcGxveW1lbnRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwaTogSVJlc3RBcGk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmV0YWluRGVwbG95bWVudHM/OiBib29sZWFuO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIERlcGxveW1lbnQgZXh0ZW5kcyBSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50SWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGFwaTogSVJlc3RBcGk7XG5cbiAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogTGF0ZXN0RGVwbG95bWVudFJlc291cmNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEZXBsb3ltZW50UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5yZXNvdXJjZSA9IG5ldyBMYXRlc3REZXBsb3ltZW50UmVzb3VyY2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgcmVzdEFwaTogcHJvcHMuYXBpLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLnJldGFpbkRlcGxveW1lbnRzKSB7XG4gICAgICB0aGlzLnJlc291cmNlLmFwcGx5UmVtb3ZhbFBvbGljeShSZW1vdmFsUG9saWN5LlJFVEFJTik7XG4gICAgfVxuXG4gICAgdGhpcy5hcGkgPSBwcm9wcy5hcGk7XG4gICAgdGhpcy5kZXBsb3ltZW50SWQgPSBMYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVzb3VyY2UucmVmIH0pO1xuXG4gICAgaWYgKHByb3BzLmFwaSBpbnN0YW5jZW9mIFJlc3RBcGlCYXNlKSB7XG4gICAgICBwcm9wcy5hcGkuX2F0dGFjaERlcGxveW1lbnQodGhpcyk7XG4gICAgfVxuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkVG9Mb2dpY2FsSWQoZGF0YTogYW55KSB7XG4gICAgdGhpcy5yZXNvdXJjZS5hZGRUb0xvZ2ljYWxJZChkYXRhKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBRdW90aW5nIGZyb20gQ2xvdWRGb3JtYXRpb24ncyBkb2NzOlxuICAgKlxuICAgKiAgIElmIHlvdSBjcmVhdGUgYW4gQVdTOjpBcGlHYXRld2F5OjpSZXN0QXBpIHJlc291cmNlIGFuZCBpdHMgbWV0aG9kcyAodXNpbmdcbiAgICogICBBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCkgaW4gdGhlIHNhbWUgdGVtcGxhdGUgYXMgeW91ciBkZXBsb3ltZW50LCB0aGVcbiAgICogICBkZXBsb3ltZW50IG11c3QgZGVwZW5kIG9uIHRoZSBSZXN0QXBpJ3MgbWV0aG9kcy4gVG8gY3JlYXRlIGEgZGVwZW5kZW5jeSxcbiAgICogICBhZGQgYSBEZXBlbmRzT24gYXR0cmlidXRlIHRvIHRoZSBkZXBsb3ltZW50LiBJZiB5b3UgZG9uJ3QsIEFXU1xuICAgKiAgIENsb3VkRm9ybWF0aW9uIGNyZWF0ZXMgdGhlIGRlcGxveW1lbnQgcmlnaHQgYWZ0ZXIgaXQgY3JlYXRlcyB0aGUgUmVzdEFwaVxuICAgKiAgIHJlc291cmNlIHRoYXQgZG9lc24ndCBjb250YWluIGFueSBtZXRob2RzLCBhbmQgQVdTIENsb3VkRm9ybWF0aW9uXG4gICAqICAgZW5jb3VudGVycyB0aGUgZm9sbG93aW5nIGVycm9yOiBUaGUgUkVTVCBBUEkgZG9lc24ndCBjb250YWluIGFueSBtZXRob2RzLlxuICAgKlxuICAgKiBAcGFyYW0gbWV0aG9kIFRoZSBtZXRob2QgdG8gYWRkIGFzIGEgZGVwZW5kZW5jeSBvZiB0aGUgZGVwbG95bWVudFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1hcGlnYXRld2F5LWRlcGxveW1lbnQuaHRtbFxuICAgKiBAc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzYxNjVcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2FkZE1ldGhvZERlcGVuZGVuY3kobWV0aG9kOiBNZXRob2QpIHtcbiAgICAvLyBhZGRpbmcgYSBkZXBlbmRlbmN5IGJldHdlZW4gdGhlIGNvbnN0cnVjdHMgdXNpbmcgYG5vZGUuYWRkRGVwZW5kZW5jeSgpYFxuICAgIC8vIHdpbGwgY3JlYXRlIGFkZGl0aW9uYWwgZGVwZW5kZW5jaWVzIGJldHdlZW4gYEFXUzo6QXBpR2F0ZXdheTo6RGVwbG95bWVudGBcbiAgICAvLyBhbmQgdGhlIGBBV1M6OkxhbWJkYTo6UGVybWlzc2lvbmAgcmVzb3VyY2VzIChjaGlsZHJlbiB1bmRlciBNZXRob2QpLFxuICAgIC8vIGNhdXNpbmcgY3ljbGljIGRlcGVuZGVuY3kgZXJyb3JzLiBIZW5jZSwgZmFsbGluZyBiYWNrIHRvIGRlY2xhcmluZ1xuICAgIC8vIGRlcGVuZGVuY2llcyBiZXR3ZWVuIHRoZSB1bmRlcmx5aW5nIENmblJlc291cmNlcy5cbiAgICB0aGlzLm5vZGUuYWRkRGVwZW5kZW5jeShtZXRob2Qubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuUmVzb3VyY2UpO1xuICB9XG59XG5cbmludGVyZmFjZSBMYXRlc3REZXBsb3ltZW50UmVzb3VyY2VQcm9wcyB7XG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICByZWFkb25seSByZXN0QXBpOiBJUmVzdEFwaTtcbn1cblxuY2xhc3MgTGF0ZXN0RGVwbG95bWVudFJlc291cmNlIGV4dGVuZHMgQ2ZuRGVwbG95bWVudCB7XG4gIHByaXZhdGUgcmVhZG9ubHkgaGFzaENvbXBvbmVudHMgPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIHJlYWRvbmx5IG9yaWdpbmFsTG9naWNhbElkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgYXBpOiBJUmVzdEFwaTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29yZUNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IExhdGVzdERlcGxveW1lbnRSZXNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICByZXN0QXBpSWQ6IHByb3BzLnJlc3RBcGkucmVzdEFwaUlkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcGkgPSBwcm9wcy5yZXN0QXBpO1xuICAgIHRoaXMub3JpZ2luYWxMb2dpY2FsSWQgPSB0aGlzLnN0YWNrLmdldExvZ2ljYWxJZCh0aGlzKTtcbiAgICB0aGlzLm92ZXJyaWRlTG9naWNhbElkKExhenkudW5jYWNoZWRTdHJpbmcoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmNhbGN1bGF0ZUxvZ2ljYWxJZCgpIH0pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbGxvd3MgYWRkaW5nIGFyYml0cmFyeSBkYXRhIHRvIHRoZSBoYXNoZWQgbG9naWNhbCBJRCBvZiB0aGlzIGRlcGxveW1lbnQuXG4gICAqIFRoaXMgY2FuIGJlIHVzZWQgdG8gY291cGxlIHRoZSBkZXBsb3ltZW50IHRvIHRoZSBBUEkgR2F0ZXdheSBtb2RlbC5cbiAgICovXG4gIHB1YmxpYyBhZGRUb0xvZ2ljYWxJZChkYXRhOiB1bmtub3duKSB7XG4gICAgLy8gaWYgdGhlIGNvbnN0cnVjdCBpcyBsb2NrZWQsIGl0IG1lYW5zIHdlIGFyZSBhbHJlYWR5IHN5bnRoZXNpemluZyBhbmQgdGhlblxuICAgIC8vIHdlIGNhbid0IG1vZGlmeSB0aGUgaGFzaCBiZWNhdXNlIHdlIG1pZ2h0IGhhdmUgYWxyZWFkeSBjYWxjdWxhdGVkIGl0LlxuICAgIGlmICh0aGlzLm5vZGUubG9ja2VkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBtb2RpZnkgdGhlIGxvZ2ljYWwgSUQgd2hlbiB0aGUgY29uc3RydWN0IGlzIGxvY2tlZCcpO1xuICAgIH1cblxuICAgIHRoaXMuaGFzaENvbXBvbmVudHMucHVzaChkYXRhKTtcbiAgfVxuXG4gIHByaXZhdGUgY2FsY3VsYXRlTG9naWNhbElkKCkge1xuICAgIGNvbnN0IGhhc2ggPSBbLi4udGhpcy5oYXNoQ29tcG9uZW50c107XG5cbiAgICBpZiAodGhpcy5hcGkgaW5zdGFuY2VvZiBSZXN0QXBpIHx8IHRoaXMuYXBpIGluc3RhbmNlb2YgU3BlY1Jlc3RBcGkpIHsgLy8gSWdub3JlIElSZXN0QXBpIHRoYXQgYXJlIGltcG9ydGVkXG5cbiAgICAgIC8vIEFkZCBDZm5SZXN0QXBpIHRvIHRoZSBsb2dpY2FsIGlkIHNvIGEgbmV3IGRlcGxveW1lbnQgaXMgdHJpZ2dlcmVkIHdoZW4gYW55IG9mIGl0cyBwcm9wZXJ0aWVzIGNoYW5nZS5cbiAgICAgIGNvbnN0IGNmblJlc3RBcGlDRiA9ICh0aGlzLmFwaS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBhbnkpLl90b0Nsb3VkRm9ybWF0aW9uKCk7XG4gICAgICBoYXNoLnB1c2godGhpcy5zdGFjay5yZXNvbHZlKGNmblJlc3RBcGlDRikpO1xuICAgIH1cblxuICAgIGxldCBsaWQgPSB0aGlzLm9yaWdpbmFsTG9naWNhbElkO1xuXG4gICAgLy8gaWYgaGFzaCBjb21wb25lbnRzIHdlcmUgYWRkZWQgdG8gdGhlIGRlcGxveW1lbnQsIHdlIHVzZSB0aGVtIHRvIGNhbGN1bGF0ZVxuICAgIC8vIGEgbG9naWNhbCBJRCBmb3IgdGhlIGRlcGxveW1lbnQgcmVzb3VyY2UuXG4gICAgaWYgKGhhc2gubGVuZ3RoID4gMCkge1xuICAgICAgY29uc3QgbWQ1ID0gY3J5cHRvLmNyZWF0ZUhhc2goJ21kNScpO1xuICAgICAgaGFzaC5tYXAoeCA9PiB0aGlzLnN0YWNrLnJlc29sdmUoeCkpLmZvckVhY2goYyA9PiBtZDUudXBkYXRlKEpTT04uc3RyaW5naWZ5KGMpKSk7XG4gICAgICBsaWQgKz0gbWQ1LmRpZ2VzdCgnaGV4Jyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGxpZDtcbiAgfVxufVxuIl19