import ec2 = require('@aws-cdk/aws-ec2');
import { Construct, Resource } from '@aws-cdk/cdk';
import { HostedZoneAttributes, IHostedZone } from './hosted-zone-ref';
import { CfnHostedZone } from './route53.generated';
export interface CommonHostedZoneProps {
    /**
     * The name of the domain. For resource record types that include a domain
     * name, specify a fully qualified domain name.
     */
    readonly zoneName: string;
    /**
     * Any comments that you want to include about the hosted zone.
     *
     * @default none
     */
    readonly comment?: string;
    /**
     * The Amazon Resource Name (ARN) for the log group that you want Amazon Route 53 to send query logs to.
     *
     * @default disabled
     */
    readonly queryLogsLogGroupArn?: string;
}
/**
 * Properties of a new hosted zone
 */
export interface HostedZoneProps extends CommonHostedZoneProps {
    /**
     * A VPC that you want to associate with this hosted zone. When you specify
     * this property, a private hosted zone will be created.
     *
     * You can associate additional VPCs to this private zone using `addVpc(vpc)`.
     *
     * @default public (no VPCs associated)
     */
    readonly vpcs?: ec2.IVpcNetwork[];
}
export declare class HostedZone extends Resource implements IHostedZone {
    static fromHostedZoneId(scope: Construct, id: string, hostedZoneId: string): IHostedZone;
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope: Construct, id: string, attrs: HostedZoneAttributes): IHostedZone;
    readonly hostedZoneId: string;
    readonly zoneName: string;
    readonly hostedZoneNameServers?: string[];
    /**
     * VPCs to which this hosted zone will be added
     */
    protected readonly vpcs: CfnHostedZone.VPCProperty[];
    constructor(scope: Construct, id: string, props: HostedZoneProps);
    export(): HostedZoneAttributes;
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc: ec2.IVpcNetwork): void;
}
export interface PublicHostedZoneProps extends CommonHostedZoneProps {
}
export interface IPublicHostedZone extends IHostedZone {
}
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
export declare class PublicHostedZone extends HostedZone implements IPublicHostedZone {
    static fromPublicHostedZoneId(scope: Construct, id: string, publicHostedZoneId: string): IPublicHostedZone;
    constructor(scope: Construct, id: string, props: PublicHostedZoneProps);
    addVpc(_vpc: ec2.IVpcNetwork): void;
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate: PublicHostedZone, opts?: ZoneDelegationOptions): void;
}
/**
 * Options available when creating a delegation relationship from one PublicHostedZone to another.
 */
export interface ZoneDelegationOptions {
    /**
     * A comment to add on the DNS record created to incorporate the delegation.
     *
     * @default none
     */
    readonly comment?: string;
    /**
     * The TTL (Time To Live) of the DNS delegation record in DNS caches.
     *
     * @default 172800
     */
    readonly ttl?: number;
}
export interface PrivateHostedZoneProps extends CommonHostedZoneProps {
    /**
     * A VPC that you want to associate with this hosted zone.
     *
     * Private hosted zones must be associated with at least one VPC. You can
     * associated additional VPCs using `addVpc(vpc)`.
     */
    readonly vpc: ec2.IVpcNetwork;
}
export interface IPrivateHostedZone extends IHostedZone {
}
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
export declare class PrivateHostedZone extends HostedZone implements IPrivateHostedZone {
    static fromPrivateHostedZoneId(scope: Construct, id: string, privateHostedZoneId: string): IPrivateHostedZone;
    constructor(scope: Construct, id: string, props: PrivateHostedZoneProps);
}
