"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
class TestApp {
    constructor() {
        this.app = new cdk.App();
        const account = '123456789012';
        const region = 'bermuda-triangle';
        this.app.node.setContext(`availability-zones:${account}:${region}`, [`${region}-1a`]);
        this.stack = new cdk.Stack(this.app, 'MyStack', { env: { account, region } });
    }
    synthesizeTemplate() {
        return this.app.synthesizeStack(this.stack.name);
    }
}
module.exports = {
    'default properties': {
        'public hosted zone'(test) {
            const app = new TestApp();
            new lib_1.PublicHostedZone(app.stack, 'HostedZone', { zoneName: 'test.public' });
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.exactlyMatchTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.public."
                        }
                    }
                }
            }));
            test.done();
        },
        'private hosted zone'(test) {
            const app = new TestApp();
            const vpcNetwork = new ec2.VpcNetwork(app.stack, 'VPC');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetwork });
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPCB9E5F0B4' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        },
        'when specifying multiple VPCs'(test) {
            const app = new TestApp();
            const vpcNetworkA = new ec2.VpcNetwork(app.stack, 'VPC1');
            const vpcNetworkB = new ec2.VpcNetwork(app.stack, 'VPC2');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetworkA })
                .addVpc(vpcNetworkB);
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPC17DE2CF87' },
                                    VPCRegion: 'bermuda-triangle'
                                },
                                {
                                    VPCId: { Ref: 'VPC2C1F0E711' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        }
    },
    'exporting and importing works'(test) {
        const stack1 = new cdk.Stack();
        const stack2 = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack1, 'Zone', {
            zoneName: 'cdk.local',
        });
        const zoneRef = zone.export();
        const importedZone = lib_1.HostedZone.fromHostedZoneAttributes(stack2, 'Imported', zoneRef);
        new lib_1.TxtRecord(importedZone, 'Record', {
            zone: importedZone,
            recordName: 'lookHere',
            recordValue: 'SeeThere'
        });
        assert_1.expect(stack1).to(assert_1.exactlyMatchTemplate({
            Resources: {
                ZoneA5DE4B68: {
                    Type: "AWS::Route53::HostedZone",
                    Properties: {
                        Name: "cdk.local."
                    }
                }
            },
            Outputs: {
                ZoneHostedZoneId413B8768: {
                    Value: { Ref: "ZoneA5DE4B68" },
                    Export: { Name: "Stack:ZoneHostedZoneId413B8768" }
                }
            }
        }));
        assert_1.expect(stack2).to(assert_1.haveResource("AWS::Route53::RecordSet", {
            HostedZoneId: { "Fn::ImportValue": "Stack:ZoneHostedZoneId413B8768" },
            Name: "lookHere.cdk.local.",
            ResourceRecords: ["\"SeeThere\""],
            Type: "TXT"
        }));
        test.done();
    },
    'adds period to name if not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            Name: 'zonename.'
        }));
        test.done();
    },
    'fails if zone name ends with a trailing dot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new lib_1.HostedZone(stack, 'MyHostedZone', { zoneName: 'zonename.' }), /zone name must not end with a trailing dot/);
        test.done();
    },
    'a hosted zone can be assiciated with a VPC either upon creation or using "addVpc"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc1 = new ec2.VpcNetwork(stack, 'VPC1');
        const vpc2 = new ec2.VpcNetwork(stack, 'VPC2');
        const vpc3 = new ec2.VpcNetwork(stack, 'VPC3');
        // WHEN
        const zone = new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename',
            vpcs: [vpc1, vpc2]
        });
        zone.addVpc(vpc3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            VPCs: [
                {
                    VPCId: {
                        Ref: "VPC17DE2CF87"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC2C1F0E711"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC3CB5FCDA8"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                }
            ]
        }));
        test.done();
    },
    'public zone cannot be associated with a vpc (runtime error)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'MyHostedZone', { zoneName: 'zonename' });
        const vpc = new ec2.VpcNetwork(stack, 'VPC');
        // THEN
        test.throws(() => zone.addVpc(vpc), /Cannot associate public hosted zones with a VPC/);
        test.done();
    },
    'setting up zone delegation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'TopZone', { zoneName: 'top.test' });
        const delegate = new lib_1.PublicHostedZone(stack, 'SubZone', { zoneName: 'sub.top.test' });
        // WHEN
        zone.addDelegation(delegate, { ttl: 1337 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'NS',
            Name: 'sub.top.test.',
            HostedZoneId: zone.node.resolve(zone.hostedZoneId),
            ResourceRecords: zone.node.resolve(delegate.hostedZoneNameServers),
            TTL: '1337',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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