"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionsIntegration = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const integration_1 = require("../integration");
const aws_1 = require("./aws");
/**
 * Options to integrate with various StepFunction API.
 *
 * @stability stable
 */
class StepFunctionsIntegration {
    /**
     * Integrates a Synchronous Express State Machine from AWS Step Functions to an API Gateway method.
     *
     * @stability stable
     * @example
     *
     *    const stateMachine = new stepfunctions.StateMachine(this, 'MyStateMachine', {
     *       definition: stepfunctions.Chain.start(new stepfunctions.Pass(this, 'Pass')),
     *    });
     *
     *    const api = new apigateway.RestApi(this, 'Api', {
     *       restApiName: 'MyApi',
     *    });
     *    api.root.addMethod('GET', apigateway.StepFunctionsIntegration.startExecution(stateMachine));
     */
    static startExecution(stateMachine, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_StepFunctionsExecutionIntegrationOptions(options);
        return new StepFunctionsExecutionIntegration(stateMachine, options);
    }
}
exports.StepFunctionsIntegration = StepFunctionsIntegration;
_a = JSII_RTTI_SYMBOL_1;
StepFunctionsIntegration[_a] = { fqn: "@aws-cdk/aws-apigateway.StepFunctionsIntegration", version: "1.138.0" };
class StepFunctionsExecutionIntegration extends aws_1.AwsIntegration {
    constructor(stateMachine, options = {}) {
        super({
            service: 'states',
            action: 'StartSyncExecution',
            options: {
                credentialsRole: options.credentialsRole,
                integrationResponses: integrationResponse(),
                passthroughBehavior: integration_1.PassthroughBehavior.NEVER,
                requestTemplates: requestTemplates(stateMachine, options),
                ...options,
            },
        });
        this.stateMachine = stateMachine;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        this.stateMachine.grantExecution(principal, 'states:StartSyncExecution');
        let stateMachineName;
        if (this.stateMachine instanceof sfn.StateMachine) {
            const stateMachineType = this.stateMachine.stateMachineType;
            if (stateMachineType !== sfn.StateMachineType.EXPRESS) {
                throw new Error('State Machine must be of type "EXPRESS". Please use StateMachineType.EXPRESS as the stateMachineType');
            }
            //if not imported, extract the name from the CFN layer to reach the
            //literal value if it is given (rather than a token)
            stateMachineName = this.stateMachine.node.defaultChild.stateMachineName;
        }
        else {
            //imported state machine
            stateMachineName = `StateMachine-${this.stateMachine.stack.node.addr}`;
        }
        let deploymentToken;
        if (stateMachineName !== undefined && !core_1.Token.isUnresolved(stateMachineName)) {
            deploymentToken = JSON.stringify({ stateMachineName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
/**
 * Defines the integration response that passes the result on success,
 * or the error on failure, from the synchronous execution to the caller.
 *
 * @returns integrationResponse mapping
 */
function integrationResponse() {
    const errorResponse = [
        {
            /**
             * Specifies the regular expression (regex) pattern used to choose
             * an integration response based on the response from the back end.
             * In this case it will match all '4XX' HTTP Errors
             */
            selectionPattern: '4\\d{2}',
            statusCode: '400',
            responseTemplates: {
                'application/json': `{
            "error": "Bad request!"
          }`,
            },
        },
        {
            /**
             * Match all '5XX' HTTP Errors
             */
            selectionPattern: '5\\d{2}',
            statusCode: '500',
            responseTemplates: {
                'application/json': '"error": $input.path(\'$.error\')',
            },
        },
    ];
    const integResponse = [
        {
            statusCode: '200',
            responseTemplates: {
                /* eslint-disable */
                'application/json': [
                    '#set($inputRoot = $input.path(\'$\'))',
                    '#if($input.path(\'$.status\').toString().equals("FAILED"))',
                    '#set($context.responseOverride.status = 500)',
                    '{',
                    '"error": "$input.path(\'$.error\')"',
                    '"cause": "$input.path(\'$.cause\')"',
                    '}',
                    '#else',
                    '$input.path(\'$.output\')',
                    '#end',
                ].join('\n'),
            },
        },
        ...errorResponse,
    ];
    return integResponse;
}
/**
 * Defines the request template that will be used for the integration
 * @param stateMachine
 * @param options
 * @returns requestTemplate
 */
function requestTemplates(stateMachine, options) {
    const templateStr = templateString(stateMachine, options);
    const requestTemplate = {
        'application/json': templateStr,
    };
    return requestTemplate;
}
/**
 * Reads the VTL template and returns the template string to be used
 * for the request template.
 *
 * @param stateMachine
 * @param includeRequestContext
 * @param options
 * @reutrns templateString
 */
function templateString(stateMachine, options) {
    var _b, _c, _d;
    let templateStr;
    let requestContextStr = '';
    const includeHeader = (_b = options.headers) !== null && _b !== void 0 ? _b : false;
    const includeQueryString = (_c = options.querystring) !== null && _c !== void 0 ? _c : true;
    const includePath = (_d = options.path) !== null && _d !== void 0 ? _d : true;
    if (options.requestContext && Object.keys(options.requestContext).length > 0) {
        requestContextStr = requestContext(options.requestContext);
    }
    templateStr = fs.readFileSync(path.join(__dirname, 'stepfunctions.vtl'), { encoding: 'utf-8' });
    templateStr = templateStr.replace('%STATEMACHINE%', stateMachine.stateMachineArn);
    templateStr = templateStr.replace('%INCLUDE_HEADERS%', String(includeHeader));
    templateStr = templateStr.replace('%INCLUDE_QUERYSTRING%', String(includeQueryString));
    templateStr = templateStr.replace('%INCLUDE_PATH%', String(includePath));
    templateStr = templateStr.replace('%REQUESTCONTEXT%', requestContextStr);
    return templateStr;
}
function requestContext(requestContextObj) {
    const context = {
        accountId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.accountId) ? '$context.identity.accountId' : undefined,
        apiId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiId) ? '$context.apiId' : undefined,
        apiKey: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiKey) ? '$context.identity.apiKey' : undefined,
        authorizerPrincipalId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.authorizerPrincipalId) ? '$context.authorizer.principalId' : undefined,
        caller: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.caller) ? '$context.identity.caller' : undefined,
        cognitoAuthenticationProvider: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationProvider) ? '$context.identity.cognitoAuthenticationProvider' : undefined,
        cognitoAuthenticationType: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationType) ? '$context.identity.cognitoAuthenticationType' : undefined,
        cognitoIdentityId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityId) ? '$context.identity.cognitoIdentityId' : undefined,
        cognitoIdentityPoolId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityPoolId) ? '$context.identity.cognitoIdentityPoolId' : undefined,
        httpMethod: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.httpMethod) ? '$context.httpMethod' : undefined,
        stage: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.stage) ? '$context.stage' : undefined,
        sourceIp: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.sourceIp) ? '$context.identity.sourceIp' : undefined,
        user: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.user) ? '$context.identity.user' : undefined,
        userAgent: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userAgent) ? '$context.identity.userAgent' : undefined,
        userArn: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userArn) ? '$context.identity.userArn' : undefined,
        requestId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.requestId) ? '$context.requestId' : undefined,
        resourceId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourceId) ? '$context.resourceId' : undefined,
        resourcePath: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourcePath) ? '$context.resourcePath' : undefined,
    };
    const contextAsString = JSON.stringify(context);
    // The VTL Template conflicts with double-quotes (") for strings.
    // Before sending to the template, we replace double-quotes (") with @@ and replace it back inside the .vtl file
    const doublequotes = '"';
    const replaceWith = '@@';
    return contextAsString.split(doublequotes).join(replaceWith);
}
//# sourceMappingURL=data:application/json;base64,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