"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const sqs = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': expectedActions,
                    'Effect': 'Allow',
                    'Resource': {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn',
                        ],
                    },
                },
            ],
            'Version': '2012-10-17',
        },
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const q = new sqs.Queue(stack, 'Queue');
        test.deepEqual(q.fifo, false);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                },
            },
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new core_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'DLQ581697C4': {
                    'Type': 'AWS::SQS::Queue',
                },
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'RedrivePolicy': {
                            'deadLetterTargetArn': {
                                'Fn::GetAtt': [
                                    'DLQ581697C4',
                                    'Arn',
                                ],
                            },
                            'maxReceiveCount': 3,
                        },
                    },
                },
            },
        });
        test.done();
    },
    'message retention period must be between 1 minute to 14 days'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new sqs.Queue(stack, 'MyQueue', {
            retentionPeriod: core_1.Duration.seconds(30),
        }), /message retention period must be 60 seconds or more/);
        test.throws(() => new sqs.Queue(stack, 'AnotherQueue', {
            retentionPeriod: core_1.Duration.days(15),
        }), /message retention period must be 1209600 seconds of less/);
        test.done();
    },
    'message retention period can be provided as a parameter'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'my-retention-period', {
            type: 'Number',
            default: 30,
        });
        // WHEN
        new sqs.Queue(stack, 'MyQueue', {
            retentionPeriod: core_1.Duration.seconds(parameter.valueAsNumber),
        });
        // THEN
        assert_1.expect(stack).toMatch({
            'Parameters': {
                'myretentionperiod': {
                    'Type': 'Number',
                    'Default': 30,
                },
            },
            'Resources': {
                'MyQueueE6CA6235': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'MessageRetentionPeriod': {
                            'Ref': 'myretentionperiod',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sqs:*'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyQueueE6CA6235': {
                    'Type': 'AWS::SQS::Queue',
                },
                'MyQueuePolicy6BBEDDAC': {
                    'Type': 'AWS::SQS::QueuePolicy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sqs:*',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': 'arn',
                                    },
                                    'Resource': '*',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'Queues': [
                            {
                                'Ref': 'MyQueueE6CA6235',
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'export and import': {
        'importing works correctly'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imports = sqs.Queue.fromQueueArn(stack, 'Imported', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            // THEN
            // "import" returns an IQueue bound to `Fn::ImportValue`s.
            test.deepEqual(stack.resolve(imports.queueArn), 'arn:aws:sqs:us-east-1:123456789012:queue1');
            test.deepEqual(stack.resolve(imports.queueUrl), { 'Fn::Join': ['', ['https://sqs.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/', { Ref: 'AWS::AccountId' }, '/queue1']] });
            test.deepEqual(stack.resolve(imports.queueName), 'queue1');
            test.done();
        },
        'importing fifo and standard queues are detected correctly'(test) {
            const stack = new core_1.Stack();
            const stdQueue = sqs.Queue.fromQueueArn(stack, 'StdQueue', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            const fifoQueue = sqs.Queue.fromQueueArn(stack, 'FifoQueue', 'arn:aws:sqs:us-east-1:123456789012:queue2.fifo');
            test.deepEqual(stdQueue.fifo, false);
            test.deepEqual(fifoQueue.fifo, true);
            test.done();
        },
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'service:hello', 'service:world'), 'service:hello', 'service:world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new core_1.Stack();
            const queue = sqs.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url',
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:PurgeQueue',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl',
                            ],
                            'Effect': 'Allow',
                            'Resource': 'arn:aws:sqs:us-east-1:123456789012:queue1',
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': { 'Fn::GetAtt': ['CustomKey1E6D0D07', 'Arn'] },
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': {
                    'Fn::GetAtt': [
                        'QueueKey39FCBAE6',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS_MANAGED });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'Queue4A7E3555': {
                        'Type': 'AWS::SQS::Queue',
                        'Properties': {
                            'KmsMasterKeyId': 'alias/aws/sqs',
                        },
                    },
                },
            });
            test.done();
        },
        'grant also affects key on encrypted queue'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', {
                encryption: sqs.QueueEncryption.KMS,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('someone'),
            });
            // WHEN
            queue.grantSendMessages(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:SendMessage',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl',
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] },
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['QueueKey39FCBAE6', 'Arn'] },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'test ".fifo" suffixed queues register as fifo'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            queueName: 'MyQueue.fifo',
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'QueueName': 'MyQueue.fifo',
                        'FifoQueue': true,
                    },
                },
            },
        });
        test.done();
    },
    'test a fifo queue is observed when the "fifo" property is specified'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            fifo: true,
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'FifoQueue': true,
                    },
                },
            },
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.resolve(queue.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.deepEqual(stack.resolve(queue.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zcXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnNxcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQXVEO0FBQ3ZELHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsd0NBQThEO0FBRTlELDhCQUE4QjtBQXlaOUIsU0FBUyxTQUFTLENBQUMsTUFBeUQsRUFBRSxHQUFHLGVBQXlCO0lBQ3hHLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUM5QyxNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRTlDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFFekIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1FBQ2hELGdCQUFnQixFQUFFO1lBQ2hCLFdBQVcsRUFBRTtnQkFDWDtvQkFDRSxRQUFRLEVBQUUsZUFBZTtvQkFDekIsUUFBUSxFQUFFLE9BQU87b0JBQ2pCLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osaUJBQWlCOzRCQUNqQixLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxTQUFTLEVBQUUsWUFBWTtTQUN4QjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQztBQTdhRCxpQkFBUztJQUNQLG9CQUFvQixDQUFDLElBQVU7UUFDN0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXhDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUU5QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFLGlCQUFpQjtpQkFDMUI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLGVBQWUsRUFBRSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV2RixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxhQUFhLEVBQUU7b0JBQ2IsTUFBTSxFQUFFLGlCQUFpQjtpQkFDMUI7Z0JBQ0QsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRSxpQkFBaUI7b0JBQ3pCLFlBQVksRUFBRTt3QkFDWixlQUFlLEVBQUU7NEJBQ2YscUJBQXFCLEVBQUU7Z0NBQ3JCLFlBQVksRUFBRTtvQ0FDWixhQUFhO29DQUNiLEtBQUs7aUNBQ047NkJBQ0Y7NEJBQ0QsaUJBQWlCLEVBQUUsQ0FBQzt5QkFDckI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ2hELGVBQWUsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUN0QyxDQUFDLEVBQUUscURBQXFELENBQUMsQ0FBQztRQUUzRCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ3JELGVBQWUsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztTQUNuQyxDQUFDLEVBQUUsMERBQTBELENBQUMsQ0FBQztRQUVoRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseURBQXlELENBQUMsSUFBVTtRQUNsRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLFNBQVMsR0FBRyxJQUFJLG1CQUFZLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1lBQy9ELElBQUksRUFBRSxRQUFRO1lBQ2QsT0FBTyxFQUFFLEVBQUU7U0FDWixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDOUIsZUFBZSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNwQixZQUFZLEVBQUU7Z0JBQ1osbUJBQW1CLEVBQUU7b0JBQ25CLE1BQU0sRUFBRSxRQUFRO29CQUNoQixTQUFTLEVBQUUsRUFBRTtpQkFDZDthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLGlCQUFpQixFQUFFO29CQUNqQixNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osd0JBQXdCLEVBQUU7NEJBQ3hCLEtBQUssRUFBRSxtQkFBbUI7eUJBQzNCO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0RBQStELENBQUMsSUFBVTtRQUN4RSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDOUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNoRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDO1lBQ2xCLFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMxQyxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLGlCQUFpQixFQUFFO29CQUNqQixNQUFNLEVBQUUsaUJBQWlCO2lCQUMxQjtnQkFDRCx1QkFBdUIsRUFBRTtvQkFDdkIsTUFBTSxFQUFFLHVCQUF1QjtvQkFDL0IsWUFBWSxFQUFFO3dCQUNaLGdCQUFnQixFQUFFOzRCQUNoQixXQUFXLEVBQUU7Z0NBQ1g7b0NBQ0UsUUFBUSxFQUFFLE9BQU87b0NBQ2pCLFFBQVEsRUFBRSxPQUFPO29DQUNqQixXQUFXLEVBQUU7d0NBQ1gsS0FBSyxFQUFFLEtBQUs7cUNBQ2I7b0NBQ0QsVUFBVSxFQUFFLEdBQUc7aUNBQ2hCOzZCQUNGOzRCQUNELFNBQVMsRUFBRSxZQUFZO3lCQUN4Qjt3QkFDRCxRQUFRLEVBQUU7NEJBQ1I7Z0NBQ0UsS0FBSyxFQUFFLGlCQUFpQjs2QkFDekI7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQkFBbUIsRUFBRTtRQUNuQiwyQkFBMkIsQ0FBQyxJQUFVO1lBQ3BDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLDJDQUEyQyxDQUFDLENBQUM7WUFFdkcsT0FBTztZQUVQLDBEQUEwRDtZQUMxRCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLDJDQUEyQyxDQUFDLENBQUM7WUFDN0YsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLFVBQVUsRUFDMUQsQ0FBRSxFQUFFLEVBQUUsQ0FBRSxjQUFjLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsU0FBUyxDQUFFLENBQUUsRUFBRSxDQUFDLENBQUM7WUFDbkksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUMzRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMkRBQTJELENBQUMsSUFBVTtZQUNwRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsMkNBQTJDLENBQUMsQ0FBQztZQUN4RyxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLGdEQUFnRCxDQUFDLENBQUM7WUFDL0csSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3JDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxRQUFRLEVBQUU7UUFDUixzQkFBc0IsQ0FBQyxJQUFVO1lBQy9CLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsRUFDM0Msb0JBQW9CLEVBQ3BCLDZCQUE2QixFQUM3QixpQkFBaUIsRUFDakIsbUJBQW1CLEVBQ25CLHdCQUF3QixDQUN6QixDQUFDO1lBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1CQUFtQixDQUFDLElBQVU7WUFDNUIsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxFQUN4QyxpQkFBaUIsRUFDakIsd0JBQXdCLEVBQ3hCLGlCQUFpQixDQUNsQixDQUFDO1lBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELFlBQVksQ0FBQyxJQUFVO1lBQ3JCLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQ2pDLGdCQUFnQixFQUNoQix3QkFBd0IsRUFDeEIsaUJBQWlCLENBQ2xCLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNEJBQTRCLENBQUMsSUFBVTtZQUNyQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxlQUFlLEVBQUUsZUFBZSxDQUFDLEVBQzlELGVBQWUsRUFDZixlQUFlLENBQ2hCLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUNBQXFDLENBQUMsSUFBVTtZQUM5QyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDM0QsUUFBUSxFQUFFLDJDQUEyQztnQkFDckQsUUFBUSxFQUFFLG1CQUFtQjthQUM5QixDQUFDLENBQUM7WUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBRXpDLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFdkIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRTtnQ0FDUixnQkFBZ0I7Z0NBQ2hCLHdCQUF3QjtnQ0FDeEIsaUJBQWlCOzZCQUNsQjs0QkFDRCxRQUFRLEVBQUUsT0FBTzs0QkFDakIsVUFBVSxFQUFFLDJDQUEyQzt5QkFDeEQ7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFLFlBQVk7aUJBQ3hCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxrQkFBa0IsRUFBRTtRQUNsQixvREFBb0QsQ0FBQyxJQUFVO1lBQzdELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFFMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztZQUM1QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLG1CQUFtQixFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFFMUUsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFDMUMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGlCQUFpQixFQUFFO2dCQUMvQyxnQkFBZ0IsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLG1CQUFtQixFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQ25FLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1GQUFtRixDQUFDLElBQVU7WUFDNUYsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFFdkUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUM7WUFDaEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGlCQUFpQixFQUFFO2dCQUMvQyxnQkFBZ0IsRUFBRTtvQkFDaEIsWUFBWSxFQUFFO3dCQUNaLGtCQUFrQjt3QkFDbEIsS0FBSztxQkFDTjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHlDQUF5QyxDQUFDLElBQVU7WUFDbEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7WUFDL0UsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztnQkFDcEIsV0FBVyxFQUFFO29CQUNYLGVBQWUsRUFBRTt3QkFDZixNQUFNLEVBQUUsaUJBQWlCO3dCQUN6QixZQUFZLEVBQUU7NEJBQ1osZ0JBQWdCLEVBQUUsZUFBZTt5QkFDbEM7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMkNBQTJDLENBQUMsSUFBVTtZQUNwRCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDMUMsVUFBVSxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRzthQUNwQyxDQUFDLENBQUM7WUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLFNBQVMsQ0FBQzthQUMvQyxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsS0FBSyxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBRTlCLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFO2dDQUNSLGlCQUFpQjtnQ0FDakIsd0JBQXdCO2dDQUN4QixpQkFBaUI7NkJBQ2xCOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixVQUFVLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxlQUFlLEVBQUUsS0FBSyxDQUFFLEVBQUU7eUJBQ3pEO3dCQUNEOzRCQUNFLFFBQVEsRUFBRTtnQ0FDUixhQUFhO2dDQUNiLGFBQWE7Z0NBQ2IsZ0JBQWdCO2dDQUNoQixzQkFBc0I7NkJBQ3ZCOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixVQUFVLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxrQkFBa0IsRUFBRSxLQUFLLENBQUUsRUFBRTt5QkFDNUQ7cUJBQ0Y7b0JBQ0QsU0FBUyxFQUFFLFlBQVk7aUJBQ3hCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7WUFDMUMsU0FBUyxFQUFFLGNBQWM7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBRWpDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osV0FBVyxFQUFFLGNBQWM7d0JBQzNCLFdBQVcsRUFBRSxJQUFJO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFFQUFxRSxDQUFDLElBQVU7UUFDOUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtZQUMxQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUVqQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsWUFBWSxFQUFFO3dCQUNaLFdBQVcsRUFBRSxJQUFJO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGNBQWMsQ0FBQyxJQUFVO1FBQ3ZCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFNUMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsMEJBQTBCLEVBQUUsQ0FBQyxFQUFFO1lBQ2hFLFVBQVUsRUFBRSxFQUFDLFNBQVMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGVBQWUsRUFBRSxXQUFXLENBQUUsRUFBRSxFQUFDO1lBQzNFLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLFVBQVUsRUFBRSxzQkFBc0I7WUFDbEMsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSxLQUFLO1NBQ2pCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsQ0FBQyxFQUFFO1lBQzNELFVBQVUsRUFBRSxFQUFDLFNBQVMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGVBQWUsRUFBRSxXQUFXLENBQUUsRUFBRSxFQUFDO1lBQzNFLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLFVBQVUsRUFBRSxpQkFBaUI7WUFDN0IsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSxTQUFTO1NBQ3JCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0IHsgQ2ZuUGFyYW1ldGVyLCBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBzcXMgZnJvbSAnLi4vbGliJztcblxuLyogZXNsaW50LWRpc2FibGUgcXVvdGUtcHJvcHMgKi9cblxuZXhwb3J0ID0ge1xuICAnZGVmYXVsdCBwcm9wZXJ0aWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBxID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJyk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChxLmZpZm8sIGZhbHNlKTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICAnUmVzb3VyY2VzJzoge1xuICAgICAgICAnUXVldWU0QTdFMzU1NSc6IHtcbiAgICAgICAgICAnVHlwZSc6ICdBV1M6OlNRUzo6UXVldWUnLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnd2l0aCBhIGRlYWQgbGV0dGVyIHF1ZXVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBkbHEgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnRExRJyk7XG4gICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBkZWFkTGV0dGVyUXVldWU6IHsgcXVldWU6IGRscSwgbWF4UmVjZWl2ZUNvdW50OiAzIH0gfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgJ1Jlc291cmNlcyc6IHtcbiAgICAgICAgJ0RMUTU4MTY5N0M0Jzoge1xuICAgICAgICAgICdUeXBlJzogJ0FXUzo6U1FTOjpRdWV1ZScsXG4gICAgICAgIH0sXG4gICAgICAgICdRdWV1ZTRBN0UzNTU1Jzoge1xuICAgICAgICAgICdUeXBlJzogJ0FXUzo6U1FTOjpRdWV1ZScsXG4gICAgICAgICAgJ1Byb3BlcnRpZXMnOiB7XG4gICAgICAgICAgICAnUmVkcml2ZVBvbGljeSc6IHtcbiAgICAgICAgICAgICAgJ2RlYWRMZXR0ZXJUYXJnZXRBcm4nOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAnRExRNTgxNjk3QzQnLFxuICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ21heFJlY2VpdmVDb3VudCc6IDMsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21lc3NhZ2UgcmV0ZW50aW9uIHBlcmlvZCBtdXN0IGJlIGJldHdlZW4gMSBtaW51dGUgdG8gMTQgZGF5cycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScsIHtcbiAgICAgIHJldGVudGlvblBlcmlvZDogRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgfSksIC9tZXNzYWdlIHJldGVudGlvbiBwZXJpb2QgbXVzdCBiZSA2MCBzZWNvbmRzIG9yIG1vcmUvKTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBzcXMuUXVldWUoc3RhY2ssICdBbm90aGVyUXVldWUnLCB7XG4gICAgICByZXRlbnRpb25QZXJpb2Q6IER1cmF0aW9uLmRheXMoMTUpLFxuICAgIH0pLCAvbWVzc2FnZSByZXRlbnRpb24gcGVyaW9kIG11c3QgYmUgMTIwOTYwMCBzZWNvbmRzIG9mIGxlc3MvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtZXNzYWdlIHJldGVudGlvbiBwZXJpb2QgY2FuIGJlIHByb3ZpZGVkIGFzIGEgcGFyYW1ldGVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBhcmFtZXRlciA9IG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2ssICdteS1yZXRlbnRpb24tcGVyaW9kJywge1xuICAgICAgdHlwZTogJ051bWJlcicsXG4gICAgICBkZWZhdWx0OiAzMCxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScsIHtcbiAgICAgIHJldGVudGlvblBlcmlvZDogRHVyYXRpb24uc2Vjb25kcyhwYXJhbWV0ZXIudmFsdWVBc051bWJlciksXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgICdQYXJhbWV0ZXJzJzoge1xuICAgICAgICAnbXlyZXRlbnRpb25wZXJpb2QnOiB7XG4gICAgICAgICAgJ1R5cGUnOiAnTnVtYmVyJyxcbiAgICAgICAgICAnRGVmYXVsdCc6IDMwLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgICdSZXNvdXJjZXMnOiB7XG4gICAgICAgICdNeVF1ZXVlRTZDQTYyMzUnOiB7XG4gICAgICAgICAgJ1R5cGUnOiAnQVdTOjpTUVM6OlF1ZXVlJyxcbiAgICAgICAgICAnUHJvcGVydGllcyc6IHtcbiAgICAgICAgICAgICdNZXNzYWdlUmV0ZW50aW9uUGVyaW9kJzoge1xuICAgICAgICAgICAgICAnUmVmJzogJ215cmV0ZW50aW9ucGVyaW9kJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkVG9Qb2xpY3kgd2lsbCBhdXRvbWF0aWNhbGx5IGNyZWF0ZSBhIHBvbGljeSBmb3IgdGhpcyBxdWV1ZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScpO1xuICAgIHF1ZXVlLmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIGFjdGlvbnM6IFsnc3FzOionXSxcbiAgICAgIHByaW5jaXBhbHM6IFtuZXcgaWFtLkFyblByaW5jaXBhbCgnYXJuJyldLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICAnUmVzb3VyY2VzJzoge1xuICAgICAgICAnTXlRdWV1ZUU2Q0E2MjM1Jzoge1xuICAgICAgICAgICdUeXBlJzogJ0FXUzo6U1FTOjpRdWV1ZScsXG4gICAgICAgIH0sXG4gICAgICAgICdNeVF1ZXVlUG9saWN5NkJCRUREQUMnOiB7XG4gICAgICAgICAgJ1R5cGUnOiAnQVdTOjpTUVM6OlF1ZXVlUG9saWN5JyxcbiAgICAgICAgICAnUHJvcGVydGllcyc6IHtcbiAgICAgICAgICAgICdQb2xpY3lEb2N1bWVudCc6IHtcbiAgICAgICAgICAgICAgJ1N0YXRlbWVudCc6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAnQWN0aW9uJzogJ3NxczoqJyxcbiAgICAgICAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgICAgICAgJ1ByaW5jaXBhbCc6IHtcbiAgICAgICAgICAgICAgICAgICAgJ0FXUyc6ICdhcm4nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICdSZXNvdXJjZSc6ICcqJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAnVmVyc2lvbic6ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnUXVldWVzJzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ1JlZic6ICdNeVF1ZXVlRTZDQTYyMzUnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZXhwb3J0IGFuZCBpbXBvcnQnOiB7XG4gICAgJ2ltcG9ydGluZyB3b3JrcyBjb3JyZWN0bHknKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBpbXBvcnRzID0gc3FzLlF1ZXVlLmZyb21RdWV1ZUFybihzdGFjaywgJ0ltcG9ydGVkJywgJ2Fybjphd3M6c3FzOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6cXVldWUxJyk7XG5cbiAgICAgIC8vIFRIRU5cblxuICAgICAgLy8gXCJpbXBvcnRcIiByZXR1cm5zIGFuIElRdWV1ZSBib3VuZCB0byBgRm46OkltcG9ydFZhbHVlYHMuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKGltcG9ydHMucXVldWVBcm4pLCAnYXJuOmF3czpzcXM6dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjpxdWV1ZTEnKTtcbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUoaW1wb3J0cy5xdWV1ZVVybCksIHsgJ0ZuOjpKb2luJzpcbiAgICAgICAgWyAnJywgWyAnaHR0cHM6Ly9zcXMuJywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSwgJy4nLCB7IFJlZjogJ0FXUzo6VVJMU3VmZml4JyB9LCAnLycsIHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sICcvcXVldWUxJyBdIF0gfSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKGltcG9ydHMucXVldWVOYW1lKSwgJ3F1ZXVlMScpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpbXBvcnRpbmcgZmlmbyBhbmQgc3RhbmRhcmQgcXVldWVzIGFyZSBkZXRlY3RlZCBjb3JyZWN0bHknKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBzdGRRdWV1ZSA9IHNxcy5RdWV1ZS5mcm9tUXVldWVBcm4oc3RhY2ssICdTdGRRdWV1ZScsICdhcm46YXdzOnNxczp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOnF1ZXVlMScpO1xuICAgICAgY29uc3QgZmlmb1F1ZXVlID0gc3FzLlF1ZXVlLmZyb21RdWV1ZUFybihzdGFjaywgJ0ZpZm9RdWV1ZScsICdhcm46YXdzOnNxczp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOnF1ZXVlMi5maWZvJyk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGRRdWV1ZS5maWZvLCBmYWxzZSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChmaWZvUXVldWUuZmlmbywgdHJ1ZSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdncmFudHMnOiB7XG4gICAgJ2dyYW50Q29uc3VtZU1lc3NhZ2VzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICB0ZXN0R3JhbnQoKHEsIHApID0+IHEuZ3JhbnRDb25zdW1lTWVzc2FnZXMocCksXG4gICAgICAgICdzcXM6UmVjZWl2ZU1lc3NhZ2UnLFxuICAgICAgICAnc3FzOkNoYW5nZU1lc3NhZ2VWaXNpYmlsaXR5JyxcbiAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICAgICdzcXM6RGVsZXRlTWVzc2FnZScsXG4gICAgICAgICdzcXM6R2V0UXVldWVBdHRyaWJ1dGVzJyxcbiAgICAgICk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2dyYW50U2VuZE1lc3NhZ2VzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICB0ZXN0R3JhbnQoKHEsIHApID0+IHEuZ3JhbnRTZW5kTWVzc2FnZXMocCksXG4gICAgICAgICdzcXM6U2VuZE1lc3NhZ2UnLFxuICAgICAgICAnc3FzOkdldFF1ZXVlQXR0cmlidXRlcycsXG4gICAgICAgICdzcXM6R2V0UXVldWVVcmwnLFxuICAgICAgKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZ3JhbnRQdXJnZScodGVzdDogVGVzdCkge1xuICAgICAgdGVzdEdyYW50KChxLCBwKSA9PiBxLmdyYW50UHVyZ2UocCksXG4gICAgICAgICdzcXM6UHVyZ2VRdWV1ZScsXG4gICAgICAgICdzcXM6R2V0UXVldWVBdHRyaWJ1dGVzJyxcbiAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICApO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdncmFudCgpIGlzIGdlbmVyYWwgcHVycG9zZScodGVzdDogVGVzdCkge1xuICAgICAgdGVzdEdyYW50KChxLCBwKSA9PiBxLmdyYW50KHAsICdzZXJ2aWNlOmhlbGxvJywgJ3NlcnZpY2U6d29ybGQnKSxcbiAgICAgICAgJ3NlcnZpY2U6aGVsbG8nLFxuICAgICAgICAnc2VydmljZTp3b3JsZCcsXG4gICAgICApO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdncmFudHMgYWxzbyB3b3JrIG9uIGltcG9ydGVkIHF1ZXVlcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHF1ZXVlID0gc3FzLlF1ZXVlLmZyb21RdWV1ZUF0dHJpYnV0ZXMoc3RhY2ssICdJbXBvcnQnLCB7XG4gICAgICAgIHF1ZXVlQXJuOiAnYXJuOmF3czpzcXM6dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjpxdWV1ZTEnLFxuICAgICAgICBxdWV1ZVVybDogJ2h0dHBzOi8vcXVldWUtdXJsJyxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnVXNlcicpO1xuXG4gICAgICBxdWV1ZS5ncmFudFB1cmdlKHVzZXIpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgJ1BvbGljeURvY3VtZW50Jzoge1xuICAgICAgICAgICdTdGF0ZW1lbnQnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICAgJ3NxczpQdXJnZVF1ZXVlJyxcbiAgICAgICAgICAgICAgICAnc3FzOkdldFF1ZXVlQXR0cmlidXRlcycsXG4gICAgICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgICAnUmVzb3VyY2UnOiAnYXJuOmF3czpzcXM6dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjpxdWV1ZTEnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgICdWZXJzaW9uJzogJzIwMTItMTAtMTcnLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdxdWV1ZSBlbmNyeXB0aW9uJzoge1xuICAgICdlbmNyeXB0aW9uTWFzdGVyS2V5IGNhbiBiZSBzZXQgdG8gYSBjdXN0b20gS01TIGtleScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgY29uc3Qga2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdDdXN0b21LZXknKTtcbiAgICAgIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBlbmNyeXB0aW9uTWFzdGVyS2V5OiBrZXkgfSk7XG5cbiAgICAgIHRlc3Quc2FtZShxdWV1ZS5lbmNyeXB0aW9uTWFzdGVyS2V5LCBrZXkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6U1FTOjpRdWV1ZScsIHtcbiAgICAgICAgJ0ttc01hc3RlcktleUlkJzogeyAnRm46OkdldEF0dCc6IFsgJ0N1c3RvbUtleTFFNkQwRDA3JywgJ0FybicgXSB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2Ega21zIGtleSB3aWxsIGJlIGFsbG9jYXRlZCBpZiBlbmNyeXB0aW9uID0ga21zIGJ1dCBhIG1hc3RlciBrZXkgaXMgbm90IHNwZWNpZmllZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLktNUyB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6S01TOjpLZXknKSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTUVM6OlF1ZXVlJywge1xuICAgICAgICAnS21zTWFzdGVyS2V5SWQnOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAnUXVldWVLZXkzOUZDQkFFNicsXG4gICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2l0IGlzIHBvc3NpYmxlIHRvIHVzZSBhIG1hbmFnZWQga21zIGtleScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLktNU19NQU5BR0VEIH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgICAgJ1Jlc291cmNlcyc6IHtcbiAgICAgICAgICAnUXVldWU0QTdFMzU1NSc6IHtcbiAgICAgICAgICAgICdUeXBlJzogJ0FXUzo6U1FTOjpRdWV1ZScsXG4gICAgICAgICAgICAnUHJvcGVydGllcyc6IHtcbiAgICAgICAgICAgICAgJ0ttc01hc3RlcktleUlkJzogJ2FsaWFzL2F3cy9zcXMnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2dyYW50IGFsc28gYWZmZWN0cyBrZXkgb24gZW5jcnlwdGVkIHF1ZXVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywge1xuICAgICAgICBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLktNUyxcbiAgICAgIH0pO1xuICAgICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzb21lb25lJyksXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgcXVldWUuZ3JhbnRTZW5kTWVzc2FnZXMocm9sZSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICAnUG9saWN5RG9jdW1lbnQnOiB7XG4gICAgICAgICAgJ1N0YXRlbWVudCc6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0FjdGlvbic6IFtcbiAgICAgICAgICAgICAgICAnc3FzOlNlbmRNZXNzYWdlJyxcbiAgICAgICAgICAgICAgICAnc3FzOkdldFF1ZXVlQXR0cmlidXRlcycsXG4gICAgICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgICAnUmVzb3VyY2UnOiB7ICdGbjo6R2V0QXR0JzogWyAnUXVldWU0QTdFMzU1NScsICdBcm4nIF0gfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICAgJ2ttczpEZWNyeXB0JyxcbiAgICAgICAgICAgICAgICAna21zOkVuY3J5cHQnLFxuICAgICAgICAgICAgICAgICdrbXM6UmVFbmNyeXB0KicsXG4gICAgICAgICAgICAgICAgJ2ttczpHZW5lcmF0ZURhdGFLZXkqJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgJ0VmZmVjdCc6ICdBbGxvdycsXG4gICAgICAgICAgICAgICdSZXNvdXJjZSc6IHsgJ0ZuOjpHZXRBdHQnOiBbICdRdWV1ZUtleTM5RkNCQUU2JywgJ0FybicgXSB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgICdWZXJzaW9uJzogJzIwMTItMTAtMTcnLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICd0ZXN0IFwiLmZpZm9cIiBzdWZmaXhlZCBxdWV1ZXMgcmVnaXN0ZXIgYXMgZmlmbycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnUXVldWUnLCB7XG4gICAgICBxdWV1ZU5hbWU6ICdNeVF1ZXVlLmZpZm8nLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocXVldWUuZmlmbywgdHJ1ZSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgJ1Jlc291cmNlcyc6IHtcbiAgICAgICAgJ1F1ZXVlNEE3RTM1NTUnOiB7XG4gICAgICAgICAgJ1R5cGUnOiAnQVdTOjpTUVM6OlF1ZXVlJyxcbiAgICAgICAgICAnUHJvcGVydGllcyc6IHtcbiAgICAgICAgICAgICdRdWV1ZU5hbWUnOiAnTXlRdWV1ZS5maWZvJyxcbiAgICAgICAgICAgICdGaWZvUXVldWUnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rlc3QgYSBmaWZvIHF1ZXVlIGlzIG9ic2VydmVkIHdoZW4gdGhlIFwiZmlmb1wiIHByb3BlcnR5IGlzIHNwZWNpZmllZCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnUXVldWUnLCB7XG4gICAgICBmaWZvOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocXVldWUuZmlmbywgdHJ1ZSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgJ1Jlc291cmNlcyc6IHtcbiAgICAgICAgJ1F1ZXVlNEE3RTM1NTUnOiB7XG4gICAgICAgICAgJ1R5cGUnOiAnQVdTOjpTUVM6OlF1ZXVlJyxcbiAgICAgICAgICAnUHJvcGVydGllcyc6IHtcbiAgICAgICAgICAgICdGaWZvUXVldWUnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rlc3QgbWV0cmljcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdRdWV1ZScpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocXVldWUubWV0cmljTnVtYmVyT2ZNZXNzYWdlc1NlbnQoKSksIHtcbiAgICAgIGRpbWVuc2lvbnM6IHtRdWV1ZU5hbWU6IHsgJ0ZuOjpHZXRBdHQnOiBbICdRdWV1ZTRBN0UzNTU1JywgJ1F1ZXVlTmFtZScgXSB9fSxcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9TUVMnLFxuICAgICAgbWV0cmljTmFtZTogJ051bWJlck9mTWVzc2FnZXNTZW50JyxcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHF1ZXVlLm1ldHJpY1NlbnRNZXNzYWdlU2l6ZSgpKSwge1xuICAgICAgZGltZW5zaW9uczoge1F1ZXVlTmFtZTogeyAnRm46OkdldEF0dCc6IFsgJ1F1ZXVlNEE3RTM1NTUnLCAnUXVldWVOYW1lJyBdIH19LFxuICAgICAgbmFtZXNwYWNlOiAnQVdTL1NRUycsXG4gICAgICBtZXRyaWNOYW1lOiAnU2VudE1lc3NhZ2VTaXplJyxcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuXG5mdW5jdGlvbiB0ZXN0R3JhbnQoYWN0aW9uOiAocTogc3FzLlF1ZXVlLCBwcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsKSA9PiB2b2lkLCAuLi5leHBlY3RlZEFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ015UXVldWUnKTtcbiAgY29uc3QgcHJpbmNpcGFsID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnVXNlcicpO1xuXG4gIGFjdGlvbihxdWV1ZSwgcHJpbmNpcGFsKTtcblxuICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAnUG9saWN5RG9jdW1lbnQnOiB7XG4gICAgICAnU3RhdGVtZW50JzogW1xuICAgICAgICB7XG4gICAgICAgICAgJ0FjdGlvbic6IGV4cGVjdGVkQWN0aW9ucyxcbiAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICAnUmVzb3VyY2UnOiB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgJ015UXVldWVFNkNBNjIzNScsXG4gICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgICdWZXJzaW9uJzogJzIwMTItMTAtMTcnLFxuICAgIH0sXG4gIH0pKTtcbn1cbiJdfQ==