# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['volcano_cooking',
 'volcano_cooking.configurations',
 'volcano_cooking.helper_scripts',
 'volcano_cooking.modules.convert',
 'volcano_cooking.modules.create',
 'volcano_cooking.plotting']

package_data = \
{'': ['*']}

install_requires = \
['PyWavelets>=1.1.1,<2.0.0',
 'cftime>=1.5.0,<2.0.0',
 'click>=8.0.1,<9.0.0',
 'cosmoplots>=0.1.5,<0.2.0',
 'dask>=2022.3.0,<2023.0.0',
 'importlib-metadata>=5.0.0,<6.0.0',
 'matplotlib>=3.4.2,<4.0.0',
 'netCDF4>=1.5.8,<2.0.0',
 'numpy>=1.21.1,<2.0.0',
 'scipy>=1.7.1,<2.0.0',
 'superposed-pulses>=1.2,<2.0',
 'wget>=3.2,<4.0',
 'xarray>=0.21.1,<2022.12.0']

entry_points = \
{'console_scripts': ['sfrc-sparse2lin = volcano_cooking.sparse_to_lin:main',
                     'view-frc = volcano_cooking.view_force:main',
                     'volcano-cooking = volcano_cooking.__main__:main']}

setup_kwargs = {
    'name': 'volcano-cooking',
    'version': '0.9.3',
    'description': 'Make some volcanoes and simulate them in CESM2',
    'long_description': '# Volcano Cooking\n\n[![PyPI version](https://img.shields.io/pypi/v/volcano-cooking)](https://pypi.org/project/volcano-cooking/)\n[![codecov](https://codecov.io/gh/engeir/volcano-cooking/branch/main/graph/badge.svg?token=8I5VE7LYA4)](https://codecov.io/gh/engeir/volcano-cooking)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n> Let\'s make some volcanoes erupt!\n\n[![asciicast](https://asciinema.org/a/6eOsLnlOikscbXYLJqclxytD3.svg)](https://asciinema.org/a/6eOsLnlOikscbXYLJqclxytD3)\n\n__NOTE:__\nThe created dates *must* start before the model start. Running CESM2 from year 1850 with\nthe first eruption in 1850 will make it crash. Setting the first eruption to 1849,\nhowever, will make it run. The same goes for the end, the model must stop prior to the\nlast event, otherwise it will crash. This project will make sure one event is placed\nahead of the `init` year, but the end will vary depending on number of events created\nand their frequency.\n\n## Install\n\nThe package is published on [PyPI] and installable via `pip`:\n\n```bash\npip install volcano-cooking\n```\n\n<!-- FIXME: add reference to available classes -->\nIf you only plan on using the already present generator classes and only need the CLI\nprogram, it is recommended to install this with [pipx][pipx-source]:\n\n```bash\npipx install volcano-cooking\n```\n\n## Usage\n\n### The basics\n\nThere are mainly two CLI programs coming with this project. The main program is\n`volcano-cooking`, which will create a `.nc` and `.npz` file in the `data/output`\ndirectory. With the `view-frc` program you can quickly view the content of the created\nfiles in a plot. There is also a third program called `sfrc-sparse2lin` which converts\narrays of only non-zero events at arbitrary times to linearly spaced events where times\nbetween the non-zero events are filled with zeros.\n\n### `volcano-cooking`\n\nOnce installed, run from within this directory with:\n\n```bash\nvolcano-cooking\n```\n\nAll created files are saved to a `data/output` directory that will be created inside the\ncurrent directory from where the `volcano-cooking` command is run.\n\nFor more information, see\n\n```bash\nvolcano-cooking --help\n```\n\n#### Option 0 (default, using NCL-script)\n\n##### TL;DR\n\n```console\n$ volcano-cooking -f 1 -s 100\nGenerating with \'GenerateFPP\'...\n$ volcano-cooking --run-ncl\n Copyright (C) 1995-2019 - All Rights Reserved\n University Corporation for Atmospheric Research\n NCAR Command Language Version 6.6.2\n The use of this software is governed by a License Agreement.\n See http://www.ncl.ucar.edu/ for more details.\n(0)     in data/originals/createVolcEruptV3.ncl\n...\n  long_name :   SO2 elevated emissions from explosive volcanoes\n  _FillValue :  9.96921e+36\n(0)     saving stratvolc\n(0)     File creation complete: data/cesm/VolcanEESMv3.11Enger_SO2_850-2016_Mscale_Zreduc_2deg_c20220502-140023.nc\nLog file created at data/cesm/logs/20220502-140022.log\nFixing the attributes of the altitude_int coordinate...\n$ volcano-cooking --package-last\nSuccessfully placed all latest source files in the \'source-files\' directory.\n$ ls source-files\n20220502-140022.log                     VolcanEESMv3.11Enger_SO2_850-2016_Mscale_Zreduc_2deg_c20220502-140023.nc\nsynthetic_volcanoes_20220502_135956.nc  synthetic_volcanoes_20220502_135956.npz\n```\n\n##### Dependencies\n\nThis option needs\n\n- `volcano-cooking` installed\n- A coordinate file (~ 10 kB)\n- [`ncl`](https://www.ncl.ucar.edu/Download/) executable\n\n##### Create source file for forcing\n\nRun command `volcano-cooking` with the options you want. See `volcano-cooking --help`.\n\n##### Create forcing file for CESM2\n\n> Running the `volcano-cooking --run-ncl` script depends on having `ncl` installed. See\n> installation instructions [here](https://www.ncl.ucar.edu/Download/).\n\nAfter having run the `volcano-cooking` command, the forcing file for CESM2 can be\ngenerated by running\n\n```bash\nvolcano-cooking --run-ncl\n```\n\nIf the needed coordinate files are missing, you will be asked if you want to download\nthem. If you want to use different files, or change the default resolution (default is 2\ndegrees), edit __.env__ (see [__./examples/.env.example__](./examples/.env.example))\naccordingly. In this case, you also need to manually download whatever coordinate file\nyou want to use. See section [Setting up manually](#setting-up-manually).\n\n##### Wrap up\n\nThe last created files, source files, logs and final output, can be nicely collected and\nplaced in a directory named `source-files` with command:\n\n```sh\nvolcano-cooking --package-last\n```\n\n<details><summary><h5>Setting up manually</h5></summary><br>\n\nTo be able to create forcing files used by CESM2 from the newly created synthetic file,\nwe need a script from the [emissions][data_source_files] directory. These are scripts\nthat use the forcing file this project generates to make a new, full forcing file that\nCESM2 accepts (examples of such files can be found [here][volc-frc-complete]). For\nexample, `createVolcEruptV3.ncl` can be found in the [emissions][data_source_files]\ndirectory. This need a `common.ncl` file, found [here][common-ncl], in addition to other\nstandard `ncl` libraries. Make sure to edit `createVolcEruptV3.ncl` to read the created\nfile and that the first and last year cover those used in the created file. A working\nversion of `createVolcEruptV3.ncl` that uses input files generated by `volcano-cooking`\ncan be found in [data/originals](data/originals). To see what was changed from the\noriginal, run `diff data/originals/createVolcEruptV3.ncl.original\nsrc/volcano_cooking/createVolcEruptV3.ncl`.\n\nCoordinate files are needed when running `createVolcEruptV3.ncl` or similar scripts, and\nare located [here][coord-file]. For example `fv_1.9x2.5_L30.nc` which can be used with\ntwo degrees resolution in the atmosphere model. The following commands will download\n1 and 2 degree resolution coordinate files, respectively, to the `data/originals`\ndirectory:\n\n```bash\nwget --no-check-certificate https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/coords/fv_0.9x1.25_L30.nc --directory-prefix data/originals\nwget --no-check-certificate https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/coords/fv_1.9x2.5_L30.nc --directory-prefix data/originals\n```\n\n</details>\n\n#### Option 1 (directly change forcing file)\n\n##### TL;DR\n\n```console\n$ volcano-cooking -f 1 -s 100 -o\nGenerating with \'GenerateFPP\'...\n```\n\n##### Dependencies\n\nThis option needs\n\n- `volcano-cooking` installed\n- A coordinate file (~ 10 kB)\n- Original CESM2 forcing file (~ 2.2 GB)\n\n##### Run library\n\nNow the only thing we need to do is running `volcano-cooking` with the flag `-o`, and\nchoose the type of forcing we want (see `volcano-cooking --lst`):\n\n```bash\nvolcano-cooking -f 1 -s 100 -o\n```\n\n<details><summary><h5>Get forcing and coordinate files manually</h5></summary><br>\n\n> Manually downloading the files and placing them in the correct directory is *not*\n> needed. Running the command as shown above will ask you if you want to download the\n> files, and place them where they need to be.\n\nThis option relies on having a working forcing file and coordinate file at hand. We will\nuse the forcing file that CESM2 places in the `stratvolc` directory of the `cam` model.\nDownload from [this link][stratvolc-forcing] and place it in the `data/originals`\ndirectory, or run command:\n\n```bash\nwget --no-check-certificate https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/chem/stratvolc/VolcanEESMv3.11_SO2_850-2016_Mscale_Zreduc_2deg_c191125.nc --directory-prefix data/originals\n```\n\nIt\'s 2.2\xa0GB file, so it will take some time.\n\nWe will also need a coordinate file, specifically `fv_1.9x2.5_L30.nc` which is found\n[here][coord-file]. This file is small and quick to download. From the command line:\n\n```bash\nwget --no-check-certificate https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/coords/fv_0.9x1.25_L30.nc --directory-prefix data/originals\nwget --no-check-certificate https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/coords/fv_1.9x2.5_L30.nc --directory-prefix data/originals\n```\n\n</details>\n\n#### Examples\n\nWhen running the command `volcano-cooking --run-ncl`, a few environment variables will\nbe used, which can be controlled by setting them in a `.env` file. See\n[`.env.example`](./examples/.env.example) to see some default values. With this you can\nfor example easily change the grid resolution to be `1deg` rather than `2deg` (default).\n\nThe [__examples__](./examples/) directory also include an example on how to use the\n`--file` option. Cloning this repository and running `volcano-cooking --file json.json`\nfrom *inside* the __examples__ directory will result in some output files generated to a\nnew __data__ directory inside __examples__. If you further rename `.env.example` →\n`.env` you may also run `volcano-cooking --run-ncl` and `volcano-cooking --package-last`\n(this assumes you follow option 0, see below).\n\nFinally, there is also a script [`custom_generator.py`](./examples/custom_generator.py)\nwhich show how you might define your own generator classes and functions. Run as `python\ncustom_generator.py`.\n\n### `view-frc`\n\nAfter running the `volcano-cooking` command, files with names `synthetic_volcanoes_...`\nare created with file type `.nc` and `.npz`. These can be viewed with\n\n```bash\nview-frc <file.nc>\n```\n\nAn optional flag can be sent to the `view-frc` program that will save the plot:\n`view-frc -s <file.nc>`.\n\nFor more information, see\n\n```bash\nview-frc --help\n```\n\n### `sfrc-sparse2lin`\n\nThis program is available as a convenience tool in the rare case when you have a forcing\nfile with only the events of eruptions. The program will re-write time array to be\nlinearly spaced and place the values from the forcing values into a new forcing array\nthat matches the new linearly spaced time axis.\n\nIt assumes the file sent to it has `yoes`(year, YYYY), `moes` (month, MM), `does`\n(days, DD) and `tes` (emissions, Tg) fields if it is a `.npz` file. If a `.nc` file is\nused, it must have a dataset with variables "Year_of_Emission", "Month_of_Emission",\n"Day_of_Emission" and "Total_Emission".\n\n## Extra\n\nThis assumes you are using the package from a python script, and not just the CLI.\n\n### Compare created file with a similar used in a default CESM2 experiment\n\nA similar file to those that are created is needed to be able to use some scripts in the\n`helper_scripts` directory. By default, it assumes the file is named\n`volcan-eesm_global_2015_so2-emissions-database_v1.0.nc` and that it is placed inside\nthe `data/originals` directory. You can find this file [here][volc-frc].\n\n## Contributing\n\nTo contribute to the project, clone and install the full development version (uses\n[poetry] for dependencies):\n\n```bash\ngit clone https://github.com/engeir/volcano-cooking.git\ncd volcano-cooking\npoetry install\npre-commit install\n```\n\nBefore committing new changes to a branch you may run the full test suite with:\n\n```bash\nnox\n```\n\nYou will need [Poetry], [nox] and [nox-poetry] installed for this.\n\n[common-ncl]: http://svn.code.sf.net/p/codescripts/code/trunk/ncl/lib/common.ncl\n[coord-file]: https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/coords/\n[data_source_files]: https://svn.code.sf.net/p/codescripts/code/trunk/ncl/emission\n[nox-poetry]: https://nox-poetry.readthedocs.io/\n[nox]: https://nox.thea.codes/en/stable/\n[pipx-source]: https://github.com/pypa/pipx\n[poetry]: https://python-poetry.org\n[pypi]: https://pypi.org/\n[stratvolc-forcing]: https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/chem/stratvolc/VolcanEESMv3.11_SO2_850-2016_Mscale_Zreduc_2deg_c191125.nc\n[volc-frc-complete]: https://svn-ccsm-inputdata.cgd.ucar.edu/trunk/inputdata/atm/cam/chem/stratvolc/\n[volc-frc]: http://catalogue.ceda.ac.uk/uuid/bfbd5ec825fa422f9a858b14ae7b2a0d\n',
    'author': 'engeir',
    'author_email': 'eirroleng@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
