# AUTOGENERATED! DO NOT EDIT! File to edit: 00r_stats.ipynb (unless otherwise specified).

__all__ = ['StatisticalTest', 'OneSampleStats', 'MultipleIndependentSamplesStats', 'MixedModelANOVAStats']

# Cell
from typing import Tuple, Dict, List, Optional, Union
from abc import ABC, abstractmethod

import pandas as pd
import numpy as np
import pingouin as pg
import itertools

from .database import Database
from .plots_refactored import OneSamplePlots, MultipleIndependentSamplesPlots, MixedModelANOVAPlots

# Cell
class StatisticalTest(ABC):

    @property
    @abstractmethod
    def name_displayed_in_gui(self):
        pass

    @property
    @abstractmethod
    def plot_handler(self):
        # set the corresponding plot_handler class
        pass


    @abstractmethod
    def add_test_specific_information_to_df_infos(self) -> Dict:
        df_infos = self.lut['df_infos'].copy()
        # add additional infos here like:
        # df_infos['additional_info'] = value
        # or leave it like this / remove these comments, if there is nothing to add
        return df_infos


    @abstractmethod
    def add_test_specific_information_to_summary_stats(self) -> Dict:
        summary_stats = self.lut['summary_stats'].copy()
        # add additional infos here like:
        # summary_stats['additional_info'] = value
        # keys that have to be added:
        # 'performed_test': str
        # 'full_test_results': pd.DataFrame
        # 'p-value': float
        # 'stars_str': str
        return summary_stats


    @abstractmethod
    def add_test_specific_information_to_pairwise_comparisons(self) -> Dict:
        pairwise_comparisons = self.lut['pairwise_comparisons'].copy()
        # add additional infos here like:
        # pairwise_comparisons['additional_info'] = value
        # or leave it like this / remove these comments, if there is nothing to add
        return pairwise_comparisons


    def compute(self, database: Database) -> Database:
        self.df = database.data.copy()
        self.lut = self.initialize_lut_with_basic_df_infos()
        self.lut['df_infos'] = self.add_test_specific_information_to_df_infos()
        for group_id in self.lut['df_infos']['all_group_ids']:
            if 'session_column_name' in self.lut['df_infos'].keys():
                for session_id in self.lut['df_infos']['all_session_ids']:
                    self.lut['group_level_stats'][(group_id, session_id)] = self.add_normality_check_on_group_level(group_id = group_id, session_id = session_id)
            else:
                self.lut['group_level_stats'][group_id] = self.add_normality_check_on_group_level(group_id = group_id)
        self.lut['summary_stats'] = self.initialize_summary_stats()
        self.lut['summary_stats'] = self.add_test_specific_information_to_summary_stats()
        self.lut['pairwise_comparisons'] = self.add_test_specific_information_to_pairwise_comparisons()
        database.stats_results = self.lut.copy()
        return database


    def initialize_lut_with_basic_df_infos(self) -> Dict:
        lut = {'df_infos': dict(),
               'summary_stats': dict(),
               'group_level_stats': dict(),
               'pairwise_comparisons': dict()}
        df_infos = {'data_column_name': self.df.columns[0],
                    'data_column_values': self.df.iloc[:, 0].values.copy(),
                    'group_column_name': self.df.columns[1],
                    'group_column_values': self.df.iloc[:, 1].values.copy(),
                    'all_group_ids': list(self.df.iloc[:, 1].unique()),
                    'n_groups': len(list(self.df.iloc[:, 1].unique()))}
        lut['df_infos'] = df_infos
        return lut


    def add_normality_check_on_group_level(self, group_id: str, session_id: Optional[str]=None) -> Dict:
        data_column_name = self.lut['df_infos']['data_column_name']
        group_column_name = self.lut['df_infos']['group_column_name']
        if type(session_id) == str:
            session_column_name = self.lut['df_infos']['session_column_name']
            data = self.df.loc[(self.df[group_column_name] == group_id)
                               & (self.df[session_column_name] == session_id), data_column_name].values.copy()
        else:
            data = self.df.loc[self.df[group_column_name] == group_id, data_column_name].values.copy()
        normality_test_results = pg.normality(data)
        is_normally_distributed = normality_test_results['normal'][0]
        normality_check = {'data': data,
                           'normality_test_results': normality_test_results,
                           'is_normally_distributed': is_normally_distributed}
        return normality_check


    def initialize_summary_stats(self) -> Dict:
        are_all_normally_distributed = []
        for group_id in self.lut['df_infos']['all_group_ids']:
            if 'session_column_name' in self.lut['df_infos'].keys():
                for session_id in self.lut['df_infos']['all_session_ids']:
                    are_all_normally_distributed.append(self.lut['group_level_stats'][(group_id, session_id)]['is_normally_distributed'])
            else:
                are_all_normally_distributed.append(self.lut['group_level_stats'][group_id]['is_normally_distributed'])
        if self.lut['df_infos']['n_groups'] > 1:
            data_all_groups = []
            for group_id in self.lut['df_infos']['all_group_ids']:
                if 'session_column_name' in self.lut['df_infos'].keys():
                    for session_id in self.lut['df_infos']['all_session_ids']:
                        data_all_groups.append(self.lut['group_level_stats'][(group_id, session_id)]['data'])
                else:
                    data_all_groups.append(self.lut['group_level_stats'][group_id]['data'])
            homoscedasticity_test_results = pg.homoscedasticity(data_all_groups)
            variance_is_equal = homoscedasticity_test_results['equal_var'].values[0]
            if all(are_all_normally_distributed) and variance_is_equal:
                use_parametric = True
            else:
                use_parametric = False
            summary_stats = {'all_normally_distributed': all(are_all_normally_distributed),
                             'homoscedasticity_test_results': homoscedasticity_test_results,
                             'variance_is_equal': variance_is_equal,
                             'use_parametric': use_parametric}
        else:
            if all(are_all_normally_distributed):
                use_parametric = True
            else:
                use_parametric = False
            summary_stats = {'all_normally_distributed': all(are_all_normally_distributed),
                             'use_parametric': use_parametric}
        return summary_stats


    def get_stars_string(self, p: float) -> str:
        if p <= 0.001:
            stars_string = '***'
        elif p <= 0.01:
            stars_string = '**'
        elif p <= 0.05:
            stars_string = '*'
        else:
            stars_string = 'n.s.'
        return stars_string

# Cell
class OneSampleStats(StatisticalTest):

    @property
    def name_displayed_in_gui(self):
        return 'One sample test: comparison of values from one sample to a reference value'

    @property
    def plot_handler(self):
        return OneSamplePlots


    def add_test_specific_information_to_df_infos(self) -> Dict:
        df_infos = self.lut['df_infos'].copy()
        df_infos['fixed_value_column_name'] = self.df.columns[2]
        df_infos['fixed_value'] = self.df.iloc[0, 2]
        df_infos['group_id'] = self.df.iloc[:, 1].unique()[0]
        return df_infos


    def add_test_specific_information_to_summary_stats(self) -> Dict:
        summary_stats = self.lut['summary_stats'].copy()
        data = self.lut['df_infos']['data_column_values']
        fixed_value = self.lut['df_infos']['fixed_value']
        if summary_stats['use_parametric']:
            performed_test = 'one sample t-test'
            full_test_results = pg.ttest(x = data, y = fixed_value, paired = False, alternative = 'two-sided')
        else:
            performed_test =  'one sample wilcoxon rank-sum test'
            full_test_results = pg.wilcoxon(data - fixed_value, alternative = 'two-sided')
        summary_stats['performed_test'] = performed_test
        summary_stats['full_test_results'] = full_test_results
        summary_stats['p_value'] = full_test_results['p-val'].values[0]
        summary_stats['stars_str'] = self.get_stars_string(p = summary_stats['p_value'])
        return summary_stats


    def add_test_specific_information_to_pairwise_comparisons(self) -> Dict:
        pairwise_comparisons = self.lut['pairwise_comparisons'].copy()
        # add additional infos here like:
        # pairwise_comparisons['additional_info'] = value
        # or leave it like this / remove these comments, if there is nothing to add
        return pairwise_comparisons

# Cell
class MultipleIndependentSamplesStats(StatisticalTest):

    @property
    def name_displayed_in_gui(self):
        return 'Pairwise comparison of values from two or more independent samples'

    @property
    def plot_handler(self):
        return MultipleIndependentSamplesPlots


    def add_test_specific_information_to_df_infos(self) -> Dict:
        df_infos = self.lut['df_infos'].copy()
        return df_infos


    def add_test_specific_information_to_summary_stats(self) -> Dict:
        summary_stats = self.lut['summary_stats'].copy()
        df_infos = self.lut['df_infos'].copy()
        if len(df_infos['all_group_ids']) > 2:
            if summary_stats['use_parametric']:
                performed_test = 'One-way ANOVA'
                full_test_results = pg.anova(data = self.df, dv = df_infos['data_column_name'], between = df_infos['group_column_name'])
                p_value = full_test_results['p-unc'].values[0]
            else:
                performed_test = 'Kruskal-Wallis-ANOVA'
                full_test_results = pg.kruskal(data = self.df, dv = df_infos['data_column_name'], between = df_infos['group_column_name'])
                p_value = full_test_results['p-unc'].values[0]
        if len(df_infos['all_group_ids']) == 2:
            id_group_a, id_group_b = df_infos['all_group_ids']
            group_column_name = df_infos['group_column_name']
            data_group_a = self.df.loc[self.df[group_column_name] == id_group_a, df_infos['data_column_name']].values
            data_group_b = self.df.loc[self.df[group_column_name] == id_group_b, df_infos['data_column_name']].values
            if summary_stats['use_parametric']:
                performed_test = 'unpaired two sample t-test (with Welch-correction if applicable)'
                full_test_results = pg.ttest(x = data_group_a, y = data_group_b, alternative = 'two-sided', correction = 'auto')
                p_value = full_test_results['p-val'].values[0]
            else:
                performed_test = 'Mann-Whitney U test'
                full_test_results = pg.mwu(x = data_group_a, y = data_group_b, alternative = 'two-sided')
                p_value = full_test_results['p-val'].values[0]
        summary_stats['performed_test'] = performed_test
        summary_stats['full_test_results'] = full_test_results
        summary_stats['p_value'] = p_value
        summary_stats['stars_str'] = self.get_stars_string(p = summary_stats['p_value'])
        return summary_stats


    def add_test_specific_information_to_pairwise_comparisons(self) -> Dict:
        pairwise_comparisons = self.lut['pairwise_comparisons'].copy()
        summary_stats = self.lut['summary_stats'].copy()
        df_infos = self.lut['df_infos'].copy()
        pairwise_comparisons = pg.pairwise_ttests(data = self.df, dv = df_infos['data_column_name'], between = df_infos['group_column_name'],
                                                  parametric = summary_stats['use_parametric'], padjust='holm')
        return pairwise_comparisons

# Cell
class MixedModelANOVAStats(StatisticalTest):

    @property
    def name_displayed_in_gui(self):
        return 'Mixed-model ANOVA: compare two or more independent samples, all measured at several timepoints'

    @property
    def plot_handler(self):
        return MixedModelANOVAPlots


    def add_test_specific_information_to_df_infos(self) -> Dict:
        df_infos = self.lut['df_infos'].copy()
        df_infos['subject_column_name'] = self.df.columns[2]
        df_infos['session_column_name'] = self.df.columns[3]
        df_infos['all_session_ids'] = list(self.df[df_infos['session_column_name']].unique())
        return df_infos


    def add_test_specific_information_to_summary_stats(self) -> Dict:
        summary_stats = self.lut['summary_stats'].copy()
        df_infos = self.lut['df_infos'].copy()
        for group_id in df_infos['all_group_ids']:
            for session_id in df_infos['all_session_ids']:
                self.lut['group_level_stats'][(group_id, session_id)]['mean'] = self.lut['group_level_stats'][(group_id, session_id)]['data'].mean()
        if summary_stats['use_parametric'] == False:
            user_warning_0 = 'Warning: Please be aware that the data require non-parametric testing.\n'
            user_warning_1 = 'However, this is not implemented yet and a parametric Mixed-model ANOVA is computed instead.'
            print(user_warning_0 + user_warning_1)
        summary_stats['performed_test'] = 'Mixed-model ANOVA'
        full_test_results = pg.mixed_anova(data = self.df, dv = df_infos['data_column_name'], within = df_infos['session_column_name'],
                                           subject = df_infos['subject_column_name'], between = df_infos['group_column_name'])
        summary_stats['full_test_results'] = full_test_results
        summary_stats['p_value'] = full_test_results['p-unc'].values[0] # group comparison (also possible: 1 -> session, 2 -> interaction)
        summary_stats['stars_str'] = self.get_stars_string(p = summary_stats['p_value'])
        return summary_stats


    def add_test_specific_information_to_pairwise_comparisons(self) -> Dict:
        pairwise_comparisons = self.lut['pairwise_comparisons'].copy()
        summary_stats = self.lut['summary_stats'].copy()
        df_infos = self.lut['df_infos'].copy()
        pairwise_comparisons = pg.pairwise_ttests(data = self.df, dv = df_infos['data_column_name'], between = df_infos['group_column_name'],
                                                  within = df_infos['session_column_name'], subject = df_infos['subject_column_name'],
                                                  parametric = summary_stats['use_parametric'], padjust='holm')
        return pairwise_comparisons