"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckAlbProps = exports.GetActiveListener = exports.AddFargateTarget = exports.AddLambdaTarget = exports.AddListener = exports.ObtainAlb = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const elbt = require("aws-cdk-lib/aws-elasticloadbalancingv2-targets");
const utils_1 = require("./utils");
const alb_defaults_1 = require("./alb-defaults");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Returns the correct ALB Load Balancer to use in this construct, either an existing
 * one provided as an argument or create  new one otherwise.
 */
function ObtainAlb(scope, id, vpc, publicApi, existingLoadBalancerObj, loadBalancerProps, logAccessLogs, loggingBucketProps) {
    let loadBalancer;
    if (existingLoadBalancerObj) {
        loadBalancer = existingLoadBalancerObj;
    }
    else {
        const consolidatedProps = utils_1.consolidateProps({}, loadBalancerProps, { vpc, internetFacing: publicApi });
        loadBalancer = new elb.ApplicationLoadBalancer(scope, `${id}-alb`, consolidatedProps);
        if (logAccessLogs === undefined || logAccessLogs === true) {
            const consolidatedLoggingBucketProps = utils_1.consolidateProps(s3_bucket_defaults_1.DefaultLoggingBucketProps(), loggingBucketProps);
            const loggingBucket = s3_bucket_helper_1.createAlbLoggingBucket(scope, id, consolidatedLoggingBucketProps);
            loadBalancer.logAccessLogs(loggingBucket);
        }
    }
    return loadBalancer;
}
exports.ObtainAlb = ObtainAlb;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function AddListener(scope, id, loadBalancer, listenerProps) {
    let consolidatedListenerProps;
    consolidatedListenerProps = utils_1.consolidateProps(alb_defaults_1.DefaultListenerProps(loadBalancer), listenerProps);
    //  create the listener
    const listener = new elb.ApplicationListener(scope, `${id}-listener`, consolidatedListenerProps);
    loadBalancer.listeners.push(listener);
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTP) {
        // This will use core.printWarning in the actual construct
        utils_1.printWarning("AWS recommends encrypting traffic to an Application Load Balancer using HTTPS.");
        if (listenerProps.certificates?.length > 0) {
            throw new Error("HTTP listeners cannot use a certificate");
        }
    }
    else {
        if (!listenerProps.certificates || listenerProps.certificates.length === 0) {
            throw new Error("A listener using HTTPS protocol requires a certificate");
        }
        listener.addCertificates(`${id}-cert`, listenerProps.certificates);
    }
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTPS) {
        const opt = {
            port: "443",
            protocol: "HTTPS",
        };
        const httpListener = new elb.ApplicationListener(scope, `${id}-redirect`, {
            loadBalancer,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect(opt),
        });
        loadBalancer.listeners.push(httpListener);
    }
    return listener;
}
exports.AddListener = AddListener;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Creates a Target Group for Lambda functions and adds the
 * provided functions as a target to that group. Then adds
 * the new Target Group to the provided Listener.
 */
function AddLambdaTarget(scope, id, currentListener, lambdaFunction, ruleProps, targetProps) {
    //  Create the target and assign it to a new target group
    const lambdaTarget = new elbt.LambdaTarget(lambdaFunction);
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, {
        targets: [lambdaTarget],
        targetGroupName: targetProps ? targetProps.targetGroupName : undefined,
        healthCheck: targetProps ? targetProps.healthCheck : undefined
    });
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble AddApplicationTargetGroupProps
    const consolidatedTargetProps = utils_1.consolidateProps({}, ruleProps, { targetGroups: [newTargetGroup] });
    currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    newTargetGroup.setAttribute('stickiness.enabled', undefined);
    return newTargetGroup;
}
exports.AddLambdaTarget = AddLambdaTarget;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function AddFargateTarget(scope, id, currentListener, fargateService, ruleProps, targetProps) {
    if (targetProps?.protocol !== elb.ApplicationProtocol.HTTPS) {
        utils_1.printWarning('AWS recommends using HTTPS protocol for Target Groups in production applications');
    }
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, targetProps);
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble an AddApplicationTargetGroupProps object
    const consolidatedTargetProps = utils_1.consolidateProps({ targetGroups: [newTargetGroup] }, ruleProps);
    currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    newTargetGroup.addTarget(fargateService);
    return newTargetGroup;
}
exports.AddFargateTarget = AddFargateTarget;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Looks for the listener associated with Target Groups
 * If there is a single listener, this returns it whether it is HTTP or HTTPS
 * If there are 2 listeners, it finds the HTTPS listener (we assume the HTTP listener redirects to HTTPS)
 */
function GetActiveListener(listeners) {
    let listener;
    if (listeners.length === 0) {
        throw new Error(`There are no listeners in the ALB`);
    }
    if (listeners.length === 1) {
        listener = listeners[0];
    }
    else {
        listener = listeners.find(i => i.node.children[0].protocol === "HTTPS");
    }
    return listener;
}
exports.GetActiveListener = GetActiveListener;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckAlbProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.listenerProps?.certificateArns) {
        errorMessages += "certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates\n";
        errorFound = true;
    }
    if (((props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length === 0) ||
        !props.existingLoadBalancerObj) &&
        !props.listenerProps) {
        errorMessages += "When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP\n";
        errorFound = true;
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        props.listenerProps) {
        errorFound = true;
        errorMessages += "This load balancer already has a listener, listenerProps may not be specified\n";
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        !props.ruleProps) {
        errorFound = true;
        errorMessages += "When adding a second target to an existing listener, there must be rules provided\n";
    }
    // Check construct specific invalid inputs
    if (props.existingLoadBalancerObj && !props.existingVpc) {
        errorFound = true;
        errorMessages += "An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use.\n";
    }
    if (props.loadBalancerProps?.vpc) {
        errorFound = true;
        errorMessages += 'Specify any existing VPC at the construct level, not within loadBalancerProps.\n';
    }
    if (props.existingLoadBalancerObj) {
        utils_1.printWarning("The public/private property of an existing ALB must match the props.publicApi setting provided.");
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckAlbProps = CheckAlbProps;
//# sourceMappingURL=data:application/json;base64,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