"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultIntegrationResponses = exports.DefaultRegionalRestApiProps = exports.DefaultGlobalRestApiProps = exports.DefaultRegionalLambdaRestApiProps = exports.DefaultGlobalLambdaRestApiProps = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const api = require("aws-cdk-lib/aws-apigateway");
/**
 * Private function to configure an api.RestApiProps
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultRestApiProps(_endpointType, _logGroup) {
    return {
        endpointConfiguration: {
            types: _endpointType
        },
        cloudWatchRole: false,
        // Configure API Gateway Access logging
        deployOptions: {
            accessLogDestination: new api.LogGroupLogDestination(_logGroup),
            accessLogFormat: api.AccessLogFormat.jsonWithStandardFields(),
            loggingLevel: api.MethodLoggingLevel.INFO,
            dataTraceEnabled: false,
            tracingEnabled: true
        },
        defaultMethodOptions: {
            authorizationType: api.AuthorizationType.IAM
        }
    };
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Provides the default set of properties for Edge/Global Lambda backed RestApi
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultGlobalLambdaRestApiProps(_existingLambdaObj, _logGroup) {
    const baseProps = DefaultRestApiProps([api.EndpointType.EDGE], _logGroup);
    const extraProps = {
        handler: _existingLambdaObj,
    };
    return Object.assign(baseProps, extraProps);
}
exports.DefaultGlobalLambdaRestApiProps = DefaultGlobalLambdaRestApiProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Provides the default set of properties for Regional Lambda backed RestApi
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultRegionalLambdaRestApiProps(_existingLambdaObj, _logGroup) {
    const baseProps = DefaultRestApiProps([api.EndpointType.REGIONAL], _logGroup);
    const extraProps = {
        handler: _existingLambdaObj,
    };
    return Object.assign(baseProps, extraProps);
}
exports.DefaultRegionalLambdaRestApiProps = DefaultRegionalLambdaRestApiProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Provides the default set of properties for Edge/Global RestApi
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultGlobalRestApiProps(_logGroup) {
    return DefaultRestApiProps([api.EndpointType.EDGE], _logGroup);
}
exports.DefaultGlobalRestApiProps = DefaultGlobalRestApiProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Provides the default set of properties for Regional RestApi
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultRegionalRestApiProps(_logGroup) {
    return DefaultRestApiProps([api.EndpointType.REGIONAL], _logGroup);
}
exports.DefaultRegionalRestApiProps = DefaultRegionalRestApiProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @returns The set of default integration responses for status codes 200 and 500.
 */
function DefaultIntegrationResponses() {
    return [
        {
            statusCode: "200"
        },
        {
            statusCode: "500",
            responseTemplates: {
                "text/html": "Error"
            },
            selectionPattern: "500"
        }
    ];
}
exports.DefaultIntegrationResponses = DefaultIntegrationResponses;
//# sourceMappingURL=data:application/json;base64,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