"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, api) {
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = api.node.findChild('Resource');
    const cfnAccount = new apigateway.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependency(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = api.latestDeployment?.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps?.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const consolidatedApiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
        api = new apigateway.LambdaRestApi(scope, 'LambdaRestApi', consolidatedApiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        api = new apigateway.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = (apiGatewayProps?.cloudWatchRole !== false) ? configureCloudwatchRoleForApi(scope, api) : undefined;
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api,
                stage: api.deploymentStage
            }]
    };
    const plan = api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (apiGatewayProps?.defaultMethodOptions?.apiKeyRequired === true) {
        // Configure Usage Plan with API Key
        const key = api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return { api, role: cwRole };
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps?.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    const consolidatedApiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
    const api = new apigateway.RestApi(scope, 'RestApi', consolidatedApiGatewayProps);
    let cwRole;
    // Configure API access logging
    if (apiGatewayProps?.cloudWatchRole !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api,
                stage: api.deploymentStage
            }]
    };
    const plan = api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (apiGatewayProps?.defaultMethodOptions?.apiKeyRequired === true) {
        // Configure Usage Plan with API Key
        const key = api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return { api, role: cwRole };
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const configureLambdaRestApiResponse = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return { api: configureLambdaRestApiResponse.api, role: configureLambdaRestApiResponse.role, group: logGroup };
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(existingLambdaObj, logGroup);
    const configureLambdaRestApiResponse = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return { api: configureLambdaRestApiResponse.api, role: configureLambdaRestApiResponse.role, group: logGroup };
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const configureRestApiResponse = configureRestApi(scope, defaultProps, apiGatewayProps);
    return { api: configureRestApiResponse.api, role: configureRestApiResponse.role, logGroup };
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const configureRestApiResponse = configureRestApi(scope, defaultProps, apiGatewayProps);
    return { api: configureRestApiResponse.api, role: configureRestApiResponse.role, logGroup };
}
exports.RegionalRestApi = RegionalRestApi;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function addProxyMethodToApiResource(params) {
    // Make sure the user hasn't also specified the application/json content-type in the additionalRequestTemplates optional property
    if (params.additionalRequestTemplates && 'application/json' in params.additionalRequestTemplates) {
        throw new Error(`Request Template for the application/json content-type must be specified in the requestTemplate property and not in the additionalRequestTemplates property `);
    }
    const requestTemplates = {
        "application/json": params.requestTemplate,
        ...params.additionalRequestTemplates
    };
    // Use user-provided integration responses, otherwise fallback to the default ones we provide.
    const integrationResponses = params.integrationResponses ?? apiDefaults.DefaultIntegrationResponses();
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: apigateway.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates,
            integrationResponses
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    const newProps = utils_1.consolidateProps(baseProps, params.awsIntegrationProps);
    apiGatewayIntegration = new apigateway.AwsIntegration(newProps);
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    // Setup the API Gateway method
    const overridenProps = utils_1.consolidateProps(defaultMethodOptions, params.methodOptions);
    return params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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