"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultCloudFrontDisributionForMediaStoreProps = exports.DefaultCloudFrontWebDistributionForS3Props = exports.DefaultCloudFrontWebDistributionForApiGatewayProps = void 0;
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const cdk = require("aws-cdk-lib");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, setHttpSecurityHeaders, cfFunction, responseHeadersPolicy) {
    const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
    const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
    let defaultBehavior = {
        origin: new origins.HttpOrigin(apiEndPointDomainName, {
            originPath: `/${apiEndPoint.deploymentStage.stageName}`
        }),
        ...getFunctionAssociationsProp(setHttpSecurityHeaders, cfFunction),
        viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
    };
    if (responseHeadersPolicy) {
        defaultBehavior = { ...defaultBehavior, responseHeadersPolicy };
    }
    return {
        defaultBehavior,
        enableLogging: true,
        logBucket: loggingBucket,
    };
}
exports.DefaultCloudFrontWebDistributionForApiGatewayProps = DefaultCloudFrontWebDistributionForApiGatewayProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, setHttpSecurityHeaders, originPath, cfFunction, responseHeadersPolicy) {
    let origin;
    if (originPath) {
        origin = new origins.S3Origin(sourceBucket, { originPath });
    }
    else {
        origin = new origins.S3Origin(sourceBucket);
    }
    let defaultBehavior = {
        origin,
        viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        ...getFunctionAssociationsProp(setHttpSecurityHeaders, cfFunction)
    };
    if (responseHeadersPolicy) {
        defaultBehavior = { ...defaultBehavior, responseHeadersPolicy };
    }
    return {
        defaultBehavior,
        enableLogging: true,
        logBucket: loggingBucket,
        defaultRootObject: 'index.html'
    };
}
exports.DefaultCloudFrontWebDistributionForS3Props = DefaultCloudFrontWebDistributionForS3Props;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function DefaultCloudFrontDisributionForMediaStoreProps(mediastoreContainer, loggingBucket, originRequestPolicy, setHttpSecurityHeaders, customHeaders, cfFunction, responseHeadersPolicy) {
    const mediaStoreContainerUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split('://', mediastoreContainer.attrEndpoint));
    const mediaStoreContainerDomainName = cdk.Fn.select(0, cdk.Fn.split('/', mediaStoreContainerUrlWithoutProtocol));
    const httpOrigin = customHeaders ?
        new origins.HttpOrigin(mediaStoreContainerDomainName, { customHeaders }) :
        new origins.HttpOrigin(mediaStoreContainerDomainName);
    let defaultBehavior = {
        origin: httpOrigin,
        viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
        cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
        originRequestPolicy,
        ...getFunctionAssociationsProp(setHttpSecurityHeaders, cfFunction),
    };
    if (responseHeadersPolicy) {
        defaultBehavior = { ...defaultBehavior, responseHeadersPolicy };
    }
    return {
        defaultBehavior,
        enableLogging: true,
        logBucket: loggingBucket
    };
}
exports.DefaultCloudFrontDisributionForMediaStoreProps = DefaultCloudFrontDisributionForMediaStoreProps;
function getFunctionAssociationsProp(setHttpSecurityHeaders, cfFunction) {
    return (setHttpSecurityHeaders && cfFunction) ? {
        functionAssociations: [
            {
                eventType: aws_cloudfront_1.FunctionEventType.VIEWER_RESPONSE,
                function: cfFunction
            }
        ]
    } : {};
}
//# sourceMappingURL=data:application/json;base64,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