"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const cdk = require("aws-cdk-lib");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps, responseHeadersPolicyProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, httpSecurityHeaders, cloudfrontFunction, responseHeadersPolicyProps ? new cloudfront.ResponseHeadersPolicy(scope, 'ResponseHeadersPolicy', responseHeadersPolicyProps) : undefined);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return { distribution: cfDistribution, cloudfrontFunction, loggingBucket };
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders = true, originPath, cloudFrontLoggingBucketProps, responseHeadersPolicyProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, httpSecurityHeaders, originPath, cloudfrontFunction, responseHeadersPolicyProps ? new cloudfront.ResponseHeadersPolicy(scope, 'ResponseHeadersPolicy', responseHeadersPolicyProps) : undefined);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return { distribution: cfDistribution, cloudfrontFunction, loggingBucket };
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps, responseHeadersPolicyProps) {
    let originRequestPolicy;
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, httpSecurityHeaders, cloudFrontDistributionProps?.customHeaders, cloudfrontFunction, responseHeadersPolicyProps ? new cloudfront.ResponseHeadersPolicy(scope, 'ResponseHeadersPolicy', responseHeadersPolicyProps) : undefined);
    let cfprops;
    cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return { distribution: cfDistribution, loggingBucket, requestPolicy: originRequestPolicy, cloudfrontFunction };
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
function getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps) {
    const isLoggingDisabled = cloudFrontDistributionProps?.enableLogging === false;
    const userSuppliedLogBucket = cloudFrontDistributionProps?.logBucket;
    if (userSuppliedLogBucket && cloudFrontLoggingBucketProps) {
        throw Error('Either cloudFrontDistributionProps.logBucket or cloudFrontLoggingBucketProps can be set.');
    }
    return isLoggingDisabled
        ? undefined
        : userSuppliedLogBucket ?? s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket', cloudFrontLoggingBucketProps ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), cloudFrontLoggingBucketProps) : s3_bucket_defaults_1.DefaultS3Props());
}
function getCloudfrontFunction(httpSecurityHeaders, scope) {
    return httpSecurityHeaders ? defaultCloudfrontFunction(scope) : undefined;
}
//# sourceMappingURL=data:application/json;base64,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