"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildLogGroup = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const logs = require("aws-cdk-lib/aws-logs");
const utils_1 = require("./utils");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildLogGroup(scope, logGroupId, logGroupProps) {
    let consolidatedLogGroupProps;
    // Override user provided CW LogGroup props with the DefaultLogGroupProps
    consolidatedLogGroupProps = utils_1.consolidateProps(cloudwatch_log_group_defaults_1.DefaultLogGroupProps(), logGroupProps);
    // Set the LogGroup Id
    const adjustedLogGroupId = logGroupId ? logGroupId : 'CloudWatchLogGroup';
    // Create the CW Log Group
    const logGroup = new logs.LogGroup(scope, adjustedLogGroupId, consolidatedLogGroupProps);
    // If required, suppress the Cfn Nag WARNINGS
    if (consolidatedLogGroupProps.retention === logs.RetentionDays.INFINITE) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W86',
                reason: 'Retention period for CloudWatchLogs LogGroups are set to \'Never Expire\' to preserve customer data indefinitely'
            }
        ]);
    }
    if (!consolidatedLogGroupProps.encryptionKey) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W84',
                reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
            }
        ]);
    }
    return logGroup;
}
exports.buildLogGroup = buildLogGroup;
//# sourceMappingURL=data:application/json;base64,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