"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildCognitoForSearchService = exports.setupCognitoForSearchService = exports.buildIdentityPool = exports.buildUserPoolClient = exports.buildUserPool = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const cognito = require("aws-cdk-lib/aws-cognito");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const cognito_defaults_1 = require("./cognito-defaults");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildUserPool(scope, userPoolProps) {
    let cognitoUserPoolProps;
    cognitoUserPoolProps = utils_1.consolidateProps(cognito_defaults_1.DefaultUserPoolProps, userPoolProps);
    const userPool = new cognito.UserPool(scope, 'CognitoUserPool', cognitoUserPoolProps);
    // Set the advancedSecurityMode to ENFORCED
    const cfnUserPool = userPool.node.findChild('Resource');
    cfnUserPool.userPoolAddOns = {
        advancedSecurityMode: 'ENFORCED'
    };
    // Add Cfn Nag suppress for the cognito SMS role policy
    const userPoolSmsRole = userPool.node.tryFindChild('smsRole');
    if (userPoolSmsRole) {
        utils_1.addCfnSuppressRules(userPool, [
            {
                id: 'W11',
                reason: `Allowing * resource on permissions policy since its used by Cognito to send SMS messages via sns:Publish`
            }
        ]);
    }
    return userPool;
}
exports.buildUserPool = buildUserPool;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildUserPoolClient(scope, userPool, cognitoUserPoolClientProps) {
    let userPoolClientProps;
    userPoolClientProps = utils_1.consolidateProps(cognito_defaults_1.DefaultUserPoolClientProps(userPool), cognitoUserPoolClientProps);
    return new cognito.UserPoolClient(scope, 'CognitoUserPoolClient', userPoolClientProps);
}
exports.buildUserPoolClient = buildUserPoolClient;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildIdentityPool(scope, userpool, userpoolclient, identityPoolProps) {
    let cognitoIdentityPoolProps = cognito_defaults_1.DefaultIdentityPoolProps(userpoolclient.userPoolClientId, userpool.userPoolProviderName);
    cognitoIdentityPoolProps = utils_1.consolidateProps(cognitoIdentityPoolProps, identityPoolProps);
    const idPool = new cognito.CfnIdentityPool(scope, 'CognitoIdentityPool', cognitoIdentityPoolProps);
    return idPool;
}
exports.buildIdentityPool = buildIdentityPool;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function setupCognitoForSearchService(scope, domainName, options) {
    // Create the domain for Cognito UserPool
    const userpooldomain = new cognito.CfnUserPoolDomain(scope, 'UserPoolDomain', {
        domain: domainName,
        userPoolId: options.userpool.userPoolId
    });
    userpooldomain.addDependency(options.userpool.node.findChild('Resource'));
    // Setup the IAM Role for Cognito Authorized Users
    const cognitoPrincipal = new iam.FederatedPrincipal('cognito-identity.amazonaws.com', {
        'StringEquals': { 'cognito-identity.amazonaws.com:aud': options.identitypool.ref },
        'ForAnyValue:StringLike': { 'cognito-identity.amazonaws.com:amr': 'authenticated' }
    }, 'sts:AssumeRoleWithWebIdentity');
    const cognitoAuthorizedRole = new iam.Role(scope, 'CognitoAuthorizedRole', {
        assumedBy: cognitoPrincipal,
        inlinePolicies: {
            CognitoAccessPolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'es:ESHttp*'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`]
                    })
                ]
            })
        }
    });
    // Attach the IAM Role for Cognito Authorized Users
    new cognito.CfnIdentityPoolRoleAttachment(scope, 'IdentityPoolRoleMapping', {
        identityPoolId: options.identitypool.ref,
        roles: {
            authenticated: cognitoAuthorizedRole.roleArn
        }
    });
    return cognitoAuthorizedRole;
}
exports.setupCognitoForSearchService = setupCognitoForSearchService;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildCognitoForSearchService(scope, domainName) {
    const userPool = buildUserPool(scope);
    const userPoolClient = buildUserPoolClient(scope, userPool);
    const identityPool = buildIdentityPool(scope, userPool, userPoolClient);
    const cognitoAuthorizedRole = setupCognitoForSearchService(scope, domainName, {
        userpool: userPool,
        identitypool: identityPool,
        userpoolclient: userPoolClient
    });
    return [userPool, userPoolClient, identityPool, cognitoAuthorizedRole];
}
exports.buildCognitoForSearchService = buildCognitoForSearchService;
//# sourceMappingURL=data:application/json;base64,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